﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include "../../panel/panel_PanelText.h"
#include "../../panel/panel_PanelContainer.h"
#include "../../ui/ui.h"

#include "../Scene.h"
#include "../../ThreadMessageChannel.h"
#include "../../AppletMessageThread.h"
#include "MenuPageReactor.h"


namespace scene{ namespace menu{
    class SceneMenuPageBase;

    class SceneMenuPageParameter
    {
    public:
        explicit SceneMenuPageParameter(const std::shared_ptr<ThreadMessageResultReporter>& pReporter, const std::shared_ptr<void>& pUserArgument) NN_NOEXCEPT
            : m_pReporter(pReporter)
            , m_pParentPage()
            , m_pUserArgument(pUserArgument)
        {
            ResetResponseImpl();
        }

        explicit SceneMenuPageParameter(const std::shared_ptr<SceneMenuPageBase>& pParentPage, const std::shared_ptr<void>& pUserArgument) NN_NOEXCEPT
            : m_pReporter()
            , m_pParentPage(pParentPage)
            , m_pUserArgument(pUserArgument)
        {
            ResetResponseImpl();
        }

        std::shared_ptr<ThreadMessageResultReporter> GetReporter() NN_NOEXCEPT
        {
            return m_pReporter;
        }

        std::shared_ptr<SceneMenuPageBase> GetParentPage() NN_NOEXCEPT
        {
            return m_pParentPage.lock();
        }

        template<typename T>
        std::shared_ptr<T> GetUserArgument() NN_NOEXCEPT
        {
            return std::static_pointer_cast<T>(m_pUserArgument);
        }

        void RequestCloseMenu() NN_NOEXCEPT
        {
            m_IsCloseMenuRequested = true;
        }

        void RequestSendMessage(const AppletMessageMenuResponse& message) NN_NOEXCEPT
        {
            m_RequestedMessageToSend = message;
        }

    private:
        // 呼出元が子ページの結果を取得するための関数。
        friend class SceneMenuPageBase;

        bool IsCloseMenuRequested() const NN_NOEXCEPT
        {
            return m_IsCloseMenuRequested;
        }

        nn::util::optional<AppletMessageMenuResponse> GetRequestedMessageToSend() const NN_NOEXCEPT
        {
            return m_RequestedMessageToSend;
        }

    private:
        void ResetResponseImpl() NN_NOEXCEPT
        {
            m_IsCloseMenuRequested = false;
            m_RequestedMessageToSend = nn::util::nullopt;
        }

    private:
        // メニューのレスポンスを返す先
        // 以下のパターンがある：
        //
        // +-----------+------------++-----------------------
        // | Reportert | ParentPage ||
        // +-----------+------------++-----------------------
        // | non-null  | NULL       || Message 処理によって開かれたメニューの RootPage。
        // | NULL      | NULL       || Message 処理以外の手段で開かれたメニューの RootPage。
        // | NULL      | non-null   || 子ページ
        // +-----------+------------++-----------------------
        //
        std::shared_ptr<ThreadMessageResultReporter> m_pReporter;
        std::weak_ptr<SceneMenuPageBase> m_pParentPage;

        std::shared_ptr<void> m_pUserArgument;

        bool m_IsCloseMenuRequested;
        nn::util::optional<AppletMessageMenuResponse> m_RequestedMessageToSend;
    };

    class SceneMenuPageBase
        : public Scene
        , public std::enable_shared_from_this<SceneMenuPageBase>
    {
    private:
        enum PageState
        {
            PageState_Created = 0,
            PageState_EnteringPage,
            PageState_InPage,
            PageState_OutPage,
            PageState_WaitLayer,
            PageState_Closed,
        };

    public:
        explicit SceneMenuPageBase(
            const std::shared_ptr<void>& pArg,
            ui::ItemAlignment itemAlignment = ui::ItemAlignment_TopLeft
            ) NN_NOEXCEPT;
        ~SceneMenuPageBase() NN_NOEXCEPT;

    public:
        virtual void OnDestroying() NN_NOEXCEPT NN_OVERRIDE;
        virtual SceneUpdateResult Update() NN_NOEXCEPT NN_OVERRIDE;
        virtual std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT NN_OVERRIDE;

    protected:
        template<typename T>
        std::shared_ptr<T> GetUserArgument() NN_NOEXCEPT
        {
            if(m_pParameter)
            {
                return m_pParameter->GetUserArgument<T>();
            }
            return nullptr;
        }

        void SetPanelName(const std::string& value) NN_NOEXCEPT;

        // このページのメニューのアイテムを設定します。
        // RootMenuItem の子に設定されます。
        void SetMenuItem(const std::shared_ptr<ui::IMenuItem>& value) NN_NOEXCEPT;

        // このページの Root のアイテムを取得します。
        std::shared_ptr<ui::IMenuItem> GetRootMenuItem() NN_NOEXCEPT;

        // このページに入る際に呼び出される関数
        virtual void OnEnteringPage(const std::shared_ptr<ui::IMenuPage>& pPage) NN_NOEXCEPT;

        // このページから出る際に呼び出される関数
        virtual void OnLeavingPage(const std::shared_ptr<ui::IMenuPage>& pPage) NN_NOEXCEPT;

        // このページが閉じられる際に呼び出される関数
        virtual void OnClosingPage(const std::shared_ptr<ui::IMenuPage>& pPage) NN_NOEXCEPT;

    private:
        SceneUpdateResult UpdateImpl() NN_NOEXCEPT;
        void CloseImpl() NN_NOEXCEPT;

        // ユーザー入力からリクエストを収集する。
        // - HandleTouchUp
        // - HandleTouched
        // - HandleTouchMove
        // - HandleTouchDown
        // - HandleNegativeButtonPressed
        // - HandlePositiveButtonPressed
        // - HandleUpButtonPressed
        // - HandleDownButtonPressed
        // - HandleRightButtonPressed
        // - HandleLeftButtonPressed
        // - HandleLeftAnalogStickInput
        // - HandleRightAnalogStickInput
        ui::MenuButtonResult GatherRequestFromUserInputImpl(const std::shared_ptr<MenuPageReactor>& pReactor) NN_NOEXCEPT;

        // 子ページの戻り値からリクエストを収集する。
        void GatherRequestFromChildPageResultImpl(const std::shared_ptr<MenuPageReactor>& pReactor) NN_NOEXCEPT;

        // ページ内の要求を処理する。
        // * IMenuPage::LoadValue
        //      - LoadValue
        // * IMenuPage::StoreValue
        //      - StoreValue
        // * IMenuPage::ChangeEnablity
        //      - SetEnablity
        // * IMenuPage::ChangeFocus
        //      - OnOutFocus
        //      - OnInFocus
        //      - NotifyFocusedItemChanged
        // * IMenuPage::UpdateLayout
        //      - UpdateLayoutRecursively
        // * IMenuPage::UpdatePanel
        //      - UpdatePanel
        //
        // (!!) LoadValue は UI の状態を変更するため HID 入力の処理と同時に行えない。
        //      基本的にページに入る際に読み込むだけのはずなので EnteringPage の際に実行する。
        void ProcessPageRequestImpl(const std::shared_ptr<MenuPageReactor>& pReactor, bool isLoadValueEnabled) NN_NOEXCEPT;

        // ページ遷移の要求を処理する。そのまま return すること。
        // * IMenuPage::SendMessage
        // * IMenuPage::SleepSystem
        // * IMenuPage::ShutdownSystem
        // * IMenuPage::CloseMenu
        // * IMenuPage::ClosePage
        // * IMenuPage::OpenPage
        // * IMenuPage::PushMenuLayer
        // * IMenuPage::PopMenuLayer
        SceneUpdateResult ProcessTransitionRequestImpl(const std::shared_ptr<MenuPageReactor>& pReactor, const std::pair<PageState, SceneUpdateResult>& defaultValue) NN_NOEXCEPT;


    private:
        void CancelUserInputTouchImpl() NN_NOEXCEPT;

    public:
        struct TouchState
        {
            // このページが Down を受け取ったか
            bool m_IsTouching = false;
            std::weak_ptr<ui::IMenuItem> m_pHeldItem = {};
            nn::util::Vector3f m_Position = {};
        };

    private:

        // このページに渡されたパラメータ。
        std::shared_ptr<SceneMenuPageParameter> m_pParameter;
        std::shared_ptr<ui::MenuItemBox> m_pMenuItemRoot;

        // ページのアクティブ状態
        struct
        {
            PageState m_State = PageState_Created;

            // 子ページに渡したパラメータ。子ページを開いている場合非 nullptr。
            std::shared_ptr<SceneMenuPageParameter> m_pChildParameter;
        } m_PageActivity;

        // フォーカス状態
        std::weak_ptr<ui::IMenuItem> m_pFocusedItem;

        // タッチ状態
        TouchState m_TouchState;
    };

}}
