﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "MenuPageList.h"
#include "../debug/debug_Activity.h"
#include "../../ui/ui.h"
#include "../../config/DebugMenuStyle.h"
#include "../../Config.h"

namespace scene{ namespace menu{

    namespace detail
    {
        class DebugModuleConfigHelper
        {
        public:
            typedef config::DebugMenuStyle StyleType;
            typedef ui::MenuItemColumn TableType;
        public:
            static void AddModuleHeader(
                const std::shared_ptr<TableType>& pTable,
                const std::shared_ptr<debug::ActivityCommon>& pCommon,
                const std::shared_ptr<std::vector<std::weak_ptr<ui::IMenuItem>>>& pOptionList,
                const std::shared_ptr<debug::LayoutEditorController>& pController
            ) NN_NOEXCEPT
            {
                const StyleType style;
                pTable->AddCell(
                    style.MakeOptionModuleNameCellStyle(),
                    std::make_shared<ui::MenuItemCheckBox>(style.MakeOptionModuleNameCheckBoxStyle(), pCommon->name.value)
                    ->SetValueAccessor(
                        [pCommon](){ return pCommon->isEnabled != 0; },
                        [pCommon](bool value){ pCommon->isEnabled = value ? 1 : 0; }
                    )
                    ->SetValueChangedCallbackFunction([pOptionList](bool v, auto p){
                        p->StoreValue();
                        p->UpdateLayout();

                        for(auto& e : *pOptionList)
                        {
                            // Enablity の設定
                            if(auto pItem = e.lock())
                            {
                                p->SetEnablity(pItem, v);
                            }
                        }
                    })
                );
                pTable->AddCell(style.MakeOptionSpacerCellStyle(), nullptr);

                // Layout ボタン
                if(pController)
                {
                    typedef debug::LayoutEditorController::Anchor Anchor;
                    auto pLabel = std::make_shared<ui::MenuItemLabel>(style.MakeOptionKeyLabelStyle(), "Position");
                    auto pDropDown = std::make_shared<ui::MenuItemDropDownList>(style.MakeOptionValueDropDownListStyle())
                        ->AddDropDownItem("TopLeft", {}, {},
                            [pController](bool v, auto p){
                                if(v)
                                {
                                    pController->SetPositionAnchor(Anchor::Anchor_TopLeft);
                                    pController->RefreshPositionToAnchor();
                                    p->UpdatePanel();
                                }
                            }
                        )
                        ->AddDropDownItem("TopRight", {}, {},
                            [pController](bool v, auto p){
                                if(v)
                                {
                                    pController->SetPositionAnchor(Anchor::Anchor_TopRight);
                                    pController->RefreshPositionToAnchor();
                                    p->UpdatePanel();
                                }
                            }
                        )
                        ->AddDropDownItem("BottomLeft", {}, {},
                            [pController](bool v, auto p){
                                if(v)
                                {
                                    pController->SetPositionAnchor(Anchor::Anchor_BottomLeft);
                                    pController->RefreshPositionToAnchor();
                                    p->UpdatePanel();
                                }
                            }
                        )
                        ->AddDropDownItem("BottomRight", {}, {},
                            [pController](bool v, auto p){
                                if(v)
                                {
                                    pController->SetPositionAnchor(Anchor::Anchor_BottomRight);
                                    pController->RefreshPositionToAnchor();
                                    p->UpdatePanel();
                                }
                            }
                        )
                    ;

                    pTable->AddCell(std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
                        ->AddCell(style.MakeOptionKeyCellStyle(),
                            pLabel
                        )
                        ->AddCell(style.MakeOptionValueCellStyle(),
                            std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                            ->SetInnerItem(pDropDown)
                        )
                    );
                    pOptionList->push_back(pDropDown);

                    pTable->AddCell(style.MakeOptionSpacerCellStyle(), nullptr);
                }
            }

        private:
            template<typename PointerType, typename Entry>
            static void AddOptionValueDropDownEntryImpl(
                const std::shared_ptr<ui::MenuItemDropDownList>& pDropDown,
                const std::shared_ptr<debug::LayoutEditorController>& pController,
                PointerType pValue,
                const Entry& entry
            ) NN_NOEXCEPT
            {
                auto v = entry.second;
                pDropDown->AddDropDownItem(
                    entry.first,
                    [pValue, v](){ return *pValue == v; },
                    [pValue, v, pController](bool value){ if(value){ *pValue = v; pController->RefreshPositionToAnchor(); } },
                    [](bool, auto p){ p->StoreValue(); p->UpdatePanel(); }
                );
            }

            template<typename PointerType, typename Entry, typename... Tail>
            static void AddOptionValueDropDownEntryImpl(
                const std::shared_ptr<ui::MenuItemDropDownList>& pDropDown,
                const std::shared_ptr<debug::LayoutEditorController>& pController,
                PointerType pValue,
                const Entry& head,
                Tail... tail
            ) NN_NOEXCEPT
            {
                AddOptionValueDropDownEntryImpl(pDropDown, pController, pValue, head);
                AddOptionValueDropDownEntryImpl(pDropDown, pController, pValue, tail...);
            }

        public:
            template<typename PointerType, typename... NameAndValue>
            static void AddOptionValueDropDownList(
                const std::shared_ptr<TableType>& pTable,
                const std::shared_ptr<std::vector<std::weak_ptr<ui::IMenuItem>>>& pOptionList,
                const std::shared_ptr<debug::LayoutEditorController>& pController,
                const std::string& name,
                PointerType pValue,
                NameAndValue... list
            ) NN_NOEXCEPT
            {
                const StyleType style;

                auto pLabel = std::make_shared<ui::MenuItemLabel>(style.MakeOptionKeyLabelStyle(), name);
                auto pDropDown = std::make_shared<ui::MenuItemDropDownList>(style.MakeOptionValueDropDownListStyle());
                AddOptionValueDropDownEntryImpl(pDropDown, pController, pValue, list...);

                pTable->AddCell(std::make_shared<ui::MenuItemRow>(ui::MenuItemTableStyle())
                    ->AddCell(style.MakeOptionKeyCellStyle(),
                        pLabel
                    )
                    ->AddCell(style.MakeOptionValueCellStyle(),
                        std::make_shared<ui::MenuItemBox>(style.MakeOptionValueBorderStyle())
                        ->SetInnerItem(pDropDown)
                    )
                );
                pOptionList->push_back(pLabel);
                pOptionList->push_back(pDropDown);

                pTable->AddCell(style.MakeOptionSpacerCellStyle(), nullptr);
            }


        };
    }

    // DebugMenu の DebugModule の設定項目を定義するためのクラス。
    class DebugModuleConfigBuilder
    {
    public:
        DebugModuleConfigBuilder(
            const std::shared_ptr<ui::MenuItemColumn>& pTable,
            const std::shared_ptr<debug::ActivityCommon>& pCommon,
            const std::shared_ptr<debug::LayoutEditor>& pLayoutEditor
        ) NN_NOEXCEPT
        {
            m_pOptionList = std::make_shared<std::vector<std::weak_ptr<ui::IMenuItem>>>();
            m_pTable = pTable;
            m_pController = nullptr;

            if(pLayoutEditor)
            {
                debug::LayoutEditor::ModuleHandle handle = pLayoutEditor->FindModule(pCommon->name.value);
                if(handle.IsValid())
                {
                    m_pController = std::make_shared<debug::LayoutEditorController>(pLayoutEditor, handle);
                }
            }


            detail::DebugModuleConfigHelper::AddModuleHeader(pTable, pCommon, m_pOptionList, m_pController);
        }

        template<typename PointerType, typename... NameAndValue>
        DebugModuleConfigBuilder& AddOptionDropDownList(
            const std::string& name,
            PointerType pValue,
            NameAndValue... list
        ) NN_NOEXCEPT
        {
            detail::DebugModuleConfigHelper::AddOptionValueDropDownList(m_pTable, m_pOptionList, m_pController, name, pValue, list...);
            return *this;
        }


    private:
        std::shared_ptr<ui::MenuItemColumn> m_pTable;
        std::shared_ptr<std::vector<std::weak_ptr<ui::IMenuItem>>> m_pOptionList;
        std::shared_ptr<debug::LayoutEditorController> m_pController;
    };

}}
