﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_StringUtil.h>
#include <nn/nn_SdkAssert.h>
#include "Utf8StreamUtility.h"

namespace scene { namespace debug { namespace logutil {

Utf8StreamUtility::Utf8StreamUtility() NN_NOEXCEPT
{
    Clear();
}

void Utf8StreamUtility::Clear() NN_NOEXCEPT
{
    m_Count = 0;
    m_Bytes = 0;
    m_ExpectedBytes = 0;
}

bool Utf8StreamUtility::Push(char c) NN_NOEXCEPT
{
    nn::Bit8 uc = static_cast<nn::Bit8>(c);

    if (uc == 0)
    {
        return End();
    }

    if (m_Bytes == 0)
    {
        if ((uc & 0x80) == 0) // 00..7F は 1 バイト文字
        {
            m_Count++;
            return true;
        }
        else if ((uc & 0xE0) == 0xC0) // C2..DF は 2 バイト文字
        {
            m_ExpectedBytes = 2;
        }
        else if ((uc & 0xF0) == 0xE0) // E0..EF は 3 バイト文字
        {
            m_ExpectedBytes = 3;
        }
        else if ((uc & 0xF8) == 0xF0) // F0..F7 は 4 バイト文字
        {
            m_ExpectedBytes = 4;
        }
        else
        {
            // 不正なシーケンス、もしくは、5 バイト以上である。
            return false;
        }
    }
    else if ((uc & 0xC0) != 0x80) // [80-BF]
    {
        // 2 バイト目以降は (10xxxxxx) でなければならない。
        return false;
    }

    m_Code[m_Bytes++] = uc;

    if (m_Bytes == m_ExpectedBytes)
    {
        // 110yyyyx 10xxxxxx -> y = 0x1E 0x00
        if (m_ExpectedBytes == 2 && (m_Code[0] & 0x1E) == 0) // C0-C1
        {
            // 2 バイト文字 (11 ビット長) なのに 7 ビットに収まるため、冗長なコードと判定する。
            return false;
        }
        // 1110yyyy 10yxxxxx 10xxxxxx -> y = 0x0F 0x20 0x00
        if (m_ExpectedBytes == 3 && (m_Code[0] & 0x0F) == 0 && (m_Code[1] & 0x20) == 0) // E0-8X, E0-9X
        {
            // 3 バイト文字 (16 ビット長) なのに 11 ビットに収まるため、冗長なコードと判定する。
            return false;
        }
        // 11110yyy 10yyxxxx 10xxxxxx 10xxxxxx -> y = 0x07 0x30 0x00 0x00
        if (m_ExpectedBytes == 4 && (m_Code[0] & 0x07) == 0 && (m_Code[1] & 0x30) == 0) // F0..F7-8X
        {
            // 4 バイト文字 (21 ビット長) なのに 16 ビットに収まるため、冗長なコードと判定する。
            return false;
        }
        m_Bytes = 0;
        m_Count++;
    }

    return true;
}

bool Utf8StreamUtility::End() NN_NOEXCEPT
{
    return (m_Bytes == 0);
}

int32_t Utf8StreamUtility::GetCount() NN_NOEXCEPT
{
    return m_Count;
}

bool Utf8StreamUtility::Verify(const char* string) NN_NOEXCEPT
{
    return Verify(nullptr, string);
}

bool Utf8StreamUtility::Verify(int* outCount, const char* string) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(string);

    Utf8StreamUtility stream;

    while (*string)
    {
        if (!stream.Push(*string++))
        {
            return false;
        }
    }
    if (!stream.End())
    {
        return false;
    }
    if (outCount)
    {
        *outCount = stream.GetCount();
    }

    return true;
}

bool Utf8StreamUtility::Verify(const char* string, size_t length) NN_NOEXCEPT
{
    return Verify(nullptr, string, length);
}

bool Utf8StreamUtility::Verify(int* outCount, const char* string, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(string);

    Utf8StreamUtility stream;

    while (*string && length > 0)
    {
        if (!stream.Push(*string++))
        {
            return false;
        }
        length--;
    }
    if (!stream.End())
    {
        return false;
    }
    if (outCount)
    {
        *outCount = stream.GetCount();
    }

    return true;
}

int Utf8StreamUtility::Substring(std::string* output, size_t outputSize, const char* input, int maxCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES(outputSize > 0);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES(maxCount > 0);

    int prevCount = 0;
    int copyLength = 0;

    const char* p = input;

    Utf8StreamUtility stream;

    while (*p)
    {
        if (!stream.Push(*p++))
        {
            return -1;
        }

        int length = static_cast<int>(p - input);

        if (stream.GetCount() != prevCount)
        {
            copyLength = length;
            prevCount = stream.GetCount();
        }
        if (stream.GetCount() == maxCount)
        {
            break;
        }
        if (static_cast<size_t>(length) == outputSize - 1)
        {
            break;
        }
    }

    *output = std::string(input, copyLength + 1);

    return copyLength;
}

int Utf8StreamUtility::GetBytesSizeOfWidth(const char* input, int maxCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES(maxCount > 0);

    int prevCount = 0;
    int copyLength = 0;

    const char* p = input;

    Utf8StreamUtility stream;

    while (*p)
    {
        if (!stream.Push(*p++))
        {
            return -1;
        }

        int length = static_cast<int>(p - input);

        if (stream.GetCount() != prevCount)
        {
            copyLength = length;
            prevCount = stream.GetCount();
        }
        if (stream.GetCount() == maxCount)
        {
            break;
        }
    }

    return copyLength;
}

int Utf8StreamUtility::Slice(std::string* output, size_t outputSize, const char** input, int maxCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(*input);
    int copyLength = Substring(output, outputSize, *input, maxCount);
    if (copyLength > 0) {
        *input += copyLength;
    }
    return copyLength;
}


int Utf8StreamUtility::SliceSubstring(std::string* output, size_t outputSize, const char* input, const std::string separator, int maxCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES(outputSize > 0);
    NN_SDK_REQUIRES(maxCount > 0);

    int prevCount = 0;
    int copyLength = 0;

    const char* p = input;

    Utf8StreamUtility stream;

    // *p != '\0'
    while (*p)
    {
        if (!stream.Push(*p++))
        {
            return -1;
        }

        int length = static_cast<int>(p - input);

        if (stream.GetCount() != prevCount)
        {
            copyLength = length;
            prevCount = stream.GetCount();
        }
        if (stream.GetCount() == maxCount)
        {
            break;
        }
        if (std::strncmp(separator.c_str(), p - (separator.size() - 1) , separator.size() ) == 0)
        {
            break;
        }
        if (static_cast<size_t>(length) == outputSize - 1)
        {
            break;
        }
    }

    *output = std::string(input, copyLength + 1);
    return copyLength;
}

}}}
