﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "debug_ReadWriteActivity.h"

#include <string>
#include <vector>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs.h>
#include "../../Config.h"

#include <nn/fs/fs_SdCardForDebug.h>

#define NN_DEVOVL_LOG_DEV_WRITEACTIVITY(...) //NN_DEVOVL_LOG("[dev]" __VA_ARGS__)
#define NN_DEVOVL_LOG_DEV_READACTIVITY(...) //NN_DEVOVL_LOG("[dev]" __VA_ARGS__)

// FS
namespace {

#define NN_DEVOVL_SD_MOUNTNAME "SD"
#define NN_DEVOVL_SETTING_FILENAME "DevOverlayDisp.setting.txt"
#define NN_DEVOVL_SETTING_FILEPATH NN_DEVOVL_SD_MOUNTNAME ":/" NN_DEVOVL_SETTING_FILENAME

    bool MountSdCard() NN_NOEXCEPT
    {
        auto result = nn::fs::MountSdCardForDebug(NN_DEVOVL_SD_MOUNTNAME);
        if(result.IsFailure())
        {
            NN_DEVOVL_LOG_ERR("Failed to mount SD card (%d-%d)\n", result.GetModule(), result.GetDescription());
            return false;
        }
        return true;
    }

    void UnmountSdCard() NN_NOEXCEPT
    {
        nn::fs::Unmount(NN_DEVOVL_SD_MOUNTNAME);
    }

    bool OpenFileForSave(nn::fs::FileHandle* pOutFileHandle, int64_t fileSize) NN_NOEXCEPT
    {
        auto createResult = nn::fs::CreateFile(NN_DEVOVL_SETTING_FILEPATH, fileSize);
        if(createResult.IsSuccess())
        {
            NN_DEVOVL_LOG_DEV_WRITEACTIVITY("creating file succeeded\n");

            nn::fs::FileHandle h = {};
            auto openResult = nn::fs::OpenFile(&h, NN_DEVOVL_SETTING_FILEPATH, nn::fs::OpenMode::OpenMode_Write | nn::fs::OpenMode_AllowAppend);

            if(openResult.IsSuccess())
            {
                NN_DEVOVL_LOG_DEV_WRITEACTIVITY("opening file succeeded\n");
                *pOutFileHandle = h;
                return true;
            }
            else
            {
                NN_DEVOVL_LOG_DEV_WRITEACTIVITY("opening file failed(%d-%d)\n", openResult.GetModule(), openResult.GetDescription());
                return false;
            }
        }
        else if(nn::fs::ResultPathAlreadyExists::Includes(createResult))
        {
            NN_DEVOVL_LOG_DEV_WRITEACTIVITY("file already exists\n");

            nn::fs::FileHandle h = {};
            auto openResult = nn::fs::OpenFile(&h, NN_DEVOVL_SETTING_FILEPATH, nn::fs::OpenMode::OpenMode_Write | nn::fs::OpenMode_AllowAppend);

            if(openResult.IsSuccess())
            {
                NN_DEVOVL_LOG_DEV_WRITEACTIVITY("opening file succeeded\n");
                auto resizeResult = nn::fs::SetFileSize(h, fileSize);

                if(resizeResult.IsSuccess())
                {
                    NN_DEVOVL_LOG_DEV_WRITEACTIVITY("resizing file succeeded\n");
                    *pOutFileHandle = h;
                    return true;
                }
                else
                {
                    NN_DEVOVL_LOG_DEV_WRITEACTIVITY("resizing file failed\n");
                    nn::fs::CloseFile(h);
                    return false;
                }
            }
            else
            {
                NN_DEVOVL_LOG_DEV_WRITEACTIVITY("opening file failed(%d-%d)\n", openResult.GetModule(), openResult.GetDescription());
                return false;
            }

        }
        else
        {
            NN_DEVOVL_LOG_DEV_WRITEACTIVITY("creating file failed(%d-%d)\n", createResult.GetModule(), createResult.GetDescription());
            return false;
        }
    }

    bool WriteToFile(nn::fs::FileHandle h, const char* data, size_t size) NN_NOEXCEPT
    {
        auto writeResult = nn::fs::WriteFile(h, 0, data, size, nn::fs::WriteOption::MakeValue(0));
        return writeResult.IsSuccess();
    }

    bool ReadWholeFile(const char** pOutEnd, char* buf, size_t size) NN_NOEXCEPT
    {
        nn::fs::FileHandle h = {};
        auto openResult = nn::fs::OpenFile(&h, NN_DEVOVL_SETTING_FILEPATH, nn::fs::OpenMode_Read);
        if(openResult.IsFailure())
        {
            return false;
        }
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(h); };

        int64_t fileSize = 0;
        auto fileSizeResult = nn::fs::GetFileSize(&fileSize, h);
        if(fileSizeResult.IsFailure())
        {
            return false;
        }

        if(fileSize > size)
        {
            return false;
        }

        auto readResult = nn::fs::ReadFile(h, 0, buf, fileSize);
        if(readResult.IsFailure())
        {
            return false;
        }

        *pOutEnd = buf + fileSize;
        return true;
    }

}


// Serialization
namespace scene{ namespace debug{

    namespace {

#define NN_DEVOVL_BEGIN_SERIALIZATION() \
    auto& common = src.common;          \
    char* cur = buf;

#define NN_DEVOVL_END_SERIALIZATION()   \
    *pOutNext = cur;                    \
    return true;

#define NN_DEVOVL_FORMATVALUE_I(format, ...)  \
    do{                     \
        int len = nn::util::SNPrintf(cur, pEnd - cur, format, __VA_ARGS__);    \
        if(len >= pEnd - cur)   \
        {                   \
            return false;   \
        }                   \
        NN_SDK_LOG("%s", cur);    \
        cur += len;         \
    }                       \
    while(NN_STATIC_CONDITION(false));

#define NN_DEVOLV_SERIALIZE_COMMON()    \
    if(!SerializeActivityCommon(&cur, cur, pEnd, src.common, src.GetWidth(), src.GetHeight())) \
    {                   \
        return false;   \
    }

#define NN_DEVOVL_SERIALIZE_INTVALUE(key, v)   \
    NN_DEVOVL_FORMATVALUE_I("%s." key "=%d\n", common.name.value, v)

        bool SerializeActivityCommon(char** pOutNext, char* buf, char* pEnd, const ActivityCommon& src, int width, int height) NN_NOEXCEPT
        {
            char* cur = buf;
            // 設定ファイルを書き換えれば位置を自由に変えられる。
            // 「width と height を書いておくので適当にやってね」という趣旨。
            NN_DEVOVL_FORMATVALUE_I("// module '%s': width=%d, height=%d\n", src.name.value, width, height);
            NN_DEVOVL_FORMATVALUE_I("%s.isEnabled=%s\n", src.name.value, (src.isEnabled != 0) ? "1" : "0");
            NN_DEVOVL_FORMATVALUE_I("%s.positionX=%d\n", src.name.value, src.positionX);
            NN_DEVOVL_FORMATVALUE_I("%s.positionY=%d\n", src.name.value, src.positionY);
            *pOutNext = cur;
            return true;
        }


        bool SerializeActivityClock(char** pOutNext, char* buf, char* pEnd, const ActivityClock& src) NN_NOEXCEPT
        {
            NN_DEVOVL_BEGIN_SERIALIZATION();
            NN_DEVOLV_SERIALIZE_COMMON();
            NN_DEVOVL_SERIALIZE_INTVALUE("clockType", src.clockType);
            NN_DEVOVL_END_SERIALIZATION();
        }

        bool SerializeActivityLogViewer(char** pOutNext, char* buf, char* pEnd, const ActivityLogViewer& src) NN_NOEXCEPT
        {
            NN_DEVOVL_BEGIN_SERIALIZATION();
            NN_DEVOLV_SERIALIZE_COMMON();
            NN_DEVOVL_SERIALIZE_INTVALUE("width", src.width);
            NN_DEVOVL_SERIALIZE_INTVALUE("height", src.height);
            NN_DEVOVL_SERIALIZE_INTVALUE("isBackground", src.isBackground);
            NN_DEVOVL_SERIALIZE_INTVALUE("fontSize", static_cast<int>(src.fontSize));
            NN_DEVOVL_END_SERIALIZATION();
        }

        bool SerializeActivityHidInputViewer(char** pOutNext, char* buf, char* pEnd, const ActivityHidInputViewer& src) NN_NOEXCEPT
        {
            NN_DEVOVL_BEGIN_SERIALIZATION();
            NN_DEVOLV_SERIALIZE_COMMON();
            NN_DEVOVL_SERIALIZE_INTVALUE("scale", src.scale);
            NN_DEVOVL_SERIALIZE_INTVALUE("slotCount", src.slotCount);
            NN_DEVOVL_END_SERIALIZATION();
        }

        bool SerializeActivityPowerConsumptionMeter(char** pOutNext, char* buf, char* pEnd, const ActivityPowerConsumptionMeter& src) NN_NOEXCEPT
        {
            NN_DEVOVL_BEGIN_SERIALIZATION();
            NN_DEVOLV_SERIALIZE_COMMON();
            NN_DEVOVL_SERIALIZE_INTVALUE("width", src.width);
            NN_DEVOVL_SERIALIZE_INTVALUE("height", src.height);
            NN_DEVOVL_END_SERIALIZATION();
        }

        bool SerializeActivitySystemSharedTextureViewer(char** pOutNext, char* buf, char* pEnd, const ActivitySystemSharedTextureViewer& src) NN_NOEXCEPT
        {
            NN_DEVOVL_BEGIN_SERIALIZATION();
            NN_DEVOLV_SERIALIZE_COMMON();
            NN_DEVOVL_SERIALIZE_INTVALUE("scale", src.scale);
            NN_DEVOVL_END_SERIALIZATION();
        }

        bool SerializeActivity(char** pOutNext, char* buf, char* pEnd, const Activity& src) NN_NOEXCEPT
        {
            NN_DEVOVL_LOG_DEV_WRITEACTIVITY("serializing activity...\n");
            char* cur = buf;


        #define NN_DEVOVL_SERIALIZE_MODULEACTIVITY(module, data)  \
            if(NN_STATIC_CONDITION(Activity##module::IsSupported))  \
            {                                       \
                NN_DEVOVL_LOG_DEV_WRITEACTIVITY("serializing activity" #module "...\n");  \
                if(!SerializeActivity##module(&cur, cur, pEnd, src.activity##module))   \
                {                                   \
                    return false;                   \
                }                                   \
            }

            NN_DEVOVL_FOREACH_DEBUGMODULE(NN_DEVOVL_SERIALIZE_MODULEACTIVITY, _);

        #undef NN_DEVOVL_SERIALIZE_MODULEACTIVITY

            NN_DEVOVL_LOG_DEV_WRITEACTIVITY("serializing activity...success\n");
            *pOutNext = cur;
            return true;
        }

    }

    static const size_t WorkBufferSize = 8 * 1024;

    bool ReadWriteActivity::WriteToSdCard(const Activity& value) NN_NOEXCEPT
    {
        // 一時的にマウント
        if(!MountSdCard())
        {
            return false;
        }
        NN_UTIL_SCOPE_EXIT{ UnmountSdCard(); };

        // Serialize
        std::vector<char> buf(WorkBufferSize);
        char* pEnd = nullptr;
        if(!SerializeActivity(&pEnd, buf.data(), buf.data() + buf.size(), value))
        {
            return false;
        }

        auto size = pEnd - buf.data();
        NN_DEVOVL_LOG_DEV_WRITEACTIVITY("serialized size=%llu\n", size);
        NN_ABORT_UNLESS_LESS(size, buf.size());

        nn::fs::FileHandle h = {};
        if(!OpenFileForSave(&h, size))
        {
            return false;
        }
        NN_UTIL_SCOPE_EXIT{ nn::fs::FlushFile(h); nn::fs::CloseFile(h); };

        if(!WriteToFile(h, buf.data(), size))
        {
            return false;
        }

        return true;
    }

    namespace {
        typedef std::pair<const char*, const char*> RangeType;

        RangeType GetLine(const char** pOutNext, const char* p, const char* pEnd) NN_NOEXCEPT
        {
            const char* q = p;
            int n = 0;
            for(; q < pEnd; q++)
            {
                char c = *q;
                if(c == '\r')
                {
                    if(q + 1 < pEnd && *(q + 1) == '\n')
                    {
                        q++;
                    }
                    break;
                }
                else if(c == '\n')
                {
                    break;
                }
                else
                {
                    n++;
                }
            }

            *pOutNext = std::min(q + 1, pEnd);
            return std::make_pair(p, p + n);
        }

        RangeType SkipWhiteSpace(const RangeType& range) NN_NOEXCEPT
        {
            auto p = range.first;
            while(p < range.second)
            {
                char c = *p;
                if(c == ' ' || c == '\t')
                {
                    continue;
                }
                break;
            }
            return std::make_pair(p, range.second);
        }

        RangeType RemoveComment(const RangeType& range) NN_NOEXCEPT
        {
            ptrdiff_t n = range.second - range.first;
            if(n <= 0)
            {
                return range;
            }

            if(n >= 1 && *range.first == '#')
            {
                return std::make_pair(range.second, range.second);
            }

            if(n >= 2 && *range.first == '/' && *(range.first + 1) == '/')
            {
                return std::make_pair(range.second, range.second);
            }

            return range;
        }

        RangeType SplitIdentifier(RangeType* pOutIdentifier, const RangeType& range) NN_NOEXCEPT
        {
            auto p = range.first;
            for(; p < range.second; p++)
            {
                char c = *p;
                if( (c >= 'A' && c <= 'Z') ||
                    (c >= 'a' && c <= 'z') ||
                    (p != range.first && (c >= '0' && c <= '9'))
                    )
                {
                    continue;
                }
                break;
            }

            *pOutIdentifier = std::make_pair(range.first, p);
            return std::make_pair(p, range.second);
        }

        RangeType SplitChar(RangeType* pOutValue, const RangeType& range, char c) NN_NOEXCEPT
        {
            auto p = range.first;
            if(p < range.second && *p == c)
            {
                p += 1;
            }

            *pOutValue= std::make_pair(range.first, p);
            return std::make_pair(p, range.second);
        }

        // 雑にパースする。オーバーフローしてもあまり気にしない。
        typedef int64_t IntegerValueType;
        RangeType ParseInteger(int64_t* pOutValue, const RangeType& range) NN_NOEXCEPT
        {
            bool isNegative = false;

            auto r = SkipWhiteSpace(range);

            // 符号を読む
            RangeType sign = {};
            r = SplitChar(&sign, r, '-');
            if(sign.first < sign.second)
            {
                isNegative = true;
            }
            else
            {
                r = SplitChar(&sign, r, '+');
            }

            r = SkipWhiteSpace(r);

            // 数値
            int64_t v = 0;
            for(; r.first < r.second; r.first++)
            {
                char c = *r.first;
                if(c >= '0' && c <= '9')
                {
                    v *= 10;
                    v += c - '0';
                }
                else
                {
                    break;
                }
            }

            if(isNegative)
            {
                v *= -1;
            }

            *pOutValue = v;
            return r;
        }

        bool CheckAreStringSame(const RangeType& range0, const RangeType& range1) NN_NOEXCEPT
        {
            auto p = range0.first;
            auto q = range1.first;
            for(; (p < range0.second && *p != '\0') && (q < range1.second && *q != '\0'); p++, q++)
            {
                if(*p != *q)
                {
                    return false;
                }
            }

            return (p == range0.second || *p == '\0') && (q == range1.second || *q == '\0');
        }

        #define NN_DEVOVL_PARSEVALUE_INTEGER(name, minValue, maxValue)  \
            {                                                   \
                char key[] = #name;                             \
                if(CheckAreStringSame(keyName, std::make_pair(key, key + sizeof(key)))) \
                {                                               \
                    IntegerValueType v;                         \
                    ParseInteger(&v, value);                    \
                    v = std::min<IntegerValueType>(v, (maxValue));  \
                    v = std::max<IntegerValueType>(v, (minValue));  \
                    act.name = static_cast<decltype(act.name)>(v);  \
                    return true;                                \
                }                                               \
            }

        bool ReadValueForCommon(ActivityCommon& act, const RangeType& keyName, const RangeType& value) NN_NOEXCEPT
        {
            NN_DEVOVL_PARSEVALUE_INTEGER(isEnabled, 0, 1);
            NN_DEVOVL_PARSEVALUE_INTEGER(positionX, -9999, 9999);
            NN_DEVOVL_PARSEVALUE_INTEGER(positionY, -9999, 9999);
            return false;
        }

        bool ReadValueForClock(ActivityClock& act, const RangeType& keyName, const RangeType& value) NN_NOEXCEPT
        {
            if(ReadValueForCommon(act.common, keyName, value))
            {
                return true;
            }
            NN_DEVOVL_PARSEVALUE_INTEGER(clockType, 0, 99);
            return false;
        }

        bool ReadValueForLogViewer(ActivityLogViewer& act, const RangeType& keyName, const RangeType& value) NN_NOEXCEPT
        {
            if(ReadValueForCommon(act.common, keyName, value))
            {
                return true;
            }
            NN_DEVOVL_PARSEVALUE_INTEGER(width, 0, 9999);
            NN_DEVOVL_PARSEVALUE_INTEGER(height, 0, 9999);
            NN_DEVOVL_PARSEVALUE_INTEGER(isBackground, 0, 1);
            NN_DEVOVL_PARSEVALUE_INTEGER(fontSize, 0, 999);
            return false;
        }

        bool ReadValueForHidInputViewer(ActivityHidInputViewer& act, const RangeType& keyName, const RangeType& value) NN_NOEXCEPT
        {
            if(ReadValueForCommon(act.common, keyName, value))
            {
                return true;
            }
            NN_DEVOVL_PARSEVALUE_INTEGER(scale, 1, 99);
            NN_DEVOVL_PARSEVALUE_INTEGER(slotCount, 1, 5); // あんまり多いとコマンドメモリが足りない気がする
            return false;
        }

        bool ReadValueForPowerConsumptionMeter(ActivityPowerConsumptionMeter& act, const RangeType& keyName, const RangeType& value) NN_NOEXCEPT
        {
            if(ReadValueForCommon(act.common, keyName, value))
            {
                return true;
            }
            NN_DEVOVL_PARSEVALUE_INTEGER(width, 0, 9999);
            NN_DEVOVL_PARSEVALUE_INTEGER(height, 0, 9999);
            return false;
        }

        bool ReadValueForSystemSharedTextureViewer(ActivitySystemSharedTextureViewer& act, const RangeType& keyName, const RangeType& value) NN_NOEXCEPT
        {
            if(ReadValueForCommon(act.common, keyName, value))
            {
                return true;
            }
            NN_DEVOVL_PARSEVALUE_INTEGER(scale, 0, 8);
            return false;
        }

        bool ReadValue(Activity& act, const RangeType& moduleName, const RangeType& keyName, const RangeType& value) NN_NOEXCEPT
        {
        #define NN_DEVOVL_DEBUGMODULE_READVALUE(module, data)   \
            if(Activity##module::IsSupported && CheckAreStringSame(moduleName, std::make_pair(act.activity##module.common.name.value, &act.activity##module.common.name.value[0] + sizeof(act.activity##module.common.name))))   \
            {                                                   \
                return ReadValueFor##module(act.activity##module, keyName, value);  \
            }

            NN_DEVOVL_FOREACH_DEBUGMODULE(NN_DEVOVL_DEBUGMODULE_READVALUE, _);

        #undef NN_DEVOVL_DEBUGMODULE_READVALUE
            return false;
        }

        bool ParseLine(Activity& act, const RangeType& range) NN_NOEXCEPT
        {
            auto lineRange = range;

            // 空白行やコメント行は無視
            lineRange = SkipWhiteSpace(lineRange);
            lineRange = RemoveComment(lineRange);
            if(lineRange.first >= lineRange.second)
            {
                NN_DEVOVL_LOG_DEV_READACTIVITY("skiping comment line\n");
                return false;
            }

            // <ModuleName>.<KeyName>=<Value>
            // 間の空白は無視。
            RangeType moduleName = {};
            lineRange = SkipWhiteSpace(lineRange);
            lineRange = SplitIdentifier(&moduleName, lineRange);
            if(moduleName.first >= moduleName.second)
            {
                NN_DEVOVL_LOG_DEV_READACTIVITY("skiping invalid line(no module-name)\n");
                return false;
            }

            RangeType separator = {};
            lineRange = SkipWhiteSpace(lineRange);
            lineRange = SplitChar(&separator, lineRange, '.');
            if(separator.first >= separator.second)
            {
                NN_DEVOVL_LOG_DEV_READACTIVITY("skiping invalid line(missing '.')\n");
                return false;
            }

            RangeType keyName = {};
            lineRange = SkipWhiteSpace(lineRange);
            lineRange = SplitIdentifier(&keyName, lineRange);
            if(keyName.first >= keyName.second)
            {
                NN_DEVOVL_LOG_DEV_READACTIVITY("skiping invalid line(no key-name)\n");
                return false;
            }

            lineRange = SkipWhiteSpace(lineRange);
            lineRange = SplitChar(&separator, lineRange, '=');
            if(separator.first >= separator.second)
            {
                NN_DEVOVL_LOG_DEV_READACTIVITY("skiping invalid line(missing '=')\n");
                return false;
            }

            NN_DEVOVL_LOG_DEV_READACTIVITY("line[%s][%s]=[%s]\n",
                std::string(moduleName.first, moduleName.second).c_str(),
                std::string(keyName.first, keyName.second).c_str(),
                std::string(lineRange.first, lineRange.second).c_str()
            );

            return ReadValue(act, moduleName, keyName, lineRange);
        }
    }


    bool ReadWriteActivity::ReadFromSdCard(Activity* pOutValue) NN_NOEXCEPT
    {
        // 一時的にマウント
        if(!MountSdCard())
        {
            return false;
        }
        NN_UTIL_SCOPE_EXIT{ UnmountSdCard(); };

        // ロード
        std::vector<char> buf(WorkBufferSize, ' ');
        const char* pEnd = buf.data();

        if(!ReadWholeFile(&pEnd, buf.data(), buf.size()))
        {
            return false;
        }
        NN_ABORT_UNLESS_MINMAX(pEnd - buf.data(), 0, buf.size());

        //NN_SDK_LOG("file:\n%s\n", buf.data());

        Activity act = {};

        // 1 行ずつ処理
        const char* p = buf.data();
        while(p < pEnd)
        {
            auto range = GetLine(&p, p, pEnd);
            if(!ParseLine(act, range))
            {
                NN_DEVOVL_LOG_DEV_READACTIVITY("ignored line: %s\n", std::string(range.first, range.second).c_str());
            }
        }

        *pOutValue = act;
        return true;
    }

}}

