﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "debug_LogViewer.h"

#include <string>
#include <sstream>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_CharacterEncoding.h>

#include "../../framework/Framework.h"

namespace scene{ namespace debug{

    LogViewerImpl::LogViewerImpl() NN_NOEXCEPT
        : m_Mutex(false)
    {
    }

    LogViewerImpl::~LogViewerImpl() NN_NOEXCEPT
    {
        DetachLogStorage();
    }

    bool LogViewerImpl::IsLogStorageAttached() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        return m_pLogStorage.lock() != nullptr;
    }

    void LogViewerImpl::AttachLogStorage(const std::shared_ptr<ILogStorage>& pLogStorage) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        NN_ASSERT(m_pLogStorage.lock() == nullptr);

        m_pLogStorage = pLogStorage;
        UpdateAutoLineFeedImpl();
    }

    void LogViewerImpl::DetachLogStorage() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if(auto p = m_pLogStorage.lock())
        {
            p->SetAutoLineFeed(0, nullptr, nullptr);
        }
        m_pLogStorage.reset();
    }

    void LogViewerImpl::SetPosition(int x, int y) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_Rectangle.x = x;
        m_Rectangle.y = y;
    }

    void LogViewerImpl::SetSize(int w, int h) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_Rectangle.width = w;
        m_Rectangle.height = h;

        UpdateAutoLineFeedImpl();
    }

    void LogViewerImpl::SetTextSize(float size) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_TextSize = size;
    }

    float LogViewerImpl::GetCharacterWidth(const char* character, size_t length) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        return GetCharacterWidthImpl(character, length);
    }

    int LogViewerImpl::GetLineHeight() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        return GetLineHeightImpl();
    }

    void LogViewerImpl::ScrollUp(int scrollLineCount) NN_NOEXCEPT
    {
        if(scrollLineCount <= 0)
        {
            return;
        }

        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // LineIndex を小さくしすぎたとしても Update() で一番上に戻されるので小さくなりすぎても気にしない。
        if(m_LineIndex > scrollLineCount)
        {
            m_LineIndex -= scrollLineCount;
        }
        else
        {
            m_LineIndex = 0;
        }

        m_IsBoundToMostRecentLine = false;
    }

    void LogViewerImpl::ScrollDown(int scrollLineCount) NN_NOEXCEPT
    {
        if(scrollLineCount <= 0)
        {
            return;
        }

        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if(auto pLogStorage = m_pLogStorage.lock())
        {
            uint64_t storedIdxBeg = 0;
            uint64_t storedIdxEnd = 0;
            pLogStorage->GetLineRange(&storedIdxBeg, &storedIdxEnd);
            int displayedLineCountMax = GetRenderedLineCountMaxImpl();

            m_LineIndex += scrollLineCount;

            if(m_LineIndex + displayedLineCountMax > storedIdxEnd)
            {
                m_IsBoundToMostRecentLine = true;
                // 最下段に拘束した場合 LineIndex の値は無視されるので大きくなりすぎても気にしない。
            }
        }
    }

    void LogViewerImpl::MoveToBottom() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_IsBoundToMostRecentLine = true;
    }

    void LogViewerImpl::SetScrollbarVisibility(bool value) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_IsScrollbarVisible = value;
    }

    void LogViewerImpl::Update() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        auto pLogStorage = m_pLogStorage.lock();

        if(pLogStorage)
        {
            int lineHeight = GetLineHeightImpl();
            int lineCountMax = GetRenderedLineCountMaxImpl();

            uint64_t idxBeg;
            uint64_t idxEnd;
            uint64_t storedIdxBeg = 0;
            uint64_t storedIdxEnd = 0;
            bool isFitToBottom = false;
            {
                pLogStorage->GetLineRange(&storedIdxBeg, &storedIdxEnd);

                if(storedIdxEnd - storedIdxBeg < lineCountMax) // 一致する場合最後の行が見切れているかもしれない
                {
                    // 全部表示できるなら上から表示する
                    idxBeg = storedIdxBeg;
                    idxEnd = storedIdxEnd;
                }
                else
                {
                    if(m_IsBoundToMostRecentLine)
                    {
                        idxBeg = storedIdxEnd - lineCountMax;
                        idxEnd = storedIdxEnd;
                        isFitToBottom = true;
                    }
                    else
                    {
                        idxBeg = m_LineIndex;
                        if(idxBeg < storedIdxBeg)
                        {
                            idxBeg = storedIdxBeg;
                        }

                        idxEnd = idxBeg + lineCountMax;
                        if(idxEnd > storedIdxEnd)
                        {
                            idxEnd = storedIdxEnd;
                        }
                    }
                }

                NN_ASSERT(idxBeg <= idxEnd);
                if(idxBeg >= idxEnd)
                {
                    idxEnd = idxBeg;
                }
            }

            m_Renderer.Update(m_Rectangle, pLogStorage, idxBeg, idxEnd, storedIdxBeg, storedIdxEnd, m_TextSize, lineHeight, isFitToBottom, m_IsScrollbarVisible);
            m_LineIndex = idxBeg;
        }
        else
        {
            m_Renderer.Update(m_Rectangle, nullptr, 0, 0, 0, 0, 0, 0, false, m_IsScrollbarVisible);
        }
    }

    std::shared_ptr<panel::IPanel> LogViewerImpl::GetPanel() NN_NOEXCEPT
    {
        return m_Renderer.GetPanel();
    }

    //--------------------------------

    void LogViewerImpl::UpdateAutoLineFeedImpl() NN_NOEXCEPT
    {
        if(auto p = m_pLogStorage.lock())
        {
            p->SetAutoLineFeed(
                m_Rectangle.width,
                [](const char* c, size_t len, void* ptr)
                {
                    return reinterpret_cast<LogViewerImpl*>(ptr)->GetCharacterWidth(c, len);
                },
                this
            );
        }
    }
    float LogViewerImpl::GetCharacterWidthImpl(const char* character, size_t length) NN_NOEXCEPT
    {
        int w = 0;
        int h = 0;
        Framework::CalculateTextRenderingSize(&w, &h, character, m_TextSize, framework::TextWriterUsage_Text);

        return w;
    }

    int LogViewerImpl::GetLineHeightImpl() NN_NOEXCEPT
    {
        int w = 0;
        int h = 0;
        Framework::CalculateTextRenderingSize(&w, &h, "x", m_TextSize, framework::TextWriterUsage_Text);

        return h;
    }

    int LogViewerImpl::GetRenderedLineCountMaxImpl() NN_NOEXCEPT
    {
        auto lineHeight = GetLineHeightImpl();
        if(lineHeight <= 0)
        {
            return 0;
        }

        int lineCountMax = (m_Rectangle.height + lineHeight - 1) / lineHeight;
        if(lineCountMax < 0)
        {
            return 0;
        }
        return lineCountMax;
    }

    //--------------------------------------------------

    namespace detail
    {
        LogViewerRenderer::LogViewerRenderer() NN_NOEXCEPT
        {
            m_pRootPanel = std::make_shared<panel::PanelContainer>();
            m_pRootPanel->SetPanelName("logview");

            m_pLinePanelContainer = std::make_shared<panel::PanelContainer>();

            m_pScrollbarBackground = std::make_shared<panel::PanelText>();
            m_pScrollbarKnob = std::make_shared<panel::PanelText>();
        }

        std::shared_ptr<panel::IPanel> LogViewerRenderer::GetPanel() NN_NOEXCEPT
        {
            return m_pRootPanel;
        }

        void LogViewerRenderer::Update(
            const panel::PanelRectangle& rect,
            const std::shared_ptr<ILogStorage>& pLogStorage,
            uint64_t begIdx,
            uint64_t endIdx,
            uint64_t storageBegIdx,
            uint64_t storageEndIdx,
            float fontSize,
            int lineHeight,
            bool isFitToBottom,
            bool isScrollbarVisible
        ) NN_NOEXCEPT
        {
            m_pRootPanel->SetPosition(rect.x, rect.y);
            m_pRootPanel->SetSize(rect.width, rect.height);
            m_pRootPanel->SetColor(m_BackgroundColor);
            m_pRootPanel->SetVisibility(panel::PanelVisibility::Transparent);

            auto logRect       = panel::PanelRectangle(0            , 0, rect.width - m_ScrollbarWidth, rect.height);
            auto scrollbarRect = panel::PanelRectangle(logRect.width, 0, m_ScrollbarWidth             , rect.height);

            m_pLinePanelContainer->SetPosition(0, 0);
            m_pLinePanelContainer->SetSize(logRect.width, logRect.height);
            m_pLinePanelContainer->SetColor(m_BackgroundColor);
            m_pLinePanelContainer->SetVisibility(panel::PanelVisibility::Visible);

            std::vector<std::shared_ptr<panel::IPanel>> panelList;
            std::vector<std::shared_ptr<panel::IPanel>> linePanelList;

            // 先頭からログ行を入れていく
            {
                UpdateLinePanelListImpl(linePanelList, logRect, pLogStorage, begIdx, endIdx, fontSize, lineHeight, isFitToBottom);
                m_pLinePanelContainer->SetChildren(linePanelList);
                panelList.push_back(m_pLinePanelContainer);
            }

            // 必要ならスクロールバーを表示
            if(isScrollbarVisible)
            {
                UpdateScrollbarImpl(panelList, scrollbarRect, begIdx, endIdx, storageBegIdx, storageEndIdx);
            }

            m_pRootPanel->SetChildren(panelList);

            // ログ行部分だけ残して記録
            m_LinePanelList.swap(linePanelList);
            m_LineIndexBase = begIdx;
        }

        void LogViewerRenderer::UpdateLinePanelListImpl(
            std::vector<std::shared_ptr<panel::IPanel>>& outList,
            const panel::PanelRectangle& rect,
            const std::shared_ptr<ILogStorage>& pLogStorage,
            uint64_t begIdx,
            uint64_t endIdx,
            float fontSize,
            int lineHeight,
            bool isFitToBottom
        ) NN_NOEXCEPT
        {
            if(!pLogStorage || endIdx <= begIdx)
            {
                return;
            }

            uint64_t n = endIdx - begIdx;

            int positionOffsetY = 0;
            if(isFitToBottom)
            {
                int textHeight = lineHeight * n;
                positionOffsetY = rect.height - textHeight;
            }

            // 新しいログを優先的に描画
            for(uint64_t i = 0; i < n; i++)
            {
                uint64_t lineIdx = endIdx - 1 - i;
                int posY = lineHeight * (n - 1 - i) + positionOffsetY;
                NN_ASSERT_RANGE(lineIdx, begIdx, endIdx);

                auto p = AcquireLinePanelImpl(lineIdx, pLogStorage, rect.width, lineHeight, fontSize);
                auto pText = panel::PanelText::IsText(p);
                pText->SetPosition(0, posY, 0);
                outList.push_back(p);
            }

        }

        std::shared_ptr<panel::IPanel> LogViewerRenderer::AcquireLinePanelImpl(
            uint64_t lineIndex,
            const std::shared_ptr<ILogStorage>& pLogStorage,
            int lineWidth,
            int lineHeight,
            float fontSize
        ) NN_NOEXCEPT
        {
            if(lineIndex >= m_LineIndexBase && lineIndex < m_LineIndexBase + m_LinePanelList.size())
            {
                size_t i = m_LinePanelList.size() - 1 - (lineIndex - m_LineIndexBase);
                return m_LinePanelList[i];
            }

            LogLineString s = {};
            pLogStorage->GetLineString(&s, lineIndex);

            auto p = std::make_shared<panel::PanelText>();
            p->SetVisibility(panel::PanelVisibility::Visible);
            p->SetText(s.value);
            p->SetColor(m_BackgroundColor);
            p->SetSize(lineWidth, lineHeight);
            p->SetTextColor(m_LogFontColor);
            p->SetTextPosition(0, 4);
            p->SetTextSize(fontSize);

            return p;
        }

        void LogViewerRenderer::UpdateScrollbarImpl(
            std::vector<std::shared_ptr<panel::IPanel>>& outList,
            const panel::PanelRectangle& rect,
            uint64_t begIdx,
            uint64_t endIdx,
            uint64_t storageBegIdx,
            uint64_t storageEndIdx
        ) NN_NOEXCEPT
        {
            // 背景
            m_pScrollbarBackground->SetVisibility(panel::PanelVisibility::Visible);
            m_pScrollbarBackground->SetPosition(rect.x, rect.y);
            m_pScrollbarBackground->SetSize(rect.width, rect.height);
            m_pScrollbarBackground->SetColor(m_ScrollbarBackgroundColor);
            outList.push_back(m_pScrollbarBackground);

            // つまみ
            uint64_t nStorage = storageEndIdx - storageBegIdx;
            if(nStorage == 0)
            {
                return;
            }

            uint64_t begOffset = begIdx - storageBegIdx;
            uint64_t endOffset = endIdx - storageBegIdx;

            int begPosition = static_cast<int>(std::floor(static_cast<float>(begOffset) / static_cast<float>(nStorage) * rect.height)) + rect.y;
            int endPosition = static_cast<int>(std::ceil (static_cast<float>(endOffset) / static_cast<float>(nStorage) * rect.height)) + rect.y;
            begPosition = std::max(rect.y, begPosition);
            endPosition = std::min(rect.y + rect.height, endPosition);

            if(endPosition <= begPosition)
            {
                return;
            }

            m_pScrollbarKnob->SetVisibility(panel::PanelVisibility::Visible);
            m_pScrollbarKnob->SetPosition(rect.x, begPosition);
            m_pScrollbarKnob->SetSize(rect.width, endPosition - begPosition);
            m_pScrollbarKnob->SetColor(m_ScrollbarKnobColor);
            outList.push_back(m_pScrollbarKnob);
        }


    }

}}

