﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "debug_LogStorageLm.h"

#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <nn/lm/lm_LogGetter.h>
#include "logutil/TextReader.h"
#include "../../Config.h"

namespace scene{ namespace debug{

    LogStorageLm::LogStorageLm() NN_NOEXCEPT
        : m_Mutex(false)
        , m_IsActive(false)
    {
        m_HeadIndex = 0;
        m_TailIndex = 0;
        std::memset(m_LineList, 0, sizeof(m_LineList));
        std::memset(m_LogBuffer, 0, sizeof(m_LogBuffer));

        m_LineWidthMax = 0;
        m_CharWidthGetterFunction = nullptr;
        m_CharWidthGetterUserPtr = nullptr;
    }

    LogStorageLm::~LogStorageLm() NN_NOEXCEPT
    {
        Deactivate();
    }

    void LogStorageLm::GetLineRange(uint64_t* pOutLineBegin, uint64_t* pOutLineEnd) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        *pOutLineBegin = m_HeadIndex;
        *pOutLineEnd   = m_TailIndex;
    }

    void LogStorageLm::GetLineString(LogLineString* pOutValue, uint64_t line) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if(line < m_HeadIndex || line >= m_TailIndex)
        {
            std::memset(pOutValue, 0, sizeof(LogLineString));
            return;
        }

        uint64_t index = line % LineCountMax;
        *pOutValue = m_LineList[index];
        pOutValue->value[LogLineSizeMax - 1] = 0; // 念の為。
    }

    void LogStorageLm::Activate() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if(m_IsActive)
        {
            return;
        }

        nn::lm::StartLogging();
        m_IsActive = true;
    }

    void LogStorageLm::Deactivate() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if(!m_IsActive)
        {
            return;
        }

        nn::lm::StopLogging();
        m_IsActive = false;
    }

    void LogStorageLm::SetAutoLineFeed(float lineWidthMax, float (*charWidthGetterFunction)(const char*, size_t, void*), void* charWidthGetterUserPtr) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_LineWidthMax = lineWidthMax;
        m_CharWidthGetterFunction = charWidthGetterFunction;
        m_CharWidthGetterUserPtr = charWidthGetterUserPtr;
    }

    void LogStorageLm::Update() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        if(!m_IsActive)
        {
            return;
        }

        auto addLogLine = [&](const std::string& value)
        {
            if(m_TailIndex == m_HeadIndex + LineCountMax)
            {
                m_HeadIndex++;
            }
            auto& e = m_LineList[m_TailIndex % LineCountMax];
            nn::util::Strlcpy(e.value, value.c_str(), static_cast<int>(sizeof(e.value)));
            //NN_DEVOVL_LOG("[log]%llu:%s\n", m_TailIndex, e.value);
            m_TailIndex++;
        };

        int64_t receivedSize = 0;
        uint32_t droppedCount;
        nn::lm::GetLog(m_LogBuffer, sizeof(m_LogBuffer), &receivedSize, &droppedCount);
        NN_ABORT_UNLESS_MINMAX(receivedSize, 0, static_cast<int64_t>(sizeof(m_LogBuffer)));

        if(receivedSize > 0)
        {
            logutil::TextReader tr;
            tr.Initialize(m_LogBuffer, receivedSize, m_LineWidthMax, m_CharWidthGetterFunction, m_CharWidthGetterUserPtr);

            std::string line;
            int64_t position = 0;
            while(position < receivedSize)
            {
                tr.ReadLine(&line, &position, position);
                addLogLine(line);
            }

            if (droppedCount > 0)
            {
                std::string dropLog("<< Some logs were dropped >>\n");
                addLogLine(dropLog);
            }
        }
    }



}}

