﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Color.h>
#include "../../panel/panel_PanelContainer.h"
#include "../../panel/panel_PanelText.h"
#include "../../panel/panel_PanelHistorical.h"

namespace scene{ namespace debug{

    class HistoricalGraph
    {
        NN_DISALLOW_COPY(HistoricalGraph);
    private:
        class SeriesEntry
        {
        public:
            SeriesEntry() NN_NOEXCEPT;

            void SetLineWidth(int value) NN_NOEXCEPT;
            void SetLineColor(const nn::util::Color4f& value) NN_NOEXCEPT;
            void SetValueRange(float valueMin, float ValueMax) NN_NOEXCEPT;

            void ResetValue() NN_NOEXCEPT;
            void PushValue(float value) NN_NOEXCEPT;

            void Update(int width, int height, int scrollSpeed) NN_NOEXCEPT;
            std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT;

        public:
            std::shared_ptr<panel::PanelText> m_PointPanel;

            float m_ValueMin;
            float m_ValueMax;
            int m_LineWidth;
            nn::util::Color4f m_LineColor;

            float m_CurrentValue;

            // 前のフレームの描画範囲。 isnan(PreviousValue) の場合不定。
            float m_PreviousValue;
            int m_PreviousYMin;
            int m_PreviousYMax;
        };

    public:
        static const size_t RequiredMemorySizePerSeries = sizeof(SeriesEntry);
        static const size_t RequiredMemoryAlignment = NN_ALIGNOF(SeriesEntry);

    public:
        HistoricalGraph() NN_NOEXCEPT;

        // Holder を使用する場合、Holder により初期化されるのでこの関数を直接呼ばないこと。
        void Initialize(int entryCount, void* memory, size_t memorySize) NN_NOEXCEPT;

        // Holder を使用する場合、Holder により破棄されるのでこの関数を直接呼ばないこと。
        void Finalize() NN_NOEXCEPT;

        // 描画領域の位置
        void SetPosition(int x, int y) NN_NOEXCEPT;

        // 描画領域のサイズ[px]
        void SetSize(int width, int height) NN_NOEXCEPT;

        // 背景色
        void SetBackgroundColor(const nn::util::Color4f& value) NN_NOEXCEPT;

        // スクロール速度[px/frame]
        // 古い値が左に流れていくときに正。
        void SetScrollSpeed(int value) NN_NOEXCEPT;

        void SetEntryLineWidth(int entryIndex, int value) NN_NOEXCEPT;
        void SetEntryLineColor(int entryIndex, const nn::util::Color4f& value) NN_NOEXCEPT;
        void SetValueRange(int entryIndex, float valueMin, float valueMax) NN_NOEXCEPT;
        void ResetEntryValue(int entryIndex) NN_NOEXCEPT;
        void PushEntryValue(int entryIndex, float value) NN_NOEXCEPT;

        void Update() NN_NOEXCEPT;
        std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT;
        void SetPanelName(const std::string& value) NN_NOEXCEPT;

    private:
        // 以下は対象外
        // - m_EntryCount
        // - m_pEntryList
        // - m_RootPanel
        void ResetParameterImpl() NN_NOEXCEPT;

    private:
        SeriesEntry* m_pEntryList;
        int m_EntryCount;

        int m_PositionX;
        int m_PositionY;
        int m_Width;
        int m_Height;
        nn::util::Color4f m_BackgroundColor;

        int m_ScroollSpeed;

        std::shared_ptr<panel::PanelContainer> m_RootPanel;
        std::shared_ptr<panel::PanelHistorical> m_HistoryPanel;
    };

    template<int EntryCount>
    class HistoricalGraphHolder
    {
        NN_DISALLOW_COPY(HistoricalGraphHolder);
        NN_DISALLOW_MOVE(HistoricalGraphHolder);
    public:
        HistoricalGraphHolder() NN_NOEXCEPT
        {
            m_Impl.Initialize(EntryCount, m_Memory, sizeof(m_Memory));
        }

        ~HistoricalGraphHolder() NN_NOEXCEPT
        {
            m_Impl.Finalize();
        }

        HistoricalGraph* operator->() NN_NOEXCEPT
        {
            return &m_Impl;
        }

        const HistoricalGraph* operator->() const NN_NOEXCEPT
        {
            return &m_Impl;
        }

    private:
        HistoricalGraph m_Impl;
        NN_ALIGNAS(HistoricalGraph::RequiredMemoryAlignment) char m_Memory[HistoricalGraph::RequiredMemorySizePerSeries * EntryCount];
    };

}}
