﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "debug_HidInputViewer.h"

#include <sstream>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../../framework/Framework.h"
#include "../../framework/Hid.h"
#include "../../framework/HidHandlingModeManager.h"
#include "../../panel/panel_Icon.h"

namespace scene{ namespace debug{
    namespace {
        static nn::util::Color4f GetBackgroundColor() NN_NOEXCEPT
        {
            return {0, 0, 0, .75f};
        }

        static nn::util::Color4f GetButtonColorFree() NN_NOEXCEPT
        {
            return {.25f, .25f, .25f, .75f};
        }

        static nn::util::Color4f GetButtonColorHeld() NN_NOEXCEPT
        {
            return {1, 0, 0, 1};
        }

        static nn::util::Color4f GetIconColor() NN_NOEXCEPT
        {
            return {1, 1, 1, 1};
        }

        static nn::util::Color4f GetAnalogStickColorX(int value) NN_NOEXCEPT
        {
            NN_UNUSED(value);
            return {1, 0, 0, 1};
        }

        static nn::util::Color4f GetAnalogStickColorY(int value) NN_NOEXCEPT
        {
            NN_UNUSED(value);
            return {0, 1, 0, 1};
        }

        static nn::util::Color4f GetLedOnColor(bool value) NN_NOEXCEPT
        {
            return {0, .5f, 0, .75f};
        }
    }

    class HidInputViewerImpl::NpadView
    {
    public:
        static const int IconFontUnitSize = 7;
        static const int IconUnitX = 0;
        static const int IconUnitY = 0;
        static const int IconUnitCountX = 8;
        static const int IconUnitCountY = 8;

        static const int ButtonXBaseL    = IconUnitCountX  + 1;
        static const int ButtonXBaseDrct = ButtonXBaseL    + 5;
        static const int ButtonXBaseMinu = ButtonXBaseDrct + 7;
        static const int ButtonXBaseLftS = ButtonXBaseMinu + 3;
        static const int ButtonXBaseRgtS = ButtonXBaseLftS + 5;
        static const int ButtonXBasePlus = ButtonXBaseRgtS + 2;
        static const int ButtonXBaseAbxy = ButtonXBasePlus + 3;
        static const int ButtonXBaseR    = ButtonXBaseAbxy + 7;
        static const int ButtonXEnd      = ButtonXBaseR    + 4;

        static const int ButtonsUnitCountX = ButtonXEnd - ButtonXBaseL;
        static const int ButtonsUnitCountY = 6;

        static const int StickXBaseL = ButtonXBaseL;
        static const int StickXBaseR = ButtonXBaseRgtS;
        static const int StickYBase  = ButtonsUnitCountY;
        static const int StickUnitCountX = 16;
        static const int StickUnitCountY = 1;

        static const int UnitCountX = ButtonXEnd;
        static const int UnitCountY = 8;


    public:
        explicit NpadView(int unitSize) NN_NOEXCEPT
        {
            auto makePanel = [&](int x, int y, int w, int h) -> std::shared_ptr<panel::PanelText>
            {
                auto p = std::make_shared<panel::PanelText>();
                NN_SDK_ASSERT_NOT_NULL(p);
                p->SetVisibility(panel::PanelVisibility::Invisible);
                p->SetPosition(unitSize * x, unitSize * y);
                p->SetSize(unitSize * w, unitSize * h);
                return p;
            };

            m_pContainer = std::make_shared<panel::PanelContainer>();
            m_pContainer->SetVisibility(panel::PanelVisibility::Visible);
            m_pContainer->SetColor(GetBackgroundColor());
            m_pContainer->SetPosition(0, 0);
            m_pContainer->SetSize(unitSize * UnitCountX, unitSize * UnitCountY);

            m_pIconContainer = std::make_shared<panel::PanelContainer>();
            m_pIconContainer->SetVisibility(panel::PanelVisibility::Visible);
            m_pIconContainer->SetColor(GetBackgroundColor());
            m_pIconContainer->SetPosition(unitSize * IconUnitX, unitSize * IconUnitY);
            m_pIconContainer->SetSize(unitSize * IconUnitCountX, unitSize * IconUnitCountY);
            {
                m_pIconContainer->AddChild(m_pIcon = makePanel(0, 0, IconUnitCountX, IconUnitCountY));
                m_pIconContainer->AddChild(m_pLed0 = makePanel(0, 0, 2, 2));
                m_pIconContainer->AddChild(m_pLed1 = makePanel(2, 0, 2, 2));
                m_pIconContainer->AddChild(m_pLed2 = makePanel(4, 0, 2, 2));
                m_pIconContainer->AddChild(m_pLed3 = makePanel(6, 0, 2, 2));
            }
            m_pContainer->AddChild(m_pIconContainer);

            m_pContainer->AddChild(m_pButtonA = makePanel(ButtonXBaseAbxy + 4, 2, 2, 2));
            m_pContainer->AddChild(m_pButtonB = makePanel(ButtonXBaseAbxy + 2, 4, 2, 2));
            m_pContainer->AddChild(m_pButtonX = makePanel(ButtonXBaseAbxy + 2, 0, 2, 2));
            m_pContainer->AddChild(m_pButtonY = makePanel(ButtonXBaseAbxy + 0, 2, 2, 2));

            m_pContainer->AddChild(m_pButtonL  = makePanel(ButtonXBaseL + 0, 4, 4, 2));
            m_pContainer->AddChild(m_pButtonZL = makePanel(ButtonXBaseL + 0, 0, 4, 3));
            m_pContainer->AddChild(m_pButtonR  = makePanel(ButtonXBaseR + 0, 4, 4, 2));
            m_pContainer->AddChild(m_pButtonZR = makePanel(ButtonXBaseR + 0, 0, 4, 3));

            m_pContainer->AddChild(m_pButtonRight = makePanel(ButtonXBaseDrct + 4, 2, 2, 2));
            m_pContainer->AddChild(m_pButtonDown  = makePanel(ButtonXBaseDrct + 2, 4, 2, 2));
            m_pContainer->AddChild(m_pButtonUp    = makePanel(ButtonXBaseDrct + 2, 0, 2, 2));
            m_pContainer->AddChild(m_pButtonLeft  = makePanel(ButtonXBaseDrct + 0, 2, 2, 2));

            m_pContainer->AddChild(m_pButtonMinus  = makePanel(ButtonXBaseMinu + 0, 0, 2, 2));
            m_pContainer->AddChild(m_pButtonPlus   = makePanel(ButtonXBasePlus + 0, 0, 2, 2));
            m_pContainer->AddChild(m_pButtonStickL = makePanel(ButtonXBaseMinu + 0, 3, 2, 2));
            m_pContainer->AddChild(m_pButtonStickR = makePanel(ButtonXBasePlus + 0, 4, 2, 2));

            m_pContainer->AddChild(m_pButtonLeftSL  = makePanel(ButtonXBaseLftS + 0, 0, 1, 3));
            m_pContainer->AddChild(m_pButtonLeftSR  = makePanel(ButtonXBaseLftS + 0, 3, 1, 3));
            m_pContainer->AddChild(m_pButtonRightSR = makePanel(ButtonXBaseRgtS + 0, 0, 1, 3));
            m_pContainer->AddChild(m_pButtonRightSL = makePanel(ButtonXBaseRgtS + 0, 3, 1, 3));

            m_pContainer->AddChild(m_pAnalogStickLX = makePanel(0, 0, 0, 0)); // 位置とサイズは Update で設定する
            m_pContainer->AddChild(m_pAnalogStickLY = makePanel(0, 0, 0, 0)); // 位置とサイズは Update で設定する
            m_pContainer->AddChild(m_pAnalogStickRX = makePanel(0, 0, 0, 0)); // 位置とサイズは Update で設定する
            m_pContainer->AddChild(m_pAnalogStickRY = makePanel(0, 0, 0, 0)); // 位置とサイズは Update で設定する

            m_UnitSize = unitSize;
        }

        void Update(const HidNpadState& state) NN_NOEXCEPT
        {
            UpdateIconImpl(state);
            UpdateButtonsImpl(state);
        }

    private:
        void UpdateIconImpl(const HidNpadState& state) NN_NOEXCEPT
        {
            if(state.m_Style == HidNpadStyle_None)
            {
                m_pIconContainer->SetVisibility(panel::PanelVisibility::Invisible);
                m_pIcon->SetVisibility(panel::PanelVisibility::Invisible);
                m_pLed0->SetVisibility(panel::PanelVisibility::Invisible);
                m_pLed1->SetVisibility(panel::PanelVisibility::Invisible);
                m_pLed2->SetVisibility(panel::PanelVisibility::Invisible);
                m_pLed3->SetVisibility(panel::PanelVisibility::Invisible);
                return;
            }

            m_pIconContainer->SetVisibility(panel::PanelVisibility::Visible);
            m_pIcon->SetVisibility(panel::PanelVisibility::Visible);
            m_pIcon->SetColor(GetBackgroundColor());
            m_pIcon->SetTextColor(GetIconColor());
            m_pIcon->SetTextSize(IconFontUnitSize * m_UnitSize);

            const char* text = "";
            switch(state.m_Style)
            {
            case HidNpadStyle_Handheld:
                text = panel::IconToString::GetUtf8(panel::Icon_SwitchDevice);
                break;
            case HidNpadStyle_FullKey:
                text = panel::IconToString::GetUtf8(panel::Icon_SwitchProController);
                break;
            case HidNpadStyle_JoyDual:
                text = panel::IconToString::GetUtf8(panel::Icon_JoyConDual);
                break;
            case HidNpadStyle_JoyDualLeftOnly:
                text = panel::IconToString::GetUtf8(panel::Icon_JoyConLeftVertical);
                break;
            case HidNpadStyle_JoyDualRightOnly:
                text = panel::IconToString::GetUtf8(panel::Icon_JoyConRightVertical);
                break;
            case HidNpadStyle_JoyLeft:
                text = panel::IconToString::GetUtf8(panel::Icon_JoyConLeftHorizontal);
                break;
            case HidNpadStyle_JoyRight:
                text = panel::IconToString::GetUtf8(panel::Icon_JoyConRightHorizontal);
                break;
            default:
                text = "?";
            }
            m_pIcon->SetText(text);

            // テキスト位置を調整
            {
                int w = 0;
                int h = 0;
                Framework::CalculateTextRenderingSize(&w, &h, text, IconFontUnitSize * m_UnitSize, framework::TextWriterUsage_Icon);

                int pw = IconUnitCountX * m_UnitSize;
                int ph = IconUnitCountY * m_UnitSize;

                // 中央寄せ
                int x = (pw - w) / 2;
                int y = (ph - h) / 2;
                m_pIcon->SetTextPosition(x, y);
            }

            // LED パターンを更新
            auto updateLed = [&](std::shared_ptr<panel::PanelText>& p, bool value)
            {
                if(value)
                {
                    p->SetVisibility(panel::PanelVisibility::Visible);
                    p->SetColor(GetLedOnColor(value));
                }
                else
                {
                    p->SetVisibility(panel::PanelVisibility::Invisible);
                }
            };

            updateLed(m_pLed0, (state.m_LedPattern & (1 << 0)) != 0);
            updateLed(m_pLed1, (state.m_LedPattern & (1 << 1)) != 0);
            updateLed(m_pLed2, (state.m_LedPattern & (1 << 2)) != 0);
            updateLed(m_pLed3, (state.m_LedPattern & (1 << 3)) != 0);

        }

        void UpdateButtonsImpl(const HidNpadState& state) NN_NOEXCEPT
        {
            // LeftSide
            switch(state.m_Style)
            {
            case HidNpadStyle_Handheld:         // 両方
            case HidNpadStyle_FullKey:          // 両方
            case HidNpadStyle_JoyDual:          // 両方
            case HidNpadStyle_JoyDualLeftOnly:  // 左のみ
            case HidNpadStyle_JoyLeft:          // 左のみ
                UpdateLeftSideButtonsImpl(state);
                UpdateLeftSideAnalogStickImpl(state);
                break;
            default:
                InvalidateLeftSideButtonsImpl();
                InvalidateLeftSideAnalogStickImpl();
            }

            // RightSide
            switch(state.m_Style)
            {
            case HidNpadStyle_Handheld:         // 両方
            case HidNpadStyle_FullKey:          // 両方
            case HidNpadStyle_JoyDual:          // 両方
            case HidNpadStyle_JoyDualRightOnly: // 右のみ
            case HidNpadStyle_JoyRight:         // 右のみ
                UpdateRightSideButtonsImpl(state);
                UpdateRightSideAnalogStickImpl(state);
                break;
            default:
                InvalidateRightSideButtonsImpl();
                InvalidateRightSideAnalogStickImpl();
            }
        }


        void InvalidateLeftSideButtonsImpl() NN_NOEXCEPT
        {
            m_pButtonZL->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonL->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonLeft->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonUp->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonDown->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonRight->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonMinus->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonStickL->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonLeftSL->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonLeftSR->SetVisibility(panel::PanelVisibility::Invisible);
        }

        void InvalidateRightSideButtonsImpl() NN_NOEXCEPT
        {
            m_pButtonRightSR->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonRightSL->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonPlus->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonStickR->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonY->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonX->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonB->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonA->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonZR->SetVisibility(panel::PanelVisibility::Invisible);
            m_pButtonR->SetVisibility(panel::PanelVisibility::Invisible);
        }

#define NN_DEVOVL_UPDATE_BUTTONPANEL(BUTTON)    \
            m_pButton ## BUTTON ->SetVisibility(panel::PanelVisibility::Visible);   \
            m_pButton ## BUTTON ->SetColor(state.m_Button.BUTTON ? GetButtonColorHeld() : GetButtonColorFree());

        void UpdateLeftSideButtonsImpl(const HidNpadState& state) NN_NOEXCEPT
        {
            NN_DEVOVL_UPDATE_BUTTONPANEL(ZL);
            NN_DEVOVL_UPDATE_BUTTONPANEL(L);
            NN_DEVOVL_UPDATE_BUTTONPANEL(Left);
            NN_DEVOVL_UPDATE_BUTTONPANEL(Up);
            NN_DEVOVL_UPDATE_BUTTONPANEL(Down);
            NN_DEVOVL_UPDATE_BUTTONPANEL(Right);
            NN_DEVOVL_UPDATE_BUTTONPANEL(Minus);
            NN_DEVOVL_UPDATE_BUTTONPANEL(StickL);
            NN_DEVOVL_UPDATE_BUTTONPANEL(LeftSL);
            NN_DEVOVL_UPDATE_BUTTONPANEL(LeftSR);
        }

        void UpdateRightSideButtonsImpl(const HidNpadState& state) NN_NOEXCEPT
        {
            NN_DEVOVL_UPDATE_BUTTONPANEL(RightSL);
            NN_DEVOVL_UPDATE_BUTTONPANEL(RightSR);
            NN_DEVOVL_UPDATE_BUTTONPANEL(Plus);
            NN_DEVOVL_UPDATE_BUTTONPANEL(StickR);
            NN_DEVOVL_UPDATE_BUTTONPANEL(Y);
            NN_DEVOVL_UPDATE_BUTTONPANEL(X);
            NN_DEVOVL_UPDATE_BUTTONPANEL(B);
            NN_DEVOVL_UPDATE_BUTTONPANEL(A);
            NN_DEVOVL_UPDATE_BUTTONPANEL(ZR);
            NN_DEVOVL_UPDATE_BUTTONPANEL(R);
        }
#undef NN_DEVOVL_UPDATE_BUTTONPANEL

        void InvalidateLeftSideAnalogStickImpl() NN_NOEXCEPT
        {
            m_pAnalogStickLX->SetVisibility(panel::PanelVisibility::Invisible);
            m_pAnalogStickLY->SetVisibility(panel::PanelVisibility::Invisible);
        }

        void InvalidateRightSideAnalogStickImpl() NN_NOEXCEPT
        {
            m_pAnalogStickRX->SetVisibility(panel::PanelVisibility::Invisible);
            m_pAnalogStickRY->SetVisibility(panel::PanelVisibility::Invisible);
        }

        static void CalculateAnalogStickBarSignedLength(int* pOutLengthX, int* pOutLengthY, const HidAnalogStickValue& stick, int lengthMax) NN_NOEXCEPT
        {
            int lenX = 0;
            int lenY = 0;
            if(stick.x > 0)
            {
                lenX = static_cast<int>(std::ceil(stick.x * lengthMax));
            }
            else
            {
                lenX = static_cast<int>(std::floor(stick.x * lengthMax));
            }

            if(stick.y > 0)
            {
                lenY = static_cast<int>(std::ceil(stick.y * lengthMax));
            }
            else
            {
                lenY = static_cast<int>(std::floor(stick.y * lengthMax));
            }

            *pOutLengthX = lenX;
            *pOutLengthY = -lenY; // DevOverlayDisp の中では下向き正にしているが、 nn::hid としては上向きが正なので再度反転しておく。
        }

        template<typename F>
        static void SetStickPanel(std::shared_ptr<panel::PanelText>& p, int len, int baseX, int baseY, int lengthMax, int height, F colorFunction) NN_NOEXCEPT
        {
            if(len == 0)
            {
                p->SetVisibility(panel::PanelVisibility::Invisible);
                return;
            }

            p->SetVisibility(panel::PanelVisibility::Visible);
            p->SetColor(colorFunction(len));

            if(len > 0)
            {
                p->SetPosition(baseX + lengthMax, baseY);
                p->SetSize(len, height);
            }
            else
            {
                p->SetPosition(baseX + lengthMax + len, baseY);
                p->SetSize(-len, height);
            }
        }

        void UpdateLeftSideAnalogStickImpl(const HidNpadState& state) NN_NOEXCEPT
        {
            int lengthMax = m_UnitSize * StickUnitCountX / 2;
            int xBase = m_UnitSize * StickXBaseL;
            int yBase = m_UnitSize * StickYBase;

            int lenX = 0;
            int lenY = 0;
            CalculateAnalogStickBarSignedLength(&lenX, &lenY, state.m_AnalogStick.stickL, lengthMax);
            SetStickPanel(m_pAnalogStickLX, lenX, xBase, yBase             , lengthMax, m_UnitSize, GetAnalogStickColorX);
            SetStickPanel(m_pAnalogStickLY, lenY, xBase, yBase + m_UnitSize, lengthMax, m_UnitSize, GetAnalogStickColorY);
        }

        void UpdateRightSideAnalogStickImpl(const HidNpadState& state) NN_NOEXCEPT
        {
            int lengthMax = m_UnitSize * StickUnitCountX / 2;
            int xBase = m_UnitSize * StickXBaseR;
            int yBase = m_UnitSize * StickYBase;

            int lenX = 0;
            int lenY = 0;
            CalculateAnalogStickBarSignedLength(&lenX, &lenY, state.m_AnalogStick.stickR, lengthMax);
            SetStickPanel(m_pAnalogStickRX, lenX, xBase, yBase             , lengthMax, m_UnitSize, GetAnalogStickColorX);
            SetStickPanel(m_pAnalogStickRY, lenY, xBase, yBase + m_UnitSize, lengthMax, m_UnitSize, GetAnalogStickColorY);
        }

        //--------------------------------------------------

    public:
        void SetPosition(int x, int y) NN_NOEXCEPT
        {
            m_pContainer->SetPosition(x, y);
        }

        int GetWidth() const NN_NOEXCEPT
        {
            return UnitCountX * m_UnitSize;
        }

        int GetHeight() const NN_NOEXCEPT
        {
            return UnitCountY * m_UnitSize;
        }

        std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT
        {
            return m_pContainer;
        }

    private:
        int m_UnitSize;
        std::shared_ptr<panel::PanelContainer> m_pContainer;
        std::shared_ptr<panel::PanelContainer> m_pIconContainer;

        std::shared_ptr<panel::PanelText> m_pLed0;
        std::shared_ptr<panel::PanelText> m_pLed1;
        std::shared_ptr<panel::PanelText> m_pLed2;
        std::shared_ptr<panel::PanelText> m_pLed3;

        std::shared_ptr<panel::PanelText> m_pIcon;

        std::shared_ptr<panel::PanelText> m_pButtonA;
        std::shared_ptr<panel::PanelText> m_pButtonB;
        std::shared_ptr<panel::PanelText> m_pButtonX;
        std::shared_ptr<panel::PanelText> m_pButtonY;

        std::shared_ptr<panel::PanelText> m_pButtonL;
        std::shared_ptr<panel::PanelText> m_pButtonR;
        std::shared_ptr<panel::PanelText> m_pButtonZL;
        std::shared_ptr<panel::PanelText> m_pButtonZR;

        std::shared_ptr<panel::PanelText> m_pButtonUp;
        std::shared_ptr<panel::PanelText> m_pButtonDown;
        std::shared_ptr<panel::PanelText> m_pButtonLeft;
        std::shared_ptr<panel::PanelText> m_pButtonRight;

        std::shared_ptr<panel::PanelText> m_pButtonPlus;
        std::shared_ptr<panel::PanelText> m_pButtonMinus;
        std::shared_ptr<panel::PanelText> m_pButtonStickL;
        std::shared_ptr<panel::PanelText> m_pButtonStickR;

        std::shared_ptr<panel::PanelText> m_pButtonLeftSL;
        std::shared_ptr<panel::PanelText> m_pButtonLeftSR;
        std::shared_ptr<panel::PanelText> m_pButtonRightSL;
        std::shared_ptr<panel::PanelText> m_pButtonRightSR;

        std::shared_ptr<panel::PanelText> m_pAnalogStickLX;
        std::shared_ptr<panel::PanelText> m_pAnalogStickLY;
        std::shared_ptr<panel::PanelText> m_pAnalogStickRX;
        std::shared_ptr<panel::PanelText> m_pAnalogStickRY;

    };

    //---------------------------------------------


    HidInputViewerImpl::HidInputViewerImpl(int unitSize, int viewSlotCount) NN_NOEXCEPT
        : m_UnitSize(unitSize)
        , m_ViewSlotCount(viewSlotCount)
    {
        if(m_UnitSize <= 0)
        {
            m_UnitSize = 1;
        }
        if(m_ViewSlotCount <= 0)
        {
            m_ViewSlotCount = 1;
        }

        m_pPanel = std::make_shared<panel::PanelContainer>();
        NN_SDK_ASSERT_NOT_NULL(m_pPanel);
        m_pPanel->SetPanelName("hidviw");

        m_NpadViewList.reserve(m_ViewSlotCount);
        for(size_t i = 0; i < m_ViewSlotCount; i++)
        {
            m_NpadViewList.push_back(std::make_shared<NpadView>(m_UnitSize));
            NN_SDK_ASSERT_NOT_NULL(m_NpadViewList[i]);
        }

        framework::g_HidHandlingModeManager.BeginObservation();
    }

    HidInputViewerImpl::~HidInputViewerImpl() NN_NOEXCEPT
    {
        framework::g_HidHandlingModeManager.EndObservation();
    }

    std::shared_ptr<panel::IPanel> HidInputViewerImpl::GetPanel() NN_NOEXCEPT
    {
        return m_pPanel;
    }

    void HidInputViewerImpl::SetPosition(int x, int y) NN_NOEXCEPT
    {
        m_pPanel->SetPosition(x, y);
    }

    void HidInputViewerImpl::Update() NN_NOEXCEPT
    {
        if(m_ViewSlotCount > 0)
        {
            std::vector<int> indexList;
            indexList.resize(m_ViewSlotCount, -1);
            int n = Hid::GetNpadCount();
            for(int k = 0; k < n; k++)
            {
                auto state = Hid::GetNpadState(k);

                // コントローラがなければ無視
                if(state.m_Style == HidNpadStyle_None)
                {
                    continue;
                }

                // 最初に見つけた Handheld を末尾に登録
                if(state.m_Style == HidNpadStyle_Handheld && indexList.back() == -1)
                {
                    m_NpadViewList.back()->Update(state);
                    indexList.back() = k;
                    continue;
                }

                // [0, m_ViewSlotCount - 1) までは同じインデックスのスロットに登録
                if(k < m_ViewSlotCount - 1)
                {
                    m_NpadViewList[k]->Update(state);
                    indexList[k] = k;
                    continue;
                }

                // それ以上のインデックスはひとまず置いておく。
                // このループでは Handheld であれば登録する。
            }

            // 保留しておいたものを登録
            for(int k = m_ViewSlotCount - 1; k < n; k++)
            {
                auto state = Hid::GetNpadState(k);

                // コントローラがなければ無視
                if(state.m_Style == HidNpadStyle_None)
                {
                    continue;
                }

                // このコントローラが未登録であることを確認＆空いているスロットを探す
                int t = -1;
                for(int i = 0; i < m_ViewSlotCount; i++)
                {
                    // 登録済なら無視させる
                    if(indexList[i] == k)
                    {
                        t = -1;
                        break;
                    }

                    // 最初に見つけた空きスロットを覚えておく
                    if(t < 0 && indexList[i] < 0)
                    {
                        t = i;
                        // 更に後ろのスロットに登録済かもしれないのでループは継続
                    }
                }

                // 既に登録済であるか、空きが無ければ無視する
                if(t < 0)
                {
                    continue;
                }

                // 空いているところに登録
                m_NpadViewList[t]->Update(state);
                indexList[t] = k;
            }

            // 登録しなかったスロットをクリア
            for(int t = 0; t < m_ViewSlotCount; t++)
            {
                if(indexList[t] < 0)
                {
                    m_NpadViewList[t]->Update({});
                }
            }
        }

        int widthMax = 0;
        int heightSum = 0;
        std::vector<std::shared_ptr<panel::IPanel>> viewPanelList;
        for(auto& e : m_NpadViewList)
        {
            e->SetPosition(0, heightSum);
            widthMax = std::max(widthMax, e->GetWidth());
            heightSum += e->GetHeight() + m_ViewSlotSpacing;
            viewPanelList.push_back(e->GetPanel());
        }
        if(heightSum > m_ViewSlotSpacing)
        {
            heightSum -= m_ViewSlotSpacing;
        }
        m_pPanel->SetChildren(viewPanelList);

        m_pPanel->SetVisibility(panel::PanelVisibility::Visible);
        m_pPanel->SetColor({0, 0, 0, 1});
        m_pPanel->SetSize(widthMax, heightSum);
    }

    std::pair<int, int> HidInputViewerImpl::CalculateViewSize(int unitSize, int slotCount) NN_NOEXCEPT
    {
        int slotSpacing = 0;

        int slotWidth  = unitSize * NpadView::UnitCountX;
        int slotHeight = unitSize * NpadView::UnitCountY;

        int width  = slotWidth;
        int height = slotHeight * slotCount + slotSpacing * (slotCount - 1);

        return std::make_pair(width, height);
    }

    std::pair<int, int> CalculateHidInputViewerSize(int unitSize, int slotCount) NN_NOEXCEPT
    {
        return HidInputViewerImpl::CalculateViewSize(unitSize, slotCount);
    }

}}

