﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneAppLaunch.h"

#include <nn/util/util_FormatString.h>
#include <nn/ae.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "../../framework/Framework.h"

namespace scene{ namespace applaunch{

    SceneAppLaunch::SceneAppLaunch(const std::shared_ptr<void>& pArg) NN_NOEXCEPT
    {
        auto p = std::static_pointer_cast<SceneAppLaunchParameter>(pArg);
        m_ApplicationId = p->GetApplicationId();
        m_pIsHideRequested = p->GetHideRequestedPointer();
        NN_ASSERT_NOT_NULL(m_pIsHideRequested);

        m_StartTick = nn::os::GetSystemTick();

        auto resultReadLogoData = CheckReadingApplicationLogoDataImpl();

        m_pPanelContainer = std::make_shared<panel::PanelContainer>();
        m_pPanelContainer->SetVisibility(panel::PanelVisibility::Transparent);
        m_pPanelContainer->SetSize(ScreenWidth, ScreenHeight);
        m_pPanelContainer->SetPanelName("launch");

        m_pHeaderPanel = std::make_shared<panel::PanelText>();
        m_pHeaderPanel->SetColor(nn::util::Color4f(0, 0, 0, 1));
        m_pHeaderPanel->SetPosition(0, 0, 0);
        m_pHeaderPanel->SetSize(1280, 100);

        char titleInfoString[256] = "";
        nn::util::SNPrintf(titleInfoString, sizeof(titleInfoString), "%016llX", m_ApplicationId);

        m_pHeaderPanel->SetTextColor(nn::util::Color4f(1, 1, 1, 1));
        m_pHeaderPanel->SetTextPosition(30, 20);
        m_pHeaderPanel->SetTextSize(20);
        if (resultReadLogoData.IsSuccess())
        {
            m_pHeaderPanel->SetText(std::string("Launching Application: ") + titleInfoString);
        }
        else
        {
            m_pHeaderPanel->SetText(std::string("Launching Application: Logo data corrupted"));
        }

        m_pFooterPanel = std::make_shared<panel::PanelText>();
        m_pFooterPanel->SetColor(nn::util::Color4f(0, 0, 0, 1));
        m_pFooterPanel->SetPosition(0, 720 - 100, 0);
        m_pFooterPanel->SetSize(1280, 100);

        m_pFooterPanel->SetTextColor(nn::util::Color4f(1, 1, 1, 1));
        m_pFooterPanel->SetTextPosition(880, 40);
        m_pFooterPanel->SetTextSize(30);
#if defined(NN_BUILD_APISET_NX)
        m_pFooterPanel->SetText("NintendoSDK NX Addon");
#elif defined(NN_NINTENDO_SDK)
        m_pFooterPanel->SetText("NintendoSDK");
#endif

        m_pPanelContainer->AddChild(m_pHeaderPanel);
        m_pPanelContainer->AddChild(m_pFooterPanel);

        if (!resultReadLogoData.IsSuccess())
        {
            // ロゴデータの破損を検知したらアプリを強制終了
            nn::ae::TerminateApplicationAndSetReason(resultReadLogoData);
        }
    }

    nn::Result SceneAppLaunch::CheckReadingApplicationLogoDataImpl() NN_NOEXCEPT
    {
        // TORIAEZU:
        // バッファが大き目ではあるが、他の表示と共存する必要があるので静的領域に確保しておく。
        static char s_Buffer[LogoDataBufferSize];

        char* buffer = s_Buffer;
        NN_UTIL_SCOPE_EXIT{ std::memset(buffer, 0, LogoDataBufferSize); };

        size_t nintendoLogoDataSize;
        std::memset(buffer, 0, LogoDataBufferSize);
        auto result1 = nn::ns::GetApplicationLogoData(&nintendoLogoDataSize, buffer, NintendoLogoDataSize, m_ApplicationId, "NintendoLogo.png");

        size_t startupMovieDataSize;
        std::memset(buffer, 0, LogoDataBufferSize);
        auto result2 = nn::ns::GetApplicationLogoData(&startupMovieDataSize, buffer, StartupMovieDataSize, m_ApplicationId, "StartupMovie.gif");

        if (result1 <= nn::ns::ResultApplicationLogoCorrupted() ||
            result2 <= nn::ns::ResultApplicationLogoCorrupted())
        {
            return nn::ns::ResultApplicationLogoCorrupted();
        }
        NN_RESULT_SUCCESS;
    }


    nn::TimeSpan SceneAppLaunch::GetShownTimeMin() NN_NOEXCEPT
    {
        return nn::TimeSpan::FromMilliSeconds(1630);
    }

    SceneUpdateResult SceneAppLaunch::Update() NN_NOEXCEPT
    {
        auto currentTick = nn::os::GetSystemTick();
        auto shownTime = (currentTick - m_StartTick).ToTimeSpan();

        if(*m_pIsHideRequested && shownTime >= GetShownTimeMin())
        {
            return SceneUpdateResult::GetPopResult();
        }

        return SceneUpdateResult::GetKeepResult();
    }

    std::shared_ptr<panel::IPanel> SceneAppLaunch::GetPanel() NN_NOEXCEPT
    {
        return m_pPanelContainer;
    }

}}
