﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/util/util_Color.h>
#include "../panel/panel_IPanel.h"
#include "SceneIndex.h"

namespace scene{

    enum SceneSwitchMode
    {
        SceneSwitchMode_Keep,
        SceneSwitchMode_Replace,
        SceneSwitchMode_Push,
        SceneSwitchMode_Pop,
        SceneSwitchMode_Incomplete,
    };

    typedef uint32_t SceneSwitchModeType;

    struct SceneUpdateResult
    {
        SceneSwitchModeType switchMode;
        SceneIndexType nextSceneIndex;
        std::shared_ptr<void> pNextSceneArgument;

        static SceneUpdateResult GetDefault() NN_NOEXCEPT
        {
            auto r = SceneUpdateResult();
            r.switchMode = SceneSwitchMode_Keep;
            r.nextSceneIndex = 0;
            r.pNextSceneArgument = nullptr;
            return r;
        }

        static SceneUpdateResult GetKeepResult() NN_NOEXCEPT
        {
            auto r = GetDefault();
            r.switchMode = SceneSwitchMode_Keep;
            return r;
        }

        static SceneUpdateResult GetReplaceResult(SceneIndex nextIndex, std::shared_ptr<void>&& pArg) NN_NOEXCEPT
        {
            auto r = GetDefault();
            r.switchMode = SceneSwitchMode_Replace;
            r.nextSceneIndex = nextIndex;
            r.pNextSceneArgument = std::move(pArg);
            return r;
        }

        static SceneUpdateResult GetPushResult(SceneIndex nextIndex, std::shared_ptr<void>&& pArg) NN_NOEXCEPT
        {
            auto r = GetDefault();
            r.switchMode = SceneSwitchMode_Push;
            r.nextSceneIndex = nextIndex;
            r.pNextSceneArgument = std::move(pArg);
            return r;
        }

        static SceneUpdateResult GetPopResult() NN_NOEXCEPT
        {
            auto r = GetDefault();
            r.switchMode = SceneSwitchMode_Pop;
            return r;
        }

        static SceneUpdateResult GetIncompleteResult() NN_NOEXCEPT
        {
            auto r = GetDefault();
            r.switchMode = SceneSwitchMode_Incomplete;
            return r;
        }
    };

    class Scene
    {
    public:
        Scene() NN_NOEXCEPT;
        virtual ~Scene() NN_NOEXCEPT;

        // シーンを作った直後に呼ばれる。
        virtual void OnCreated() NN_NOEXCEPT;

        // シーンを破棄する直前に呼ばれる。
        virtual void OnDestroying() NN_NOEXCEPT;

        virtual SceneUpdateResult Update() NN_NOEXCEPT;

        // 下にあるシーンを描画するか。デフォルトは false。
        virtual bool IsTransparent() const NN_NOEXCEPT;
        // IsTransparent() == false の場合に使用される背景のクリア色。デフォルトは無色 (0, 0, 0, 0)。
        virtual nn::util::Color4f GetBackgroundColor() const NN_NOEXCEPT;

        virtual std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT;
    };
}
