﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "panel_PanelFunction.h"

#include <string>
#include <sstream>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../Config.h"
#include "panel_PanelContainer.h"

#ifdef NN_SDK_BUILD_RELEASE
#else
#define NN_DEVOVL_ENABLE_DEVELOPMENTAL_FUNCTION
#endif

#ifdef NN_DEVOVL_ENABLE_DEVELOPMENTAL_FUNCTION
namespace panel{

    namespace {
        const char IndentUnitString[] = " ";
    }

    const char* PanelFunction::GetPanelTypeNameStringForDevelop(PanelTypeType type) NN_NOEXCEPT
    {
        switch(type)
        {
        #define NN_DEVOVL_CASE_PANELTYPE(TYPE) \
        case PanelType_##TYPE: return #TYPE ;

            NN_DEVOVL_FOREACH_PANELTYPE(NN_DEVOVL_CASE_PANELTYPE)

        #undef NN_DEVOVL_CASE_PANELTYPE
        default: return "(invalid)";
        }
    }

    namespace {
        std::string MakePanelDescription(const std::shared_ptr<IPanel>& panel) NN_NOEXCEPT
        {
            if(panel == nullptr)
            {
                return "(null)";
            }

            std::string value;
            value.reserve(256);

            value += [](const std::shared_ptr<IPanel>& panel) -> std::string
            {
                auto name = panel->GetPanelName();

                if(name.empty())
                {
                    std::string s = PanelFunction::GetPanelTypeNameStringForDevelop(panel->GetType());
                    s.resize(4, ' ');
                    return s;
                }

                return name;
            }(panel);

            value += "@";

            value += [](int64_t id) -> std::string
            {
                std::stringstream ss;
                ss << id;
                return ss.str();
            }(panel->GetPanelIdentifier());

            value += [](PanelVisibility v) -> const char*
            {
                switch(v)
                {
                case PanelVisibility::Invisible:   return "I";
                case PanelVisibility::Transparent: return "T";
                case PanelVisibility::Visible:     return "V";
                default: return "-";
                }
            }(panel->GetVisibility());

            value += [](const std::shared_ptr<IPanel>& p) -> const char*
            {
                auto o = PanelFunction::CheckPanelOcclusion(p);
                switch(o)
                {
                case PanelOcclusion_Background: return "B";
                case PanelOcclusion_Foreground: return "F";
                case PanelOcclusion_Partial:    return "P";
                default: return "-";
                }
            }(panel);

            value += [](const std::shared_ptr<IPanel>& p) -> std::string
            {
                std::stringstream ss;
                auto pos = p->GetPosition();
                auto size = p->GetSize();
                ss << "(x" << pos.GetX() << "y" << pos.GetY() << "w" << size.GetX() << "h" << size.GetY() << ")";
                return ss.str();
            }(panel);

            // キャッシュ状態
            value += [](const std::shared_ptr<IPanel>& p) -> std::string
            {
                std::stringstream ss;
                auto c = p->GetRenderingRecycleCache();
                ss << ":";
                if(c.IsValid())
                {
                    ss << (c.IsForegroud() ? "F" : "-");
                }
                else
                {
                    ss << "(invalid)";
                }
                return ss.str();
            }(panel);

            return value;
        }
    }

    void PanelFunction::PrintPanelTreeForDevelop(const std::shared_ptr<IPanel>& panel, int indent) NN_NOEXCEPT
    {
        std::string indentString;
        for(int i = 0; i < indent; i++)
        {
            indentString += IndentUnitString;
        }

        std::string description = MakePanelDescription(panel);

        NN_DEVOVL_PANEL_LOG_DUMP("%s%s\n", indentString.c_str(), description.c_str());

        if(auto pContainter = PanelContainer::IsContainer(panel))
        {
            int n = pContainter->GetChildrenCount();
            for(int i = 0; i < n; i++)
            {
                auto pChild = pContainter->GetChild(i);
                PrintPanelTreeForDevelop(pChild, indent + 1);
            }
        }
    }
}

#else

// Release 用
namespace panel{
    const char* PanelFunction::GetPanelTypeNameStringForDevelop(PanelTypeType) NN_NOEXCEPT
    {
        return "";
    }

    void PanelFunction::PrintPanelTreeForDevelop(const std::shared_ptr<IPanel>&, int) NN_NOEXCEPT
    {
    }
}

#endif


