﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "panel_Drawer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../../framework/Framework.h"
#include "../panel_PanelFunction.h"

namespace panel{ namespace detail{

    namespace {
        // 頂点シェーダの ConstantBuffer 。
        // PanelVertexShader.glsl に合わせる。
        struct VertexConstants
        {
            //float position[4];
            float textureRegion[4];
            //float size[2];
            int isFlipYEnabled;
        };

        // ピクセルシェーダの ConstantBuffer 。
        // PanelPixelShader.glsl に合わせる。
        struct PixelConstants
        {
            float color[4];
            int isTextureEnabled;
        };

        class DrawingResource{
        public:
            explicit DrawingResource(Drawer* pDrawer) NN_NOEXCEPT
            {
                {
                    nn::gfx::BufferInfo info;
                    info.SetDefault();
                    info.SetSize(sizeof(VertexConstants));
                    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
                    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
                    {
                        size_t size = sizeof(VertexConstants);
                        size_t alignment = nn::gfx::Buffer::GetBufferAlignment(Framework::GetDevice(), info);
                        ptrdiff_t offset = Framework::AllocateConstantBufferPoolMemory(size, alignment);
                        m_VertexConstantBuffer.Initialize(Framework::GetDevice(), info, Framework::GetConstantBufferMemoryPool(), offset, size);
                        m_VertexConstantOffset = offset;
                    }
                    else
                    {
                        m_VertexConstantBuffer.Initialize(Framework::GetDevice(), info, nullptr, 0, 0);
                        m_VertexConstantOffset = -1;
                    }
                }
                {
                    nn::gfx::BufferInfo info;
                    info.SetDefault();
                    info.SetSize(sizeof(PixelConstants));
                    info.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
                    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
                    {
                        size_t size = sizeof(PixelConstants);
                        size_t alignment = nn::gfx::Buffer::GetBufferAlignment(Framework::GetDevice(), info);
                        ptrdiff_t offset = Framework::AllocateConstantBufferPoolMemory(size, alignment);
                        m_PixelConstantBuffer.Initialize(Framework::GetDevice(), info, Framework::GetConstantBufferMemoryPool(), offset, size);
                        m_PixelConstantOffset = offset;
                    }
                    else
                    {
                        m_PixelConstantBuffer.Initialize(Framework::GetDevice(), info, nullptr, 0, 0);
                        m_PixelConstantOffset = -1;
                    }
                }
                m_ObjectId = pDrawer->AcquireObjectIdForDebug();
            };

            ~DrawingResource() NN_NOEXCEPT
            {
                m_VertexConstantBuffer.Finalize(Framework::GetDevice());
                m_PixelConstantBuffer.Finalize(Framework::GetDevice());

                if(m_VertexConstantOffset >= 0)
                {
                    Framework::FreeConstantBufferPoolMemory(m_VertexConstantOffset);
                }
                if(m_PixelConstantOffset >= 0)
                {
                    Framework::FreeConstantBufferPoolMemory(m_PixelConstantOffset);
                }
            };

            nn::gfx::Buffer* GetVertexConstantBuffer() NN_NOEXCEPT
            {
                return &m_VertexConstantBuffer;
            }

            nn::gfx::Buffer* GetPixelConstantBuffer() NN_NOEXCEPT
            {
                return &m_PixelConstantBuffer;
            }

            VertexConstants* MapVertexConstants() NN_NOEXCEPT
            {
                return m_VertexConstantBuffer.Map<VertexConstants>();
            }

            void UnmapVertexConstants() NN_NOEXCEPT
            {
                m_VertexConstantBuffer.FlushMappedRange(0, sizeof(VertexConstants));
                m_VertexConstantBuffer.Unmap();
            }

            PixelConstants* MapPixelConstants() NN_NOEXCEPT
            {
                return m_PixelConstantBuffer.Map<PixelConstants>();
            }

            void UnmapPixelConstants() NN_NOEXCEPT
            {
                m_PixelConstantBuffer.FlushMappedRange(0, sizeof(PixelConstants));
                m_PixelConstantBuffer.Unmap();
            }

            void SetTexture(const std::shared_ptr<framework::ITextureView> value) NN_NOEXCEPT
            {
                m_Texture = value;
            }

        private:
            nn::gfx::Buffer m_VertexConstantBuffer;
            nn::gfx::Buffer m_PixelConstantBuffer;
            ptrdiff_t m_VertexConstantOffset;
            ptrdiff_t m_PixelConstantOffset;
            std::shared_ptr<framework::ITextureView> m_Texture;
            int m_ObjectId; // for debug
        };
    }

    void Drawer::DrawWithShader(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const PanelRectangle& rect,
        const PanelRectangle& cropRect,
        const nn::util::Color4f& color,
        const std::shared_ptr<framework::ITextureView>& pTexture,
        bool isAlphaBlendEnabled
    ) NN_NOEXCEPT
    {
        auto srect = rect.Product(cropRect);
        auto pTexDescriptor = pTexture ? pTexture->GetTextureDescriptor() : nullptr;
        bool isTextureValid = pTexture && pTexDescriptor;

        NN_DEVOVL_PANEL_LOG_DRAW("shader rect(%d,%d;%dx%d) crop(%d,%d;%dx%d)\n", rect.x, rect.y, rect.width, rect.height, srect.x, srect.y, srect.width, srect.height);
        if(srect.IsEmpty())
        {
            return;
        }

        Framework::MakeSetViewportCommand(pCommandBuffer, rect);
        Framework::MakeSetScissorCommand(pCommandBuffer, srect);
        pCommandBuffer->SetShader(m_CommonPanelShader.GetShader(0), nn::gfx::ShaderStageBit_All);
        pCommandBuffer->SetRasterizerState(&m_CommonRasterizerState);
        if(isAlphaBlendEnabled)
        {
            pCommandBuffer->SetBlendState(m_AlphaBlendState.Get());
        }
        else
        {
            pCommandBuffer->SetBlendState(m_SolidBlendState.Get());
        }
        pCommandBuffer->SetDepthStencilState(&m_CommonDepthStencilState);

        auto pDrawResource = std::make_shared<DrawingResource>(this);
        Framework::LockFrameCommandResource(pDrawResource); // 描画が終わるまで Framework が参照カウントを取っておく

        {
            auto p = pDrawResource->MapVertexConstants();
            p->textureRegion[0] = 0;
            p->textureRegion[1] = 0;
            p->textureRegion[2] = 0;
            p->textureRegion[3] = 0;
            if(isTextureValid)
            {
                float poolWidth  = static_cast<float>(pTexture->GetPoolWidth());
                float poolHeight = static_cast<float>(pTexture->GetPoolHeight());
                float x = static_cast<float>(pTexture->GetViewOriginX());
                float y = static_cast<float>(pTexture->GetViewOriginY());
                float w = static_cast<float>(pTexture->GetViewWidth());
                float h = static_cast<float>(pTexture->GetViewHeight());
                p->textureRegion[0] = /*0*/ x / poolWidth;
                p->textureRegion[1] = /*0*/ y / poolHeight;
                p->textureRegion[2] = /*1*/ w / poolWidth;
                p->textureRegion[3] = /*1*/ h / poolHeight;
            }
            p->isFlipYEnabled = 0; // panel->GetFlipYEnabled() ? 1 : 0;
            pDrawResource->UnmapVertexConstants();
            nn::gfx::GpuAddress a;
            pDrawResource->GetVertexConstantBuffer()->GetGpuAddress(&a);
            pCommandBuffer->SetConstantBuffer(0, nn::gfx::ShaderStage_Vertex, a, sizeof(VertexConstants));
        }

        {
            auto p = pDrawResource->MapPixelConstants();
            p->color[0] = color.GetR();
            p->color[1] = color.GetG();
            p->color[2] = color.GetB();
            p->color[3] = color.GetA();
            p->isTextureEnabled = isTextureValid ?  1 : 0;
            pDrawResource->UnmapPixelConstants();
            nn::gfx::GpuAddress a;
            pDrawResource->GetPixelConstantBuffer()->GetGpuAddress(&a);
            pCommandBuffer->SetConstantBuffer(0, nn::gfx::ShaderStage_Pixel, a, sizeof(PixelConstants));
            NN_DEVOVL_PANEL_LOG_DRAW("  color(%2f,%2f,%2f;%2f)\n", color.GetR(), color.GetG(), color.GetB(), color.GetA());
        }

        if(isTextureValid)
        {
            pDrawResource->SetTexture(pTexture);
            pCommandBuffer->SetTextureAndSampler(0, nn::gfx::ShaderStage_Pixel, *pTexDescriptor, Framework::GetDefaultSamplerDescriptorSlot());
        }

        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_ConstantBuffer); // フレームの頭で 1 回だけやればいい気がするが…
        pCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, 0);
    }

}}

