﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Allocator.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DescriptorPoolAllocator.h>
#include <nn/gfx/util/gfx_MemoryPoolAllocator.h>

namespace nns { namespace gfx {

//! @brief フレームワークで使用する Reallocate 関数の型です。
typedef void* (*ReallocateFunctionWithUserData)(void* addr, size_t newSize, void* pUserData);

//! @brief フレームワークで管理するメモリプールのタイプです。
//! @details
//! 最適化のため、できる限り用途に合ったタイプを選択することを推奨します。@n
//! MemoryPoolType_Others は速度やメモリ効率を考慮しなくてよい場合に限り使用するようにしてください。
enum MemoryPoolType
{
    MemoryPoolType_CommandBuffer,   // CpuUncached_GpuUncached
    MemoryPoolType_RenderTarget,    // CpuInvisible_GpuCached_Compressible
    MemoryPoolType_ConstantBuffer,  // CpuUncached_GpuCached
    MemoryPoolType_Shader,          // CpuCached_GpuCached_ShaderCode
    MemoryPoolType_Data,            // CpuCached_GpuCached
    MemoryPoolType_Others,          // CpuCached_GpuCached_Compressible_ShaderCode
    MemoryPoolType_End
};

//! @brief フレームワークで管理する頂点ステートのタイプです。
enum VertexStateType
{
    VertexStateType_Float3_Float2,
    VertexStateType_End
};

//! @brief フレームワークで管理するラスタライザステートのタイプです。
enum RasterizerStateType
{
    RasterizerStateType_FillSolid_CullNone,
    RasterizerStateType_End
};

//! @brief フレームワークで管理するブレンドステートのタイプです。
enum BlendStateType
{
    BlendStateType_Disabled,
    BlendStateType_Alpha,
    BlendStateType_End
};

//! @brief フレームワークで管理する深度ステンシルステートのタイプです。
enum DepthStencilStateType
{
    DepthStencilStateType_Disabled,
    DepthStencilStateType_End
};

//! @brief フレームワークで管理するサンプラのタイプです。
enum SamplerType
{
    SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Repeat,
    SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Mirror,
    SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Clamp,
    SamplerType_FilterMode_MinLinear_MagLinear_MipPoint_AddressMode_Border,
    SamplerType_End
};

//! @brief フレームワークで管理するコマンドバッファのラッパーです。
struct CommandBuffer
{
    nn::gfx::CommandBuffer object;
    ptrdiff_t commandMemoryOffset;
    size_t commandMemorySize;
    void* pControlMemory;
    size_t controlMemorySize;
};

//------------------------------------------------------------------------------
//! @brief フレームワークを初期化するための情報を表すクラスです。
//------------------------------------------------------------------------------
class FrameworkInfo
{
public:
    //! @brief デフォルトコンストラクタです。
    FrameworkInfo() NN_NOEXCEPT
    {
        SetDefault();
    }

    //! @brief 各パラメータを既定値に設定するためのヘルパー関数です。
    //! @details 設定される値はゼロ初期化とは異なります。
    void SetDefault() NN_NOEXCEPT
    {
        SetDisplayWidth(0);
        SetDisplayHeight(0);

        SetMemoryPoolSize(MemoryPoolType_CommandBuffer, 16 * 1024 * 1024);
        SetMemoryPoolSize(MemoryPoolType_RenderTarget, 16 * 1024 * 1024);
        SetMemoryPoolSize(MemoryPoolType_ConstantBuffer, 16 * 1024 * 1024);
        SetMemoryPoolSize(MemoryPoolType_Shader, 8 * 1024 * 1024);
        SetMemoryPoolSize(MemoryPoolType_Data, 8 * 1024 * 1024);
        SetMemoryPoolSize(MemoryPoolType_Others, 0);

        SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_BufferView, 0);
        SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_TextureView, 4096);
        SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_Sampler, 4096);

        SetBufferCount(1);
        SetRootCommandBufferCommandMemorySize(8 * 1024 * 1024);
        SetRootCommandBufferControlMemorySize(1 * 1024 * 1024);

        m_ColorBufferFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;
        m_DepthStencilBufferFormat = nn::gfx::ImageFormat_D32_Float;

        SetSwapChainBufferCount(2);
        SetSwapChainFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);

        SetDebugMode(nn::gfx::DebugMode_Disable);
    }

    //! @brief ディスプレイ幅を設定します。
    //! @param[in]  displayWidth  ディスプレイ幅です。
    void SetDisplayWidth(int displayWidth) NN_NOEXCEPT
    {
        m_DisplayWidth = displayWidth;
    }

    //! @brief ディスプレイ幅を取得します。
    //! @return ディスプレイ幅を返します。
    int GetDisplayWidth() const NN_NOEXCEPT
    {
        return m_DisplayWidth;
    }

    //! @brief ディスプレイ高さを設定します。
    //! @param[in]  displayHeight  ディスプレイ高さです。
    void SetDisplayHeight(int displayHeight) NN_NOEXCEPT
    {
        m_DisplayHeight = displayHeight;
    }

    //! @brief ディスプレイ高さを取得します。
    //! @return ディスプレイ高さを返します。
    int GetDisplayHeight() const NN_NOEXCEPT
    {
        return m_DisplayHeight;
    }

    //! @brief バッファのマルチバッファリングの数を設定します。
    //! @param[in]  bufferCount  マルチバッファリングする数です。
    void SetBufferCount(int bufferCount) NN_NOEXCEPT
    {
        m_BufferCount = bufferCount;
    }

    //! @brief バッファのマルチバッファリングの数を取得します。
    //! @return マルチバッファリングする数を返します。
    int GetBufferCount() const NN_NOEXCEPT
    {
        return m_BufferCount;
    }

    //! @brief スワップチェーンのフォーマットを設定します。
    //! @param[in]  format  設定するスワップチェーンのフォーマットです。
    void SetSwapChainFormat(nn::gfx::ImageFormat format) NN_NOEXCEPT
    {
        m_SwapChainFormat = format;
    }

    //! @brief スワップチェーンのフォーマットを取得します。
    //! @return 設定されたフォーマットを返します。
    nn::gfx::ImageFormat GetSwapChainFormat() const NN_NOEXCEPT
    {
        return m_SwapChainFormat;
    }

    //! @brief スワップチェーンのバッファ数を設定します。
    //! @param[in]  bufferCount  バッファの数です。
    void SetSwapChainBufferCount(int bufferCount) NN_NOEXCEPT
    {
        m_SwapChainBufferCount = bufferCount;
    }

    //! @brief スワップチェーンのバッファ数を取得します。
    //! @return バッファの数を返します。
    int GetSwapChainBufferCount() const NN_NOEXCEPT
    {
        return m_SwapChainBufferCount;
    }

    //! @brief メモリプールのサイズを設定します。
    //! @param[in]  type  メモリプールのタイプです。
    //! @param[in]  size  バイトでのメモリプールのサイズです。
    void SetMemoryPoolSize(MemoryPoolType type, size_t size) NN_NOEXCEPT
    {
        m_MemoryPoolSize[type] = size;
    }

    //! @brief メモリプールのサイズを取得します。
    //! @param[in]  type  メモリプールのタイプです。
    //! @return バイトでのメモリプールのサイズを返します。
    size_t GetMemoryPoolSize(MemoryPoolType type) const NN_NOEXCEPT
    {
        return m_MemoryPoolSize[type];
    }

    //! @brief デスクリプタプールのスロット数を設定します。
    //! @param[in]  type  デスクリプタプールのタイプです。
    //! @param[in]  slotCount  スロット数です。
    void SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType type, int slotCount) NN_NOEXCEPT
    {
        m_DescriptorPoolSlotCount[type] = slotCount;
    }

    //! @brief デスクリプタプールのスロット数を取得します。
    //! @param[in]  type  デスクリプタプールのタイプです。
    //! @return スロット数を返します。
    int GetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType type) const NN_NOEXCEPT
    {
        return m_DescriptorPoolSlotCount[type];
    }

    //! @brief ルートコマンドバッファのコマンドメモリサイズを設定します。
    //! @param[in]  commandMemorySize  バイトでのコマンドメモリサイズです。
    void SetRootCommandBufferCommandMemorySize(size_t commandMemorySize) NN_NOEXCEPT
    {
        m_RootCommandBufferCommandMemorySize = commandMemorySize;
    }

    //! @brief ルートコマンドバッファのコマンドメモリサイズを取得します。
    //! @return バイトでのコマンドメモリサイズを返します。
    size_t GetRootCommandBufferCommandMemorySize() const NN_NOEXCEPT
    {
        return m_RootCommandBufferCommandMemorySize;
    }

    //! @brief ルートコマンドバッファのコントロールメモリサイズを設定します。
    //! @param[in]  commandMemorySize  バイトでのコントロールメモリサイズです。
    void SetRootCommandBufferControlMemorySize(size_t controlMemorySize) NN_NOEXCEPT
    {
        m_RootCommandBufferControlMemorySize = controlMemorySize;
    }

    //! @brief ルートコマンドバッファのコントロールメモリサイズを取得します。
    //! @return バイトでのコントロールメモリサイズを返します。
    size_t GetRootCommandBufferControlMemorySize() const NN_NOEXCEPT
    {
        return m_RootCommandBufferControlMemorySize;
    }

    //! @brief カラーバッファのフォーマットを設定します。
    //! @param[in]  format  設定するカラーバッファのフォーマットです。
    void SetColorBufferFormat(nn::gfx::ImageFormat format) NN_NOEXCEPT
    {
        m_ColorBufferFormat = format;
    }

    //! @brief カラーバッファのフォーマットを取得します。
    //! @return 設定されたカラーバッファフォーマットを返します。
    nn::gfx::ImageFormat GetColorBufferFormat() const NN_NOEXCEPT
    {
        return m_ColorBufferFormat;
    }

    //! @brief デプスステンシルバッファのフォーマットを設定します。
    //! @param[in]  format  設定するデプスステンシルバッファのフォーマットです。
    void SetDepthStencilBufferFormat(nn::gfx::ImageFormat format) NN_NOEXCEPT
    {
        m_DepthStencilBufferFormat = format;
    }

    //! @brief デプスステンシルバッファのフォーマットを取得します。
    //! @return 設定されたデプスステンシルバッファフォーマットを返します。
    nn::gfx::ImageFormat GetDepthStencilBufferFormat() const NN_NOEXCEPT
    {
        return m_DepthStencilBufferFormat;
    }

    //! @brief デバッグモードを設定します。
    //! @param[in]  value  デバッグモード
    void SetDebugMode(nn::gfx::DebugMode value) NN_NOEXCEPT
    {
        m_DebugMode = value;
    }

    //! @brief デバッグモードを取得します。
    //! @return 設定されたデバッグモードを返します。
    nn::gfx::DebugMode GetDebugMode() const NN_NOEXCEPT
    {
        return m_DebugMode;
    }

private:
    int m_DisplayWidth;
    int m_DisplayHeight;
    size_t m_MemoryPoolSize[MemoryPoolType_End];
    int m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_End];
    int m_BufferCount;
    int m_SwapChainBufferCount;
    nn::gfx::ImageFormat m_SwapChainFormat;
    size_t m_RootCommandBufferCommandMemorySize;
    size_t m_RootCommandBufferControlMemorySize;
    nn::gfx::ImageFormat m_ColorBufferFormat;
    nn::gfx::ImageFormat m_DepthStencilBufferFormat;
    nn::gfx::DebugMode m_DebugMode;
};



//------------------------------------------------------------------------------
//! @brief グラフィックスフレームワーククラスです。
//! @details 描画に必要な GFX の手続きを単純化します。
//------------------------------------------------------------------------------
class GraphicsFramework
{
public:
    typedef nns::gfx::MemoryPoolType MemoryPoolType;
    typedef nns::gfx::VertexStateType VertexStateType;
    typedef nns::gfx::RasterizerStateType RasterizerStateType;
    typedef nns::gfx::BlendStateType BlendStateType;
    typedef nns::gfx::DepthStencilStateType DepthStencilStateType;
    typedef nns::gfx::SamplerType SamplerType;
    typedef nns::gfx::CommandBuffer CommandBuffer;
    typedef nns::gfx::FrameworkInfo FrameworkInfo;

    //------------------------------------------------------------------------------
    // 初期化・終了処理および初期化パラメータ取得
    //------------------------------------------------------------------------------

    //! @brief デフォルトコンストラクタです。
    GraphicsFramework() NN_NOEXCEPT;

    //! @brief グラフィックスシステムを初期化します。
    //! @param[in]    graphicsSystemMemorySize  グラフィックスシステムに渡すメモリのサイズです。
    //! @param[in]    pAllocateFunction         メモリ確保用コールバック関数です。
    //! @param[in]    pFreeFunction             メモリ解放用コールバック関数です。
    //! @param[in]    pReallocateFunction       メモリ再確保用コールバック関数です。
    //! @param[in]    pAllocateFunctionUserData メモリ確保、再確保、解放時に呼ばれるユーザ定義のパラメータです。
    static void InitializeGraphicsSystem(
        size_t graphicsSystemMemorySize,
        nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
        nn::FreeFunctionWithUserData pFreeFunction,
        ReallocateFunctionWithUserData pReallocateFunction,
        void* pAllocateFunctionUserData
    ) NN_NOEXCEPT;

    //! @brief グラフィックスシステムを初期化します。
    //! @param[in]    graphicsSystemMemorySize  グラフィックスシステムに渡すメモリのサイズです。
    static void InitializeGraphicsSystem(
        size_t graphicsSystemMemorySize
    ) NN_NOEXCEPT;

    //! @brief 初期化します。
    //! @param[in]    info                      初期化用の情報です。
    //! @param[in]    pAllocateFunction         メモリ確保用コールバック関数です。
    //! @param[in]    pFreeFunction             メモリ解放用コールバック関数です。
    //! @param[in]    pAllocateFunctionUserData メモリ確保、解放時に呼ばれるユーザ定義のパラメータです。
    void Initialize(
        const FrameworkInfo& info,
        nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
        nn::FreeFunctionWithUserData pFreeFunction,
        void* pAllocateFunctionUserData
    ) NN_NOEXCEPT;

    //! @brief 初期化します。
    //! @param[in]    info                      初期化用の情報です。
    void Initialize(
        const FrameworkInfo& info
    ) NN_NOEXCEPT;

    //! @brief 終了処理です。
    void Finalize() NN_NOEXCEPT;

    //! @brief このオブジェクトが初期化済かを返します。
    //! @return 初期化済の場合 true を返します。それ以外の場合 false を返します。
    //! @details
    //! Initialize() により初期化済になります。
    //! Finalize() により未初期化になります。
    bool IsInitialized() const NN_NOEXCEPT;

    //! @brief ディスプレイ幅を取得します。
    //! @return ディスプレイ幅を返します。
    int GetDisplayWidth() const NN_NOEXCEPT;

    //! @brief ディスプレイ高さを取得します。
    //! @return ディスプレイ高さを返します。
    int GetDisplayHeight() const NN_NOEXCEPT;

    //! @brief バッファのマルチバッファリングの数を取得します。
    //! @return マルチバッファリングする数を返します。
    int GetBufferCount() const NN_NOEXCEPT;

    //! @brief メモリ確保用コールバック関数を取得します。
    //! @return メモリ確保用コールバック関数を返します。
    nn::AlignedAllocateFunctionWithUserData GetAllocateFunction() const NN_NOEXCEPT;

    //! @brief メモリ解放用コールバック関数を取得します。
    //! @return メモリ解放用コールバック関数を返します。
    nn::FreeFunctionWithUserData GetFreeFunction() const NN_NOEXCEPT;

    //! @brief メモリ確保時に呼ばれるユーザ定義のパラメータを取得します。
    //! @return メモリ確保時に呼ばれるユーザ定義のパラメータを返します。
    void* GetAllocateFunctionUserData() const NN_NOEXCEPT;

    //! @brief メモリ領域を確保します。
    //! @param[in]    size       確保したいメモリのサイズです。
    //! @param[in]    alignment  確保するメモリのアライメントです。
    //! @return 確保したメモリの先頭へのポインタが返ります。
    //! @details
    //! 初期化時に渡したメモリ確保関数とユーザ定義のパラメータを使用します。
    void* AllocateMemory(size_t size, size_t alignment) NN_NOEXCEPT;

    //! @brief メモリ領域を解放します。
    //! @param[in]    ptr       開放するメモリ領域の先頭アドレスです。
    //! @details
    //! 初期化時に渡したメモリ解放関数とユーザ定義のパラメータを使用します。
    void FreeMemory(void* ptr) NN_NOEXCEPT;

    //------------------------------------------------------------------------------
    // 組み込み GFX オブジェクト取得
    // フレームワークが内部で生成した GFX オブジェクトを取得・利用できます。
    //------------------------------------------------------------------------------

    //! @brief VI ディスプレイを取得します。
    //! @return VI ディスプレイを返します。
    nn::vi::Display* GetDisplay() NN_NOEXCEPT;

    //! @brief VI レイヤーを取得します。
    //! @return VI レイヤーを返します。
    nn::vi::Layer* GetLayer() NN_NOEXCEPT;

    //! @brief デバイスを取得します。
    //! @return デバイスを返します。
    nn::gfx::Device* GetDevice() NN_NOEXCEPT;

    //! @brief キューを取得します。
    //! @return キューを返します。
    nn::gfx::Queue* GetQueue() NN_NOEXCEPT;

    //! @brief スワップチェーンを取得します。
    //! @return スワップチェーンを返します。
    nn::gfx::SwapChain* GetSwapChain() NN_NOEXCEPT;

    //! @brief フェンスを取得します。
    //! @return フェンスを返します。
    nn::gfx::Fence* GetFence() NN_NOEXCEPT;

    //! @brief ビューポートシザーステートを取得します。
    //! @return ビューポートシザーステートを返します。
    nn::gfx::ViewportScissorState* GetViewportScissorState() NN_NOEXCEPT;

    //! @brief 頂点ステートを取得します。
    //! @param[in]    type 頂点ステートのタイプです。
    //! @return 頂点ステートを返します。
    nn::gfx::VertexState* GetVertexState(VertexStateType type) NN_NOEXCEPT;

    //! @brief ラスタライザステートを取得します。
    //! @param[in]    type ラスタライザステートのタイプです。
    //! @return ラスタライザステートを返します。
    nn::gfx::RasterizerState* GetRasterizerState(RasterizerStateType type) NN_NOEXCEPT;

    //! @brief ブレンドステートを取得します。
    //! @param[in]    type ブレンドステートのタイプです。
    //! @return ブレンドステートを返します。
    nn::gfx::BlendState* GetBlendState(BlendStateType type) NN_NOEXCEPT;

    //! @brief 深度ステンシルステートを取得します。
    //! @param[in]    type 深度ステンシルステートのタイプです。
    //! @return 深度ステンシルステートを返します。
    nn::gfx::DepthStencilState* GetDepthStencilState(DepthStencilStateType type) NN_NOEXCEPT;

    //! @brief サンプラを取得します。
    //! @param[in]    type サンプラのタイプです。
    //! @return サンプラを返します。
    nn::gfx::Sampler* GetSampler(SamplerType type) NN_NOEXCEPT;

    //---------------------------------------
    // カラーバッファ, 深度ステンシルバッファ
    //---------------------------------------

    //! @brief カラーターゲットビューを取得します。
    //! @return カラーターゲットビューを返します。
    nn::gfx::ColorTargetView* GetColorTargetView() NN_NOEXCEPT;

#if 0
    //! @brief 深度ステンシルバッファを取得します。
    //! @return 深度ステンシルバッファを返します。
    nn::gfx::Texture* GetDepthStencilBuffer() NN_NOEXCEPT;

    //! @brief 深度ステンシルビューを取得します。
    //! @return 深度ステンシルビューを返します。
    nn::gfx::DepthStencilView* GetDepthStencilView() NN_NOEXCEPT;
#endif

    //---------------------------------------
    // メモリプール
    //---------------------------------------

    //! @brief メモリプールを取得します。
    //! @param[in]    type メモリプールのタイプです。
    //! @return メモリプールを返します。
    nn::gfx::MemoryPool* GetMemoryPool(MemoryPoolType type) NN_NOEXCEPT;

    //! @brief メモリプールアロケータを取得します。
    //! @param[in]    type メモリプールのタイプです。
    //! @return メモリプールアロケータを返します。
    nn::gfx::util::MemoryPoolAllocator* GetMemoryPoolAllocator(MemoryPoolType type) NN_NOEXCEPT;

    //! @brief メモリプールの区間を確保します。
    //! @param[in]    type メモリプールのタイプです。
    //! @param[in]    size 確保したい区間のサイズです。
    //! @param[in]    alignment 確保する区間のアライメントです。
    //! @return 確保した区間の先頭のオフセットが返ります。
    ptrdiff_t AllocatePoolMemory(MemoryPoolType type, size_t size, size_t alignment) NN_NOEXCEPT;

    //! @brief 確保したメモリプールの区間を解放します。
    //! @param[in]    type メモリプールのタイプです。
    //! @param[in]    offset 解放する区間の先頭オフセットです。
    void FreePoolMemory(MemoryPoolType type, ptrdiff_t offset) NN_NOEXCEPT;

    //---------------------------------------
    // デスクリプタプール
    //---------------------------------------

    //! @brief デスクリプタプールを取得します。
    //! @param[in]    type デスクリプタプールのタイプです。
    //! @return デスクリプタプールを返します。
    nn::gfx::DescriptorPool* GetDescriptorPool(nn::gfx::DescriptorPoolType type) NN_NOEXCEPT;

    //! @brief デスクリプタプールアロケータを取得します。
    //! @param[in]    type デスクリプタプールのタイプです。
    //! @return デスクリプタプールアロケータを返します。
    nn::gfx::util::DescriptorPoolAllocator* GetDescriptorPoolAllocator(nn::gfx::DescriptorPoolType type) NN_NOEXCEPT;

    //! @brief デスクリプタプールの区間を確保します。
    //! @param[in]    type デスクリプタプールのタイプです。
    //! @param[in]    count 確保したい区間の長さです。
    //! @return 確保した区間の先頭のインデックスが返ります。
    int AllocateDescriptorSlot(nn::gfx::DescriptorPoolType type, int count) NN_NOEXCEPT;

    //! @brief 確保したデスクリプタプールの区間を解放します。
    //! @param[in]    type デスクリプタプールのタイプです。
    //! @param[in]    indexSlot 解放する区間の先頭インデックスです。
    void FreeDescriptorSlot(nn::gfx::DescriptorPoolType type, int indexSlot) NN_NOEXCEPT;

    //! @brief バッファデスクリプタプールの指定されるスロットにバッファビューを設定します。
    //! @param[in]    indexSlot 対象のスロット番号です。
    //! @param[in]    gpuAddress 対象のバッファの GPU アドレスです。
    //! @param[in]    size 設定するバッファのサイズです。
    void SetBufferViewToDescriptorPool(int indexSlot, const nn::gfx::GpuAddress& gpuAddress, size_t size);

    //! @brief テクスチャデスクリプタプールの指定されるスロットにテクスチャビューを設定します。
    //! @param[in]    indexSlot 対象のスロット番号です。
    //! @param[in]    pTextureView 対象のテクスチャビューです。
    void SetTextureViewToDescriptorPool(int indexSlot, const nn::gfx::TextureView* pTextureView);

    //! @brief サンプラデスクリプタプールの指定されるスロットにサンプラを設定します。
    //! @param[in]    indexSlot 対象のスロット番号です。
    //! @param[in]    pSampler 対象のテクスチャビューです。
    void SetSamplerToDescriptorPool(int indexSlot, const nn::gfx::Sampler* pSampler);

    //------------------------------------------------------------------------------
    // フレームワーク操作
    //------------------------------------------------------------------------------

    //! @brief ルートコマンドバッファを取得します。
    //! @param[in]    bufferIndex バッファのインデックスです。
    //! @return ルートコマンドバッファを返します。
    nn::gfx::CommandBuffer* GetRootCommandBuffer(int bufferIndex) NN_NOEXCEPT;

    //! @brief シェーダスクラッチメモリのサイズを設定します。
    //! @param[in]    size バイトでのメモリサイズです。
    void SetShaderScratchMemorySize(size_t size) NN_NOEXCEPT;

    //! @brief ルートコマンドバッファへのコマンドの追加を開始します。
    //! @param[in]    bufferIndex バッファのインデックスです。
    void BeginFrame(int bufferIndex) NN_NOEXCEPT;

    //! @brief ルートコマンドバッファへのコマンドの追加を終了します。
    //! @param[in]    bufferIndex バッファのインデックスです。
    void EndFrame(int bufferIndex) NN_NOEXCEPT;

    //! @brief キューへルートコマンドバッファを提出します。
    //! @param[in]    bufferIndex バッファのインデックスです。
    void ExecuteCommand(int bufferIndex) NN_NOEXCEPT;

    //! @brief スワップチェーンの内容をディスプレイに描写します。
    void Present() NN_NOEXCEPT;

    //------------------------------------------------------------------------------
    // 外部 GFX オブジェクトの初期化関数
    // 組み込み GFX オブジェクトとは別に、以下の関数を使用して GFX オブジェクトを生成・使用することができます。
    //------------------------------------------------------------------------------

    //! @brief コマンドバッファを初期化します。
    //! @param[out]   pOutCommandBuffer 初期化したコマンドバッファを返します。
    //! @param[in]    info 初期化用の情報です。
    //! @param[in]    commandMemorySize バイトでのコマンドメモリサイズです。
    //! @param[in]    controlMemorySize バイトでのコントロールメモリサイズです。
    void InitializeCommandBuffer(GraphicsFramework::CommandBuffer* pOutCommandBuffer, nn::gfx::CommandBuffer::InfoType& info, size_t commandMemorySize, size_t controlMemorySize) NN_NOEXCEPT;

    //! @brief コマンドバッファを破棄します。
    //! @param[in]    pCommandBuffer 破棄するコマンドバッファです。
    void FinalizeCommandBuffer(GraphicsFramework::CommandBuffer* pCommandBuffer) NN_NOEXCEPT;

    //! @brief コマンドメモリとコントロールメモリを設定します。
    //! @param[in]    pCommandBuffer 設定するコマンドバッファです。
    void ResetCommandBuffer(GraphicsFramework::CommandBuffer* pCommandBuffer) NN_NOEXCEPT;

    //------------------------------------------------------------------------------
    // デフォルトのメモリアロケータ
    //------------------------------------------------------------------------------

    //! @brief アライメントを指定してメモリ領域を確保します。
    //! @param[in]    size      確保するメモリのサイズです。
    //! @param[in]    alignment 確保するメモリのアライメントです。
    //! @param[in]    pUserData ユーザ定義のパラメータです。
    //! @return 確保したメモリの先頭へのポインタが返ります。
    static void* DefaultAllocateFunction(size_t size, size_t alignment, void* pUserData);

    //! @brief メモリ領域を解放します。
    //! @param[in]    ptr       開放するメモリ領域の先頭アドレスです。
    //! @param[in]    pUserData ユーザ定義のパラメータです。
    static void DefaultFreeFunction(void* ptr, void* pUserData);

    //! @brief メモリ領域を指定したサイズで再確保します。
    //! @param[in]    ptr       確保済みメモリ領域の先頭アドレスです。
    //! @param[in]    newSize   新しく割り当てるメモリのサイズです。
    //! @param[in]    pUserData ユーザ定義のパラメータです。
    //! @return 再確保したメモリの先頭へのポインタが返ります。
    static void* DefaultReallocateFunction(void* ptr, size_t newSize, void* pUserData);

private:
    struct MemoryPool
    {
        nn::gfx::MemoryPool object;
        nn::gfx::util::MemoryPoolAllocator allocator;
        void* pMemory;
        size_t size;
    };

    struct VertexState
    {
        nn::gfx::VertexState object;
        void* pMemory;
    };

    struct BlendState
    {
        nn::gfx::BlendState object;
        void* pMemory;
    };

    struct ShaderScratchMemoryInfo
    {
        ptrdiff_t offset;
        size_t size;
    };

    static void* MemoryPoolAllocatorMalloc(size_t size, void* pUserData);
    static void MemoryPoolAllocatorFree(void* ptr, void* pUserData);
    static void* DescriptorPoolAllocatorMalloc(size_t size, void* pUserData);
    static void DescriptorPoolAllocatorFree(void* ptr, void* pUserData);
    void ResetRootCommandBuffer(int bufferIndex) NN_NOEXCEPT;
    void BeginRootCommandBuffer(int bufferIndex) NN_NOEXCEPT;
    void EndRootCommandBuffer(int bufferIndex) NN_NOEXCEPT;
    void CreateShaderScratchMemory(int bufferIndex) NN_NOEXCEPT;

    void InitializeVertexState(GraphicsFramework::VertexState* pOutVertexState, nn::gfx::VertexState::InfoType& info) NN_NOEXCEPT;
    void FinalizeVertexState(GraphicsFramework::VertexState* pVertexState) NN_NOEXCEPT;
    void InitializeBlendState(GraphicsFramework::BlendState* pOutBlendState, nn::gfx::BlendState::InfoType& info) NN_NOEXCEPT;
    void FinalizeBlendState(GraphicsFramework::BlendState* pBlendState) NN_NOEXCEPT;

    nn::vi::Display* m_pDisplay;
    nn::vi::Layer* m_pLayer;
    nn::gfx::Device m_Device;
    nn::gfx::Queue m_Queue;
    nn::gfx::SwapChain m_SwapChain;
    nn::gfx::Fence m_pFence;
    nn::gfx::ViewportScissorState m_ViewportScissorState;

    GraphicsFramework::VertexState m_VertexState[VertexStateType_End];
    nn::gfx::RasterizerState m_RasterizerState[RasterizerStateType_End];
    GraphicsFramework::BlendState m_BlendState[BlendStateType_End];
    nn::gfx::DepthStencilState m_DepthStencilState[DepthStencilStateType_End];
    nn::gfx::Sampler m_Sampler[SamplerType_End];

    GraphicsFramework::MemoryPool m_MemoryPool[MemoryPoolType_End];

    nn::gfx::DescriptorPool m_DescriptorPool[nn::gfx::DescriptorPoolType_End];
    nn::gfx::util::DescriptorPoolAllocator m_DescriptorPoolAllocator[nn::gfx::DescriptorPoolType_End];
    int m_DescriptorPoolSlotCount[nn::gfx::DescriptorPoolType_End];

    GraphicsFramework::CommandBuffer* m_pCommandBuffer;
    ShaderScratchMemoryInfo* m_pShaderScratchMemory;
    size_t m_ShaderScratchMemorySize;

    nn::gfx::ColorTargetView** m_pColorTargetView;
    //nn::gfx::Texture* m_pDepthStencilBuffer;
    //nn::gfx::DepthStencilView* m_pDepthStencilView;

    nn::AlignedAllocateFunctionWithUserData m_AllocateFunction;
    nn::FreeFunctionWithUserData m_FreeFunction;
    void* m_pAllocateFunctionUserData;

    int m_DisplayWidth;
    int m_DisplayHeight;
    int m_BufferCount;

    bool m_IsInitialized;
};

} // namespace gfx
} // namespace nns
