﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SharedTexture.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/am/am_Shim.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include "../../../../../Eris/Sources/Libraries/vi/vi_CommonUtility.h"
#include "Framework.h"
#include "../Config.h"

#include <nvn/nvn_FuncPtrInline.h>

namespace framework{

#ifdef NN_DEVOVL_IS_SYSTEM
    namespace {
        nn::Result GetSharedBufferHandle(nn::vi::fbshare::SharedBufferHandle* pOutBufferHandle) NN_NOEXCEPT
        {
            return nn::am::GetSelfController()->GetSystemSharedBufferHandle(pOutBufferHandle);
        }

        nn::Result GetSharedBufferMemoryHandleId(
            nn::vi::native::NativeMemoryHandleId* pOutValue,
            uint64_t* pOutSize,
            nn::vi::fbshare::SharedMemoryPoolLayout* pOutLayout,
            nn::vi::fbshare::SharedBufferHandle hBuffer
        ) NN_NOEXCEPT
        {
            auto pService = nn::vi::GetSystemService();
            return pService->GetSharedBufferMemoryHandleId(pOutValue, pOutSize, pOutLayout, hBuffer, nn::applet::GetAppletResourceUserId());
        }
    }
#endif

    SharedTexturePool::SharedTexturePool() NN_NOEXCEPT
        : m_Layout({})
    {
    }

    nn::Result SharedTexturePool::Initialize() NN_NOEXCEPT
    {
#ifdef NN_DEVOVL_IS_SYSTEM
        NN_DEVOVL_FW_LOG("initializing shared texture pool...\n");
        auto pDevice = Framework::GetDevice();
        auto pNvnDevice = pDevice->ToData()->pNvnDevice;

        // buffer handle
        NN_DEVOVL_FW_LOG("  getting shared buffer handle\n");
        nn::vi::fbshare::SharedBufferHandle hBuffer = {};
        {
            auto result = GetSharedBufferHandle(&hBuffer);
            if(result.IsFailure())
            {
                NN_DEVOVL_FW_LOG("  -> failure %d-%d\n", result.GetModule(), result.GetDescription());
                return result;
            }
        }
        NN_DEVOVL_FW_LOG("  -> success %llu\n", hBuffer);

        // mem handle
        NN_DEVOVL_FW_LOG("  getting shared buffer memory handle id\n");
        nn::vi::native::NativeMemoryHandleId memId = {};
        uint64_t memSize = 0;
        nn::vi::fbshare::SharedMemoryPoolLayout layout = {};
        NN_RESULT_DO(GetSharedBufferMemoryHandleId(&memId, &memSize, &layout, hBuffer));
        NN_DEVOVL_FW_LOG("  -> %lld\n", memId);

        // memory pool
        NN_DEVOVL_FW_LOG("  initializing shared memory pool\n");
        NN_RESULT_DO(m_MemoryPool.Initialize(pNvnDevice, memId, memSize, nn::vi::fbshare::SharedTextureMemoryPoolOption_None));

        // texture pool
        NN_DEVOVL_FW_LOG("  initializing shared texture pool\n");
        m_TexturePool.Initialize(pNvnDevice, &m_MemoryPool, layout, NVN_FORMAT_RGBA8_SRGB);

        // gfx texture
        NN_DEVOVL_FW_LOG("  initializing gfx texture list\n");
        for(int i = 0; i < layout.count; i++)
        {
            NVNtexture* pNvnTexture;
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_TexturePool.AcquireTexture(&pNvnTexture, i));
            nn::gfx::InteroperationNvn::ConvertToGfxTexture(&m_GfxTextureList[i], pNvnTexture);
        }

        // descriptor slot
        NN_DEVOVL_FW_LOG("  initializing desctiptor slot list\n");
        for(int i = 0; i < layout.count; i++)
        {
            nn::gfx::TextureViewInfo info;
            info.SetDefault();
            info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
            info.SetImageDimension(nn::gfx::ImageDimension_2d);
            info.SetTexturePtr(&m_GfxTextureList[i]);
            nn::gfx::TextureView view;
            view.Initialize(pDevice, info);
            NN_UTIL_SCOPE_EXIT{ view.Finalize(pDevice); };

            auto slotIdx = Framework::AllocateTextureDescriptorSlot();
            Framework::GetTextureDescriptorPool()->SetTextureView(slotIdx, &view);
            Framework::GetTextureDescriptorPool()->GetDescriptorSlot(&m_DescriptorSlotList[i], slotIdx);
        }


        NN_DEVOVL_FW_LOG("initialized shared texture pool\n");
        m_Layout = layout;
        NN_RESULT_SUCCESS;
#else
        m_Layout = {};
        NN_RESULT_SUCCESS;
#endif
    }

    void SharedTexturePool::Finalize() NN_NOEXCEPT
    {
        // どうせ finalize しないので未実装
        NN_ABORT("not implemented");
    }

    int SharedTexturePool::GetTextureCount() const NN_NOEXCEPT
    {
        return m_Layout.count;
    }

    std::shared_ptr<SharedTexture> SharedTexturePool::GetTexture(int index) NN_NOEXCEPT
    {
        // SharedTexture 自体は proxy オブジェクトなので index のチェックはしない
        return std::make_shared<SharedTexture>(this, index);
    }

    nn::gfx::Texture* SharedTexturePool::GetGfxTexture(int index) NN_NOEXCEPT
    {
        if(index < 0 || index >= m_Layout.count)
        {
            return nullptr;
        }
        return &m_GfxTextureList[index];
    }

    const nn::gfx::DescriptorSlot* SharedTexturePool::GetDescriptorSlot(int index) const NN_NOEXCEPT
    {
        if(index < 0 || index >= m_Layout.count)
        {
            return nullptr;
        }
        return &m_DescriptorSlotList[index];
    }

    //-------------------------------------------------------------------

    SharedTexture::SharedTexture(SharedTexturePool* pPool, int index) NN_NOEXCEPT
        : m_pPool(pPool)
        , m_Index(index)
    {
    }

    SharedTexture::~SharedTexture() NN_NOEXCEPT
    {
    }

    nn::gfx::Texture* SharedTexture::Get() NN_NOEXCEPT
    {
        return m_pPool->GetGfxTexture(m_Index);
    }

    const nn::gfx::Texture* SharedTexture::Get() const NN_NOEXCEPT
    {
        return m_pPool->GetGfxTexture(m_Index);
    }

    int SharedTexture::GetWidth() const NN_NOEXCEPT
    {
        if(auto p = Get())
        {
            return nvnTextureGetWidth(p->ToData()->pNvnTexture);
        }
        return 0;
    }

    int SharedTexture::GetHeight() const NN_NOEXCEPT
    {
        if(auto p = Get())
        {
            return nvnTextureGetHeight(p->ToData()->pNvnTexture);
        }
        return 0;
    }

    nn::gfx::ImageFormat SharedTexture::GetFormat() const NN_NOEXCEPT
    {
        return nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;
    }

    const nn::gfx::DescriptorSlot* SharedTexture::GetDescriptorSlot() const NN_NOEXCEPT
    {
        return m_pPool->GetDescriptorSlot(m_Index);
    }

}

