﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ResShader.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include "Framework.h"

void ResShader::Initialize(const char* shaderFilePath)
{
    //NN_LOG("\nLoad ShaderFile: %s\n",shaderFilePath);
    nn::Result result;

    void* pFileBuffer = NULL;
    int64_t fileSize = 0;
    {
        nn::fs::FileHandle hFile;
        result = nn::fs::OpenFile(&hFile, shaderFilePath, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = nn::fs::GetFileSize(&fileSize, hFile);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        pFileBuffer = Framework::Allocate(static_cast<size_t>(fileSize), NN_ALIGNOF(nn::util::BinaryFileHeader));
        NN_ASSERT_NOT_NULL(pFileBuffer);

        result = nn::fs::ReadFile(hFile, 0, pFileBuffer, static_cast<size_t>(fileSize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::fs::CloseFile(hFile);
    }

    nn::gfx::Buffer buffer;
    void* pShaderBinary = NULL;
    {
        size_t alignment = reinterpret_cast<nn::gfx::ResShaderFile*>(pFileBuffer)->ToData().fileHeader.GetAlignment();
        m_BinarySize = static_cast<size_t>(fileSize);


        nn::gfx::Buffer::InfoType info;
        {
            info.SetDefault();
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
            info.SetSize(m_BinarySize);
        }

        if(NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
        {
            auto pMemoryPool = Framework::GetShaderMemoryPool();
            auto offset = Framework::AllocateShaderPoolMemory(m_BinarySize, alignment);
            NN_ASSERT_NOT_EQUAL(offset, -1);
            m_BinaryOffset = offset;

            buffer.Initialize(
                Framework::GetDevice(),
                info,
                pMemoryPool,
                offset,
                m_BinarySize
            );
        }
        else
        {
            m_BinaryOffset = -1;
            buffer.Initialize(Framework::GetDevice(), info, NULL, 0, 0);
        }

        pShaderBinary = buffer.Map();
        std::memcpy(pShaderBinary, pFileBuffer, static_cast<size_t>(fileSize));
    }

    {
        Framework::Free(pFileBuffer);
        pFileBuffer = NULL;
    }

    m_pResShaderFile = nn::gfx::ResShaderFile::ResCast(pShaderBinary);
    m_pResShaderContainer = m_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL(m_pResShaderContainer);
    size_t containerOffset = m_BinaryOffset + nn::util::BytePtr(m_pResShaderFile).Distance(m_pResShaderContainer);
    m_pResShaderContainer->Initialize(
        Framework::GetDevice(),
        Framework::GetShaderMemoryPool(),
        containerOffset,
        m_BinarySize
    );

    nn::gfx::ResShaderVariation* pVariation = m_pResShaderContainer->GetResShaderVariation(0);
    NN_ASSERT_NOT_NULL(pVariation);

    nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_Success;
    m_CodeType = nn::gfx::ShaderCodeType_Binary;
    nn::gfx::ResShaderProgram* pResShaderProgram = pVariation->GetResShaderProgram(m_CodeType);
    if (pResShaderProgram)
    {
        shaderResult = pResShaderProgram->Initialize(Framework::GetDevice());
    }

    if (!pResShaderProgram || shaderResult != nn::gfx::ShaderInitializeResult_Success)
    {
        m_CodeType = nn::gfx::ShaderCodeType_Source;
        pResShaderProgram = pVariation->GetResShaderProgram(m_CodeType);
        if (pResShaderProgram)
        {
            shaderResult = pResShaderProgram->Initialize(Framework::GetDevice());
        }
    }
    NN_ASSERT(pResShaderProgram && shaderResult == nn::gfx::ShaderInitializeResult_Success);

    int variationCount = m_pResShaderContainer->GetShaderVariationCount();
    for (int idxVariation = 1; idxVariation < variationCount; idxVariation++)
    {
        pVariation = m_pResShaderContainer->GetResShaderVariation(idxVariation);
        NN_ASSERT_NOT_NULL(pVariation);
        shaderResult = pVariation->GetResShaderProgram(m_CodeType)->Initialize(Framework::GetDevice());
        NN_ASSERT_EQUAL(shaderResult, nn::gfx::ShaderInitializeResult_Success);
    }

    buffer.Unmap();
    buffer.Finalize(Framework::GetDevice());

    //NN_LOG("Load Shader: SUCCESS\n\n");
}

void ResShader::Finalize() NN_NOEXCEPT
{
    m_pResShaderContainer->Finalize(Framework::GetDevice());
    if(m_BinaryOffset != -1)
    {
        Framework::FreeShaderPoolMemory(m_BinaryOffset);
    }
}

nn::gfx::Shader* ResShader::GetShader(int variation) NN_NOEXCEPT
{
    return m_pResShaderContainer->GetResShaderVariation(variation)->GetResShaderProgram(m_CodeType)->GetShader();
}
