﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HidHandlingModeManager.h"

#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ae/ae_OverlayAppletApi.h>
#include "../Config.h"

namespace nn{ namespace ae{
    void BeginOverlayUserInteractionObservationForDevelop() NN_NOEXCEPT;
}}

namespace framework{

    HidHandlingModeManager g_HidHandlingModeManager;

    HidHandlingModeManager::HidHandlingModeManager() NN_NOEXCEPT
        : m_Mutex(false)
    {
    }

    HidHandlingModeManager::~HidHandlingModeManager() NN_NOEXCEPT
    {
        // 終了するものでもないので何もしない。
    }

    void HidHandlingModeManager::BeginInteraction() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_IsInteractionRequested = true;
        UpdateHandlingModeImpl();
    }

    void HidHandlingModeManager::EndInteraction() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_IsInteractionRequested = false;
        UpdateHandlingModeImpl();
    }

    void HidHandlingModeManager::BeginObservation() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_IsObservationRequested = true;
        UpdateHandlingModeImpl();
    }

    void HidHandlingModeManager::EndObservation() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        m_IsObservationRequested = false;
        UpdateHandlingModeImpl();
    }

    void HidHandlingModeManager::UpdateHandlingModeImpl() NN_NOEXCEPT
    {
        // 新しいモードを決定
        HandlingMode newMode = HandlingMode_None;
        if(m_IsInteractionRequested)
        {
            newMode = HandlingMode_Interaction;
        }
        else if(m_IsObservationRequested)
        {
            newMode = HandlingMode_Observation;
        }

        // 変化がなければ何もしない
        if(m_CurrentHandlingMode == newMode)
        {
            return;
        }

        // 新しいモードに変更
        switch(newMode)
        {
        case HandlingMode_Interaction:
            NN_DEVOVL_FW_LOG("hid handling mode -> Interaction\n");
            nn::ae::BeginOverlayUserInteraction();
            break;
        case HandlingMode_Observation:
            NN_DEVOVL_FW_LOG("hid handling mode -> Observation\n");
            nn::ae::BeginOverlayUserInteractionObservationForDevelop();
            break;
        default:
            NN_DEVOVL_FW_LOG("hid handling mode -> None\n");
            nn::ae::EndOverlayUserInteraction();
        }

        m_CurrentHandlingMode = newMode;
    }
}

