﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Framework.h"
#include "FrameworkImpl.h"

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

namespace {
    using namespace ::framework;
}

void Framework::WindowAcquireTexture(nn::gfx::Fence* pOutFence) NN_NOEXCEPT
{
    auto pSwapChain = g_Gfw.GetSwapChain();
    auto pWindow = pSwapChain->ToData()->pNvnWindow;

    int textureIndex;
    NN_ABORT_UNLESS_EQUAL(
        nvnWindowAcquireTexture(pWindow, pOutFence->ToData()->pNvnSync, &textureIndex),
        NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS
    );

    pSwapChain->ToData()->currentScanBufferIndex = textureIndex;
}

void Framework::QueuePresentTexture() NN_NOEXCEPT
{
    auto pQueue = g_Gfw.GetQueue();
    auto pSwapChain = g_Gfw.GetSwapChain();
    nvnQueuePresentTexture(
        pQueue->ToData()->pNvnQueue,
        pSwapChain->ToData()->pNvnWindow,
        pSwapChain->ToData()->currentScanBufferIndex
    );
}

void Framework::QueueFlush() NN_NOEXCEPT
{
    auto pQueue = g_Gfw.GetQueue();
    pQueue->Flush();
}

void Framework::WaitQueueFinish() NN_NOEXCEPT
{
    auto pQueue = g_Gfw.GetQueue();
    pQueue->Sync();
}

void Framework::QueueSubmitFrameCommand() NN_NOEXCEPT
{
    auto pQueue = g_Gfw.GetQueue();
    auto pCommandBuffer = g_Gfw.GetRootCommandBuffer(0);

    pQueue->ExecuteCommand(pCommandBuffer, nullptr);
}

void Framework::QueueWaitFence(nn::gfx::Fence* pFence) NN_NOEXCEPT
{
    auto pQueue = g_Gfw.GetQueue();
    nvnQueueWaitSync(
        pQueue->ToData()->pNvnQueue,
        pFence->ToData()->pNvnSync
    );
}

void Framework::QueueSignalFence(nn::gfx::Fence* pFence) NN_NOEXCEPT
{
    auto pQueue = g_Gfw.GetQueue();
    nvnQueueFenceSync(
        pQueue->ToData()->pNvnQueue,
        pFence->ToData()->pNvnSync,
        NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE,
        0
    );
}

void Framework::WaitFence(nn::gfx::Fence* pFence) NN_NOEXCEPT
{
    nvnSyncWait(pFence->ToData()->pNvnSync, NVN_WAIT_TIMEOUT_MAXIMUM);
}

void Framework::BeginFrameCommand() NN_NOEXCEPT
{
    g_Impl.frameCount++;
    g_Impl.frameIndex = g_Impl.frameCount % static_cast<int>(g_Impl.renderedRegion.size());

    g_Gfw.BeginFrame(0);
    GetTextureDescriptorPool()->BeginUpdate();
}

void Framework::EndFrameCommand() NN_NOEXCEPT
{
    GetTextureDescriptorPool()->EndUpdate();
    g_Gfw.EndFrame(0);

    {
        auto pCommandBuffer = GetCommandBuffer();
        auto pData = nn::gfx::AccessorToData(pCommandBuffer);
        size_t commandUsed = nvnCommandBufferGetCommandMemoryUsed(pData->pNvnCommandBuffer);
        size_t controlUsed = nvnCommandBufferGetControlMemoryUsed(pData->pNvnCommandBuffer);
        g_Impl.rootCommandMemoryUsedSizeMax = std::max(g_Impl.rootCommandMemoryUsedSizeMax, commandUsed);
        g_Impl.rootControlMemoryUsedSizeMax = std::max(g_Impl.rootControlMemoryUsedSizeMax, controlUsed);
    }

}

void Framework::MakeSetFrameBufferRenderTargetCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    auto pTarget = GetCurrentColorTargetView();
    pCommandBuffer->SetRenderTargets(1, &pTarget, nullptr);
}

void Framework::LockFrameCommandResource(const std::shared_ptr<void>& p) NN_NOEXCEPT
{
    g_Impl.lockList.push_back(p);
}

void Framework::ReleaseFrameCommandResource() NN_NOEXCEPT
{
    // ロックしていたオブジェクトを解放
    {
        auto& v = g_Impl.lockList;
        for(auto it = v.rbegin(); it != v.rend(); ++it)
        {
            it->reset();
        }
        v.clear();
    }
}

void Framework::MakeClearPreviousRendererdRegionCommand(nn::gfx::CommandBuffer* pCommandBuffer, const nn::util::Color4f& color) NN_NOEXCEPT
{
    auto& renderedRegion = g_Impl.renderedRegion[g_Impl.frameIndex];

    if(renderedRegion.clearColor != color)
    {
        // クリアカラーが変わっていたら全範囲クリア
        MakeSetViewportScissorCommand(pCommandBuffer, GetScreenRectangle());
        MakeClearCommand(pCommandBuffer, color);
        renderedRegion.clearColor = color;
        renderedRegion.regionList.clear();
        return;
    }

    // 個別の範囲をクリア
    for(auto& region : renderedRegion.regionList)
    {
        MakeSetViewportScissorCommand(pCommandBuffer, region);
        MakeClearCommand(pCommandBuffer, color);
    }
    renderedRegion.regionList.clear();
}

void Framework::RegisterClearRegion(const framework::Rectangle& rect) NN_NOEXCEPT
{
    auto& renderedRegion = g_Impl.renderedRegion[g_Impl.frameIndex];
    renderedRegion.regionList.push_back(rect);
}

void Framework::SetWindowCrop(const framework::Rectangle& rect) NN_NOEXCEPT
{
    //NN_DEVOVL_FW_LOG("setting window crop to (%d,%d;%dx%d)\n", rect.x, rect.y, rect.width, rect.height);
    auto pData = nn::gfx::AccessorToData(g_Gfw.GetSwapChain());
    nvnWindowSetCrop(pData->pNvnWindow, rect.x, rect.y, rect.width, rect.height);
}
