﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "OverlayNotificationThread.h"

#include <nn/os.h>
#include <nn/ovln/ovln_ReceiverForOverlay.h>

#include "Config.h"
#include "AppletMessageThread.h"

namespace {

    enum MultiWaitIndex
    {
        MultiWaitIndex_OverlayNotification,
        MultiWaitIndex_AppletMessage,
    };

}


void OverlayNotificationThreadFunction(void* pArg) NN_NOEXCEPT
{
    auto pParam = reinterpret_cast<OverlayNotificationThreadParameter*>(pArg);

    // receiver を初期化
    nn::ovln::ReceiverForOverlayType receiver;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeReceiver(&receiver));
    NN_UTIL_SCOPE_EXIT{ nn::ovln::FinalizeReceiver(&receiver); };

    // multi-wait を作成
    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);
    NN_UTIL_SCOPE_EXIT{ nn::os::FinalizeMultiWait(&multiWait); };

    nn::os::MultiWaitHolderType receiverHolder;
    nn::ovln::InitializeMultiWaitHolder(&receiverHolder, &receiver);
    NN_UTIL_SCOPE_EXIT{ nn::os::FinalizeMultiWaitHolder(&receiverHolder); };
    receiverHolder.userData = MultiWaitIndex_OverlayNotification;

    nn::os::MultiWaitHolderType appletMessageHolder;
    pParam->pAppletMessageChannel->IntializeReceivableEventHolder(&appletMessageHolder);
    NN_UTIL_SCOPE_EXIT{ nn::os::FinalizeMultiWaitHolder(&appletMessageHolder); };
    appletMessageHolder.userData = MultiWaitIndex_AppletMessage;

    nn::os::LinkMultiWaitHolder(&multiWait, &receiverHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &appletMessageHolder);
    NN_UTIL_SCOPE_EXIT{
        nn::os::UnlinkAllMultiWaitHolder(&multiWait);
    };

    for(;;)
    {
        auto pSignaled = nn::os::WaitAny(&multiWait);
        if(pSignaled->userData == MultiWaitIndex_OverlayNotification)
        {
            nn::ovln::Message message;
            if(!nn::ovln::TryReceive(&message, &receiver))
            {
                continue;
            }

            NN_DEVOVL_OVLN_LOG("Received OvlnMessage tag=%u(0x%08X)\n", message.tag, message.tag);
            pParam->pToOverlayDisplayChannel->SendMessage(message.tag, std::make_shared<nn::ovln::Message>(message));
        }
        else if(pSignaled->userData == MultiWaitIndex_AppletMessage)
        {
            ThreadMessageType message = 0;
            std::shared_ptr<ThreadMessageResultReporter> pReporter;
            if(!pParam->pAppletMessageChannel->TryReceiveMessage(&message, &pReporter))
            {
                continue;
            }

            NN_DEVOVL_OVLN_LOG("Received AppletMessage %lld\n", message);
            if(message == AppletMessage_RequestExit)
            {
                NN_DEVOVL_OVLN_LOG("  Exit OverlayNotificationThread\n");
                break;
            }
            else
            {
                NN_DEVOVL_OVLN_LOG("  Unknown message\n");
            }
        }
    }

}
