﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "OverlayDisplayThread.h"

#include "Config.h"
#include "framework/Framework.h"
#include "framework/FrameTime.h"
#include "framework/Hid.h"
#include "panel/panel_PanelSystem.h"
#include "scene/SceneStack.h"
#include "scene/overlay/SceneOverlay.h"
#include "scene/menu/SceneMenuPageBase.h"
#include "scene/applaunch/SceneAppLaunch.h"
#include "scene/debug/debug_SceneDebug.h"
#include "scene/debug/debug_ReadWriteActivity.h"

#include "detail/SleepScheduler.h"

namespace {
    scene::SceneIndex InitialSceneIndex = scene::SceneIndex_Overlay;
}

namespace {
    class OverlayDisplayStatus
    {
    public:
        explicit OverlayDisplayStatus(OverlayDisplayThreadParameter* pParam) NN_NOEXCEPT
            : m_pAppletMessageChannel(pParam->pAppletMessageChannel)
            , m_pOverlayNotificationChannel(pParam->pOverlayNotificationChannel)
            , m_pRunMode(pParam->pRunMode)
            , m_OverlaySceneStack()
            , m_MenuSceneStack()
            , m_LogoSceneStack()
            , m_MenuType(AppletMessage_None)
            , m_IsSleeping(false)
            , m_IsExitRequested(false)
            , m_IsHideApplicationLogoRequested(false)
        {
            nn::os::InitializeMultiWait(&m_MultiWait);
            m_pAppletMessageChannel->IntializeReceivableEventHolder(&m_AppletMessageWaitHolder);
            m_pOverlayNotificationChannel->IntializeReceivableEventHolder(&m_OverlayMessageWaitHolder);
            nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_AppletMessageWaitHolder);
            nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_OverlayMessageWaitHolder);

            m_OverlaySceneStack.PushScene(
                InitialSceneIndex,
                std::make_shared<scene::overlay::SceneOverlayParameter>(pParam->pOverlayNotificationChannel, &m_MultiWait, pParam->pRunMode)
            );

            m_pRootPanel = std::make_shared<panel::PanelContainer>();
            m_pRootPanel->SetSize(ScreenWidth, ScreenHeight);
            m_pRootPanel->SetVisibility(panel::PanelVisibility::Transparent);
        }

        ~OverlayDisplayStatus() NN_NOEXCEPT
        {
            nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
            nn::os::FinalizeMultiWaitHolder(&m_OverlayMessageWaitHolder);
            nn::os::FinalizeMultiWaitHolder(&m_AppletMessageWaitHolder);
            nn::os::FinalizeMultiWait(&m_MultiWait);
        }

        bool IsAlive() NN_NOEXCEPT
        {
            return !m_OverlaySceneStack.IsEmpty();
        }

        bool HasMenuScene() const NN_NOEXCEPT
        {
            return !m_MenuSceneStack.IsEmpty();
        }

        bool HasLogoScene() const NN_NOEXCEPT
        {
            return !m_LogoSceneStack.IsEmpty();
        }

        bool HasDebugScene() const NN_NOEXCEPT
        {
            return scene::debug::g_SceneDebug.IsActive();
        }

        bool IsEventDrivenMode() const NN_NOEXCEPT
        {
            return !HasMenuScene()
                && !HasLogoScene()
                && !HasDebugScene()
                && !m_SleepScheduler.IsScheduled();
        }

        void DrawSceneStack() NN_NOEXCEPT
        {
            // スリープ中は描画しない
            if(m_IsSleeping)
            {
                m_pRootPanel->ClearChildren();
                return;
            }

            std::vector<std::shared_ptr<panel::IPanel>> newList;

            // 7. デバッグ表示
            {
                // デバッグ機能が OFF の場合は Panel が Invisible になる。
                newList.push_back(scene::debug::g_SceneDebug.GetPanel());
            }

            // 6. 電源メニュー
            if(!m_MenuSceneStack.IsEmpty() && m_MenuType == AppletMessage_LaunchPowerMenu)
            {
                m_MenuSceneStack.GatherPanel(newList);
            }

            // 5. アプリ起動ロゴ
            {
                m_LogoSceneStack.GatherPanel(newList);
            }

            // 4. クイック設定
            if(!m_MenuSceneStack.IsEmpty() && m_MenuType == AppletMessage_LaunchQuickMenu)
            {
                m_MenuSceneStack.GatherPanel(newList);
            }

            // 3. コントローラ演出
            //   未実装

            // 2. オーバーレイ通知
            {
                m_OverlaySceneStack.GatherPanel(newList);
            }
            // 1. ウェイト画面
            //   未実装

            m_pRootPanel->SetChildren(newList);

            // 描画
            {
                auto pCommandBuffer = Framework::GetCommandBuffer();
                auto pTarget = Framework::GetCurrentColorTargetView();
                pCommandBuffer->SetRenderTargets(1, &pTarget, nullptr);
                panel::g_PanelSystem.RenderPanel(Framework::GetCommandBuffer(), m_pRootPanel, Framework::GetScreenRectangle());
            }
        }

        void ProcessAppletMessage() NN_NOEXCEPT
        {
            auto pChannel = m_pAppletMessageChannel;

            ThreadMessageType message = 0;
            std::shared_ptr<ThreadMessageResultReporter> pReporter;

            // スリープ準備中の処理
            if(!m_IsSleeping && m_SleepScheduler.IsScheduled())
            {
                if(m_SleepScheduler.UpdateToSleep())
                {
                    NN_DEVOVL_DISP_LOG_AMSG("falling to sleep...\n", m_SleepScheduler.GetFrameToSleep());
                    m_IsSleeping = true;
                }
                else
                {
                    NN_DEVOVL_DISP_LOG_AMSG("%d frames to sleep...\n", m_SleepScheduler.GetFrameToSleep());
                }
            }

            // メッセージの受信
            if(m_IsSleeping)
            {
                // 起床メッセージが来るまでブロックして待つ
                pChannel->ReceiveMessage(&message, &pReporter);
            }
            else
            {
                if(!pChannel->TryReceiveMessage(&message, &pReporter))
                {
                    return;
                }
            }

            // メッセージの処理
            NN_DEVOVL_DISP_LOG_AMSG("Applet Message Received(%d)\n", static_cast<int>(message));
            switch(message)
            {
            // 終了
            case AppletMessage_RequestExit:
                m_IsExitRequested = true;
                pReporter->NotifyCompletion(ThreadMessageResult_Success);
                break;
            // スリープ
            case AppletMessage_RequestSuspend:
                // 少しまわしてからスリープに入れる
                m_SleepScheduler.Schedule(5, std::move(pReporter));
                break;
            case AppletMessage_RequestResume:
                m_IsSleeping = false;
                // メニューは閉じる
                m_MenuSceneStack.Clear();
                pReporter->NotifyCompletion(ThreadMessageResult_Success);
                break;
            // ロゴ表示
            case AppletMessage_ShowApplicationLogo:
                if(m_LogoSceneStack.IsEmpty())
                {
                    NN_SDK_ASSERT_NOT_NULL(pReporter);
                    m_IsHideApplicationLogoRequested = false;
                    auto pUserData = pReporter->GetUserData();
                    auto pArg = std::static_pointer_cast<AppletMessageApplicationLogoInfo>(pUserData);
                    auto p = std::make_shared<scene::applaunch::SceneAppLaunchParameter>(pArg->GetApplicationId(), &m_IsHideApplicationLogoRequested);
                    m_LogoSceneStack.PushScene(scene::SceneIndex_ApplicationLogo, p);
                }
                break;
            case AppletMessage_HideApplicationLogo:
                m_IsHideApplicationLogoRequested = true;
                break;
            case AppletMessage_ForceHideApplicationLogo:
                m_LogoSceneStack.Clear();
                break;
            // メニュー表示
            case AppletMessage_LaunchPowerMenu:
                if(m_MenuSceneStack.IsEmpty())
                {
                    NN_SDK_ASSERT_NOT_NULL(pReporter);
                    auto p = std::make_shared<scene::menu::SceneMenuPageParameter>(pReporter, nullptr);
                    m_MenuSceneStack.PushScene(scene::SceneIndex_PowerMenuPage, p);
                    m_MenuType = AppletMessage_LaunchPowerMenu;
                }
                break;
            case AppletMessage_LaunchQuickMenu:
                if(m_MenuSceneStack.IsEmpty())
                {
                    NN_SDK_ASSERT_NOT_NULL(pReporter);
                    auto p = std::make_shared<scene::menu::SceneMenuPageParameter>(pReporter, nullptr);
                    m_MenuSceneStack.PushScene(scene::SceneIndex_QuickMenuPage, p);
                    m_MenuType = AppletMessage_LaunchQuickMenu;
                }
                break;
            case AppletMessage_CloseMenu:
                m_MenuSceneStack.Clear();
                break;
            // 未知
            default:
                {
                    NN_DEVOVL_DISP_LOG_AMSG("  Unknown applet message(%d)\n", static_cast<int>(message));
                }
            }
        } // NOLINT(impl/function_size)

        // シーンの切り替わりがあった場合 true を返す。
        // true が返された場合、描画をスキップして再計算が必要。
        bool UpdateSceneStack() NN_NOEXCEPT
        {
            auto overlayUpdateResult = m_OverlaySceneStack.UpdateTopScene();
            auto menuUpdateResult    = m_MenuSceneStack.UpdateTopScene();
            auto logoUpdateResult    = m_LogoSceneStack.UpdateTopScene();
            scene::debug::g_SceneDebug.Update();

            bool isSceneChanging = false;
            isSceneChanging |= m_OverlaySceneStack.ProcessUpdateResult(overlayUpdateResult);
            isSceneChanging |= m_MenuSceneStack.ProcessUpdateResult(menuUpdateResult);
            isSceneChanging |= m_LogoSceneStack.ProcessUpdateResult(logoUpdateResult);

            return isSceneChanging;
        }

        void WaitAnyEvent() NN_NOEXCEPT
        {
            nn::os::WaitAny(&m_MultiWait);
        }

        nn::util::Color4f GetClearColor() NN_NOEXCEPT
        {
            nn::util::Color4f clearColor = nn::util::Color4f(0, 0, 0, 0);
            if(m_IsSleeping)
            {
                return clearColor;
            }

            // TORIAEZU:
            //   透明でない BG を持っているのは PowerMenu のみなのでそこだけ対応。
            if(!m_MenuSceneStack.IsEmpty())
            {
                clearColor = m_MenuSceneStack.GetBackgroundColor();
            }
            return clearColor;
        }

    private:
        ThreadMessageChannel* m_pAppletMessageChannel;
        ThreadMessageChannel* m_pOverlayNotificationChannel;
        RunMode*              m_pRunMode NN_IS_UNUSED_MEMBER;

        scene::SceneStack m_OverlaySceneStack;
        scene::SceneStack m_MenuSceneStack;
        scene::SceneStack m_LogoSceneStack;

        std::shared_ptr<panel::PanelContainer> m_pRootPanel;
        detail::SleepScheduler m_SleepScheduler;

        // 起動中のメニューの種類。AppletMessage_LaunchXxxMenu の値。
        // MenuSceneStack が空の場合の値は実装依存。
        // つまり、メニューを開くときに設定されて閉じる場合には放置で構わない。
        AppletMessage m_MenuType;
        bool m_IsSleeping;
        bool m_IsExitRequested;
        bool m_IsHideApplicationLogoRequested;

        nn::os::MultiWaitType m_MultiWait;
        nn::os::MultiWaitHolderType m_AppletMessageWaitHolder;
        nn::os::MultiWaitHolderType m_OverlayMessageWaitHolder;
    };

}

void OverlayDisplayThreadFunction(void* pArg) NN_NOEXCEPT
{
    auto pParam = reinterpret_cast<OverlayDisplayThreadParameter*>(pArg);
    NN_UNUSED(pParam);

    Framework::Initialize();
    NN_UTIL_SCOPE_EXIT{ Framework::Finalize(); };
    panel::g_PanelSystem.Initialize();
    NN_UTIL_SCOPE_EXIT{ panel::g_PanelSystem.Finalize(); };
    scene::debug::g_SceneDebug.Initialize();
    NN_UTIL_SCOPE_EXIT{ scene::debug::g_SceneDebug.Finalize(); };

    {
        Framework::BeginFrameCommand();
        auto pCommandBuffer = Framework::GetCommandBuffer();
        Framework::MakeInitializeCommand(pCommandBuffer);
        Framework::EndFrameCommand();
        Framework::QueueSubmitFrameCommand();
        Framework::QueueFlush();
        Framework::WaitQueueFinish();
    }

    {
        panel::detail::RendererOption option = {};
#ifdef NN_DEVOVL_ENABLE_PANEL_DUMP
        option.dumpingPanelInterval = 150;
#endif
#ifdef NN_DEVOVL_ENABLE_PRINT_PANELCOUNT
        option.printRenderedPanelCount = true;
#endif
        panel::g_PanelSystem.SetRenderOption(option);
    }

    {
        HidButtonHandleParameter param;
        param.repeatTriggerFrameBeginCount = 12;
        param.repeatTriggerFrameInterval   = 3;
        Hid::SetParameter(param);
    }

    auto status = OverlayDisplayStatus(pParam);

    nn::gfx::Fence windowFence;
    nn::gfx::Fence commandFence;
    nn::gfx::Fence presentFence;
    {
        nn::gfx::FenceInfo info;
        info.SetDefault();
        windowFence.Initialize(Framework::GetDevice(), info);
        commandFence.Initialize(Framework::GetDevice(), info);
        presentFence.Initialize(Framework::GetDevice(), info);
    }
    NN_UTIL_SCOPE_EXIT {
        windowFence.Finalize(Framework::GetDevice());
        commandFence.Finalize(Framework::GetDevice());
        presentFence.Finalize(Framework::GetDevice());
    };

    // デバッグ設定を読込む
    {
        scene::debug::Activity act = {};
        if(scene::debug::ReadWriteActivity::ReadFromSdCard(&act))
        {
            scene::debug::g_SceneDebug.SubmitActivity(act);
            scene::debug::g_SceneDebug.Update();
        }
    }

    while(status.IsAlive())
    {
        // 通知だけを行う場合には描画ループを回さない
        if(status.IsEventDrivenMode())
        {
            status.WaitAnyEvent();
        }

        status.ProcessAppletMessage();

        Hid::Update();
        while(status.UpdateSceneStack())
        {
            Hid::Update();
            // just loop
        }

        nn::util::Color4f clearColor = status.GetClearColor();
        //nn::util::Color4f clearColor = nn::util::Color4f(1, 0, 0, .5f);

        framework::FrameTime time;

        time.beginWindowAcquireTexture = nn::os::GetSystemTick();
        Framework::WindowAcquireTexture(&windowFence);
        panel::g_PanelSystem.PushRenderTargetTexture(Framework::GetCurrentTargetTexture());

        time.endWindowAcquireTexture = nn::os::GetSystemTick();
        Framework::QueueWaitFence(&windowFence);

        // コマンドの積み込みと提出
        time.beginFrameCommand = nn::os::GetSystemTick();
        Framework::BeginFrameCommand();
        Framework::MakeSetFrameBufferRenderTargetCommand(Framework::GetCommandBuffer());
        Framework::MakeSetViewportScissorCommand(Framework::GetCommandBuffer(), Framework::GetScreenRectangle());
        Framework::MakeClearCommand(Framework::GetCommandBuffer(), clearColor);
        //Framework::MakeClearPreviousRendererdRegionCommand(Framework::GetCommandBuffer(), clearColor);
        status.DrawSceneStack();
        Framework::EndFrameCommand();
        Framework::QueueSubmitFrameCommand();
        Framework::QueueSignalFence(&commandFence);
        Framework::QueueFlush();
        time.endFrameCommand = nn::os::GetSystemTick();

        time.beginWaitWindowFence = nn::os::GetSystemTick();
        Framework::WaitFence(&windowFence);
        time.endWaitWindowFence = nn::os::GetSystemTick();

        time.beginWaitCommandFence = nn::os::GetSystemTick();
        Framework::WaitFence(&commandFence);
        time.endWaitCommandFence = nn::os::GetSystemTick();

        // 描画したものがなければ Crop を縮小して DC のメモリアクセスを減らす
        if(panel::g_PanelSystem.GetRenderedPanelCount() > 0)
        {
            Framework::SetWindowCrop(Framework::GetScreenRectangle());
        }
        else
        {
            Framework::SetWindowCrop(Framework::GetIdleCropRectangle());
        }

        Framework::ReleaseFrameCommandResource();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));

        time.beginQueuePresentTexture = nn::os::GetSystemTick();
        Framework::QueuePresentTexture();
        Framework::QueueSignalFence(&presentFence);
        Framework::QueueFlush();
        time.endQueuePresentTexture = nn::os::GetSystemTick();

        time.beginWaitPresentFence = nn::os::GetSystemTick();
        Framework::WaitFence(&presentFence);
        time.endWaitPresentFence = nn::os::GetSystemTick();

        //framework::PrintFrameTime(time);
    }
}// NOLINT(impl/function_size)
