﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MemoryManagement.h"

#include "Config.h"

#ifdef NN_BUILD_TARGET_PLATFORM_OS_NN

#include <nn/mem/mem_StandardAllocator.h>
#include <new>
#include <cstdlib>
#include <cerrno>
#include <type_traits>
#include <algorithm>

namespace
{
    static const size_t HeapSize = 14 * 1024 * 1024;

    std::aligned_storage<sizeof(nn::mem::StandardAllocator), NN_ALIGNOF(nn::mem::StandardAllocator)>::type g_AllocatorStorage;
    nn::mem::StandardAllocator* g_pAllocator;

    size_t g_TotalAllocatedSizeMax;

    void UpdateTotalAllocatedSizeMax() NN_NOEXCEPT
    {
        size_t s = GetTotalAllocatedSize();
        g_TotalAllocatedSizeMax = std::max(g_TotalAllocatedSizeMax, s);
    }
}


void InitializeAllocator() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(HeapSize));
    uintptr_t address = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, HeapSize));
    g_pAllocator = new(&g_AllocatorStorage) nn::mem::StandardAllocator();
    g_pAllocator->Initialize(reinterpret_cast<void*>(address), HeapSize);
    g_TotalAllocatedSizeMax = 0;
}

size_t GetApplicationHeapSize() NN_NOEXCEPT
{
    return HeapSize;
}

size_t GetTotalAllocatedSize() NN_NOEXCEPT
{
    return HeapSize - g_pAllocator->GetTotalFreeSize();
}

size_t GetTotalAllocatedSizeMax() NN_NOEXCEPT
{
    return g_TotalAllocatedSizeMax;
}

extern "C"
{
    void* malloc(size_t size)
    {
        void* p = g_pAllocator->Allocate(size);
        if ( p == NULL )
        {
            errno = ENOMEM;
            NN_DEVOVL_LOG_ERR("malloc failed\n");
        }

        UpdateTotalAllocatedSizeMax();
        return p;
    }

    void free(void* p)
    {
        if (p)
        {
            g_pAllocator->Free(p);
        }
    }

    void* calloc(size_t num, size_t size)
    {
        size_t sum = num * size;
        void*  p   = malloc(sum);

        if ( p != NULL )
        {
            memset(p, 0, sum);
        }
        else
        {
            errno = ENOMEM;
        }

        return p;
    }

    void* realloc(void* p, size_t newSize)
    {
        void* r = g_pAllocator->Reallocate(p, newSize);
        if ( r == NULL )
        {
            NN_DEVOVL_LOG_ERR("realloc failed\n");
            errno = ENOMEM;
        }

        UpdateTotalAllocatedSizeMax();
        return r;
    }

    void* aligned_alloc(size_t alignment, size_t size)
    {
        void* p = g_pAllocator->Allocate(size, alignment);
        if ( p == NULL )
        {
            NN_DEVOVL_LOG_ERR("aligned_alloc failed\n");
            errno = ENOMEM;
        }

        UpdateTotalAllocatedSizeMax();
        return p;
    }

    size_t malloc_usable_size(void* p)
    {
        if ( p == NULL )
        {
            errno = ENOMEM;
            return 0;
        }

        return g_pAllocator->GetSizeOf(p);
    }
}


#else

void InitializeAllocator() NN_NOEXCEPT
{
}
size_t GetApplicationHeapSize() NN_NOEXCEPT
{
    return 0;
}

size_t GetTotalAllocatedSize() NN_NOEXCEPT
{
    return 0;
}

size_t GetTotalAllocatedSizeMax() NN_NOEXCEPT
{
    return 0;
}

#endif

