﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>

#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/util/util_ScopeExit.h>

#include "../Common/DevMenu_CommonScrollBox.h"
#include "../DevMenu_Config.h"
#include "../DevMenu_RootSurface.h"
#include "DevMenu_SaveData.h"
#include "DevMenu_SaveDataProperty.h"
#include "DevMenu_SaveDataUiPolicy.h"

namespace devmenu { namespace savedata {

/*********************************
 * class SaveDataListView
 *********************************/

 /**
 * @brief リストビュー
 */
class SaveDataListView : public glv::CustomVerticalListView< SaveDataPropertyType >
{
public:
    typedef std::function< void( const SaveDataListView::ItemType& item, const glv::Rect& contentRegion ) > CallbackFunction;

public:
    /**
     * @brief コンストラクタです。
     */
    explicit SaveDataListView( const glv::Rect& parentClipRegion, CallbackFunction callback ) NN_NOEXCEPT;

protected:
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE;

protected:
    std::function< void( const ItemType& item, const glv::Rect& contentRegion ) > m_OnDrawCallback;
};

/*********************************
 * class SaveDataListSceneBase
 *********************************/

class SaveDataListSceneBase : public SaveDataSceneBase
{
protected:

    class Header : public glv::Group
    {
    public:
        Header( glv::Label* pPageTitleLabel, const glv::Rect& rect ) NN_NOEXCEPT;

    public:
        void SetPageTitle( const char* titleName ) NN_NOEXCEPT { m_pPageTitle->setValue( titleName ); }

    private:
        glv::Table*   m_pTitleTable;
        glv::Label*   m_pPageTitle;
    };

    class Footer : public glv::Group
    {
    public:
        Footer( SaveDataListSceneBase* pParent, const glv::Rect& rect ) NN_NOEXCEPT;
        ~Footer() NN_NOEXCEPT {};
    };

protected:
    static const float HeaderDescLineFontSize;
    static const float HeaderDescLineLineHeight;

protected:
    /**
     * @brief コンストラクタ。
     */
    SaveDataListSceneBase( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect, bool isDefaultbackButtonCallbackUsed, const std::function< void() >& backButtonCallback = nullptr ) NN_NOEXCEPT;

    /**
     * @brief デストラクタ。
     */
    ~SaveDataListSceneBase() NN_NOEXCEPT {}

    /**
     * @brief セーブデータを問い合わせます。
     */
    virtual void QueryProperties() NN_NOEXCEPT = 0;

    /**
     * @brief プロパティコレクションを登録します。
     * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
     */
    void EntryProperties( const SaveDataListView::CollectionType* pItems ) NN_NOEXCEPT;

    /**
     * @brief 問い合わせ済プロパティデータを解放します。
     */
    virtual void FinalizeProperties() NN_NOEXCEPT;

    /**
     * @brief glv シーンレンダラ前のループ処理です。glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief リストの並び順をアプリケーション ID でソートします。
     */
    static void QuickSort( SaveDataListView::CollectionType* pItems ) NN_NOEXCEPT;

    /**
     * @brief In order to determine which scene is requesting a particular action, each List Scene needs to return its Scene Type.
     */
    virtual SceneType GetSceneType() NN_NOEXCEPT = 0;

protected:

    const AbstractOperators&                m_Op;
    SaveDataListView::CollectionType*       m_pItems;
    glv::Label*                             m_pLabelNoItem;
    Header*                                 m_pHeader;
    Footer                                  m_Footer;
    glv::Style                              m_StyleRed;
    std::function< void() >                 m_ListUserActionCallback;
    glv::View*                              m_pContainer;
    ScrollableBoxView*                      m_pScrollContainer;
};

/**********************************
 * class SaveDataTileListSceneBase
 **********************************/
class SaveDataTileListSceneBase : public SaveDataListSceneBase
{
public:
    /**
     * @brief セーブデータを検証します。
     */
    virtual void VerifySaveData() NN_NOEXCEPT;

    /**
     * @brief セーブデータを削除します。
     */
    virtual void DeleteSaveData() NN_NOEXCEPT;

    /**
     * @brief セーブデータを破壊します。
     */
    virtual void CorruptSaveData() NN_NOEXCEPT;

    void EnableFooterButtons( bool isEnabled ) NN_NOEXCEPT;

protected:
    /**
     * @brief コンストラクタ
     */
    SaveDataTileListSceneBase( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect, bool isDefaultbackButtonCallbackUsed, const std::function< void() >& backButtonCallback = nullptr ) NN_NOEXCEPT;

    /**
     * @brief SaveDataListTile を作成して ScrollContainer に追加します。
     */
    float AddSaveDataListTileToScrollContainer(
        const nn::ncm::ApplicationId& applicationId, nn::fs::SaveDataSpaceId spaceId, const glv::Rect& tileRect, bool isCurrentSelectedSaveDataValid, nn::fs::SaveDataId currentSelectedSaveDataId ) NN_NOEXCEPT;

    /**
     * @brief セーブデータを問い合わせます (実装部)。
     */
    void QueryPropertiesImpl( const nn::ncm::ApplicationId& applicationId, const std::vector< nn::fs::SaveDataSpaceId >& spaceIdList ) NN_NOEXCEPT;

    /**
     * @brief セーブデータを SD カードにエクスポートします。
     */
    virtual void ExportSaveData() NN_NOEXCEPT = 0;

    /**
     * @brief セーブデータを RAW データも含め SD カードにエクスポートします。
     */
    virtual void ExportRawSaveData() NN_NOEXCEPT = 0;

protected:
    devmenu::Button             m_ExportButton;
    devmenu::Button             m_OptionButton;
    const SaveDataPropertyType* m_pSelectedSaveDataProperty;

private:

    /**
     * @brief リスト更新通知ハンドラー (ユーザアクショントリガ)
     */
    static void OnSaveDataTileClickNotification( const glv::Notification& notification ) NN_NOEXCEPT;

    /**
     * @brief Initialize the Save Data info and show the Details scene
     */
    void ShowSaveDataDetails( const SaveDataPropertyType* const pSaveDataProperty ) NN_NOEXCEPT;

    /**
     * @brief セーブデータのVerify処理が完了した際に呼ぶコールバック関数です。
     */
    void OnCompleteVerifyProgress( const nn::Result& result ) NN_NOEXCEPT;

private:
    bool m_IsSaveDataTileFocusedOnRefresh;
};

/*********************************
 * class SaveDataListTile
 *********************************/
class SaveDataListTile : public glv::View
{
public:
    SaveDataListTile( const glv::Rect& rect, SaveDataPropertyType& item, nn::fs::SaveDataSpaceId spaceId, const std::function< void( const SaveDataPropertyType* ) >& callback ) NN_NOEXCEPT;

    void UpdateTextColor() NN_NOEXCEPT;

    virtual bool onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT NN_OVERRIDE;

private:
    static const glv::Label::Spec DefaultLabelSpec;

private:
    glv::Label*          m_pType;
    glv::Label*          m_pSaveDataId;
    glv::Label*          m_pTimeStamp; // 実際に表示されることはない
    glv::Label*          m_pSeperator;
    glv::Table           m_Table;
    glv::Style           m_StyleRed;
    SaveDataPropertyType m_SaveDataProperty;

    std::function< void( const SaveDataPropertyType* ) > m_OnFocusCallback;
};

/***********************************
* class SaveDataApplicationTileView
************************************/
class SaveDataApplicationTileView : public glv::View
{
public:
    explicit SaveDataApplicationTileView( const glv::Rect& rect ) NN_NOEXCEPT;

    void UpdateSaveDataApplicationTileViewValues( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

private:
    Table                   m_Table;
    glv::Label*             m_pApplicationTitle;
    glv::Label*             m_pIdValue;
};

/*****************************************
 * class SaveDataApplicationTileContainer
 *****************************************/
class SaveDataApplicationTileContainer : public glv::View
{
public:
    SaveDataApplicationTileContainer(
        SaveDataApplicationTileView* pSaveDataApplicationTile, const glv::Rect& rect, const nn::ncm::ApplicationId& applicationId, const std::function< void( const nn::ncm::ApplicationId* ) >& callback ) NN_NOEXCEPT;
    ~SaveDataApplicationTileContainer() NN_NOEXCEPT;

    virtual void onDraw( glv::GLV& glvContext ) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT NN_OVERRIDE;

    const nn::ncm::ApplicationId GetApplicationId() NN_NOEXCEPT;

private:
    nn::ncm::ApplicationId                                  m_ApplicationId;
    SaveDataApplicationTileView*                            m_pSaveDataApplicationTile;
    std::function< void( const nn::ncm::ApplicationId* ) >  m_OnFocusCallback;
};

/************************************
* class ApplicationSaveDataListScene
*************************************/

class ApplicationSaveDataListScene : public SaveDataListSceneBase
{
public:
    /**
     * @brief コンストラクタ。
     */
    ApplicationSaveDataListScene( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect ) NN_NOEXCEPT;

    /**
     * @brief デストラクタ。
     */
    ~ApplicationSaveDataListScene() NN_NOEXCEPT {}

    void ShowSaveDataTypeListScene() NN_NOEXCEPT;

    /**
     * @brief ユーザーセーブデータを問い合わせます。
     */
    virtual void QueryProperties() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief 問い合わせ済プロパティデータを解放します。
     */
    virtual void FinalizeProperties() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief glv シーンレンダラ前のループ処理です。glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    glv::View* GetFocusableChild() NN_NOEXCEPT;

    /**
     * @brief 描画内容をリフレッシュします。
     */
    virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE;

private:

    /**
     * @brief In order to determine which scene is requesting a particular action, each List Scene needs to return its Scene Type.
     */
    virtual SceneType GetSceneType() NN_NOEXCEPT NN_OVERRIDE { return SceneType_ApplicationSaveDataList; }

    /**
     * @brief Loads save data grouped by Application Id
     */
    void QueryPropertiesImpl( const std::vector< nn::fs::SaveDataSpaceId >& spaceIdList ) NN_NOEXCEPT;

    /**
     * @brief Handles the click of a Save Data Application Tile
     */
    static void OnApplicationTileClickNotification( const glv::Notification& notification ) NN_NOEXCEPT;

    /**
     * @brief Initialize the Save Data info and show the Details scene
     */
    void ShowSaveDataTypeListScene( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

private:

    const nn::ncm::ApplicationId* m_pSelectedApplicationId;
    bool                          m_IsApplicationTileFocusedOnRefresh;
    std::vector< SaveDataApplicationTileView* > m_SaveDataApplicationsTilesList;

    static const int                            MaxCountOfVisibleSaveDataApplicationsTiles = 6;
};

/*********************************
 * class SaveDataTypeListScene
 *********************************/

class SaveDataTypeListScene : public SaveDataTileListSceneBase
{
public:
    /**
     * @brief コンストラクタ。
     */
    SaveDataTypeListScene( const AbstractOperators& op, Page* pParentPage, const glv::Rect& rect, const std::function< void() >& backButtonCallback ) NN_NOEXCEPT;

    /**
     * @brief デストラクタ。
     */
    ~SaveDataTypeListScene() NN_NOEXCEPT {}

    /**
     * @brief ユーザーセーブデータを問い合わせます。
     */
    virtual void QueryProperties() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief 問い合わせ済プロパティデータを解放します。
     */
    virtual void FinalizeProperties() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief 選択されているアプリケーションの情報を更新します。
     */
    void UpdateSelectedApplicationInformation( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

    /**
     * @brief glv シーンレンダラ前のループ処理です。glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    glv::View* GetFocusableChild() NN_NOEXCEPT;

    /**
     * @brief ユーザーセーブデータを SD カードにエクスポートします。
     */
    virtual void ExportSaveData() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ユーザーセーブデータを RAW データも含め SD カードにエクスポートします。
     */
    virtual void ExportRawSaveData() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief 描画内容をリフレッシュします。
     */
    virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE;

private:
    void SetDefaultFocusedView() NN_NOEXCEPT;

    /**
     * @brief In order to determine which scene is requesting a particular action, each List Scene needs to return its Scene Type.
     */
    virtual SceneType GetSceneType() NN_NOEXCEPT NN_OVERRIDE { return SceneType_SaveDataTypeList; }

    /**
    * @brief セーブデータのエクスポート処理が完了した際に呼ぶコールバック関数です。
    */
    void OnCompleteExportProgress( const nn::Result& result ) NN_NOEXCEPT;

private:
    devmenu::Button*        m_pBackButton;
    nn::ncm::ApplicationId  m_ApplicationId;
};

/*********************************
 * class SystemSaveDataListScene
 *********************************/

class SystemSaveDataListScene : public SaveDataTileListSceneBase
{
public:
    SystemSaveDataListScene( const AbstractOperators& op, devmenu::Page* pParentPage, const glv::Rect& rect ) NN_NOEXCEPT;

    ~SystemSaveDataListScene() NN_NOEXCEPT {}

    /**
     * @brief セーブデータを問い合わせます。
     */
    virtual void QueryProperties() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief 問い合わせ済プロパティデータを解放します。
     */
    virtual void FinalizeProperties() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief glv シーンレンダラ前のループ処理です。glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief システムセーブデータを SD カードにエクスポートします。
     */
    virtual void ExportSaveData() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief システムセーブデータを RAW データも含め SD カードにエクスポートします。
     */
    virtual void ExportRawSaveData() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief 描画内容をリフレッシュします。
     */
    virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE;

private:

    /**
     * @brief In order to determine which scene is requesting a particular action, each List Scene needs to return its Scene Type.
     */
    virtual SceneType GetSceneType() NN_NOEXCEPT NN_OVERRIDE;
};

}} // ~namespace devmenu::savedata, ~namespace devmenu

