﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * インクルード
 */

#include <cstring>

#include "QREL.h"

/*
 * 定数定義
 */
enum ConstantValue_LoadToMatrix
{
/* 向き用フラグ（-1は上向き） */
    ScanUpper = -1,
    ScanDowner = 0,
    CellOutArea = (uint8_t)0x80,     /* 除外エリア */
};
/*
 * 内部関数定義
 */
static bool SetOutofArea(uint8_t ***matrix, VERSIONINFO *versioninfo);
static bool OutofAllPCArea(int32_t VersionInfo, uint8_t ***matrix);
static bool OutofPCArea(int32_t x, int32_t y, uint8_t ***matrix);
static short SetDataWordBit(uint8_t *Cell, uint8_t DataWord,
                             short bitCount);
/****************************************************************
 * 外部関数
 ****************************************************************/
/*
 * 関数名 QREL_LoadToMatrix
 * 機能   最終データワード列をマトリックスに配置する
 * 引数   versioninfo(IN)
 *        ecclevel(IN)
 *        ncodes(IN)
 *        code(IN)
 *        matrix(OUT)
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *        型番に則したマトリックスを作成し、それに対して
 *        機能パターンを配置し、それをよけつつ最終データワード列を配置する。
 *
 */
bool QREL_LoadToMatrix(VERSIONINFO versioninfo, int32_t ncodes,
                        uint8_t *code,
                        uint8_t ***matrix)
{
    /* 内部変数宣言 */
    bool ret = true;
    int32_t i = 0, j = 0, k = 0;
    int32_t ncells = 0;          /* 一辺あたりのセル数 */
    short   bitCount = 7;        /* データワードの完了ビット数 */
    int32_t flg = ScanUpper;    /* 向き用フラグ（-1は上向き） */

    QREL_SetLastError(ERR_NOERROR, true);
    /* 型番に則したマトリックス作成 */
    if (!VER_GetnCellNum(versioninfo.version, &ncells))
        return false;
    *matrix = (uint8_t **)QRE_Malloc(ncells * sizeof(uint8_t *));
    for (i = 0; i < ncells; i++) {
        (*matrix)[i] = (uint8_t *)QRE_Malloc(ncells);
        memset((*matrix)[i], 0x00, ncells);
    }
    /* 機能パターンをロードし配置（除外エリアコード） */
    ret = SetOutofArea(matrix, &versioninfo);
    if (!ret)
        return false;

    /* 最終データ列を配置 */
    for (j = ncells - 1; j >= 1; j -= 2) {
        if (flg == ScanUpper) {
            for (i = ncells - 1; i >= 1; i--) {
                if (k == ncodes)
                    break;
                if (j <= 6) {
                    if (i <= 6)
                        bitCount
                            = SetDataWordBit(&(*matrix)[i - 1][j - 1],
                                              *(code + k),
                                              bitCount);
                    else
                        bitCount
                            = SetDataWordBit(&(*matrix)[i][j - 1], *(code + k),
                                              bitCount);
                }
                else {
                    if (i <= 6)
                        bitCount
                            = SetDataWordBit(&(*matrix)[i - 1][j], *(code + k),
                                              bitCount);
                    else
                        bitCount = SetDataWordBit(&(*matrix)[i][j], *(code + k),
                                                   bitCount);
                }
                if (bitCount == -1) {
                    bitCount = 7;
                    k++;
                }
                if (k == ncodes)
                    break;
                if (j <= 6) {
                    if (i <= 6)
                        bitCount
                            = SetDataWordBit(&(*matrix)[i - 1][j - 1 - 1],
                                              *(code + k),
                                              bitCount);
                    else
                        bitCount
                            = SetDataWordBit(&(*matrix)[i][j - 1 - 1],
                                              *(code + k),
                                              bitCount);
                }
                else {
                    if (i <= 6)
                        bitCount
                            = SetDataWordBit(&(*matrix)[i - 1][j - 1],
                                              *(code + k),
                                              bitCount);
                    else
                        bitCount
                            = SetDataWordBit(&(*matrix)[i][j - 1], *(code + k),
                                              bitCount);
                }
                if (bitCount == -1) {
                    bitCount = 7;
                    k++;
                }
            }
            flg = ScanDowner;
        } /* SCAN_UPPER */
        else { /* SCAN_DOWNER */
            for (i = 1; i < ncells; i++) {
                if (k == ncodes)
                    break;
                if (j <= 6) {
                    if (i <= 6)
                        bitCount
                            = SetDataWordBit(&(*matrix)[i - 1][j - 1],
                                              *(code + k),
                                              bitCount);
                    else
                        bitCount
                            = SetDataWordBit(&(*matrix)[i][j - 1], *(code + k),
                                              bitCount);
                }
                else {
                    if (i <= 6)
                        bitCount
                            = SetDataWordBit(&(*matrix)[i - 1][j], *(code + k),
                                              bitCount);
                    else
                        bitCount = SetDataWordBit(&(*matrix)[i][j], *(code + k),
                                                   bitCount);
                }
                if (bitCount == -1) {
                    bitCount = 7;
                    k++;
                }
                if (k == ncodes)
                    break;
                if (j <= 6) {
                    if (i <= 6)
                        bitCount
                            = SetDataWordBit(&(*matrix)[i - 1][j - 1 - 1],
                                              *(code + k),
                                              bitCount);
                    else
                        bitCount
                            = SetDataWordBit(&(*matrix)[i][j - 1 - 1],
                                              *(code + k),
                                              bitCount);
                }
                else {
                    if (i <= 6)
                        bitCount
                            = SetDataWordBit(&(*matrix)[i - 1][j - 1],
                                              *(code + k),
                                              bitCount);
                    else
                        bitCount
                            = SetDataWordBit(&(*matrix)[i][j - 1], *(code + k),
                                              bitCount);
                }
                if (bitCount == -1) {
                    bitCount = 7;
                    k++;
                }
            }
            flg = ScanUpper;
        } /* SCAN_DOWNER */
    } /* j */

    return true;
}// NOLINT(impl/function_size)

/****************************************************************
 * 内部関数
 ****************************************************************/
/*
 * 関数名 SetOutofArea
 * 機能   データワード書き込みのために機能パターンエリアに
 *        除外コードをセットする。
 * 引数   ncodes(IN)            コード数
 *          matrix(IN/OUT)      マトリックス
 *        versioninfo(IN)       型番情報
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *
 */
bool SetOutofArea(uint8_t ***matrix, VERSIONINFO *versioninfo)
{
    int32_t i = 0, j = 0;
    int32_t ncells; /* 一辺あたりのセル数
                     * 位置検出パターン領域：左上・左下・右上（8×8）
                     */
    if (!VER_GetnCellNum(versioninfo->version, &ncells)) {
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        return false;
    }
    for (i = 0; i < 8; i++)
        for (j = 0; j < 8; j++)
            (*matrix)[i][j] = CellOutArea;
    for (i = 0; i < 8; i++)
        for (j = ncells - 1; j > ncells - 1 - 8; j--)
            (*matrix)[i][j] = CellOutArea;
    for (i = ncells - 1; i > ncells - 1 - 8; i--)
        for (j = 0; j < 8; j++)
            (*matrix)[i][j] = CellOutArea;
    /* 位置合わせパターン領域を除外する */
    OutofAllPCArea(versioninfo->version, matrix);
    /* 型番情報領域を除外する
     * 型番情報領域：２箇所
     */
    if (versioninfo->version > 6) { /* 型番７以上の場合 */
        for (i = 0; i < 6; i++)
            for (j = ncells - 1 - 8; j > ncells - 1 - 8 - 3; j--)
                (*matrix)[i][j] = CellOutArea;
        for (i = ncells - 1 - 8; i > ncells - 1 - 8 - 3; i--)
            for (j = 0; j < 6; j++)
                (*matrix)[i][j] = CellOutArea;
    }
    /* 形式情報領域：４箇所（線分）*/
    for (i = 0; i < 8; i++)
        (*matrix)[i][8] = CellOutArea;
    for (i = 0; i < 9; i++)
        (*matrix)[8][i] = CellOutArea;
    for (i = ncells - 1; i > ncells - 1 - 8; i--)
        (*matrix)[i][8] = CellOutArea;
    for (i = ncells - 1; i > ncells - 1 - 8; i--)
        (*matrix)[8][i] = CellOutArea;
     /* タイミングパターン：２箇所 */
    for (i = 0; i < ncells; i++)
        (*matrix)[i][6] = CellOutArea;
    for (i = 0; i < ncells; i++)
        (*matrix)[6][i] = CellOutArea;
    return true;
}

/****************************************************************
 * 内部関数
 ****************************************************************/
/*
 * 関数名 OutofAllPCArea
 * 機能   型番情報ごとに全ての位置合わせパターン領域を除外する。
 * 引数   matrix(IN/OUT)      マトリックス
 *        VersionInfo(IN)     型番情報
 *        bitCount(IN)
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *
 */
bool OutofAllPCArea(int32_t VersionInfo, uint8_t ***matrix)
{
    /* 変数宣言 */
    int32_t x[PopMaxNum], y[PopMaxNum];   /* 位置合わせパターンの中心座標 */
    int32_t popnum = 0;                   /* 位置合わせパターン数 */
    int32_t i = 0;
    /* 処理部 */
    if (VER_GetPOPCoord(VersionInfo, &popnum, x, y) == false) {
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        return false;
    }
    else
        for (i = 0; i < popnum; i++)
            OutofPCArea(x[i], y[i], matrix);
    return true;
}

/*
 * 関数名 OutofPCArea
 * 機能   引数で指定された中心点から位置合わせパターン領域を除外する。
 * 引数   cp(IN)            中心点のX座標
 *        matrix(IN/OUT)    マトリックス
 * 戻り値 正常終了時0以外を異常終了時0を返す
 * 説明
 *
 */
bool OutofPCArea(int32_t x, int32_t y, uint8_t ***matrix)
{
    /* 変数宣言 */
    int32_t i = 0, j = 0;           /* ループ用変数
                                     * 処理部
                                     */
    for (i = y - 2; i < y - 2 + 5; i++)
        for (j = x - 2; j < x - 2 + 5; j++)
            (*matrix)[i][j] = CellOutArea;
    return true;    /* 正常 */
}

/*
* 関数名 SetDataWordBit
* 機能   データワードをセルごとに取得判断し配置する
* 引数   Cell(IN)                セルデータ
*        DataWord(IN/OUT)        データワード格納変数
*          bitCount(IN)
* 戻り値 ビット完了数
* 説明
*
*/
short SetDataWordBit(uint8_t *Cell, uint8_t DataWord,
                      short bitCount)
{
    /* セルのデータ領域でない場合は次へジャンプ（何もしないで戻る） */
    if (*Cell == CellOutArea)
        return bitCount;
    if ((DataWord >> bitCount) & 0x01)
        *Cell = 0xFF;
    else
        *Cell = 0x00;
    return bitCount - 1;
}
