﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * インクルード
 */
#include <cstdlib>
#include <cstring>

#include "QREL.h"

/*
 * マクロ定義
 */
#if 0
#define DEBUG_LOG    /* デバッグログ出力用マクロ */
#include <cstdio>
#include <nn.h>
#endif

/*
 * 定数定義
 */
enum ConstantValue_DataEncode
{
    CountAlphabetNum = 45,
};

/*
 * 構造体定義
 */
/* モードセグメント */
typedef struct {
    int32_t start;              /* 開始バイト位置 */
    int32_t end;                /* 終了バイト位置 */
    int32_t charlength;         /* 文字数指示子
                                   漢字のときは2バイトで1文字 */
    MODE mode;                  /* モード指示子 */
    uint8_t padding[3];
} MODESEG;

/*
 * 外部変数定義
 */
static int32_t AlphaNumConvert[CountAlphabetNum] = {
    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J',
    'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T',
    'U', 'V', 'W', 'X', 'Y', 'Z', ' ', '$', '%', '*',
    '+', '-', '.', '/', ':'
};
static int32_t maxbit;  /* データコード語最大ビット数 */

/*
 * 内部関数定義
 */
static void SetBit(uint8_t *databitstream, int32_t bitpos)
{
    databitstream[bitpos / 8] |= (1 << (7 - bitpos % 8));
}

#if 0
static int32_t CalcDataBitLength(int32_t version, MODE mode, int32_t datalen);
static int32_t CalcCharLengthFieldLength(int32_t version, MODE mode);
#endif
static int32_t SetData8BitsByte(int32_t bitpos, uint8_t *code,
                             uint8_t *data,
                             int32_t datalen);
static int32_t SetDataNum(int32_t bitpos, uint8_t *code, uint8_t *data,
                       int32_t datalen);
static int32_t SetDataAlphaNum(int32_t bitpos, uint8_t *code,
                            uint8_t *data,
                            int32_t datalen);
static int32_t SetDataKanji(int32_t bitpos, uint8_t *code, uint8_t *data,
                         int32_t datalen);
static int32_t GetAlphaNumCode(int32_t alphanum);
static int32_t GetNumCode(int32_t num);
static int32_t GetKanjiGroupCode(int32_t kanji);
static int32_t MakeMixedModeSegments(int32_t version, USERDATA *userdata,
                                  MODESEG **seg);
#ifdef DEBUG_LOG
static void DumpModeSegments(const int8_t *id, int32_t version, int32_t nseg,
                              MODESEG *seg, USERDATA *userdata,
                              FILE *fp);
#endif /* DEBUG_LOG */
static int32_t PackModeSegmentArray(int32_t nseg, MODESEG *seg, int32_t pos);
static bool Join2ModeSegments(MODESEG *seg, int32_t version);
static bool MakeSymbolChainInfo(uint16_t *chain, int32_t seqno,
                                 int32_t totalcount,
                                 uint8_t parity);
/****************************************************************
 * 外部関数
 ****************************************************************/
/*
 * 関数名 QREL_DataEncode
 * 機能   型番、誤り訂正レベル、モード指示子をもとにデータをエンコード
 *          する。
 * 引数   versioninfo(IN)         型番情報へのポインタ
 *        ecclevel(IN)            誤り訂正レベルへのポインタ
 *        mode(IN)                モード指示子へのポインタ
 *        userdata(IN)            ユーザデータへのポインタ
 *          ncodes(OUT)           総コード語数格納領域へのポインタ
 *          code(OUT)             コード語のポインタ格納領域へのポインタ
 *          chaininfo(IN)         連結情報のポインタ
 *                                連結モードでないときはNULLを指定する。
 * 戻り値 正常終了時0以外を異常終了時0を返す
 *          異常終了時のエラーコードはQREL_GetLastError関数で取得できる。
 *          エラーコードは以下。
 *          ERR_ILLEGALPARAM      指定されたモード指示子が不正
 *          ERR_VERSIONINFOERROR  指定された型番が不正
 *          ERR_DATAFORMATERROR   指定されたモード指示子でエンコードできない文字
 *                                がユーザデータに含まれている
 *          ERR_ILLEGALPARAM      連結情報のパラメータが不正
 * 説明   型番、誤り訂正レベル、モード指示子をもとにデータをエンコード
 *          する。
 *
 */
bool QREL_DataEncode(VERSIONINFO *versioninfo, ECCLEVEL *ecclevel, MODE *mode,
                      USERDATA *userdata, int32_t *ncodes, uint8_t **code,
                      CHAININFO *chaininfo)
{
    int32_t bitpos;
    int32_t i;
    int32_t lengthfieldlength;
    int32_t capacity;
    ERRCORUNIT rsinfo;
    int32_t data_capacity;
    int32_t s;
    bool overflow;
    int32_t nseg;               /* モードセグメントの個数 */
    MODESEG *seg = NULL;        /* モードセグメントの配列 */
    uint16_t chain = 0;

    QREL_SetLastError(ERR_NOERROR, true);
    if (chaininfo != NULL) {
        if (!MakeSymbolChainInfo(&chain, chaininfo->seqno,
                                  chaininfo->totalcount,
                                  chaininfo->parity))
        {
            QREL_SetLastError(ERR_ILLEGALPARAM, true);
            return false;
        }
    }
    switch (*mode) {
    case MODE_NUM:
    case MODE_ALPHANUM:
    case MODE_8BITSBYTE:
        nseg = 1;
        seg = reinterpret_cast<MODESEG *>(QRE_Malloc(nseg * sizeof(MODESEG)));
        seg[0].mode = *mode;
        seg[0].start = 0;
        seg[0].end = userdata->length - 1;
        seg[0].charlength = userdata->length;
        break;

    case MODE_KANJI:
        if (userdata->length % 2) {   // BUG 20021225 追加
            QREL_SetLastError(ERR_DATAFORMATERROR, true);
            return false;
        }
        nseg = 1;
        seg = reinterpret_cast<MODESEG *>(QRE_Malloc(nseg * sizeof(MODESEG)));
        seg[0].mode = *mode;
        seg[0].start = 0;
        seg[0].end = userdata->length - 1;
        seg[0].charlength = userdata->length / 2;
        break;

    case MODE_ANY:
        nseg = MakeMixedModeSegments(versioninfo->version, userdata, &seg);
        break;

    default:
        QREL_SetLastError(ERR_ILLEGALPARAM, true);
        return false;
    }
#ifdef DEBUG_LOG
    {
        // FILE *fp;
        int32_t s, d;
        // fp = fopen("modeseg.txt", "w");
        for (s = 0; s < nseg; s++) {
            int8_t smode[128];
            switch (seg[s].mode) {
            case MODE_CHAIN:
                strcpy(smode, "CHAIN");
                break;

            case MODE_NUM:
                strcpy(smode, "NUM");
                break;

            case MODE_ALPHANUM:
                strcpy(smode, "ALPHANUM");
                break;

            case MODE_8BITSBYTE:
                strcpy(smode, "8BITBYTE");
                break;

            case MODE_KANJI:
                strcpy(smode, "KANJI");
                break;

            case MODE_TERM:
                strcpy(smode, "TERM");
                break;

            case MODE_ANY:
                strcpy(smode, "ANY");
                break;

            default:
                strcpy(smode, "ERROR");
                break;
            }
            NN_LOG("%02d - mode: %s start: %d end: %d charlength: %d\n",
                    s, smode, seg[s].start, seg[s].end, seg[s].charlength);
            if (seg[s].mode != MODE_KANJI) {
                for (d = seg[s].start; d <= seg[s].end; d++) {
                    int8_t fmt[128];
                    if (seg[s].mode == MODE_8BITSBYTE)
                        strcpy(fmt, "%02x");
                    else
                        strcpy(fmt, "%c");
                    NN_LOG(fmt, (uint8_t)userdata->data[d]);
                }
            }
            else {
                int8_t *kbuff = reinterpret_cast<int8_t *>(QRE_Malloc(userdata->length + 1));
                memset(kbuff, 0, userdata->length + 1);
                strncpy(kbuff, userdata->data + seg[s].start,
                        seg[s].end - seg[s].start + 1);
                NN_LOG("%s", kbuff);
                QRE_Free(kbuff);
            }
            NN_LOG("\n");
        }
        // fclose(fp);
    }
#endif  /* DEBUG_LOG */
    if (!VER_GetRSInfo(versioninfo->version, *ecclevel, &rsinfo)) {
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        QRE_Free(seg);
        seg = NULL;
        return false;
    }
    if (!VER_GetAllDataWord(versioninfo->version, &capacity)) {
        QREL_SetLastError(ERR_VERSIONINFOERROR, true);
        QRE_Free(seg);
        seg = NULL;
        return false;
    }
    data_capacity = capacity - rsinfo.ErrcorCodeNum;
    maxbit = data_capacity * 8;
    *ncodes = capacity;
    *code = (uint8_t *)QRE_Malloc(*ncodes);
    memset(*code, 0, *ncodes);

    bitpos = 0;
    overflow = false;
    /* 連結モード指示子 */
    if (chaininfo != NULL) {
        if (bitpos + 4 > maxbit)
            /* 容量オーバー */
            overflow = true;
        else {
            for (i = 0; i < 4; i++) {
                static MODE chainmode = MODE_CHAIN;
                if (chainmode & (1 << (4 - 1 - i)))
                    SetBit(*code, bitpos);
                bitpos++;
            }
        }
    }
    /* シンボル列指示子 */
    if (!overflow && chaininfo != NULL) {
        if (bitpos + 16 > maxbit)
            /* 容量オーバー */
            overflow = true;
        else {
            for (i = 0; i < 16; i++) {
                if (chain & (1 << (16 - 1 - i)))
                    SetBit(*code, bitpos);
                bitpos++;
            }
        }
    }
    if (!overflow) {
        for (s = 0; s < nseg; s++) {
            int32_t j;
            /* モード指示子 */
            if (bitpos + 4 > maxbit) {
                /* 容量オーバー */
                overflow = true;
                break;
            }
            for (j = 0; j < 4; j++) {
                if (seg[s].mode & (1 << (4 - 1 - j)))
                    SetBit(*code, bitpos);
                bitpos++;
            }
            /* 文字数指示子 */
            lengthfieldlength
                = CalcCharLengthFieldLength(versioninfo->version, seg[s].mode);
            if (bitpos + lengthfieldlength > maxbit) {
                /* 容量オーバー */
                overflow = true;
                break;
            }
            for (j = 0; j < lengthfieldlength; j++) {
                if (seg[s].charlength & (1 << (lengthfieldlength - 1 - j)))
                    SetBit(*code, bitpos);
                bitpos++;
            }
            /* データビット列 */
            if (bitpos
                + CalcDataBitLength(versioninfo->version, seg[s].mode,
                                     seg[s].charlength)
                - 4                      /* モード指示子のビット長 */
                - lengthfieldlength      /* 文字数指示子のビット長 */
                > maxbit)
            {
                /* 容量オーバー */
                overflow = true;
                break;
            }
            switch (seg[s].mode) {
            case MODE_8BITSBYTE:
                bitpos
                    = SetData8BitsByte(bitpos, *code,
                                       reinterpret_cast<uint8_t *>(userdata->data + seg[s].start),
                                       seg[s].charlength);
                break;

            case MODE_NUM:
                bitpos = SetDataNum(bitpos, *code,
                                    reinterpret_cast<uint8_t *>(userdata->data + seg[s] .start),
                                    seg[s].charlength);
                break;

            case MODE_ALPHANUM:
                bitpos = SetDataAlphaNum(bitpos, *code,
                                         reinterpret_cast<uint8_t *>(userdata->data + seg[s].start),
                                         seg[s].charlength);
                break;

            case MODE_KANJI:
                bitpos = SetDataKanji(bitpos, *code,
                                      reinterpret_cast<uint8_t *>(userdata->data + seg[s].start),
                                      seg[s].charlength * 2);
                break;

            default:
                QRE_Free(*code);
                *code = NULL;
                QRE_Free(seg);
                seg = NULL;
                QREL_SetLastError(ERR_ILLEGALPARAM, true);
                return false;
            }
            if (bitpos < 0) {
                QRE_Free(*code);
                *code = NULL;
                QRE_Free(seg);
                seg = NULL;
                QREL_SetLastError(ERR_DATAFORMATERROR, true);
                return false;
            }
        }
        /* 終了パターン 0000 を付加する。*/
        bitpos += 4; /* 4: B0000 */
        if (bitpos > maxbit)
            bitpos = maxbit;
         /* 8ビットバウンダリに満たない場合は0を埋める */
        while (bitpos % 8 != 0) {
            /* ビットスキップ すでにゼロクリアされている */
            bitpos++;
        }
        /* "yagi@isp21.co.jp"(8ビットバイト)->
                41 07 96 16 76 94 06 97 37 03 23 12 E6 36 F2 E6 A7 00 */

        /* 埋め草コードを埋める。 */
        i = 15;
        while (bitpos / 8 + 1 <= data_capacity) {
            static uint16_t pad = 0xec11;
            if (pad & (1 << i))
                SetBit(*code, bitpos);
            bitpos++;
            if (i == 0)
                i = 15;
            else
                i--;
        }
    }
    QRE_Free(seg);
    seg = NULL;
    if (overflow) {
        QRE_Free(*code);
        *code = NULL;
        QREL_SetLastError(ERR_TOOSMALL, true);
        return false;
    }
#ifdef DEBUG_LOG
    {
        int8_t *c = reinterpret_cast<int8_t *>(QRE_Malloc(bitpos + 1));
        int32_t i;
        // FILE *fp;
        memset(c, 0, bitpos + 1);
        for (i = 0; i < bitpos; i++) {
            int32_t t_bytepos = i / 8;
            int32_t t_bitpos = i % 8;
            c[i] = ((*code)[t_bytepos] & (1 << (7 - t_bitpos))) ? '1' : '0';
        }
        // fp = fopen("datacode1.txt", "w");
        NN_LOG("%s\n", c);
        // fclose(fp);
        QRE_Free(c);
    }
#endif /* DEBUG_YAGI */

    return true;
}// NOLINT(impl/function_size)

/****************************************************************
 * 内部関数
 ****************************************************************/
/*
 * 関数名 CalcDataBitLength
 * 機能   データビット列の長さを求める
 * 引数   version(IN)            型番
 *        mode(IN)               モード指示子
 *        datalen(IN)            ユーザデータの長さ
 *                               モード指示子が漢字のときは漢字1文字を1とする。
 * 戻り値 0以上のときはデータビット列の長さを返す。
 *          -1のときはエラーとする。
 * 説明   データビット列の長さを求める。
 */
int32_t CalcDataBitLength(int32_t version, MODE mode, int32_t datalen)
{
    int32_t bitlen;
    switch (mode) {
    case MODE_8BITSBYTE:    /* 4 + C + 8D */
        bitlen = 4;
        /* C */
        bitlen += CalcCharLengthFieldLength(version, mode);
        /* 8D */
        bitlen += 8 * datalen;
        break;

    case MODE_NUM:          /* 4 + C + 10(D DIV 3) + R */
        bitlen = 4;
        /* C */
        bitlen += CalcCharLengthFieldLength(version, mode);
        /* 10(D DIV 3) */
        bitlen += 10 * (datalen / 3);
        /* R */
        switch (datalen % 3) {
        case 0:
            bitlen += 0;
            break;

        case 1:
            bitlen += 4;
            break;

        case 2:
            bitlen += 7;
            break;

        default:
            break;
        }
        break;

    case MODE_ALPHANUM:         /* 4 + C + 11(D DIV 2) + 6(D MOD 2) */
        bitlen = 4;
        /* C */
        bitlen += CalcCharLengthFieldLength(version, mode);
        /* 11(D DIV 2) */
        bitlen += 11 * (datalen / 2);
        /* 6(D MOD 2) */
        bitlen += 6 * (datalen % 2);
        break;

    case MODE_KANJI:            /* 4 + C + 13D */
        bitlen = 4;
        /* C */
        bitlen += CalcCharLengthFieldLength(version, mode);
        /* 13D */
        bitlen += 13 * datalen;
        break;

    default:
        bitlen = -1;
        break;
    }
    return bitlen;
}// NOLINT(impl/function_size)

/*
 * 関数名 CalcCharLengthFieldLength
 * 機能   文字数指示子の長さを求める
 * 引数   version(IN)             型番
 *        mode(IN)                モード指示子
 * 戻り値 0以上のときは文字数指示子の長さのビット数を返す。
 *          -1のときはエラーとする。
 * 説明   文字数指示子の長さを求める。
 */
int32_t CalcCharLengthFieldLength(int32_t version, MODE mode)
{
    int32_t bitlen = -1;
    switch (mode) {
    case MODE_8BITSBYTE:
        if (version <= 9)
            bitlen = 8;
        else if (version <= 26)
            bitlen = 16;
        else if (version <= 40)
            bitlen = 16;
        break;

    case MODE_NUM:
        if (version <= 9)
            bitlen = 10;
        else if (version <= 26)
            bitlen = 12;
        else if (version <= 40)
            bitlen = 14;
        break;

    case MODE_ALPHANUM:
        if (version <= 9)
            bitlen = 9;
        else if (version <= 26)
            bitlen = 11;
        else if (version <= 40)
            bitlen = 13;
        break;

    case MODE_KANJI:
        if (version <= 9)
            bitlen = 8;
        else if (version <= 26)
            bitlen = 10;
        else if (version <= 40)
            bitlen = 12;
        break;

    default:
        bitlen = -1;
        break;
    }
    return bitlen;
}

/*
 * 関数名 SetData8BitsByte
 * 機能   データビット列に8ビットバイトのデータを格納する
 * 引数   bitpos(IN)            ビット位置 0, 1, 2, 3...
 *        code(IN/OUT)          データビット列
 *        data(IN)              格納するデータ
 *        datalen(IN)           格納するデータのバイト長
 * 戻り値 ビット位置を返す
 * 説明   データビット列に8ビットバイトのデータを格納する。
 */
static int32_t SetData8BitsByte(int32_t bitpos, uint8_t *code,
                             uint8_t *data,
                             int32_t datalen)
{
    int32_t i;
    for (i = 0; i < 8 * datalen; i++) {
        int32_t t_bytepos = i / 8;
        int32_t t_bitpos = 7 - i % 8;
        if (data[t_bytepos] & (1 << t_bitpos))
            SetBit(code, bitpos);
        bitpos++;
    }
    return bitpos;
}

/*
 * 関数名 SetDataNum
 * 機能   データビット列に数字のデータを格納する
 * 引数   bitpos(IN)          ビット位置 0, 1, 2, 3...
 *        code(IN/OUT)        データビット列
 *        data(IN)            格納するデータ
 *        datalen(IN)         格納するデータのバイト長
 * 戻り値 正常終了時ビット位置をまた異常終了時-1を返す
 * 説明   データビット列に数字のデータを格納する。
 *        データが数字以外の文字を含む場合戻り値-1を返す。
 */
static int32_t SetDataNum(int32_t bitpos, uint8_t *code, uint8_t *data,
                       int32_t datalen)
{
    int32_t i;
    int32_t n;

    n = datalen / 3 + (datalen % 3 ? 1 : 0);
    for (i = 0; i < n; i++) {
        int8_t s_num[3 + 1];
        int32_t num;
        int32_t bitlen;
        int32_t j;
        int32_t m;

        memset(s_num, 0, 3 + 1);
        if (i < n - 1 || datalen % 3 == 0)
            m = 3;
        else if (datalen % 3 == 2)
            m = 2;
        else
            m = 1;
        strncpy(reinterpret_cast<char *>(s_num), reinterpret_cast<char *>(data + 3 * i), m);
        // strncpy_s(s_num, 3+1, data+3*i, m);
        for (j = 0; j < 3; j++) {
            if (s_num[j] == 0)
                break;
            if (GetNumCode(s_num[j]) == -1)
                return -1;  /* 異常終了：数字以外の文字あり */
        }
        num = atoi(reinterpret_cast<char *>(s_num));
        if (i < n - 1 || datalen % 3 == 0)
            /* 10ビット */
            bitlen = 10;
        else if (datalen % 3 == 2)
            /* 7ビット */
            bitlen = 7;
        else
            /* 4ビット */
            bitlen = 4;
        for (j = bitlen - 1; j >= 0; j--) {
            if (num & 1 << j)
                SetBit(code, bitpos);
            bitpos++;
        }
    }
    return bitpos;
}

/*
 * 関数名 SetDataAlphaNum
 * 機能   データビット列に英数字のデータを格納する
 * 引数   bitpos(IN)            ビット位置 0, 1, 2, 3...
 *        code(IN/OUT)          データビット列
 *        data(IN)              格納するデータ
 *        datalen(IN)           格納するデータのバイト長
 * 戻り値 正常終了時ビット位置をまた異常終了時-1を返す
 * 説明   データビット列に英数字のデータを格納する。
 *        データが数字以外の文字を含む場合戻り値-1を返す。
 */
static int32_t SetDataAlphaNum(int32_t bitpos, uint8_t *code,
                            uint8_t *data,
                            int32_t datalen)
{
    int32_t i;
    int32_t n = datalen / 2 + ((datalen % 2) ? 1 : 0);
    for (i = 0; i < n; i++) {
        int32_t j;
        int32_t bitlen;
        int32_t num;
        if (i < n - 1 || datalen % 2 == 0) {
            int32_t a, b;
            bitlen = 11;
            a = GetAlphaNumCode(data[2 * i]);
            if (a < 0)
                return -1;  /* 異常終了：英数字以外の文字あり */
            b = GetAlphaNumCode(data[2 * i + 1]);
            if (b < 0)
                return -1;  /* 異常終了：英数字以外の文字あり */
            num = a * 45 + b;
        }
        else {
            int32_t a;
            bitlen = 6;
            a = GetAlphaNumCode(data[2 * i]);
            if (a < 0)
                return -1;  /* 異常終了：英数字以外の文字あり */
            num = a;
        }
        for (j = bitlen - 1; j >= 0; j--) {
            if (num & 1 << j)
                SetBit(code, bitpos);
            bitpos++;
        }
    }
    return bitpos;
}

/*
 * 関数名 SetDataKanji
 * 機能   データビット列に漢字のデータを格納する
 * 引数   bitpos(IN)            ビット位置 0, 1, 2, 3...
 *        code(IN/OUT)          データビット列
 *        data(IN)              格納するデータ
 *        datalen(IN)           格納するデータのバイト長
 * 戻り値 正常終了時ビット位置をまた異常終了時-1を返す
 * 説明   データビット列に漢字のデータを格納する。
 *        データが漢字以外の文字を含む場合戻り値-1を返す。
 */
static int32_t SetDataKanji(int32_t bitpos, uint8_t *code, uint8_t *data,
                         int32_t datalen)
{
    int32_t i;
    int32_t n = datalen / 2;
    if (datalen % 2)
        return -1;  /* 異常終了：漢字以外の文字あり */
    for (i = 0; i < n; i++) {
        int32_t j;
        int32_t bitlen;
        int32_t kanji, kanji1;
        int32_t num;

        kanji = data[2 * i] << 8 | data[2 * i + 1];
        switch (GetKanjiGroupCode(kanji)) {
        case 1: /* 0x8140 - 0x9ffc */
            kanji1 = kanji - 0x8140;
            break;

        case 2: /* 0xe040 - 0xebbf */
            kanji1 = kanji - 0xc140;
            break;

        default:
            return -1;  /* 異常終了：漢字以外の文字あり */
        }
        num = (kanji1 >> 8) * 0xc0 + (kanji1 & 0x00ff);
        bitlen = 13;
        for (j = bitlen - 1; j >= 0; j--) {
            if (num & 1 << j)
                SetBit(code, bitpos);
            bitpos++;
        }
    }
    return bitpos;
}

/*
 * 関数名 GetAlphaNumCode
 * 機能   英数字データのコードを取得する
 * 引数   alphanum(IN)            英数字
 * 戻り値 0以上のとき英数字データのコードを返す。
 *          コードはAlphaNumConvertで定義されたものである。
 *          -1のときは英数字以外のデータであることを示す。
 * 説明   英数字データのコードを取得する。
 */
static int32_t GetAlphaNumCode(int32_t alphanum)
{
    int32_t c;
    for (c = 0; c < CountAlphabetNum; c++)
        if (alphanum == AlphaNumConvert[c])
            break;
    return c < CountAlphabetNum ? c : -1;
}

/*
 * 関数名 GetNumCode
 * 機能   数字データのコードを取得する
 * 引数   num(IN)                数字
 * 戻り値 0以上のとき数字データのコードを返す。
 *          コードは以下とする。
 *          '0': 0 '1': 1 '2': 2 '3': 3 '4': 4
 *          '5': 5 '6': 6 '7': 7 '8': 8 '9': 9
 *          -1のときは数字以外のデータであることを示す。
 * 説明   数字データのコードを取得する。
 */
static int32_t GetNumCode(int32_t num)
{
    if (!(num < '0' || num > '9'))
        return num - '0';
    return -1;
}

/*
 * 関数名 GetKanjiGroupCode
 * 機能   漢字データのグループコードを取得する
 * 引数   kanji(IN)                漢字
 * 戻り値 0以上のとき漢字データのグループコードを返す。
 *          コードは以下とする。
 *          1: 0x8140 - 0x9ffc
 *          2: 0xe040 - 0xebbf
 *          -1のときは漢字以外のデータであることを示す。
 * 説明   漢字データのグループコードを取得する。
 */
static int32_t GetKanjiGroupCode(int32_t kanji)
{
    int32_t code;
//@ADD 2003/10/09 絵文字対応 ak Start
    int32_t iBuff;
    iBuff = kanji & 0x000000FF; // 仮ﾊﾞｯﾌｧにｾｯﾄ
//@ADD 2003/10/09 絵文字対応 ak End
    if (kanji >= 0x8140 && kanji <= 0x9ffc) {
//@CHG 2003/10/09 絵文字対応 ak Start
        // S-JISとして無効なｺｰﾄﾞの場合漢字として認めない
        if (iBuff != 0x7F && iBuff != 0xFD && iBuff != 0xFE
            && iBuff != 0xFF && iBuff > 0x5F)
            code = 1;
        else
            code = -1;
//@CHG 2003/10/09 絵文字対応 ak End
    }
    else if (kanji >= 0xe040 && kanji <= 0xebbf) {
//@CHG 2003/10/09 絵文字対応 ak Start
        // S-JISとして無効なｺｰﾄﾞの場合漢字として認めない
        if (iBuff != 0x7F && iBuff != 0xFD && iBuff != 0xFE
            && iBuff != 0xFF && iBuff > 0x5F)
            code = 2;
        else
            code = -1;
//@CHG 2003/10/09 絵文字対応 ak End
    }
    else
        code = -1;
    return code;
}

/*
 * 関数名 MakeMixedModeSegments
 * 機能   ユーザーデータを複数のモードセグメントに分割する
 * 引数   version(IN)            型番
 *          userdata(IN)         ユーザデータ
 *          seg(OUT)             モードセグメント配列の格納領域のポインタ
 * 戻り値 分割されたモードセグメント数を返す。
 * 説明   ユーザーデータを複数のモードセグメントに分割する。
 */
static int32_t MakeMixedModeSegments(int32_t version, USERDATA *userdata,
                                  MODESEG **pseg)
{
    int32_t i;
    int32_t nseg = 0;
#ifdef DEBUG_LOG
    FILE *fp;
#endif /* DEBUG_LOG */
    MODE currentmode = MODE_8BITSBYTE;
    MODESEG *seg = NULL;
    /* モードセグメントに分割する。*/
    /* 優先度は以下とする。
       漢字：8ビットバイト < 漢字
       英数字：8ビットバイト < 英数字 < 数字 */
    for (i = 0; i < userdata->length; i++) {
        if (i < userdata->length - 1) {
            /* 2文字以上あり */
            int32_t kanji = ((uint8_t)userdata->data[i]) << 8
                        | (uint8_t)userdata->data[i + 1];
            if (GetKanjiGroupCode(kanji) >= 0) {
                /* 漢字 */
                if (nseg == 0 || currentmode != MODE_KANJI) {
                    /* 先頭モードセグメントまたは新規モードセグメント */
                    nseg++;
                    seg = reinterpret_cast<MODESEG *>(QRE_Realloc(seg, nseg * sizeof(MODESEG)));
                    seg[nseg - 1].mode = MODE_KANJI;
                    seg[nseg - 1].charlength = 1;
                    seg[nseg - 1].start = i;
                    seg[nseg - 1].end = i + 1;
                }
                else {
                    /* 漢字モードセグメントの続き */
                    seg[nseg - 1].charlength++;
                    seg[nseg - 1].end += 2;
                }
                i++;    /* 漢字2バイトめをスキップ */
                currentmode = MODE_KANJI;
                continue;
            }
        }
        if (GetNumCode(userdata->data[i]) >= 0) {
            /* 数字 */
            if (nseg == 0 || currentmode != MODE_NUM) {
                /* 先頭モードセグメントまたは新規モードセグメント */
                nseg++;
                seg = reinterpret_cast<MODESEG *>(QRE_Realloc(seg, nseg * sizeof(MODESEG)));
                seg[nseg - 1].mode = MODE_NUM;
                seg[nseg - 1].charlength = 1;
                seg[nseg - 1].start = i;
                seg[nseg - 1].end = i;
            }
            else {
                /* 数字モードセグメントの続き */
                seg[nseg - 1].charlength++;
                seg[nseg - 1].end++;
            }
            currentmode = MODE_NUM;
            continue;
        }
        if (GetAlphaNumCode(userdata->data[i]) >= 0) {
            /* 英数字 */
            if (nseg == 0 || currentmode != MODE_ALPHANUM) {
                /* 先頭モードセグメントまたは新規モードセグメント */
                nseg++;
                seg = reinterpret_cast<MODESEG *>(QRE_Realloc(seg, nseg * sizeof(MODESEG)));
                seg[nseg - 1].mode = MODE_ALPHANUM;
                seg[nseg - 1].charlength = 1;
                seg[nseg - 1].start = i;
                seg[nseg - 1].end = i;
            }
            else {
                /* 英数字モードセグメントの続き */
                seg[nseg - 1].charlength++;
                seg[nseg - 1].end++;
            }
            currentmode = MODE_ALPHANUM;
            continue;
        }
        /* その他 8ビットバイト */
        if (nseg == 0 || currentmode != MODE_8BITSBYTE) {
            /* 先頭モードセグメントまたは新規モードセグメント */
            nseg++;
            seg = reinterpret_cast<MODESEG *>(QRE_Realloc(seg, nseg * sizeof(MODESEG)));
            seg[nseg - 1].mode = MODE_8BITSBYTE;
            seg[nseg - 1].charlength = 1;
            seg[nseg - 1].start = i;
            seg[nseg - 1].end = i;
        }
        else {
            /* 8ビットバイトモードセグメントの続き */
            seg[nseg - 1].charlength++;
            seg[nseg - 1].end++;
        }
        currentmode = MODE_8BITSBYTE;
    }
    /* となりあったモードセグメントの結合による最適化を行う。*/
#ifdef DEBUG_LOG
    // fp = fopen("modeseg.txt", "w");
    DumpModeSegments("0", version, nseg, seg, userdata, fp);
#endif /* DEBUG_LOG */
    for (i = 0; i < nseg - 1; i++) {
        if (Join2ModeSegments(seg + i, version)) {
            nseg = PackModeSegmentArray(nseg, seg, i + 1);
            i--; /* もう一度トライ */
#ifdef DEBUG_LOG
            {
                int8_t id[128];
                sprintf(id, "%d", i + 1);
                DumpModeSegments(id, version, nseg, seg, userdata, fp);
            }
#endif /* DEBUG_LOG */
            continue;
        }
    }
#ifdef DEBUG_LOG
    // fclose(fp);
#endif /* DEBUG_LOG */

    pseg[0] = seg;
    return nseg;
}

/*
 * 関数名 PackModeSegmentArray
 * 機能   モードセグメント配列の空きをつめる
 * 引数   nseg(IN)              モードセグメント配列の要素数
 *        seg(IN/OUT)           モードセグメント配列のポインタ
 *        pos(IN)               空きモードセグメントのインデックス
 * 戻り値 モードセグメント配列の要素数-1を返す。
 * 説明   モードセグメント配列の空きをつめる。
 */
static int32_t PackModeSegmentArray(int32_t nseg, MODESEG *seg, int32_t pos)
{
    int32_t i;
    for (i = pos; i < nseg - 1; i++)
        seg[i] = seg[i + 1];
    return nseg - 1;
}

#ifdef DEBUG_LOG
/*
 * 関数名 DumpModeSegments
 * 機能   モードセグメントをダンプする
 * 引数   id(IN)                ID
 *          version(IN)         型番
 *          nseg(IN)            モードセグメント数
 *          seg(IN)             モードセグメント配列
 *          userdata(IN)        ユーザーデータ
 *          fp(IN)              ダンプするファイルのファイルポインタ
 * 戻り値 なし。
 * 説明   モードセグメントをダンプする。
 */
static void DumpModeSegments(const int8_t *id, int32_t version, int32_t nseg,
                              MODESEG *seg, USERDATA *userdata,
                              FILE *fp)
{
    int32_t s, d;
    int32_t total_len = 0;
    int32_t len;

    NN_LOG("***** %s *****\n", id);
    for (s = 0; s < nseg; s++) {
        int8_t smode[128];
        switch (seg[s].mode) {
        case MODE_CHAIN:
            strcpy(smode, "CHAIN");
            break;

        case MODE_NUM:
            strcpy(smode, "NUM");
            break;

        case MODE_ALPHANUM:
            strcpy(smode, "ALPHANUM");
            break;

        case MODE_8BITSBYTE:
            strcpy(smode, "8BITSBYTE");
            break;

        case MODE_KANJI:
            strcpy(smode, "KANJI");
            break;

        case MODE_TERM:
            strcpy(smode, "TERM");
            break;

        case MODE_ANY:
            strcpy(smode, "ANY");
            break;

        default:
            strcpy(smode, "ERROR");
            break;
        }
        len = CalcDataBitLength(version, seg[s].mode, seg[s].charlength);
        total_len += len;
        NN_LOG("** ModeSegment#%02d %s **\n", s, smode);
        NN_LOG("start: %d end: %d charlength: %d bitlen: %d\n",
                seg[s].start, seg[s].end, seg[s].charlength, len);
        if (seg[s].mode != MODE_KANJI) {
            NN_LOG("%s", "/");
            for (d = seg[s].start; d <= seg[s].end; d++) {
                int8_t fmt[128];
                if (seg[s].mode == MODE_8BITSBYTE)
                    strcpy(fmt, "%02x");
                else
                    strcpy(fmt, "%c");
                NN_LOG(fmt, (uint8_t)userdata->data[d]);
            }
            NN_LOG("%s", "/");
        }
        else {
            int8_t *kbuff = reinterpret_cast<int8_t *>(QRE_Malloc(userdata->length + 1));
            memset(kbuff, 0, userdata->length + 1);
            strncpy(kbuff, userdata->data + seg[s].start,
                    seg[s].end - seg[s].start + 1);
            NN_LOG("/%s/", kbuff);
            QRE_Free(kbuff);
        }
        NN_LOG("\n");
    }
    NN_LOG("***** total bitlength: %d bytelength: %d *****\n",
            total_len,
            total_len / 8 + (total_len % 8 ? 1 : 0)
            );
}

#endif /* DEBUG_LOG */

/*
 * 関数名 Join2ModeSegments
 * 機能   連続した2つのモードセグメントを結合する
 * 引数   seg(IN/OUT)                モードセグメント配列
 *          version(IN)              型番
 * 戻り値 結合したとき0以外を結合しないかったとき0を返す。
 * 説明   連続した2つのモードセグメントを結合する。
 */
static bool Join2ModeSegments(MODESEG *seg, int32_t version)
{
    int32_t len1, len2, len_join;
    if (seg[0].mode == seg[1].mode) {
        /* 同じモードセグメントがつながっているときは結合する。*/
        seg[0].charlength += seg[1].charlength;
        seg[0].end = seg[1].end;
        return true;
    }
    len1 = CalcDataBitLength(version, seg[0].mode, seg[0].charlength);
    len2 = CalcDataBitLength(version, seg[1].mode, seg[1].charlength);
    if (seg[0].mode == MODE_8BITSBYTE && seg[1].mode == MODE_KANJI) {
        int32_t newsize = seg[0].charlength + 2 * seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_KANJI && seg[1].mode == MODE_8BITSBYTE) {
        int32_t newsize = 2 * seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_ALPHANUM && seg[1].mode == MODE_NUM) {
        int32_t newsize = seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_ALPHANUM, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_ALPHANUM;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_NUM && seg[1].mode == MODE_ALPHANUM) {
        int32_t newsize = seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_ALPHANUM, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_ALPHANUM;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_8BITSBYTE && seg[1].mode == MODE_ALPHANUM) {
        int32_t newsize = seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_ALPHANUM && seg[1].mode == MODE_8BITSBYTE) {
        int32_t newsize = seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_KANJI && seg[1].mode == MODE_NUM) {
        int32_t newsize = 2 * seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_NUM && seg[1].mode == MODE_KANJI) {
        int32_t newsize = seg[0].charlength + 2 * seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_KANJI && seg[1].mode == MODE_ALPHANUM) {
        int32_t newsize = 2 * seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_ALPHANUM && seg[1].mode == MODE_KANJI) {
        int32_t newsize = seg[0].charlength + 2 * seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_8BITSBYTE && seg[1].mode == MODE_NUM) {
        int32_t newsize = seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    if (seg[0].mode == MODE_NUM && seg[1].mode == MODE_8BITSBYTE) {
        int32_t newsize = seg[0].charlength + seg[1].charlength;
        len_join = CalcDataBitLength(version, MODE_8BITSBYTE, newsize);
        if (len_join < len1 + len2) {
            seg[0].mode = MODE_8BITSBYTE;
            seg[0].charlength = newsize;
            seg[0].end = seg[1].end;
            return true;
        }
    }
    return false;
}// NOLINT(impl/function_size)

/*
 * 関数名 MakeSymbolChainInfo
 * 機能   シンボル列指示子を作成する
 * 引数   chain(OUT)            シンボル列指示子のポインタ
 *        seqno(IN)             シンボル通し番号
 *        totalcount(IN)        全シンボル個数
 *        parity(IN)            パリティデータ
 * 戻り値 正常時0以外を以上時0を返す。
 * 説明   シンボル列指示子を作成する。
 *          引数が以下のときはエラーとする。
 *          1. 全シンボル個数が2未満
 *          2. 全シンボル個数が17以上
 *          3. シンボル通し番号が1未満
 *          4. シンボル通し番号が全シンボル個数+1以上
 */
static bool MakeSymbolChainInfo(uint16_t *chain, int32_t seqno,
                                 int32_t totalcount,
                                 uint8_t parity)
{
    if (totalcount < 2 || totalcount > 16)
        return false;
    if (seqno < 1 || seqno > totalcount)
        return false;
    *chain = (uint16_t)((seqno - 1) << 12 | ((totalcount - 1) << 8) | parity);

    return true;
}

MODE AnalyzeMode(uint8_t *data, uint32_t size)
{
    MODE current_mode = MODE_8BITSBYTE;
    uint32_t pos = 0;

    if (size >= 2) {
        int32_t kanji = ((data[pos] << 8) & 0xFF00) | (data[pos + 1] & 0xFF);
        if (GetKanjiGroupCode(kanji) >= 0) {
            current_mode = MODE_KANJI;
            pos += 2;
        }
    }
    if (current_mode != MODE_KANJI) {
        if (GetNumCode(data[pos]) >= 0) {
            current_mode = MODE_NUM;
        }
        else if (GetAlphaNumCode(data[pos]) >= 0) {
            current_mode = MODE_ALPHANUM;
        }
        pos++;
    }
    if (current_mode == MODE_8BITSBYTE)
        return MODE_8BITSBYTE;

    for (; pos < size; pos++) {
        if (current_mode == MODE_KANJI) {
            if (pos < size - 1) {
                int32_t kanji = ((data[pos] << 8) & 0xFF00) | (data[pos + 1] & 0xFF);
                if (GetKanjiGroupCode(kanji) >= 0) {
                    current_mode = MODE_KANJI;
                    pos++;
                    continue;
                }
            }
            return MODE_8BITSBYTE;
        }

        if (current_mode == MODE_NUM) {
            if (GetNumCode(data[pos]) >= 0) {
                continue;
            }
            else if (GetAlphaNumCode(data[pos]) >= 0) {
                current_mode = MODE_ALPHANUM;
                continue;
            }
            return MODE_8BITSBYTE;
        }
        else if (current_mode == MODE_ALPHANUM) {
            if (GetAlphaNumCode(data[pos]) >= 0) {
                continue;
            }
            return MODE_8BITSBYTE;
        }
    }

    return current_mode;
}
