﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>

#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>

#include "../Common/DevMenu_CommonAsyncTaskWithProgressView.h"
#include "../DevMenu_ModalView.h"

namespace devmenu { namespace launcher {

/**
 * @brief   クライアント上の権利状態を示す列挙体です。
 */
enum RightsState
{
    RightsState_None,                 // 権利なし
    RightsState_DeviceAuthenticated,  // 機器認証された権利有り
    RightsState_TimeLimited,          // 有効期限付きの権利有り
};

/**
 * @brief   コンテンツの可用性をチェックします。
 *
 * @param[in]  applicationId  チェック対象の AppliationId
 *
 * @return     使用可能な場合は ResultSuccess が返ります。
 */
const nn::Result CheckContentsAvailability( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

/**
 * @brief   クライアント上のアプリケーションを使用する権利をチェックします
 *
 * @param[out]  pOutHasLaunchRights  起動権利の有無
 * @param[in]   applicationId        チェック対象の AppliationId
 * @param[in]   userId               チェック対象のユーザー(Uid)
 *
 * @return  処理の結果を返します。
 *
 * @details
 *  applictionId に対してクライアント上の権利のチェックを行います。
 *  userId が InvalidUid の場合は、全ユーザーの機器認証の権利が対象になります。
 *  userId が 非 InvalidUid の場合は、そのユーザーの有効期限付きの権利が対象になります。
 *
 *  Result が ResultSuccess かつ *pOutHasLaunchRights が true の場合に起動可能です。
 *  アプリケーション起動に使用可能なチケットが存在しない場合は ResultSuccess が返りますが、
 *  *pOutHasLaunchRights は false になります。
 *  ResultSuccess 以外が返ってきた場合は、適切にエラーハンドリングするようにしてください。
 */
const nn::Result CheckApplicationRightsOnClientForApplication(
    bool* pOutHasLaunchRights, const nn::ncm::ApplicationId& applicationId,
    const nn::account::Uid& userid = nn::account::InvalidUid ) NN_NOEXCEPT;

/**
 * @brief  実行中のアプリケーションがローカル通信中かを判定します。
 *
 * @retrun ローカル通信中の場合は真を返します。
 */
bool IsLocalCommunicationRunning() NN_NOEXCEPT;

/**
 * @brief  ローカル通信を切断するか否かを問う ModalView を表示します。
 *
 * @param[in]  startModalFunction   ModalView の表示関数
 * @param[in]  disconnectCallback   ローカル通信切断時のコールバック関数
 * @param[in]  cancelCallback       ローカル通信起因でのキャンセルコールバック関数
 *
 * @details
 *  本関数はメインスレッドから呼ぶ必要があります。
 */
void AskToDisconnectLocalCommunication(
    const std::function< void( ModalView* ) >& startModalFunction,
    const std::function< void() >& disconnectCallback,
    const std::function< void() >& cancelCallback ) NN_NOEXCEPT;

/**
 * @brief   アプリケーションを使用する権利をサーバに問い合わせます。
 *
 * @param[out]  pOutRightsState       権利状態
 * @param[out]  pOutExitReason        処理の終了理由
 * @param[in]   pIsCancelRequested    処理のキャンセル要求の有無
 * @param[in]   applicationId         対象の ApplicationId
 * @param[in]   userId                対象のユーザー(Uid)
 *
 * @return  処理の結果を返します。
 *
 * @details
 *  本関数はライブラリアプレットが表示されてユーザー操作が必要になり、長時間ブロックする可能性があります。
 *
 *  アプリケーションを起動する権利がある状態の場合は、
 *  Result が ResultSuccess かつ *pOutExitReason が ExitReason_Success になります。
 *  また、このとき *pOutRightsState は RightsState_DeviceAuthenticated または RightsState_TimeLimited になります。
 *
 *  Result が ResultSuccess 以外の場合は、いずれかの処理に失敗しています。
 *  適切にエラーハンドリングするようにしてください。
 *  *pOutExitReason が ExitReason_Failure になります。
 *
 *  処理がキャンセルされた場合、*pOutExitReason が ExitReason_Canceled になります。
 *  Result は ResultSuccess が返ります。
 */
const nn::Result CheckLaunchRightsOnServer(
    RightsState* pOutRightsState,
    AsyncTaskWithProgressView::ExitReason* pOutExitReason,
    bool* pIsCancelRequested,
    const nn::ncm::ApplicationId& applicationId,
    const nn::account::Uid& userId ) NN_NOEXCEPT;

}} // ~namespace devmenu::launcher
