﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>

#include <nn/nn_Macro.h>
#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>

namespace nn {
    namespace ns {
        struct ApplicationControlProperty;
    }
    namespace arp {
        struct ApplicationLaunchProperty;
    }
}

namespace devmenu {

    class RootSurfaceContext;

namespace launcher {

/**
 * @brief       活動中のアプリケーション ID を取得します。
 *
 * @return      活動中のアプリケーション ID を返します。
 */
const nn::ncm::ApplicationId GetActiveApplicationId() NN_NOEXCEPT;

/**
 * @brief       アプリケーションが生存しているか判定します。
 *
 * @return      アプリケーションが生存中なら真を返します。
 */
bool IsApplicationAlive() NN_NOEXCEPT;

/**
 * @brief       アプリケーションが外部起動されたか判定します。
 *
 * @param[in]   pOutId   外部起動されたアプリケーションの ID
 *
 * @return      アプリケーションが外部起動なら真を返します。
 */
bool IsApplicationFloating( nn::ncm::ApplicationId* pOutId ) NN_NOEXCEPT;

/**
 * @brief       アプリケーションの FG 遷移を要求されているか判定します。
 *
 * @return      要求されているなら真を返します。
 */
bool IsApplicationRequestedToGetForeground() NN_NOEXCEPT;

/**
 * @brief       アプリケーションの終了を待機します。
 */
void WaitForApplicationExited() NN_NOEXCEPT;

/**
 * @brief       アプリケーションの終了後の状態を更新します。
 */
void UpdateApplicationStateAfterExit() NN_NOEXCEPT;

/**
 * @brief       アプリケーションの状態を更新します。
 *
 * @return      アプリケーションが生存中なら真を返します。
 */
bool UpdateApplicationState() NN_NOEXCEPT;

/**
 * @brief       外部起動されたアプリケーションを確認します。
 *
 * @return      外部起動されたアプリケーションの存在を確認できたら真を返します。
 */
bool CheckFloatingApplication() NN_NOEXCEPT;

/**
 * @brief       アプリケーションジャンプ要求の有無を確認します。
 *
 * @return      アプリケーションジャンプを要求されている場合は真を返します。
 */
bool IsApplicationJumpRequested() NN_NOEXCEPT;

/**
 * @brief       生存しているアプリケーションの管理データを取得します。
 *
 * @param[out]   pOut   生存しているアプリケーションの管理データ
 *
 * @return      取得に成功したら真を返します。
 */
bool GetApplicationControlProperty( nn::ns::ApplicationControlProperty* pOut ) NN_NOEXCEPT;

/*
 * @brief       起動中のアプリケーションの性質を更新します。
 *
 * @return      更新に成功したら真を返します。
 */
bool UpdateApplicationLaunchProperty() NN_NOEXCEPT;

/*
 * @brief       起動中のアプリケーションの起動元ストレージを取得します。
 *              事前に UpdateApplicationLaunchProperty() を呼んで起動中のアプリケーションの性質を更新しておく必要があります。
 *
 * @return      起動元ストレージの ID を返します。アプリケーションが実行中でない場合は InvalidProperty を返します。
 */
const nn::arp::ApplicationLaunchProperty GetApplicationLaunchProperty() NN_NOEXCEPT;

/*
 * @brief       アプリケーションジャンプを要求します。
 *              RootSurfaceContext で要求を確認し、要求がある場合は StartApplicationJump() が呼ばれます。
 *
 */
void RequestApplicationJump() NN_NOEXCEPT;

/**
 * @brief       アプリケーションジャンプを実行します。
 */
void StartApplicationJump() NN_NOEXCEPT;

/**
 * @brief       Library Applet 経由でのアプリケーション起動要求を処理します。
 */
void RequestLaunchTriggeredByLibraryApplet( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

/**
 * @brief       アプリケーションの起動または中断状態の解除を要求します（失敗時のエラーハンドリングを含む）。
 *
 * @param[in]   applicationId           起動または中断状態の解除を要求するアプリケーションの ID
 * @param[in]   isAutoBootRequested     自動起動要求の有無
 * @param[in]   requiresLaunchCheck     起動可能かのチェック有無
 *
 */
void RequestApplicationLaunchOrResume( const nn::ncm::ApplicationId& applicationId, bool isAutoBootRequested, bool requiresLaunchCheck = true ) NN_NOEXCEPT;

/**
 * @brief       外部起動アプリを活性化します。
 */
void ActivateFloatingApplication() NN_NOEXCEPT;

/**
 * @brief       アプリケーションにフォアグランドへの遷移を要求します
 */
void RequestApplicationForeground() NN_NOEXCEPT;

/**
 * @brief       アプリケーションのフォアグランド遷移を開始します。
 *
 * @return      要求に成功したら真を返します。
 */
bool StartApplicationForeground() NN_NOEXCEPT;

/**
 * @brief       アプリケーションに終了を要求します。
 *
 * @return      要求に成功したら真を返します。
 */
bool RequestApplicationExit() NN_NOEXCEPT;

/**
 * @brief       アプリケーションを強制終了します。
 *
 * @return      アプリケーションが実行中の場合は即座に真を返します。何らかの理由で終了していた場合は偽が返ります。
 */
bool TerminateApplication() NN_NOEXCEPT;

/**
 * @brief       ループ処理を実行します。
 */
void HandleLoop() NN_NOEXCEPT;

/**
 * @brief       MessageView 表示用の RootSurfaceContext を登録します。
 *
 * @param[in]   pRootSurface        RootSurfaceContext ポインタ
 */
void SetRootSurfaceContext( RootSurfaceContext* pRootSurface ) NN_NOEXCEPT;

/**
 * @brief       RID デモアプリケ－ショーンの終了トリガを設定します。
 *
 * @details     システムが終了トリガをかけた場合は true, アプリケーションが自身で終了トリガをかけた場合は false が返ります。
 *
 */
void SetRidDemoApplicationExitedBySystem( bool isApplicationExitedBySystem ) NN_NOEXCEPT;

/**
 * @brief       アプリケーション手動起動成功時のコールバック関数を登録します。
 *
 * @param[in]   callback            コールバック関数
 */
void SetManualLaunchSuccessCallback( const std::function< void() >& callback ) NN_NOEXCEPT;

/**
 * @brief       DevMenu が LA を起動したかどうかを返します。
 */
bool IsLibraryAppletLaunchedByDevMenu() NN_NOEXCEPT;

/*
 * @brief      ストレージ ID に対応するストレージ名を取得します。
 *
 * @return     ストレージ ID に対応するストレージ名文字列を返します。
 */
const char* GetStorageName( nn::ncm::StorageId storageId ) NN_NOEXCEPT;

/**
 * @brief       アプリケーションの起動に必要な管理プロパティデータを取得します。
 *
 * @param[out]  pOutValue           管理プロパティデータ
 * @param[in]   applicationId       起動を要求するアプリケーションの ID
 *
 * @return      取得に成功したら NN_RESULT_SUCCESS を返します。
 */
const nn::Result GetApplicationControlPropertyForLaunch( nn::ns::ApplicationControlProperty* pOutValue, const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT;

/**
 * @brief       アプリケーション起動のエラーメッセージを取得します。
 *
 * @param[in]   result           エラー Result
 * @param[in]   requiredSize     セーブデータ領域に必要な空きサイズ
 *
 * @return      エラーメッセージを返します。
 */
const std::string GetLaunchRelatedErrorMessage( const nn::Result& result, int64_t requiredSize = 0LL ) NN_NOEXCEPT;

}} // ~namespace devmenu::launcher
