﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <cstring>
#include <mutex>

#include <nn/ec/system/ec_TicketApi.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/nifm/nifm_ResultMenu.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_DynamicRightsApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    #include <nn/ae.h>
    #include <nn/ae/ae_ApplicationLaunchPropertyApi.h>
    #include "../DevMenu_Exhibition.h"
#endif // defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

#include "DevMenu_Launcher.h"
#include "DevMenu_LauncherRightsCheckApis.h"
#include "../DevMenu_Config.h"
#include "../DevMenu_Common.h"
#include "../DevMenu_Result.h"
#include "../DevMenu_RootSurface.h"
#include "../Accounts/DevMenu_AccountsSdkHelper.h"
#include "../Accounts/DevMenu_AccountsSelector.h"
#include "../Applications/DevMenu_ApplicationsCommon.h"
#include "../Common/DevMenu_CommonAsyncTaskWithProgressView.h"

#include "DevMenuCommand_Common.h"
#include "DevMenuCommand_HashUtil.h"
#include "DevMenuCommand_Result.h"

namespace devmenu { namespace launcher {

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
// SA モードでは、AE ライブラリを利用してアプリケーションをコントロールします。

namespace {

    const nn::Result SetApplicationLaunchParameter( const nn::ae::ApplicationHandle& handle, nn::applet::LaunchParameterKind kind, const void* pParameter, size_t parameterSize, size_t storageSize ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES( parameterSize <= storageSize );

        nn::applet::StorageHandle storageHandle;
        NN_RESULT_DO( nn::applet::CreateStorage( &storageHandle, storageSize ) );
        bool isReleaseRequired = true;
        NN_UTIL_SCOPE_EXIT
        {
            if ( isReleaseRequired )
            {
                nn::applet::ReleaseStorage( storageHandle );
            }
        };

        NN_RESULT_DO( nn::applet::WriteToStorage( storageHandle, 0, pParameter, parameterSize ) );
        nn::ae::PushApplicationLaunchParameter( handle, kind, storageHandle );
        isReleaseRequired = false;
        NN_RESULT_SUCCESS;
    }

    const nn::Result SetStartupUserAccount( const nn::ae::ApplicationHandle& handle, const nn::account::Uid& uid ) NN_NOEXCEPT
    {
        const size_t storageSizeForAccount = 1024;

        auto info = nn::account::MakePreselectionInfo( uid );
        NN_RESULT_DO( SetApplicationLaunchParameter( handle, nn::applet::LaunchParameterKind_Account, &info, sizeof( info ), storageSizeForAccount ) );
        NN_RESULT_SUCCESS;
    }

    const nn::Result ReadUserLaunchParameterFromSdCard( size_t* outSize, const nn::ncm::ApplicationId& applicationId, void* outBuffer, size_t bufferSize ) NN_NOEXCEPT
    {
        const char mountName[] = "sd";
        const char directoryPath[] = "NintendoSDK/ApplicationLaunchParameters";
        const char extention[] = "nxargs";

        NN_RESULT_DO( nn::fs::MountSdCardForDebug( mountName ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount( mountName );
        };

        char path[ 80 ];
        nn::util::SNPrintf( path, sizeof( path ), "%s:/%s/%016llX.%s", mountName, directoryPath, applicationId.value, extention );

        nn::fs::FileHandle handle;
        NN_RESULT_DO( nn::fs::OpenFile( &handle, path, nn::fs::OpenMode_Read ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( handle );
        };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize( &fileSize, handle ) );

        NN_RESULT_THROW_UNLESS( fileSize <= bufferSize, nn::fs::ResultInvalidSize() );

        NN_RESULT_DO( nn::fs::ReadFile( outSize, handle, 0, outBuffer, bufferSize ) );

        NN_RESULT_SUCCESS;
    }

    const nn::Result SetUserLaunchParameterIfExists( const nn::ae::ApplicationHandle& handle ) NN_NOEXCEPT
    {
        const nn::ncm::ApplicationId applicationId = nn::ae::GetApplicationId(handle);

        const size_t bufferSize = devmenu::LaunchParameterSizeMax;
        std::unique_ptr< char[] > buffer( new char[ bufferSize ] );

        size_t readSize;
        NN_RESULT_TRY( ReadUserLaunchParameterFromSdCard( &readSize, applicationId, buffer.get(), bufferSize ) )
            NN_RESULT_CATCH( nn::fs::ResultPathNotFound )
            {
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_CATCH( nn::fs::ResultSdCardAccessFailed )
            {
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_RESULT_DO( SetApplicationLaunchParameter( handle, nn::applet::LaunchParameterKind_User, buffer.get(), readSize, readSize ) );

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief       アプリケーションのセーブデータ領域を確保します。アプリケーション起動前に呼ぶ必要があります。
     *
     * @param[out]  pOutRequiredSize     セーブデータ領域確保に失敗した場合に MessageView を表示される必要な空きサイズ
     * @param[in]   applicationId        起動を要求するアプリケーションの ID
     * @param[in]   property             起動を要求するアプリケーションの管理データ
     * @param[in]   userId               起動を要求するアプリケーションがロードするユーザー ID
     *
     * @return      確保の成否を Result 値で返します。
     */
    const nn::Result EnsureApplicationSaveData( int64_t* pOutRequiredSize, const nn::ncm::ApplicationId& applicationId, const nn::ns::ApplicationControlProperty& controlProperty, const nn::account::Uid& userId ) NN_NOEXCEPT
    {
        NN_RESULT_DO( nn::fs::EnsureApplicationSaveData( pOutRequiredSize, applicationId, controlProperty, userId ) );
        nn::fs::CacheStorageTargetMedia media;
        NN_RESULT_DO( nn::fs::EnsureApplicationCacheStorage( pOutRequiredSize, &media, applicationId, controlProperty ) );
        NN_RESULT_SUCCESS;
    }

    const nn::TimeSpan TimeLimitToExitApplication = nn::TimeSpan::FromMilliSeconds( 15000 );

    // アプリケーションコントローラ
    class ApplicationController
    {
        NN_DISALLOW_COPY( ApplicationController );
        NN_DISALLOW_MOVE( ApplicationController );

    private:
        enum ApplicationState
        {
            ApplicationState_NotLaunched,       //!< アプリは未起動
            ApplicationState_PreparingToLaunch, //!< アプリは起動準備中
            ApplicationState_Launching,         //!< アプリは起動中
            ApplicationState_Running,           //!< アプリは動作中
            ApplicationState_Exited,            //!< アプリは正常終了済み
        };

        enum LaunchState
        {
            LaunchState_None,
            LaunchState_ExitApplication,
            LaunchState_WaitingForExitOfApplication,
            LaunchState_RequestedToSelectUser,
            LaunchState_CheckLocalCommunication,
            LaunchState_CheckCanStartApplication,
            LaunchState_CheckRightsOnServer,
            LaunchState_WaitingForCompletionOfCheckingRights,
            LaunchState_StartApplication,
            LaunchState_RequestedToGetForeground,
            LaunchState_WaitingForUserOperation,
            LaunchState_FailureExit,
            LaunchState_Failure,
            LaunchState_Canceled,
        };

    public:
        // 無効なアプリケーションハンドル値
        static const nn::ae::ApplicationHandle InvalidHandle;

        // 空のアプリケーション ID
        static const nn::ncm::ApplicationId EmptyId;

        // 無効なアプリケーション起動プロパティ
        static const nn::arp::ApplicationLaunchProperty InvalidLaunchProperty;

        // コンストラクタ
        ApplicationController() NN_NOEXCEPT
        :   m_RequestId( EmptyId ),
            m_CurrentId( EmptyId ),
            m_RequestHandle( InvalidHandle ),
            m_Handle( InvalidHandle ),
            m_State( ApplicationState_NotLaunched ),
            m_LaunchState( LaunchState_None ),
            m_LaunchProperty( InvalidLaunchProperty ),
            m_Mutex( true ),
            m_UserId( nn::account::InvalidUid ),
            m_NextUserId( nn::account::InvalidUid ),
            m_SaveDataSize( 0LL ),
            m_IsFloating( false ),
            m_IsFloated( false ),
            m_IsGettingForegroundRequested( false ),
            m_IsAutoBootRequested( false ),
            m_IsApplicaitonJumpRequested( false ),
            m_IsRidDemoApplicationExitedBySystem( false ),
            m_RightsState( RightsState_None ),
            m_ManualLaunchCallback( nullptr ),
            m_ResultLaunchApplication( nn::ResultSuccess() ),
            m_pOnlineLaunchRightsCheckAsyncTask( nullptr ),
            m_pApplicationExitAsyncTask( nullptr )
        {
        }

        // 活動中のアプリケーション ID を取得
        const nn::ncm::ApplicationId GetActiveId() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            return m_CurrentId;
        }

        // アプリケーションの生存判定
        bool IsAlive() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            return ( ApplicationState_PreparingToLaunch == m_State)
                || ( ApplicationState_Launching == m_State )
                || ( ApplicationState_Running == m_State );
        }

        // アプリケーションの外部起動判定
        bool IsFloating( nn::ncm::ApplicationId* pOutId ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            if ( m_IsFloating )
            {
                *pOutId = m_CurrentId;
                m_IsFloating = false;
                return true;
            }
            else
            {
                *pOutId = EmptyId;
            }
            return false;
        }

        // アプリケーションジャンプの要求有無を取得
        bool IsApplicationJumpRequested() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            return m_IsApplicaitonJumpRequested;
        }

        // アプリケーション FG 遷移の要求有無を取得
        bool IsGettingForegroundRequested() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            return m_IsGettingForegroundRequested;
        }

        // アプリケーションの終了待ち
        void WaitForExited() NN_NOEXCEPT
        {
            NN_ASSERT( InvalidHandle != m_Handle );

            {
                std::lock_guard<nn::os::Mutex> lock( m_Mutex );

                if ( InvalidHandle != m_Handle )
                {
                    auto pExitEvent = nn::ae::GetApplicationExitEvent( m_Handle );
                    nn::os::WaitSystemEvent( pExitEvent );
                }
            }

            // 終了処理が完了するのを待つ
            while ( NN_STATIC_CONDITION( true ) )
            {
                if ( LaunchState_WaitingForExitOfApplication != m_LaunchState )
                {
                    break;
                }
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
            }
        }

        // アプリケーション終了後の処理
        void OnCompleteExit() NN_NOEXCEPT
        {
            NN_ASSERT( InvalidHandle != m_Handle );
            NN_ASSERT( nn::os::TryWaitSystemEvent( nn::ae::GetApplicationExitEvent( m_Handle ) ) );

            const auto applicationExitedResult = this->GetApplicationExitedResult();
            const auto applicationResultForDebug = nn::ae::GetApplicationResultRawForDebug( m_Handle );
            const bool isApplicationExitedAbnormally = ( nn::ae::ApplicationResult_NormallyExited != applicationExitedResult ) && ( nn::ae::ApplicationResult_VoluntarilyExited != applicationExitedResult );
            bool needsErrorDialog = false;
            DEVMENU_LOG_AE( "ApplicationExitedResult=%d (result=0x%08x)\n", applicationExitedResult, applicationResultForDebug.GetInnerValueForDebug() );

            // 正常終了 && 終了したアプリケーションが体験版アプリケーション && 体験版アプリケーション自身が終了トリガをかけた
            const bool isRetailInteractiveDisplayLaunchRequired =
                false == isApplicationExitedAbnormally &&
                false == rid::IsRetailInteractiveDisplayMenu( nn::ae::GetApplicationId( m_Handle ) ) &&
                false == m_IsRidDemoApplicationExitedBySystem;

            // アプリケーションの状態を更新し、アプリケーション強制終了時のエラーダイアログ表示が必要か判別する
            this->UpdateStateAndLaunchIfRequired( &needsErrorDialog, isApplicationExitedAbnormally );

            if ( true == needsErrorDialog )
            {
                auto* pMessageView = new MessageView( false );
                const auto errorMessage = GetApplicationExitedErrorMessage( applicationExitedResult, applicationResultForDebug );
                pMessageView->AddMessage( errorMessage );
                pMessageView->AddButton( "Close" );
                m_pRootSurface->StartModal( pMessageView, true );
                DEVMENU_LOG( "%s\n", errorMessage.c_str() );
            }
            else
            {
                // Retail Interactive Display 時
                if ( true == rid::IsRetailInteractiveDisplay() )
                {
                    // 正常終了 && 終了したアプリケーションが体験版アプリケーション && 体験版アプリケーション自身が終了トリガをかけた場合
                    if ( true == isRetailInteractiveDisplayLaunchRequired )
                    {
                        DEVMENU_LOG_AE( "Retail Interactive Display demo application has been exited by itself\n" );
                        // Retail Interactive Display メニューを起動する
                        rid::LaunchRetailInteractiveDisplayMenu( false );
                    }
                }
            }
        }

        // アプリケーションの状態更新
        bool UpdateState() NN_NOEXCEPT
        {
            return this->UpdateStateAndLaunchIfRequired();
        }

    private:
        // アプリケーションの状態更新と起動
        bool UpdateStateAndLaunchIfRequired( bool* pOutNeedsErrorDialog = nullptr, bool isApplicationExitedAbnormally = false ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            // 異常終了でフローティング起動でなかった場合にのみエラーダイアログを表示する
            bool needsErrorDialog = isApplicationExitedAbnormally && ( false == m_IsFloated );

            const auto previousState = m_State;
            NN_UNUSED( previousState );
            if ( m_Handle != InvalidHandle )
            {
                // ToDo: アプリケーション終了通知を受け取ってから実行しているので、実質的に意味がないはず
                //       必ず ApplicationState_Exited になるので、おって調整する
                //       ApplicationState_Exited は未使用なので ApplicationState_NotLaunched がよさそうだが、
                //       直後に Clear() しているので削除するのが最適だと思われる
                auto pExitEvent = nn::ae::GetApplicationExitEvent( m_Handle );
                m_State = nn::os::TryWaitSystemEvent( pExitEvent ) ? ApplicationState_Exited : ApplicationState_Running;

                if ( false == this->IsAlive() )
                {
                    Clear();

                    const bool isAutoBootMode = exhibition::IsExhibitionModeEnabled() || rid::IsRetailInteractiveDisplay();
                    const bool skipsApplicationLaunch = ( true == isApplicationExitedAbnormally ) && ( false == isAutoBootMode && false == IsApplicationJump() );

                    // アプリケーション異常終了時で、次に起動するアプリケーションが自動起動 or アプリケーションジャンプの場合は起動処理に進まない
                    // isAutoBootMode が真なら必ず自動起動であるわけではないが、細かい判別は複雑になり過剰な対応なので自動起動と見なす
                    if ( false == skipsApplicationLaunch )
                    {
                        // 起動要求されたアプリがある
                        if ( EmptyId != m_RequestId )
                        {
                            const auto requestedId = m_RequestId;
                            m_RequestId = EmptyId;
                            const auto isAutoBootRequested = m_IsAutoBootRequested;
                            m_IsAutoBootRequested = false;
                            this->RequestLaunch( requestedId, isAutoBootRequested, true, true ); // 直前にアプリケーションを終了しているので isApplicationClosedBefore を true にする
                            needsErrorDialog = false;
                        }
                    }
                    else
                    {
                        // アプリケーションの異常終了により、アプリケーションの起動を行わなかった場合は、アプリケーションの起動要求を初期化する
                        m_RequestId = EmptyId;
                        m_IsAutoBootRequested = false;
                    }
                }
            }
            else
            {
                m_State = ApplicationState_NotLaunched;
            }

            if ( previousState != m_State )
            {
                DEVMENU_LOG_AE( "Update ApplicationState %d -> %d\n", previousState, m_State );
            }

            if ( nullptr != pOutNeedsErrorDialog )
            {
                *pOutNeedsErrorDialog = needsErrorDialog;
            }

            return this->IsAlive();
        }

    public:
        // 外部起動されたアプリケーション確認
        bool CheckFloater() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_Mutex);

            DEVMENU_LOG_AE( "Invoke nn::ae::PopFloatingApplicationHandle()\n" );
            auto handle = nn::ae::PopFloatingApplicationHandle();
            if ( InvalidHandle != handle )
            {
                // Note: 不要かもしれない
                Clear();

                m_Handle = handle;
                m_CurrentId = nn::ae::GetApplicationId( m_Handle );
                m_State = ApplicationState_PreparingToLaunch;
                m_IsFloating = true;

                // フローティング起動であったことを覚えておいて、アプリケーション異常終了時のエラーダイアログ表示有無の判断に使用する
                // 現状では m_IsFloating は ReleaseForeground() すると false にする必要があり、FG 遷移後は m_IsFloating でフローティング起動判定ができない
                m_IsFloated = true;

                DEVMENU_LOG_AE( "PopedApplication( 0x%016llX )\n", m_CurrentId.value );
            }
            return this->IsAlive();
        }

        // 起動中アプリケーションの性質を更新
        bool UpdateLaunchProperty() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            if ( InvalidHandle != m_Handle )
            {
                return nn::ae::GetApplicationLaunchProperty( &m_LaunchProperty, m_Handle ).IsSuccess();
            }
            return false;
        }

        // アプリケーションジャンプを要求
        void RequestApplicationJump() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );
            DEVMENU_LOG_AE( "RequestApplicationJump\n" );
            m_IsApplicaitonJumpRequested = true;
        }

        // アプリケーションジャンプの開始
        void StartApplicationJump() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            m_IsApplicaitonJumpRequested = false;
            m_RequestHandle = InvalidHandle;

            // アプリケーション ID の取得
            const auto isSuccess = nn::ae::TryPopLaunchRequestedApplication( &m_RequestHandle );
            NN_ASSERT( isSuccess );
            NN_UNUSED( isSuccess );
            const auto applicationId = nn::ae::GetApplicationId( m_RequestHandle );

            this->RequestLaunch( applicationId, false );
        }

        // LibraryApplet からのアプリケーション起動要求
        void RequestLaunchTriggeredByLibraryApplet( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            if ( this->IsRunning( applicationId ) )
            {
                nn::ae::RequestToGetForeground();

                // Library Applet を終了させてアプリケーションに戻る
                // ToDo: タイムアウト時間をもう少し長くしてタイムアウトするまでダイアログを出すとベター
                nn::ae::RequestExitLibraryAppletOfApplicationOrTerminate( m_Handle, nn::TimeSpan( nn::TimeSpan::FromSeconds( 2 ) ) );
                this->RequestToGetForeground();
            }
            else
            {
                m_RequestHandle = nn::ae::CreateApplication( applicationId ); // ApplicationJump と同じ流れに乗せるため m_RequestHandle に代入する
                this->RequestLaunch( applicationId, false );
            }
        }

        // アプリケーションの起動 or 中断解除要求
        void RequestLaunchOrResume( const nn::ncm::ApplicationId& applicationId, bool isAutoBootRequested, bool requiresLaunchCheck = true, bool isApplicationClosedBefore = false ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            const bool isTargetRunning = ( true == this->IsAlive() ) && ( applicationId == m_CurrentId );

            if ( isTargetRunning && !this->IsApplicationJump() )
            {
                this->RequestToGetForeground();
            }
            else
            {
                this->RequestLaunch( applicationId, isAutoBootRequested, requiresLaunchCheck, isApplicationClosedBefore );
            }
        }

        // アプリケーションのフォアグラウンド遷移を要求
        void RequestToGetForeground( const nn::account::Uid& userId = nn::account::InvalidUid ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            auto controlProperty = std::make_shared< nn::ns::ApplicationControlProperty >();
            if ( ApplicationState_Running == m_State )
            {
                // 起動要求されたアプリケーションがある
                if ( EmptyId != m_RequestId )
                {
                    // アプリケーションの管理データを取得
                    auto result = GetApplicationControlPropertyForLaunch( controlProperty.get(), m_RequestId );
                    if ( result.IsSuccess() )
                    {
                        // ユーザー選択が完了していない
                        if ( controlProperty->startupUserAccount != nn::ns::StartupUserAccount::None && nn::account::InvalidUid == userId )
                        {
                            // アプリケーションの起動要求をキャンセル
                            this->PostProcessOnLaunchApplication();
                        }
                    }
                }

                m_UserId = userId;
                m_IsGettingForegroundRequested = true;
            }
            else if ( ApplicationState_Launching == m_State && InvalidHandle != m_Handle )
            {
                // アプリケーションの管理データを取得(フローティング起動ではない時は ae 層での準備が整っていないので false が返る)
                if ( this->GetControlProperty( controlProperty.get() ) )
                {
                    // ユーザー選択が完了していない場合は FG 遷移させない
                    if ( controlProperty->startupUserAccount != nn::ns::StartupUserAccount::None && nn::account::InvalidUid == userId )
                    {
                        return;
                    }

                    m_UserId = userId;
                    m_IsGettingForegroundRequested = true;
                }
            }
        }

        // アプリケーションのフォアグラウンド遷移を開始
        bool StartToGetForeground() NN_NOEXCEPT
        {
            std::lock_guard< nn::os::Mutex > lock( m_Mutex );

            NN_ASSERT( m_IsGettingForegroundRequested );

            m_IsGettingForegroundRequested = false;

            if ( !this->IsAlive() )
            {
                DEVMENU_LOG_AE( "StartToGetForeground failed.\n" );
                return false;
            }

            // 中断からの復帰時も含め、Foregorund に遷移させる前に権利の可用性をチェックする
            if ( !nn::ae::CheckApplicationRightsAvailability( m_Handle ) )
            {
                // 最終的には権利回復処理を実行することになる見込みだが、
                // 現時点ではエラーメッセージを表示する
                this->HandleLaunchError( ResultApplicationSuspendedByRightsError() );
                return false;
            }

            // フローティング起動時はアプリケーション FG 遷移前に起動パラメータをセットする
            // (インストールされたアプリケーション起動時は StartApplication() 前にセットする)
            if ( m_IsFloated && ApplicationState_Launching == m_State )
            {
                // ファイル読み込み失敗でエラーが返る可能性があるが、アプリケーション起動を優先してログ出力に留める
                const auto result = this->SetApplicationLaunchParameterAll( m_Handle, m_UserId );
                if ( result.IsFailure() )
                {
                    DEVMENU_LOG( "Failed to set the user launch paramter. (result: 0x%08x)\n",  result.GetInnerValueForDebug() );
                }
            }

            DEVMENU_LOG_AE( "Invoke nn::ae::RequestApplicationGetForeground()\n" );
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ae::RequestApplicationGetForeground( m_Handle ) );

            m_State = ApplicationState_Running;
            this->UpdateLaunchProperty();

            return true;
        }

        // アプリケーションに終了要求
        const nn::Result RequestExit() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            if ( this->IsAlive() )
            {
                if ( InvalidHandle != m_Handle )
                {
                    // アプリケーションの終了を開始する
                    this->StartToExitApplication();
                }
            }

            NN_RESULT_SUCCESS;
        }

        // アプリケーションを強制終了
        const nn::Result TerminateApplication() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            if ( this->IsAlive() )
            {
                DEVMENU_LOG_AE( "Invoke nn::ae::TerminateApplication()\n" );
                if ( InvalidHandle != m_Handle )
                {
                    NN_RESULT_DO( nn::ae::TerminateApplication( m_Handle ) );
                }
            }

            NN_RESULT_SUCCESS;
        }

        // アプリケーション終了結果の取得
        nn::ae::ApplicationResult GetApplicationExitedResult() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            return nn::ae::GetApplicationResult( m_Handle );
        }

        // Rid デモアプリケーションの終了がシステムによって行われたことを設定します。
        void SetRidDemoApplicationExitedBySystem( bool isApplicationExitedBySystem ) NN_NOEXCEPT
        {
            std::lock_guard< nn::os::Mutex > lock( m_Mutex );
            m_IsRidDemoApplicationExitedBySystem = isApplicationExitedBySystem;
        }

        // アプリケーションの管理データを取得
        bool GetControlProperty( nn::ns::ApplicationControlProperty* pOut ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            if ( InvalidHandle != m_Handle )
            {
                return nn::ae::GetApplicationControlProperty( pOut, m_Handle ).IsSuccess();
            }
            return false;
        }

        // アプリケーション起動プロパティの取得
        const nn::arp::ApplicationLaunchProperty GetLaunchProperty() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            return ( ApplicationState_Running == m_State ) ? m_LaunchProperty : InvalidLaunchProperty;
        }

        // アプリケーションの起動元ストレージの取得
        nn::ncm::StorageId GetLaunchAppStorage() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            return ( ApplicationState_Running == m_State ) ? m_LaunchProperty.storageId : nn::ncm::StorageId::None;
        }

        // アプリケーションのパッチが存在するストレージの取得
        nn::ncm::StorageId GetLaunchPatchStorage() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock(m_Mutex);

            return ( ApplicationState_Running == m_State ) ? m_LaunchProperty.patchStorageId : nn::ncm::StorageId::None;
        }

        // RootSurfaceContext の登録
        void SetRootSurfaceContext( RootSurfaceContext* pRootSurface ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );
            m_pRootSurface = pRootSurface;
        }

        // 手動起動成功時のコールバック関数登録
        void SetManualLaunchSuccessCallback( std::function<void()> callback ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );
            m_ManualLaunchCallback = callback;
        }

        // 外部起動アプリを活性化します。
        void ActivateFloatingApplication() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );
            nn::ncm::ApplicationId applicationId;

            if ( true == this->IsFloating( &applicationId ) )
            {
                auto controlProperty = std::make_shared< nn::ns::ApplicationControlProperty >();

                if ( !this->GetControlProperty( controlProperty.get() ) )
                {
                    m_State = ApplicationState_Launching;
                    this->RequestToGetForeground();
                    return;
                }

                if ( controlProperty->startupUserAccount != nn::ns::StartupUserAccount::None )
                {
                    // アカウントセレクタを表示する
                    this->RequestUserAccountSelector(
                        [ this, applicationId, controlProperty ]( nn::account::Uid userId )
                        {
                            this->StartFloatingApplication( *controlProperty, userId, applicationId );
                        }, // select
                        [ this ]{ this->RequestExit(); }, // cancel
                        controlProperty->startupUserAccount );
                }
                else
                {
                    this->StartFloatingApplication( *controlProperty, nn::account::InvalidUid, applicationId );
                }
            }
        }

        /**
         * @brief   メインスレッドから呼ぶループ処理を実行します
         */
        void HandleLoop() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            switch( m_LaunchState )
            {
            case LaunchState_None:
                break;
            case LaunchState_ExitApplication:
                // 現時点ではエンバグリスクを考慮して使用しない
                break;
            case LaunchState_WaitingForExitOfApplication:
                NN_ASSERT_NOT_NULL( m_pApplicationExitAsyncTask );
                if ( m_pApplicationExitAsyncTask )
                {
                    if ( m_pApplicationExitAsyncTask->TryWait() )
                    {
                        // 非同期処理の結果を取得する
                        m_ResultExitApplication = m_pApplicationExitAsyncTask->GetResult();
                        delete m_pApplicationExitAsyncTask;
                        m_pApplicationExitAsyncTask = nullptr;

                        m_LaunchState = m_ResultExitApplication.IsSuccess() ? LaunchState_None : LaunchState_FailureExit;
                    }
                    else
                    {
                        // アプリケーション終了の進捗を更新
                        m_pApplicationExitAsyncTask->UpdateProgressView();
                    }
                }
                break;
            case LaunchState_RequestedToSelectUser:
                // 現時点ではエンバグリスクを考慮して使用しない
                break;
            case LaunchState_CheckCanStartApplication:
                m_LaunchState =
                    RightsState_DeviceAuthenticated == m_RightsState
                    ? LaunchState_StartApplication
                    : LaunchState_CheckLocalCommunication;
                break;
            case LaunchState_CheckLocalCommunication:
                if ( IsLocalCommunicationRunning() )
                {
                    AskToDisconnectLocalCommunication(
                        [&]( ModalView* pModalView ) { m_pRootSurface->StartModal( pModalView, true ); },
                        [&]() { m_LaunchState = LaunchState_CheckRightsOnServer; },
                        [&]() { m_LaunchState = LaunchState_Canceled; } );
                    // ローカル通信を切断するか否かの選択を待つ
                    m_LaunchState = LaunchState_WaitingForUserOperation;
                }
                else
                {
                    m_LaunchState = LaunchState_CheckRightsOnServer;
                }
                break;
            case LaunchState_CheckRightsOnServer:
                this->StartToCheckLaunchRightsOnServer();
                m_LaunchState = LaunchState_WaitingForCompletionOfCheckingRights;
                break;
            case LaunchState_WaitingForCompletionOfCheckingRights:
                NN_ASSERT_NOT_NULL( m_pOnlineLaunchRightsCheckAsyncTask );
                if ( m_pOnlineLaunchRightsCheckAsyncTask )
                {
                    if ( m_pOnlineLaunchRightsCheckAsyncTask->TryWait() )
                    {
                        // 非同期処理の結果を取得する
                        m_ResultLaunchApplication = m_pOnlineLaunchRightsCheckAsyncTask->GetResult();
                        delete m_pOnlineLaunchRightsCheckAsyncTask;
                        m_pOnlineLaunchRightsCheckAsyncTask = nullptr;

                        m_LaunchState = HasLaunchRights() ? LaunchState_StartApplication : LaunchState_Failure;
                    }
                    else
                    {
                        // オンラインの起動権利確認の進捗を更新
                        m_pOnlineLaunchRightsCheckAsyncTask->UpdateProgressView();
                    }
                }
                break;
            case LaunchState_StartApplication:
                m_ResultLaunchApplication = this->StartApplication();
                m_LaunchState = m_ResultLaunchApplication.IsSuccess() ? LaunchState_None : LaunchState_Failure;
                break;
            case LaunchState_RequestedToGetForeground:
                // 現時点ではエンバグリスクを考慮して使用しない
                break;
            case LaunchState_WaitingForUserOperation:
                // ユーザー操作により m_LaunchState が更新されるのを待つ
                break;
            case LaunchState_FailureExit:
                this->HandleExitError( m_ResultExitApplication );
                m_LaunchState = LaunchState_None;
                break;
            case LaunchState_Failure:
                this->HandleLaunchError( m_ResultLaunchApplication );
                m_LaunchState = LaunchState_None;
                break;
            case LaunchState_Canceled:
                this->PostProcessAtLaunchFailOrCancel();
                m_LaunchState = LaunchState_None;
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

    private:

        // アプリケーションジャンプ要求の処理実行中かの判定
        bool IsApplicationJump() NN_NOEXCEPT
        {
            // アプリケーションジャンプ時のみ、RequestLaunch() を呼んで起動要求を出す前に m_RequestHandle を取得できている前提があるので要留意
            return m_RequestHandle != InvalidHandle;
        }

        // 動作中のアプリケーション判定
        bool IsRunning( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
        {
            return m_CurrentId == applicationId;
        }

        bool HasLaunchRights( ) const NN_NOEXCEPT
        {
            return RightsState_DeviceAuthenticated == m_RightsState || RightsState_TimeLimited == m_RightsState;
        }

        // アプリケーションの起動要求
        void RequestLaunch( const nn::ncm::ApplicationId& applicationId, bool isAutoBootRequested, bool requiresLaunchCheck = true, bool isApplicationClosedBefore = false ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            const auto result = this->RequestLaunchImpl( applicationId, isAutoBootRequested, requiresLaunchCheck, isApplicationClosedBefore );

            if ( result.IsFailure() )
            {
                this->HandleLaunchError( result, applicationId );
            }
        }

        // アプリケーションの起動要求
        const nn::Result RequestLaunchImpl( const nn::ncm::ApplicationId& applicationId, bool isAutoBootRequested, bool requiresLaunchCheck, bool isApplicationClosedBefore ) NN_NOEXCEPT
        {
            DEVMENU_LOG_AE( "RequestLaunch( 0x%016llX ), AutoBoot( %s ), ApplicationJump( %s ), LaunchCheck( %s ), Exit->Launch( %s )\n",
                applicationId.value,
                isAutoBootRequested ? "true" : "false",
                IsApplicationJump() ? "true" : "false",
                requiresLaunchCheck ? "true" : "false",
                isApplicationClosedBefore ? "true" : "false"
            );

            // 起動要求中の場合は処理を行わない
            if ( ApplicationState_NotLaunched != m_State && ApplicationState_Running != m_State )
            {
                NN_RESULT_SUCCESS;
            }

            // 起動可否のチェック
            if ( requiresLaunchCheck )
            {
                bool hasLaunchRights = false;
                NN_RESULT_DO( CheckContentsAvailability( applicationId ) );
                NN_RESULT_DO( CheckApplicationRightsOnClientForApplication( &hasLaunchRights, applicationId ) );

                m_RightsState = hasLaunchRights ? RightsState_DeviceAuthenticated : RightsState_None;
            }

            m_IsAutoBootRequested = isAutoBootRequested;

            // 管理データの取得
            auto controlProperty = std::make_shared< nn::ns::ApplicationControlProperty >();
            NN_RESULT_DO( GetApplicationControlPropertyForLaunch( controlProperty.get(), applicationId ) );

            // 任意のアプリケーションが動作中ならば終了してから起動する
            if ( true == this->IsAlive() )
            {
                m_RequestId = applicationId;
                const bool isUserSelectionRequired =
                    ( nn::ns::StartupUserAccount::None != controlProperty->startupUserAccount ) || !this->HasLaunchRights();

                // アプリケーションジャンプ時は、ジャンプ元のアプリケーションを予告なく終了して対象のアプリケーションを起動する
                // 試遊台モードでは試遊台メニューへの戻りと見なして予告なく終了する
                if ( this->IsApplicationJump() || rid::IsRetailInteractiveDisplay() )
                {
                    NN_RESULT_DO( this->RequestExit() );
                    NN_RESULT_SUCCESS;
                }

                if ( isUserSelectionRequired )
                {
                    this->RequestUserAccountSelectorToStartInstalledApplication( *controlProperty );
                }
                else
                {
                    this->SelectSwitchRunningApplication();
                }
            }
            // 未起動ならば即座に起動する
            else
            {
                m_State = ApplicationState_Launching;

                // アプリケーションハンドルの登録
                if ( false == this->IsApplicationJump() )
                {
                    m_Handle = nn::ae::CreateApplication( applicationId );
                }
                else
                {
                    m_Handle = m_RequestHandle;
                }

                // ApplicationView の更新前に、自動コミットを無効にする. 自動コミット設定の変更は以下を満たさないといけない
                //   - ApplicationControlProperty を取得前 nn::ns::DisableAutoCommit() を呼ぶ
                //   - nn::ae::StartApplicaiton() 実行後 or 起動処理の失敗 or 起動のキャンセルで nn::ns::EnableAutoCommit() を呼ぶ
                nn::ns::DisableAutoCommit();
                {
                    nn::ns::ApplicationView view;
                    nn::ns::GetApplicationView( &view, &applicationId, 1 ); // アプリケーション起動に関係する処理ではないので、Result チェックは行わない
                }

                // アプリケーションの起動
                const bool isUserSelectionRequired =
                    ( nn::ns::StartupUserAccount::None != controlProperty->startupUserAccount || !this->HasLaunchRights() ) &&
                    ( false == isApplicationClosedBefore || true == this->IsApplicationJump() );
                NN_RESULT_DO( this->TryLaunch( *controlProperty, isUserSelectionRequired ) );
            }
            NN_RESULT_SUCCESS;
        }

        // 動作中のアプリケーションを終了して起動要求されたアプリケーションを起動するかを選択します
        void SelectSwitchRunningApplication() NN_NOEXCEPT
        {
            auto pMessageView = new MessageView( false );
            pMessageView->AddMessage( "Close the suspended software and launch this one?" );
            pMessageView->AddButton(
                "Cancel",
                [&]( void* pParam, nn::TimeSpan& timespan )
                {
                    NN_UNUSED( pParam );
                    timespan = nn::TimeSpan::FromMilliSeconds( 100 );
                    m_RequestId = EmptyId;
                }
            );
            pMessageView->AddButton(
                "Launch",
                [&]( void* pParam, nn::TimeSpan& timespan )
                {
                    NN_UNUSED( pParam );
                    timespan = nn::TimeSpan::FromMilliSeconds( 100 );
                    this->RequestExit();
                },
                nullptr,
                MessageView::ButtonTextColor::Green
            );

            m_pRootSurface->StartModal( pMessageView, true, false, true );
        }

        /**
         * @brief   ユーザー選択画面を表示し、ユーザー選択後にアプリケーションの起動や FG 遷移を実行します
         *
         * @param[in]  selectCallback                ユーザー選択時のコールバック関数
         * @param[in]  cancelCallback                キャンセル時のコールバック関数
         * @param[in]  startupUserAccount            アプリケーションが起動時に要求するアカウント状態
         * @param[in]  headerText                    ヘッダーに表示する文字列
         */
        void RequestUserAccountSelector(
            const std::function< void( nn::account::Uid uid )>& selectCallback,
            const std::function< void() >& cancelCallback,
            nn::ns::StartupUserAccount startupUserAccount,
            const char* headerText = "Select an user account with whom launch" ) NN_NOEXCEPT
        {
            auto pSelector = new accounts::UserAccountSelector(
                selectCallback, // ユーザー選択時のコールバック関数
                cancelCallback, // キャンセル時のコールバック関数
                nn::ns::StartupUserAccount::RequiredWithNetworkServiceAccountAvailable == startupUserAccount,
                true, // enableCancel
                headerText
            );

            if ( false == pSelector->IsAutoSelect() )
            {
                m_pRootSurface->SetPreviousFocusedView( m_pRootSurface->focusedView() );
                m_pRootSurface->StartModal( pSelector, true, m_pRootSurface->IsModalViewRunning(), true, false );
            }
            else
            {
                delete pSelector;
            }
        }

        /**
         * @brief   ユーザー選択画面を表示し、ユーザー選択後にインストールされたアプリケーションを起動します
         *
         * @param[in]  controlProperty    アプリケーションのコントロールプロパティ
         */
        void RequestUserAccountSelectorToStartInstalledApplication( const nn::ns::ApplicationControlProperty& controlProperty ) NN_NOEXCEPT
        {
            auto selectCallback = [ this ]( const nn::account::Uid& userId )->void
            {
                m_UserId = userId;
                if ( ApplicationState_Running == m_State )
                {
                    m_NextUserId = userId;
                    this->RequestExit();
                }
                else
                {
                    m_LaunchState = LaunchState_CheckCanStartApplication;
                }
                m_pRootSurface->SetKeepsInvisibleWallAtterExit( !this->HasLaunchRights() );
            };

            auto cancelCallback = [ this ]()->void
            {
                m_LaunchState = LaunchState_Canceled;
            };

            this->RequestUserAccountSelector(
                selectCallback,
                cancelCallback,
                controlProperty.startupUserAccount,
                ApplicationState_Running == m_State ? "Close the suspended software and launch with?" : "Launch with?" );
        }

        /**
         * @brief   アプリケーション起動を試行します
         *
         * @param[in]  controlProperty           アプリケーションのコントロールプロパティ
         * @param[in]  isUserSelectionRequired   ユーザー選択が必要か否か
         */
        const nn::Result TryLaunch( const nn::ns::ApplicationControlProperty& controlProperty, bool isUserSelectionRequired ) NN_NOEXCEPT
        {
            NN_ASSERT( this->HasLaunchRights() || ( !this->HasLaunchRights() && isUserSelectionRequired ) );

            if ( isUserSelectionRequired )
            {
                NN_ABORT_UNLESS( !rid::IsRetailInteractiveDisplay() );
                if ( false == m_IsAutoBootRequested ) // 通常起動時
                {
                    this->RequestUserAccountSelectorToStartInstalledApplication( controlProperty );
                }
                else // exhibition mode 自動起動時
                {
                    // ユーザーリスト取得
                    nn::account::Uid userIds[ nn::account::UserCountMax ];
                    int userCount;
                    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::ListAllUsers( &userCount, userIds, sizeof( userIds ) / sizeof( userIds[ 0 ] ) ) );
                    if ( 0 < userCount )
                    {
                        // ユーザーが 1 つ以上存在すれば、先頭のアカウントを自動でロードします
                        m_UserId = userIds[ 0 ];
                        m_LaunchState = LaunchState_CheckCanStartApplication;
                    }
                    else
                    {
                        NN_RESULT_THROW( ResultAccountNotFound() );
                    }
                }
            }
            else
            {
                m_LaunchState = LaunchState_CheckCanStartApplication;
            }
            NN_RESULT_SUCCESS;
        }

        /**
         * @brief    サーバー上のアプリケーションを使用する権利状態を確認します
         *
         * @pre
         *  - LaunchState_CheckRightsOnServer == m_LaunchState
         *  - m_pOnlineLaunchRightsCheckAsyncTask == nullptr
         *  - m_Handle.IsValid() == true
         *
         * @details
         *  サーバー上のアプリケーションの使用権利確認を非同期で処理します。
         *  本関数はブロックすることなくすぐに返ります。
         */
        void StartToCheckLaunchRightsOnServer() NN_NOEXCEPT
        {
            NN_ASSERT( LaunchState_CheckRightsOnServer == m_LaunchState );
            NN_ASSERT( m_Handle.IsValid() );
            NN_ASSERT( nullptr == m_pOnlineLaunchRightsCheckAsyncTask );

            if ( m_pOnlineLaunchRightsCheckAsyncTask )
            {
                return;
            }

            // オンラインの権利をチェックする
            m_pOnlineLaunchRightsCheckAsyncTask = new AsyncTaskWithProgressView(
                m_pRootSurface,
                "Checking application launch rights",
                [&]( AsyncTaskWithProgressView::ExitReason* pOutExitReason, bool* pIsCancelRequested ) -> nn::Result
                {
                    return CheckLaunchRightsOnServer(
                        &m_RightsState, pOutExitReason, pIsCancelRequested,
                        nn::ae::GetApplicationId( m_Handle ), m_UserId
                    );
                },
                [&]( const nn::Result& result, AsyncTaskWithProgressView::ExitReason exitReason ) -> bool
                {
                    NN_UNUSED( result );
                    return AsyncTaskWithProgressView::ExitReason_Success != exitReason;
                },
                true
            );

            m_pOnlineLaunchRightsCheckAsyncTask->Start();
        }

        // アプリケーションの開始
        const nn::Result StartApplication() NN_NOEXCEPT
        {
            const auto applicationId = nn::ae::GetApplicationId( m_Handle );

            // 管理データの取得
            // TORIAEZU: 再度取得する。RequestLaunchImpl() で取得した property を使用したいが、AsyncTask の拡張が必要
            auto controlProperty = std::make_shared< nn::ns::ApplicationControlProperty >();
            NN_RESULT_DO( GetApplicationControlPropertyForLaunch( controlProperty.get(), applicationId ) );

            // セーブデータ領域の確保
            NN_RESULT_DO( EnsureApplicationSaveData( &m_SaveDataSize, applicationId, *controlProperty, m_UserId ) );

            // アプリケーションに渡す起動パラメータをセットする
            NN_RESULT_DO( this->SetApplicationLaunchParameterAll( m_Handle, m_UserId ) );

            // 使用可能なユーザーの登録
            switch ( m_RightsState )
            {
            case RightsState_None:
                NN_ABORT( "No Rights. Cannot start application." );
            case RightsState_DeviceAuthenticated:
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ae::SetApplicationUsers( m_Handle, nn::ae::AllUser ) );
                break;
            case RightsState_TimeLimited:
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ae::SetApplicationUsers( m_Handle, &m_UserId , 1 ) );
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }

            // アプリケーションの開始
            DEVMENU_LOG_AE( "Invoke nn::ae::StartApplication( 0x%016llX )\n", applicationId.value );
            nn::ae::StartApplication( m_Handle );

            // アプリケーションの情報を更新
            m_CurrentId = applicationId;

            // アプリケーションの FG 遷移要求
            m_IsGettingForegroundRequested = true;

            // リストからの手動起動時に限定する
            if ( !m_IsAutoBootRequested && !IsApplicationJump() && nullptr != m_ManualLaunchCallback )
            {
                m_ManualLaunchCallback();
            }

            // Retail Interactive Display 向けの設定をクリアする
            m_IsRidDemoApplicationExitedBySystem = false;

            this->PostProcessOnLaunchApplication();
            DEVMENU_LOG_AE( "Launched Application( id: 0x%016llX, state: %d )\n", m_CurrentId.value, m_State );

            NN_RESULT_SUCCESS;
        }

        // フローティング起動されたアプリケーションを FG  に遷移させる
        const nn::Result StartFloatingApplication( const nn::ns::ApplicationControlProperty& controlProperty, const nn::account::Uid& userId, const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
        {
            // セーブデータ作成
            int64_t requiredSize;
            auto result = EnsureApplicationSaveData( &requiredSize, applicationId, controlProperty, userId );
            if ( result.IsSuccess() )
            {
                m_State = ApplicationState_Launching;
                this->RequestToGetForeground( userId );
            }
            else
            {
                // アプリケーションを終了させる
                NN_RESULT_DO( this->RequestExit() );

                auto pMessageView = new MessageView( false );
                CreateErrorMessageView( pMessageView, GetLaunchRelatedErrorMessage( result, requiredSize ), result );
                m_pRootSurface->StartModal( pMessageView, true, m_pRootSurface->IsModalViewRunning() );
            }

            NN_RESULT_SUCCESS;
        }

        // アプリケーション起動時に全パラメータをセットする
        static const nn::Result SetApplicationLaunchParameterAll( const nn::ae::ApplicationHandle& handle, const nn::account::Uid& uid ) NN_NOEXCEPT
        {
            NN_ASSERT( InvalidHandle != handle );

            // 選択したユーザーアカウントをセットする
            if ( nn::account::InvalidUid != uid )
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS( SetStartupUserAccount( handle, uid ) );
            }

            // 起動時引数パラメータをセットする
            NN_RESULT_DO( SetUserLaunchParameterIfExists( handle ) );

            DEVMENU_LOG_AE( "All launch parameters have been set.\n" );

            NN_RESULT_SUCCESS;
        }

        /**
         * @brief   アプリケーション起動の後処理です。成功時、失敗時、キャンセル時の共通処理のみが実行されます。
         */
        void PostProcessOnLaunchApplication() NN_NOEXCEPT
        {
            // 起動要求のクリア
            m_RequestId = EmptyId;
            m_RequestHandle = InvalidHandle;

            // 権利状態のクリア
            m_RightsState = RightsState_None;

            // 起動処理状態のクリア
            m_LaunchState = LaunchState_None;

            // 自動コミットの有効化
            nn::ns::EnableAutoCommit();
        }

        /**
         * @brief   アプリケーション起動の失敗時もしくはキャンセル時の共通後処理です。
         */
        void PostProcessAtLaunchFailOrCancel() NN_NOEXCEPT
        {
            // ToDo: 本来は全て PostProcessOnLaunchApplication() に入れたいが、
            //       成功時に ApplicationState_Running になるのはアプリケーションの FG 遷移要求を出した後であり、
            //       PostProcessOnLaunchApplication() を呼び出す箇所を調整するか、ApplicationState の取り回しを改善する必要がある
            //       ベターなのはおそらくは後者
            if ( ApplicationState_Running != m_State && InvalidHandle != m_Handle )
            {
                this->Clear();
            }
            this->PostProcessOnLaunchApplication();
        }

        // アプリケーション起動のエラーハンドリング
        void HandleLaunchError( const nn::Result& result, const nn::ncm::ApplicationId& applicationId = nn::ncm::ApplicationId::GetInvalidId() ) NN_NOEXCEPT
        {
            NN_UTIL_SCOPE_EXIT
            {
                this->PostProcessAtLaunchFailOrCancel();
            };

            if ( result.IsSuccess() )
            {
                return;
            }

            auto pMessageView = new MessageView( false );
            std::string errorMessage = "Failed to launch the application.";
            std::string buttonMessage = "Close";

            // Retail Interactive Display モードで Rid メニュー自動起動時にも exhibition 向けのエラーメッセージを取得しようとするが Result がヒットしないので問題ない
            errorMessage = m_IsAutoBootRequested ? exhibition::GetLaunchErrorMessage( result , m_SaveDataSize ) : GetLaunchRelatedErrorMessage( result, m_SaveDataSize );

            const auto requestHandle = m_RequestHandle;

            // AutoBoot 失敗時の画面表示を有効にする
            m_pRootSurface->enable( glv::Property::Visible );

            auto isModalViewExitedOnChangeIntoBackground = false;
            if ( nn::ns::ResultApplicationUpdateRecommended::Includes( result ) )
            {
                pMessageView->AddButton(
                    "Launch",
                    [ this, applicationId, requestHandle ]( void* p, nn::TimeSpan& timespan )
                    {
                        NN_UNUSED( p );
                        timespan = nn::TimeSpan::FromMilliSeconds( 100 );

                        m_RequestHandle = requestHandle; // アプリケーションジャンプ時のためにハンドルを再度設定する
                        this->RequestLaunch( applicationId, m_IsAutoBootRequested, false ); // requiresLaunchCheck を false にする
                    },
                    nullptr,
                    MessageView::ButtonTextColor::Green
                );
                buttonMessage = "Cancel";
                isModalViewExitedOnChangeIntoBackground = true;
            }

            CreateErrorMessageView( pMessageView, errorMessage, result, buttonMessage );

            m_pRootSurface->StartModal( pMessageView, true, m_pRootSurface->IsModalViewRunning(), isModalViewExitedOnChangeIntoBackground );
        }

        // アプリケーション終了のエラーハンドリング
        void HandleExitError( const nn::Result& result ) NN_NOEXCEPT
        {
            auto pMessageView = new MessageView( false );
            m_pRootSurface->enable( glv::Property::Visible );
            CreateErrorMessageView( pMessageView, "Failed to exit the application.", result );
            m_pRootSurface->StartModal( pMessageView, true, m_pRootSurface->IsModalViewRunning() );
        }

        /**
         * @brief   アプリケーションの終了を開始します。
         */
        void StartToExitApplication() NN_NOEXCEPT
        {
            NN_ASSERT( nullptr == m_pApplicationExitAsyncTask );

            if ( m_pApplicationExitAsyncTask )
            {
                return;
            }

            m_pApplicationExitAsyncTask = new AsyncTaskWithProgressView(
                m_pRootSurface,
                "Closing application",
                [&]( AsyncTaskWithProgressView::ExitReason* pOutExitReason, bool* pIsCancelRequested ) -> nn::Result
                {
                    NN_UNUSED( pIsCancelRequested );
                    DEVMENU_LOG_AE( "Invoke nn::ae::RequestApplicationExit()\n" );
                    NN_RESULT_DO( nn::ae::RequestApplicationExit( m_Handle ) );
                    NN_RESULT_DO( this->WaitApplicationExitOrTerminate( pOutExitReason ) );
                    NN_RESULT_SUCCESS;
                },
                [&]( const nn::Result& result, AsyncTaskWithProgressView::ExitReason exitReason ) -> bool
                {
                    NN_UNUSED( result );
                    return AsyncTaskWithProgressView::ExitReason_Success != exitReason;
                },
                true
            );
            m_pApplicationExitAsyncTask->Start();
            m_LaunchState = LaunchState_WaitingForExitOfApplication;
        }

        /**
         * @brief   アプリケーションの終了を待つ。タイムアウトした場合はアプリケーションの強制終了を行う
         *
         * @param[out]  pOutExitReason   　   処理の終了理由
         */
        const nn::Result WaitApplicationExitOrTerminate( AsyncTaskWithProgressView::ExitReason* pOutExitReason ) NN_NOEXCEPT
        {
            NN_ASSERT( InvalidHandle != m_Handle );
            NN_SDK_REQUIRES_NOT_NULL( pOutExitReason );
            auto exitReason = AsyncTaskWithProgressView::ExitReason_Failure;
            NN_UTIL_SCOPE_EXIT
            {
                *pOutExitReason = exitReason;
            };

            const auto startTime = nn::os::GetSystemTick();
            while ( NN_STATIC_CONDITION( true ) )
            {
                auto pExitEvent = nn::ae::GetApplicationExitEvent( m_Handle );
                if ( nn::os::TryWaitSystemEvent( pExitEvent ) == true )
                {
                    break;
                }

                const auto elapsedTime = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() - startTime );
                if ( elapsedTime > TimeLimitToExitApplication )
                {
                    // アプリケーションを強制終了
                    NN_RESULT_DO( this->TerminateApplication() );
                    NN_RESULT_SUCCESS;
                }

                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
            }

            exitReason = AsyncTaskWithProgressView::ExitReason_Success;
            NN_RESULT_SUCCESS;
        }

        // アプリケーションのハンドルを破棄
        void Clear() NN_NOEXCEPT
        {
            if ( InvalidHandle != m_Handle )
            {
                if ( this->IsAlive() )
                {
                    // この条件に入るのは、shell の L コマンドでアプリ起動
                    // された場合に限られる。この場合、アプリが終了しても
                    // pm->ns->am への Exit 通知が飛ばないので、DevMenu も
                    // アプリの終了を検知できない状態のまま、ここに来る。
                    //
                    // 対策として、強制的にアプリ終了および DevMenu を FG 遷移
                    // させて辻褄を合わせることで、アサートに失敗したり、
                    // DevMenu の内部状態がおかしくなるのを回避する。
                    //
                    // なお、L コマンドでアプリを 2 個以上同時に起動した場合は、
                    // 対策しようがないので動作保証外とする。
                    //
                    nn::ae::TerminateApplication( m_Handle );
                    nn::ae::RequestToGetForeground();
                }

                DEVMENU_LOG_AE( "Invoke nn::ae::CloseApplicationHandle()\n" );
                nn::ae::CloseApplicationHandle( m_Handle );
                m_Handle = InvalidHandle;
                m_CurrentId = EmptyId;
                m_State = ApplicationState_NotLaunched;
                m_LaunchProperty = InvalidLaunchProperty;
                m_IsFloating = false;
                m_IsFloated = false;

                // 「起動するアプリケーション選択 → ユーザー選択 → 実行中のアプリケーション終了 → 次のアプリケーション起動」の場合を想定して Uid を覚えておく
                m_UserId = m_NextUserId;
                m_NextUserId = nn::account::InvalidUid;

                // Retail Interactive Display 向けの設定をクリアする（アプリケーションが終了されているはずなのでここでも呼ぶ）
                m_IsRidDemoApplicationExitedBySystem = false;
            }
        }

        const std::string GetApplicationExitedErrorMessage( nn::ae::ApplicationResult exitedResult, const nn::Result& resultForDebug ) NN_NOEXCEPT
        {
            switch ( exitedResult )
            {
            case nn::ae::ApplicationResult_AbnormallyExited:
                return "Application has been abnormally exited.";
            case nn::ae::ApplicationResult_TerminatedByMediaLost:
                return "Application has been terminated due to media lost.";
            case nn::ae::ApplicationResult_TerminatedManually:
                return "Application has been terminated manually.";
            case nn::ae::ApplicationResult_Unexpected:
                {
                    char message[ 128 ];
                    NN_ABORT_UNLESS_GREATER( nn::util::SNPrintf( message, sizeof( message ),
                        "Application has been unexpectedly exited.\n(result: 0x%08x)", resultForDebug.GetInnerValueForDebug() ), 0 );
                    return message;
                }
            case nn::ae::ApplicationResult_NormallyExited: NN_FALL_THROUGH;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

    private:
        nn::ncm::ApplicationId              m_RequestId;                          // 起動要求されたアプリケーション ID (ハンドルを取得するまで有効）
        nn::ncm::ApplicationId              m_CurrentId;                          // 実行中のアプリケーション ID
        nn::ae::ApplicationHandle           m_RequestHandle;                      // アプリケーションジャンプ対象のアプリケーションハンドル
        nn::ae::ApplicationHandle           m_Handle;                             // 実行中のアプリケーションハンドル
        ApplicationState                    m_State;                              // アプリケーション状態
        LaunchState                         m_LaunchState;                        // 起動処理状態
        nn::arp::ApplicationLaunchProperty  m_LaunchProperty;                     // 実行中のアプリケーションの性質
        nn::os::Mutex                       m_Mutex;                              // 排他ロック
        nn::account::Uid                    m_UserId;                             // アプリケーション起動時に指定されたユーザー ID
        nn::account::Uid                    m_NextUserId;                         // 次に起動するアプリケーションにセットするユーザー ID
        int64_t                             m_SaveDataSize;                       // アプリケーション起動時に生成するセーブデータサイズ
        bool                                m_IsFloating;                         // 外部起動された状態
        bool                                m_IsFloated;                          // 外部起動されていたか否か
        bool                                m_IsGettingForegroundRequested;       // アプリケーションの FG 遷移要求の有無(画面を Unvisible にする判断材料)
        bool                                m_IsAutoBootRequested;                // exhibition mode でのアプリケーション自動起動要求の有無
        bool                                m_IsApplicaitonJumpRequested;         // アプリケーションジャンプ要求の有無
        bool                                m_IsRidDemoApplicationExitedBySystem; // Retail Interactive Display 体験版アプリケーションの終了経路
        RightsState                         m_RightsState;                        // アプリケーションを起動する権利の状態
        RootSurfaceContext*                 m_pRootSurface;                       // MessageView 表示用 (RootSurface 生成時に登録される前提)
        std::function< void() >             m_ManualLaunchCallback;               // アプリケーション手動起動成功時のコールバック関数 (フォーカス調整)

        nn::Result                          m_ResultLaunchApplication;            // アプリケーション起動の結果
        AsyncTaskWithProgressView*          m_pOnlineLaunchRightsCheckAsyncTask;  // オンラインの起動権利確認タスク

        nn::Result                          m_ResultExitApplication;              // アプリケーション終了の結果
        AsyncTaskWithProgressView*          m_pApplicationExitAsyncTask;          // アプリケーション終了タスク
    };

    // 定数宣言
    const nn::ae::ApplicationHandle ApplicationController::InvalidHandle = nn::ae::ApplicationHandle::GetInvalidHandle();
    const nn::ncm::ApplicationId ApplicationController::EmptyId = { 0 };
    const nn::arp::ApplicationLaunchProperty ApplicationController::InvalidLaunchProperty = { { 0 } };

    // 変数宣言
    ApplicationController g_ApplicationController;

} // end of unnamed namespace

const nn::ncm::ApplicationId GetActiveApplicationId() NN_NOEXCEPT
{
    return g_ApplicationController.GetActiveId();
}

bool IsApplicationAlive() NN_NOEXCEPT
{
    return g_ApplicationController.IsAlive();
}

bool IsApplicationFloating( nn::ncm::ApplicationId* pOutId ) NN_NOEXCEPT
{
    return g_ApplicationController.IsFloating(pOutId);
}

bool IsApplicationRequestedToGetForeground() NN_NOEXCEPT
{
    return g_ApplicationController.IsGettingForegroundRequested();
}

void WaitForApplicationExited() NN_NOEXCEPT
{
    g_ApplicationController.WaitForExited();
}

void UpdateApplicationStateAfterExit() NN_NOEXCEPT
{
    g_ApplicationController.OnCompleteExit();
}

bool UpdateApplicationState() NN_NOEXCEPT
{
    return g_ApplicationController.UpdateState();
}

bool CheckFloatingApplication() NN_NOEXCEPT
{
    return g_ApplicationController.CheckFloater();
}

bool IsApplicationJumpRequested() NN_NOEXCEPT
{
    return g_ApplicationController.IsApplicationJumpRequested();
}

bool GetApplicationControlProperty( nn::ns::ApplicationControlProperty* pOut ) NN_NOEXCEPT
{
    return g_ApplicationController.GetControlProperty( pOut );
}

bool UpdateApplicationLaunchProperty() NN_NOEXCEPT
{
    return g_ApplicationController.UpdateLaunchProperty();
}

const nn::arp::ApplicationLaunchProperty GetApplicationLaunchProperty() NN_NOEXCEPT
{
    return g_ApplicationController.GetLaunchProperty();
}

void RequestApplicationJump() NN_NOEXCEPT
{
    g_ApplicationController.RequestApplicationJump();
}

void StartApplicationJump() NN_NOEXCEPT
{
    g_ApplicationController.StartApplicationJump();
}

void RequestLaunchTriggeredByLibraryApplet( const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    g_ApplicationController.RequestLaunchTriggeredByLibraryApplet( applicationId );
}

void RequestApplicationLaunchOrResume( const nn::ncm::ApplicationId& applicationId, bool isAutoBootRequired, bool requiresLaunchCheck ) NN_NOEXCEPT
{
    g_ApplicationController.RequestLaunchOrResume( applicationId, isAutoBootRequired, requiresLaunchCheck );
}

void ActivateFloatingApplication() NN_NOEXCEPT
{
    g_ApplicationController.ActivateFloatingApplication();
}

void RequestApplicationForeground() NN_NOEXCEPT
{
    g_ApplicationController.RequestToGetForeground();
}

bool StartApplicationForeground() NN_NOEXCEPT
{
    return g_ApplicationController.StartToGetForeground();
}

bool RequestApplicationExit() NN_NOEXCEPT
{
    return g_ApplicationController.RequestExit().IsSuccess();
}

bool TerminateApplication() NN_NOEXCEPT
{
    return g_ApplicationController.TerminateApplication().IsSuccess();
}

void HandleLoop() NN_NOEXCEPT
{
    return g_ApplicationController.HandleLoop();
}

void SetRootSurfaceContext( RootSurfaceContext* pRootSurface ) NN_NOEXCEPT
{
    g_ApplicationController.SetRootSurfaceContext( pRootSurface );
}

void SetRidDemoApplicationExitedBySystem( bool isApplicationExitedBySystem ) NN_NOEXCEPT
{
    return g_ApplicationController.SetRidDemoApplicationExitedBySystem( isApplicationExitedBySystem );
}

void SetManualLaunchSuccessCallback( const std::function< void() >& callback ) NN_NOEXCEPT
{
    g_ApplicationController.SetManualLaunchSuccessCallback( callback );
}

const char* GetStorageName( nn::ncm::StorageId storageId ) NN_NOEXCEPT
{
    switch ( storageId )
    {
    case nn::ncm::StorageId::None:
        return "None";
    case nn::ncm::StorageId::GameCard:
        return "Card";
    case nn::ncm::StorageId::BuiltInUser:
        return "NAND";
    case nn::ncm::StorageId::SdCard:
        return "SD";
    case nn::ncm::StorageId::Host:
        return "Host";
    default:
        return "UNKNOWN";
    }
}

const nn::Result GetApplicationControlPropertyForLaunch( nn::ns::ApplicationControlProperty* pOutValue, const nn::ncm::ApplicationId& applicationId ) NN_NOEXCEPT
{
    std::unique_ptr< char[] > buffer;
    size_t dataSize;

    NN_RESULT_DO( application::GetApplicationControlData( &dataSize, &buffer, nn::ns::ApplicationControlSource::StorageOnly, applicationId ) );
    nn::ns::ApplicationControlDataAccessor accessor( buffer.get(), dataSize );
    std::memcpy( pOutValue, &accessor.GetProperty(), sizeof( nn::ns::ApplicationControlProperty ) );

    NN_RESULT_SUCCESS;
}

const std::string GetLaunchRelatedErrorMessage( const nn::Result& result, int64_t requiredSize ) NN_NOEXCEPT
{
    if ( nn::ns::ResultApplicationLaunchRightsNotFound::Includes( result ) )
    {
        return "No rights to launch the application.";
    }
    else if ( nn::ns::ResultSystemUpdateRequired::Includes( result ) )
    {
        return "System update is required to launch the application.";
    }
    else if ( nn::ns::ResultApplicationUpdateRequired::Includes( result ) )
    {
        return "Need to reset the required version of the application to launch.";
    }
    else if ( nn::ns::ResultApplicationUpdateRecommended::Includes( result ) )
    {
        return "Application can be launched but update is recommended.\nApplication vesion should be higher than or equal to notification version.";
    }
    else if ( nn::fs::ResultUsableSpaceNotEnough::Includes( result ) )
    {
        return "Failed to ensure application save data.\nRequired free space size is " + GetDelimitedNumberString( requiredSize ) + " Bytes.";
    }
    else if ( nn::fs::ResultInvalidSize::Includes( result ) )
    {
        return "Failed to set launch parameter. An nxargs file must be less than or equal to " + GetDelimitedNumberString( devmenu::LaunchParameterSizeMax ) + " Bytes.";
    }
    else if ( devmenuUtil::ResultBadApplicationForThePatch::Includes( result ) )
    {
        return "The application is not one that is used to create the patch.";
    }
    else if ( ResultNoLaunchRightsOnServer::Includes( result ) )
    {
        return "No rights to launch appliction on server.";
    }
    else if ( ResultSystemUpdateRequiredToCheckLaunchRights::Includes( result ) )
    {
        return "System update is required to check application launch rights on server.";
    }
    else if ( ResultAssignableRightsLimitExceeded::Includes( result ) )
    {
        return "This appliction is not playable now.\nIf it is being launched on other devices, try again after exitting it.";
    }
    else if ( ResultHasDeviceLinkedRightsOnlyContent::Includes( result ) )
    {
        return "This application is playable on the device you always use.";
    }
    else if ( ResultNotReleasedContent::Includes( result ) )
    {
        return "Appliction is not released and playable yet.";
    }
    else if ( nn::nifm::ResultRequestRejected::Includes( result  ) )
    {
        return "Network connection is required to check launch rights on server.";
    }
    else if ( ResultApplicationSuspendedByRightsError::Includes( result ) )
    {
        return "Cannot launch or resume application. Rights is not available.";
    }
    else
    {
        if ( !exhibition::IsExhibitionModeEnabled() )
        {
            return "Failed to launch the application.";
        }
        else
        {
            return "Failed to launch the selected application or the auto boot application.";
        }
    }
}

#else // defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
// アプリモードでは、AE ライブラリは使用できません。

const nn::ncm::ApplicationId GetActiveApplicationId() NN_NOEXCEPT
{
    return { 0 };
}

bool IsApplicationAlive() NN_NOEXCEPT
{
    return false;
}

bool IsApplicationFloating( const nn::ncm::ApplicationId* ) NN_NOEXCEPT
{
    return false;
}

bool IsApplicationRequestedToGetForeground() NN_NOEXCEPT
{
    return false;
}

bool UpdateApplicationState() NN_NOEXCEPT
{
    return false;
}

bool CheckFloatingApplication() NN_NOEXCEPT
{
    return false;
}

bool IsApplicationJumpRequested() NN_NOEXCEPT
{
    return false;
}

void RequestApplicationJump() NN_NOEXCEPT
{
}

void StartApplicationJump() NN_NOEXCEPT
{
}

void RequestLaunchTriggeredByLibraryApplet( const nn::ncm::ApplicationId& ) NN_NOEXCEPT
{
    return;
}

void RequestApplicationLaunchOrResume( const nn::ncm::ApplicationId&, bool, bool ) NN_NOEXCEPT
{
}

void RequestApplicationForeground() NN_NOEXCEPT
{
}

bool StartApplicationForeground( ) NN_NOEXCEPT
{
    return false;
}

bool RequestApplicationExit() NN_NOEXCEPT
{
    return false;
}

bool TerminateApplication() NN_NOEXCEPT
{
    return false;
}

bool GetApplicationControlProperty( nn::ns::ApplicationControlProperty* ) NN_NOEXCEPT
{
    return false;
}

const nn::arp::ApplicationLaunchProperty GetApplicationLaunchProperty() NN_NOEXCEPT
{
    return { { 0 } };
}

void HandleLoop() NN_NOEXCEPT
{
}

void SetRootSurfaceContext( RootSurfaceContext* ) NN_NOEXCEPT
{
}

void SetRidDemoApplicationExitedBySystem( bool ) NN_NOEXCEPT
{
}

void SetManualLaunchSuccessCallback( const std::function< void() >& ) NN_NOEXCEPT
{
}

const char* GetStorageName( nn::ncm::StorageId ) NN_NOEXCEPT
{
    return "UNKNOWN";
}

#endif // defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

}} // ~namespace devmenu::launcher
