﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// far の多重定義を避けるため、msgpack 内の Windows.h が glv 関連のヘッダよりも先にインクルードされている必要がある
#ifdef _USE_MATH_DEFINES
#undef _USE_MATH_DEFINES // msgpack.h 以前に _USE_MATH_DEFINES が定義されており、多重定義防止のため
#endif
#include <nn/msgpack.h>

#ifdef NN_BUILD_CONFIG_OS_WIN
#include <nn/nn_Windows.h> // msgpack.h が Windows.h を include していることに対するワークアラウンド
#endif

#include <algorithm>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_Report.h>
#include <nn/err/err_ShowErrorApiForSystem.h>
#include <nn/err/err_SystemApi.h>
#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_CalendarAdditionalInfo.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include "DevMenu_ErrorExport.h"
#include "DevMenu_ErrorReportList.h"
#include "../DevMenu_RootSurface.h"
#include "../Applications/DevMenu_ApplicationsCommon.h"
#include "../Launcher/DevMenu_Launcher.h"
#include "../Launcher/DevMenu_LauncherLibraryAppletApis.h"

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>

namespace devmenu { namespace error {

const size_t ErrorProperty::CrashReportHashSize;

namespace {

// null 終端されていない文字列 in を out にコピーする。
void CopyAsNullTerminatedString(char* out, size_t outLength, const char* in, size_t inLength)
{
    auto l = std::min(outLength - 1, inLength);
    std::memcpy(out, in, l);
    out[l] = '\0';
}

void TimestampDumper(char* output, int length, const char* format, ...) NN_NOEXCEPT
{
    std::va_list vaList;
    int64_t timeStamp;
    va_start(vaList, format);
    timeStamp = va_arg(vaList, int64_t);
    va_end(vaList);

    if( timeStamp == 0 ) // ネットワーク時計が無効な場合に 0 が設定されている。
    {
        nn::util::SNPrintf(output, sizeof(char) * length, "(Invalid)");
        return;
    }

    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    nn::time::PosixTime posixTime;
    posixTime.value = timeStamp;
    auto result = nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
    if( result.IsSuccess() )
    {
        nn::util::SNPrintf(output, sizeof(char) * length, "%04d-%02d-%02d %02d:%02d (%s)",
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute,
            calendarAdditionalInfo.timeZone.standardTimeName);
    }
    else
    {
#if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_32 ) || defined( NN_BUILD_CONFIG_OS_WIN )
        nn::util::SNPrintf(output, sizeof(char) * length, "%lld (posix time)", timeStamp);
#else
        nn::util::SNPrintf(output, sizeof(char) * length, "%ld (posix time)", timeStamp);
#endif
    }
}

const char* GetErrorPropertyTypeString(ErrorProperty::Type type) NN_NOEXCEPT
{
    switch( type )
    {
        case ErrorProperty::Type::ErrorViewer:      return "Viewer";
        case ErrorProperty::Type::ApplicationCrash: return "AppCrash";
        case ErrorProperty::Type::Fatal:            return "Fatal";
        case ErrorProperty::Type::Other:            return "-";
        default: NN_UNEXPECTED_DEFAULT;
    }
}

#if !defined(NN_DEVMENUSYSTEM)
bool IsVisibleInDevMenu(ErrorProperty::Type type) NN_NOEXCEPT
{
    return false ||
        type == ErrorProperty::Type::ErrorViewer ||
        type == ErrorProperty::Type::ApplicationCrash ||
        type == ErrorProperty::Type::Fatal;
}
#endif

bool IsErrorViewerError(const nn::msgpack::MpWalker& erptMpWalker) NN_NOEXCEPT
{
    auto visible = erptMpWalker.Find("ReportVisibilityFlag");
    if( !visible || !visible.GetBoolean().second )
    {
        return false;
    }
    auto abort = erptMpWalker.Find("AbortFlag");
    return abort ? !abort.GetBoolean().second : true;
}

bool IsApplicationCrashError(const nn::msgpack::MpWalker& erptMpWalker) NN_NOEXCEPT
{
    auto hash = erptMpWalker.Find("CrashReportHash");
    if( !hash )
    {
        return false;
    }
    auto abort = erptMpWalker.Find("ApplicationAbortFlag");
    return abort ? abort.GetBoolean().second : false;
}

bool IsFatalError(const nn::msgpack::MpWalker& erptMpWalker) NN_NOEXCEPT
{
    auto f = erptMpWalker.Find("FatalFlag");
    return f ? f.GetBoolean().second : false;
}

ErrorProperty::Type GetErrorReportType(const nn::msgpack::MpWalker& erptMpWalker) NN_NOEXCEPT
{
    if( IsApplicationCrashError(erptMpWalker) )
    {
        return ErrorProperty::Type::ApplicationCrash;
    }
    if( IsFatalError(erptMpWalker) )
    {
        return ErrorProperty::Type::Fatal;
    }
    if( IsErrorViewerError(erptMpWalker) )
    {
        return ErrorProperty::Type::ErrorViewer;
    }
    return ErrorProperty::Type::Other;
}

#if defined(NN_DEVMENUSYSTEM)
const char* GetUploadStatusString(const ErrorProperty& e) NN_NOEXCEPT
{
    return e.isUploaded ? "Uploaded" : "Pending";
}
#elif 0 // DevMenu, DevMenuApp 向け。現状アップロードが成功する機会がなく使用しないので定義しない。
const char* GetUploadStatusString(const ErrorProperty& e) NN_NOEXCEPT
{
    if( e.type == ErrorProperty::Type::ApplicationCrash )
    {
        if( e.sharePermission == nn::settings::system::ErrorReportSharePermission_Granted )
        {
            if( e.isUploaded )
            {
                return "Uploaded";
            }
            else
            {
                return "Pending";
            }
        }
        else
        {
            return "Share Not Allowed";
        }
    }
    return "";
}
#endif

} // ~devmenu::error::<anonymous>

void ErrorProperty::Prepare(int inputNumber, const nn::msgpack::MpWalker& erptMpWalker, nn::erpt::ReportFlagSet flagSet) NN_NOEXCEPT
{
    nn::util::SNPrintf(number, sizeof(number), "%d", inputNumber);

    type = GetErrorReportType(erptMpWalker);
    isUploaded = flagSet.Test<nn::erpt::ReportFlag::Transmitted>();

    const char* errorCodeStr = nullptr;
    uint32_t errorCodeStrLength = 0;
    erptMpWalker.Find("ErrorCode").GetString(&errorCodeStr, &errorCodeStrLength);
    CopyAsNullTerminatedString(rawErrorCode, sizeof(rawErrorCode), errorCodeStr, errorCodeStrLength);

    if( nn::util::Strnlen(rawErrorCode, static_cast<int>(sizeof(rawErrorCode))) == 9 ) // "2XXX-YYYY".
    {
        uint32_t category = std::strtoul(rawErrorCode, NULL, 10);
        uint32_t num = std::strtoul(rawErrorCode + 5, NULL, 10);

        // NX の category は 2XXX というフォーマットで、XXX が Result の Module に対応。
        // category がそれ以上の値の場合は Result からの生成ではないエラーコードとみなす。
        if( category >= 2000 && category <= 2000 + nn::result::detail::ResultTraits::ModuleEnd - 1 )
        {
            nn::err::ErrorCode errorCode = { category, num };
            auto result = nn::err::detail::ConvertErrorCodeToResult(errorCode);
            nn::util::SNPrintf(code, sizeof(code), "%s (result=%08x)", rawErrorCode, result.GetInnerValueForDebug());
        }
        else
        {
            nn::util::Strlcpy(code, rawErrorCode, static_cast<int>(sizeof(code)));
        }
    }
    else
    {
        nn::util::Strlcpy(code, rawErrorCode, static_cast<int>(sizeof(code)));
    }

    erptMpWalker.Find("OccurrenceTimestamp").GetInt(&posixTime.value);
    TimestampDumper(time, static_cast<int>(sizeof(time)), "", posixTime.value);

    uint32_t permission = static_cast<uint32_t>(nn::settings::system::ErrorReportSharePermission_NotConfirmed);
    erptMpWalker.Find("ErrorReportSharePermission").GetUint(&permission);
    sharePermission = static_cast<nn::settings::system::ErrorReportSharePermission>(permission);

    const char* idStr = nullptr;
    uint32_t idStrLength = 0;
    switch( type )
    {
        case Type::ApplicationCrash:
        case Type::Fatal:
            {
                erptMpWalker.Find("ProgramId").GetString(&idStr, &idStrLength);
            }
            break;
        case Type::ErrorViewer:
        case Type::Other:
            {
                erptMpWalker.Find("ApplicationID").GetString(&idStr, &idStrLength);
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    if( idStr != nullptr )
    {
        CopyAsNullTerminatedString(pid, sizeof(pid), idStr, idStrLength);
    }
    else
    {
        nn::util::Strlcpy(pid, "-", static_cast<int>(sizeof(pid)));
    }

    auto errorDescriptionField = erptMpWalker.Find("ErrorDescription");
    if( errorDescriptionField )
    {
        const char* errorDescriptionStr = nullptr;
        uint32_t errorDescriptionStrLength = 0;
        errorDescriptionField.GetString(&errorDescriptionStr, &errorDescriptionStrLength);
        description.reset(new char[ErrorProperty::MaxDescriptionLength]);
        CopyAsNullTerminatedString(description.get(), ErrorProperty::MaxDescriptionLength, errorDescriptionStr, errorDescriptionStrLength);
    }

    auto crashReportHashField = erptMpWalker.Find("CrashReportHash");
    if( crashReportHashField )
    {
        const void* binArray;
        uint32_t length;
        crashReportHashField.GetBinary(&binArray, &length);
        crashReportHash.reset(new nn::Bit8[ErrorProperty::CrashReportHashSize]);
        NN_SDK_ASSERT_EQUAL(static_cast<size_t>(length), ErrorProperty::CrashReportHashSize);
        std::memcpy(crashReportHash.get(), binArray, std::min(static_cast<size_t>(length), ErrorProperty::CrashReportHashSize));
    }
}

const glv::space_t PropertyListView::LeftMargin = 4.f;
const glv::space_t PropertyListView::NoWidth = 64.f;
const glv::space_t PropertyListView::TypeWidth = 144.f;
const glv::space_t PropertyListView::ErrorCodeWidth = 256.f;

PropertyListView::PropertyListView(const glv::Rect& parentClipRegion) NN_NOEXCEPT
    : CustomVerticalListView(parentClipRegion)
{
    SetTouchAndGo(true);
    glv::Style* pStyle = new glv::Style();
    pStyle->color = glv::Style::standard().color;
    pStyle->color.selection.set(0.1f, 0.85f, 0.2f);
    style(pStyle);
    font().size(25.f);
}

void PropertyListView::OnQueryBounds(const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight) NN_NOEXCEPT
{
    this->font().getBounds(outWidth, outHeight, item.number);
    outWidth = this->width();
}

void PropertyListView::OnDrawItem(const ItemType& item, const IndexType index, const glv::Rect& contentRegion) NN_NOEXCEPT
{
    NN_UNUSED(index);
    glv::Font& font = this->font();

    const auto NoX = contentRegion.left() + LeftMargin;
    const auto TypeX = NoX + NoWidth;
    const auto ErrorCodeX = TypeX + TypeWidth;
    font.render(item.number, NoX, contentRegion.top());
    font.render(GetErrorPropertyTypeString(item.type), TypeX, contentRegion.top());
    font.render(item.code, ErrorCodeX, contentRegion.top());

    glv::space_t outWidth, outHeight;
    font.getBounds(outWidth, outHeight, item.time);
    const glv::space_t nameExpect = contentRegion.right() - (outWidth + (paddingX() * 2) + 4.f);
    const glv::space_t nameLimit = ErrorCodeX + ErrorCodeWidth;
    font.render(item.time, (nameExpect < nameLimit) ? nameLimit : nameExpect, contentRegion.top());
}

// ErrorReportScene

ErrorReportScene::ErrorReportScene(devmenu::Page* pParent, glv::Rect rect) NN_NOEXCEPT
    : devmenu::Scene(pParent, rect, true), m_pItems(nullptr), m_pListView(nullptr), m_pLabelNoItem(nullptr)
    , m_CrashReportStatusLabel("", glv::Label::Spec(glv::Place::BL, 16, -16 - CommonValue::InitialFontSize, CommonValue::InitialFontSize))
    , m_ExportButton("Export", [&]() { ShowExportErrorReportDialog(GetRootSurfaceContext(), true); }, glv::Rect(-128 - 32, -96 + 8, 128, 44), glv::Place::BR)
#if defined( NN_DEVMENUSYSTEM )
    , m_DumpButton("Dump", [&]() { ShowExportErrorReportDialog(GetRootSurfaceContext(), false); }, glv::Rect(-128 - 32 - (128 + 16), -96 + 8, 128, 44), glv::Place::BR)
#endif
{
    // シザーボックスの領域設定
    const glv::space_t ErrorReportSceneHeaderRegion = 48.f;
    const glv::space_t ErrorReportSceneFooterRegion = 96.f;
    glv::ScissorBoxView* pListContainer = new glv::ScissorBoxView(0, ErrorReportSceneHeaderRegion, width(), height() - (ErrorReportSceneHeaderRegion + ErrorReportSceneFooterRegion));

    // リスト要素のパディング ( ボーダーを表示する場合は 2以上必要 )
    const glv::space_t ListMarginL = 16;
    const glv::space_t ListMarginR = 16;
    const glv::space_t ListMarginY = 2;

    // ヘッダ
    const glv::space_t HeaderDy = 8;
    const glv::space_t NoDx = ListMarginL + PropertyListView::LeftMargin + 4;
    const glv::space_t TypeDx = NoDx + PropertyListView::NoWidth;
    const glv::space_t ErrorCodeDx = TypeDx + PropertyListView::TypeWidth;
    *this << new glv::Label("No.", glv::Label::Spec(glv::Place::TL, NoDx, HeaderDy, CommonValue::InitialFontSize));
    *this << new glv::Label("Type", glv::Label::Spec(glv::Place::TL, TypeDx, HeaderDy, CommonValue::InitialFontSize));
    *this << new glv::Label("Error Code", glv::Label::Spec(glv::Place::TL, ErrorCodeDx, HeaderDy, CommonValue::InitialFontSize));
    *this << new glv::Label("Time of occurrence", glv::Label::Spec(glv::Place::TR, -(ListMarginL + 16), HeaderDy, CommonValue::InitialFontSize));

    // クラッシュレポート有効・無効
    UpdateCrashReportStatus(true);
    *this << m_CrashReportStatusLabel;

    // フッタ
    *this << m_ExportButton;
#if defined( NN_DEVMENUSYSTEM )
    *this << m_DumpButton;
#endif
    *this << new glv::Label(GLV_TEXT_API_STRING_UTF8("↑↓: Select an error report"), glv::Label::Spec(glv::Place::BR, -(ListMarginL + 16), -8, CommonValue::InitialFontSize));

#if !defined( APPLICATION_BUILD )

    nn::err::ErrorMessageDatabaseVersion version;
    auto getMessageVersionResult = nn::err::GetErrorMessageDatabaseVersion(&version);
    if( getMessageVersionResult.IsSuccess() )
    {
        char versionString[32];
        nn::util::SNPrintf(versionString, sizeof(versionString), "Message Version: %u.%u", version.majorVersion, version.minorVersion);
        *this << new glv::Label(versionString, glv::Label::Spec(glv::Place::BL, ListMarginR, -8, CommonValue::InitialFontSize));
    }
    else
    {
        DEVMENU_LOG("Failed to get error message version : 0x%08x\n", getMessageVersionResult.GetInnerValueForDebug());
    }
#endif

#if !defined( NN_BUILD_CONFIG_OS_WIN )

    // リストビュー
    glv::Rect clipRegion(ListMarginL, ListMarginY, pListContainer->width() - (ListMarginL + ListMarginR), pListContainer->height() - (ListMarginY * 2));
    PropertyListView* pListView = new PropertyListView(clipRegion);
    pListView->attach(OnPropertyListUpdateNotification, glv::Update::Action, this);
    *pListContainer << pListView;
    m_pListView = pListView;

#endif
    // アイテムなしメッセージ
    glv::Label* pLabelNoItem = new glv::Label("No error report to display.", glv::Label::Spec(glv::Place::CC, 0, 0, 32.f));
    *pListContainer << pLabelNoItem;
    m_pLabelNoItem = pLabelNoItem;

    *this << pListContainer;
}

void ErrorReportScene::Activate() NN_NOEXCEPT
{
    MakeErrorReportList();
    UpdateCrashReportStatus(true);
}

void ErrorReportScene::Deactivate() NN_NOEXCEPT
{
    FinalizeProperties();
}

void ErrorReportScene::UpdateCrashReportStatus(bool updateDetails) NN_NOEXCEPT
{
    std::unique_ptr< char[] > buffer;
    size_t dataSize;

    // アプリが動いていない場合
    if( launcher::IsApplicationAlive() == false )
    {
        m_CrashReportStatusLabel.setValue("Application Crash Report: not running");
        return;
    }

    if( updateDetails == false )
    {
        // 毎フレーム呼ばれる際には詳細を更新せずに返る
        return;
    }

    const auto result = application::GetApplicationControlData(&dataSize, &buffer, nn::ns::ApplicationControlSource::Storage, launcher::GetActiveApplicationId());
    if( result.IsFailure() )
    {
        bool isEnabled = nn::ns::IsApplicationCrashReportEnabled();
        m_CrashReportStatusLabel.setValue(isEnabled ? "Application Crash Report:" : "Application Crash Report: (Temporarily Disabled)");
        return;
    }

    nn::ns::ApplicationControlDataAccessor accessor(buffer.get(), dataSize);
    const nn::ns::ApplicationControlProperty& property = accessor.GetProperty();

    std::string workString = "Application Crash Report: ";
    if( property.crashReport == nn::ns::CrashReport::Allow )
    {
        workString += "Allow";
        if( true )
        {
            bool isEnabled = nn::ns::IsApplicationCrashReportEnabled();
            if( isEnabled == false )
            {
                workString += " (Temporarily Disabled)";
            }
        }
    }
    else
    {
        workString += "Deny";
    }

    m_CrashReportStatusLabel.setValue(workString.c_str());
}

glv::View* ErrorReportScene::GetFocusableChild() NN_NOEXCEPT
{
    return (nullptr == m_pItems || m_pItems->empty())
        ? static_cast< View* >(m_pLabelNoItem)
        : static_cast< View* >(m_pListView);
}

void ErrorReportScene::Refresh() NN_NOEXCEPT
{
    UpdateCrashReportStatus(false);
}

void ErrorReportScene::OnExecuteErrorDetail() NN_NOEXCEPT
{
    auto pView = new MessageView();
    const ErrorProperty *pCurrentProperty = this->m_pListView->GetSelectedValue();

    char message[128];

    nn::util::SNPrintf(message, sizeof(message), "No.%s - %s", pCurrentProperty->number, pCurrentProperty->time);
    pView->AddMessage(message);

    nn::util::SNPrintf(message, sizeof(message), "ID: %s", pCurrentProperty->pid);
    pView->AddMessage(message);

    nn::util::SNPrintf(message, sizeof(message), "Error Code: %s", pCurrentProperty->code);
    pView->AddMessage(message);

#if defined( NN_DEVMENUSYSTEM )
    if( pCurrentProperty->crashReportHash )
    {
#else
    if( pCurrentProperty->type == ErrorProperty::Type::ApplicationCrash )
    {
#endif
        auto hashPrefixLen = nn::util::SNPrintf(message, sizeof(message), "Hash: ");
        for( int i = 0; i < ErrorProperty::CrashReportHashSize; i++ )
        {
            nn::util::SNPrintf(message + hashPrefixLen + i * 2, 3, "%02x", pCurrentProperty->crashReportHash.get()[i]);
        }
        pView->AddMessage(message);
    }

#if defined( NN_DEVMENUSYSTEM )
    nn::util::SNPrintf(message, sizeof(message), "Upload Status: %s", GetUploadStatusString(*pCurrentProperty));
    pView->AddMessage(message);
#endif

    pView->AddButton("Close");

    if( pCurrentProperty->type != ErrorProperty::Type::Fatal )
    {
        pView->AddButton(
            "Error Viewer",
            [pCurrentProperty](void*, nn::TimeSpan&)
            {
                NN_ALIGNAS(8) static std::unique_ptr<char[]> pWorkBuffer(new char[nn::err::ErrorViewerStartupParamSizeMax]);
                size_t actualSize;

                if( !pCurrentProperty->description )
                {
                    nn::err::CreateErrorViewerStartupParamForRecordedError(pWorkBuffer.get(), &actualSize, nn::err::ErrorViewerStartupParamSizeMax, pCurrentProperty->rawErrorCode, nullptr, pCurrentProperty->posixTime);
                }
                else
                {
                    nn::err::CreateErrorViewerStartupParamForRecordedError(pWorkBuffer.get(), &actualSize, nn::err::ErrorViewerStartupParamSizeMax, pCurrentProperty->rawErrorCode, pCurrentProperty->description.get(), pCurrentProperty->posixTime);
                }
                launcher::ShowErrorRecord(pWorkBuffer.get(), actualSize);
            },
            this,
            MessageView::ButtonTextColor::Green
        );
    }

    GetRootSurfaceContext()->StartModal(pView, true);
}

void ErrorReportScene::OnPropertyListUpdateNotification(const glv::Notification& notification) NN_NOEXCEPT
{
    ErrorReportScene* pScene;
    const ErrorProperty* pProperty;
    if( nullptr != (pScene = notification.receiver< ErrorReportScene >()) && nullptr != (pProperty = notification.data< PropertyListView::ItemType >()) )
    {
        pScene->OnExecuteErrorDetail();
    }
}

void ErrorReportScene::EntryProperties(const PropertyListView::CollectionType* pItems) NN_NOEXCEPT
{
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    PropertyListView* pListView;
    if( nullptr != (pListView = m_pListView) )
    {
        pListView->EntryCollection(*pItems);
        if( true == pItems->empty() )
        {
            pListView->disable(focusableProperty);
            auto* const pLabelNoItem = m_pLabelNoItem;
            pLabelNoItem->enable(glv::Property::t::Visible);
            pLabelNoItem->bringToFront();
            // Focus出来るViewが無くなるのでメインメニューにフォーカスを戻す。
            GetRootSurfaceContext()->MoveFocusToMenuTabs();
        }
        else
        {
            pListView->enable(focusableProperty);
            m_pLabelNoItem->disable(glv::Property::t::Visible);
        }
    }
}

void ErrorReportScene::FinalizeProperties() NN_NOEXCEPT
{
    delete m_pItems;
    m_pItems = nullptr;
}

nn::Result ErrorReportScene::MakeErrorReportList() NN_NOEXCEPT
{
#if !defined( NN_BUILD_CONFIG_OS_WIN )

    NN_FUNCTION_LOCAL_STATIC(nn::erpt::ReportList, reportList);

    nn::erpt::Manager manager;

    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
    NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.GetReportList(reportList));

    FinalizeProperties();
    m_pItems = new PropertyListView::CollectionType();

    int count = 0;
    for( size_t ri = 0; ri < reportList.reportCount; ri++ )
    {
        nn::erpt::Report report;
        NN_ABORT_UNLESS_RESULT_SUCCESS(report.Open(reportList.Report[ri].reportId));

        int64_t reportSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(report.GetSize(&reportSize));

        std::unique_ptr<uint8_t[]> reportData(new uint8_t[static_cast<uint32_t>(reportSize)]);
        NN_RESULT_THROW_UNLESS(reportData != nullptr, nn::erpt::ResultOutOfMemory());

        uint32_t readCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(report.Read(&readCount, reportData.get(), static_cast<uint32_t>(reportSize)));

        nn::erpt::ReportFlagSet flagSet;
        NN_ABORT_UNLESS_RESULT_SUCCESS(report.GetFlags(&flagSet));

        nn::msgpack::MpWalker mpWalker;
        mpWalker.Init(reportData.get(), readCount);

#if !defined (NN_DEVMENUSYSTEM)
        auto type = GetErrorReportType(mpWalker);
        if( !IsVisibleInDevMenu(type) )
        {
            continue;
        }
#endif

        ErrorProperty ep;
        // ユーザに見せる report 番号は for 文の添字 + 1
        ep.Prepare(++count, mpWalker, flagSet);
        m_pItems->push_back(std::move(ep));
    }

    EntryProperties(m_pItems);
    UpdateButtonStatus(count > 0);
#else
    // Windows 環境では常に無効化
    UpdateButtonStatus(false);
#endif
    NN_RESULT_SUCCESS;
}

void ErrorReportScene::UpdateButtonStatus(bool hasErrorReport) NN_NOEXCEPT
{
#if !defined( NN_BUILD_CONFIG_OS_WIN )
    if( hasErrorReport )
    {
        m_ExportButton.UpdateFocusAndColor(true, true);
#if defined( NN_DEVMENUSYSTEM )
        m_DumpButton.UpdateFocusAndColor(true, true);
#endif
    }
    else
#endif
    {
        m_ExportButton.UpdateFocusAndColor(false, false);
#if defined( NN_DEVMENUSYSTEM )
        m_DumpButton.UpdateFocusAndColor(false, false);
#endif
    }
}

}} // ~namespace devmenu::error, ~namespace devmenu
