﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt.h>
#include <nn/erpt/erpt_Context.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_Report.h>
#include <nn/fs.h>
#include <nn/fs/fs_Context.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/htc/htc_Api.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/time.h>
#include <nn/util/util_StringUtil.h>

#include "../Common/DevMenu_CommonSettingsApi.h"
#include "../DevMenu_Config.h"
#include "../DevMenu_ModalView.h"
#include "../DevMenu_RootSurface.h"

#include "errors/DevMenuCommand_ErrorExport.h"
#include "DevMenuCommand_Common.h"

#if defined( NN_DEVMENUSYSTEM )
#include "errors/DevMenuCommand_ErrorDump.h"
#endif

namespace devmenu { namespace error {

namespace {

void MakeExportReportName(char outValue[], size_t outValueLength, const char* extension) NN_NOEXCEPT
{
    nn::settings::system::SerialNumber serialNumber;
    nn::settings::system::GetSerialNumber(&serialNumber);

    nn::time::PosixTime time;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&time));

    nn::time::CalendarTime calendarTime;
    auto result = nn::time::ToCalendarTime(&calendarTime, nullptr , time);
    if( result.IsSuccess() )
    {
        int l = nn::util::SNPrintf(outValue, outValueLength, "ErrorReport_%s_%04d%02d%02d-%02d%02d%02d.%s",
            serialNumber.string,
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second,
            extension);
        NN_ASSERT_LESS_EQUAL(l, static_cast<int>(outValueLength));
        NN_UNUSED(l);
    }
    else
    {
        int l = nn::util::SNPrintf(outValue, outValueLength, "ErrorReport_%s.%s", serialNumber.string, extension);
        NN_ASSERT_LESS_EQUAL(l, static_cast<int>(outValueLength));
        NN_UNUSED(l);
    }
}

nn::Result ExportReport(bool* outIsExported, const char* path) NN_NOEXCEPT
{
    nn::erpt::Manager manager;
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
    NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };

    nn::erpt::ReportList reportList;
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.GetReportList(reportList));
    if( reportList.reportCount == 0 )
    {
        *outIsExported = false;
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_DO(errors::ExportReport(path, reportList, errors::ErrorReportBinaryHeaderFlag_All));
    *outIsExported = true;
    NN_RESULT_SUCCESS;
}

void ShowExportResultMessage(devmenu::RootSurfaceContext* surface, const char* message) NN_NOEXCEPT
{
    auto resultMessageView = new devmenu::MessageView(false);
    resultMessageView->AddMessage(message);
    resultMessageView->AddButton("OK");
    surface->StartModal(resultMessageView, true, true);
}

void ShowCreateExportDirectoryFailureMessage(devmenu::RootSurfaceContext* surface, const char* directoryPath, nn::Result failureResult) NN_NOEXCEPT
{
    char resultMessage[256];
    nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "Failed to create directory\n\"%s\".\nResult = 0x%08x", directoryPath, failureResult.GetInnerValueForDebug());
    ShowExportResultMessage(surface, resultMessage);
}

void ExportReportAndShowResultMessage(devmenu::RootSurfaceContext* surface, const char* exportPath, const char* displayPath) NN_NOEXCEPT
{
    char resultMessage[256];
    bool isExported;
    auto exportResult = ExportReport(&isExported, exportPath);
    if( exportResult.IsFailure() )
    {
        int l = nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "Failed to export error reports.\nResult = 0x%08x", exportResult.GetInnerValueForDebug());
        NN_SDK_ASSERT_LESS_EQUAL(l, static_cast<int>(sizeof(resultMessage)));
        NN_UNUSED(l);
    }
    else if( !isExported )
    {
        int l = nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "No error report to export.");
        NN_SDK_ASSERT_LESS_EQUAL(l, static_cast<int>(sizeof(resultMessage)));
        NN_UNUSED(l);
    }
    else
    {
        int l = nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "Error reports are successfully exported as\n%s", displayPath);
        NN_SDK_ASSERT_LESS_EQUAL(l, static_cast<int>(sizeof(resultMessage)));
        NN_UNUSED(l);
    }
    ShowExportResultMessage(surface, resultMessage);
}

#if defined( NN_DEVMENUSYSTEM )

nn::Result DumpReport(bool* outIsDumped, const char* path) NN_NOEXCEPT
{
    nn::erpt::Manager manager;
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.Initialize());
    NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };

    nn::erpt::ReportList reportList;
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager.GetReportList(reportList));
    if( reportList.reportCount == 0 )
    {
        *outIsDumped = false;
        NN_RESULT_SUCCESS;
    }

    nn::fs::FileHandle fileHandle;
    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound) {}
    NN_RESULT_END_TRY
    NN_RESULT_DO(nn::fs::CreateFile(path, 0));
    NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    auto toolVersion = errors::GetToolVersion();
    char toolVersionStr[32];
    errors::GetToolVersionString(toolVersionStr, sizeof(toolVersionStr), toolVersion);
    char toolVersionLabel[128];
    auto toolVersionLabelLength = nn::util::SNPrintf(toolVersionLabel, sizeof(toolVersionLabel), "Dumped by DevMenuSystem ver. %s\n", toolVersionStr);

    nn::fs::WriteFile(fileHandle, 0, toolVersionLabel, toolVersionLabelLength, nn::fs::WriteOption::MakeValue(0));
    errors::BeginDumpOutputToFile(fileHandle, toolVersionLabelLength);
    NN_UTIL_SCOPE_EXIT
    {
        errors::EndDumpOutputToFile();
        nn::fs::FlushFile(fileHandle);
        nn::fs::CloseFile(fileHandle);
    };

    for( uint32_t i = 0; i < reportList.reportCount; ++i )
    {
        auto reportInfo = reportList.Report[i];
        NN_ABORT_UNLESS_RESULT_SUCCESS(errors::DumpReport(reportInfo.reportId, reportInfo.reportMetaData));
    }
    *outIsDumped = true;
    NN_RESULT_SUCCESS;
}

void DumpReportAndShowResultMessage(devmenu::RootSurfaceContext* surface, const char* dumpPath, const char* displayPath) NN_NOEXCEPT
{
    char resultMessage[256];
    bool isExported;
    auto dumpResult = DumpReport(&isExported, dumpPath);
    if( dumpResult.IsFailure() )
    {
        int l = nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "Failed to dump error reports.\nResult = 0x%08x", dumpResult.GetInnerValueForDebug());
        NN_SDK_ASSERT_LESS_EQUAL(l, static_cast<int>(sizeof(resultMessage)));
        NN_UNUSED(l);
    }
    else if( !isExported )
    {
        int l = nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "No error report to dump.");
        NN_SDK_ASSERT_LESS_EQUAL(l, static_cast<int>(sizeof(resultMessage)));
        NN_UNUSED(l);
    }
    else
    {
        int l = nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "Error reports are successfully dumped as\n%s", displayPath);
        NN_SDK_ASSERT_LESS_EQUAL(l, static_cast<int>(sizeof(resultMessage)));
        NN_UNUSED(l);
    }
    ShowExportResultMessage(surface, resultMessage);
}
#endif // ~#if defined( NN_DEVMENUSYSTEM )

} // ~devmenu::error::<anonymous>

void ShowExportErrorReportDialog(devmenu::RootSurfaceContext* pRootSurfaceContext, bool encrypt) NN_NOEXCEPT
{
    // lambda キャプチャ用に static 変数に格納
    NN_FUNCTION_LOCAL_STATIC(devmenu::RootSurfaceContext*, s_pRootSurfaceContext);
    s_pRootSurfaceContext = pRootSurfaceContext;
    NN_FUNCTION_LOCAL_STATIC(bool, s_Encrypt);
    s_Encrypt = encrypt;
    auto pMessageView = new devmenu::MessageView( true );
#if defined( NN_DEVMENUSYSTEM )
    if( !s_Encrypt )
    {
        pMessageView->AddMessage("Dump error report");
    }
    else
#endif
    {
        pMessageView->AddMessage("Export error report");
    }
    pMessageView->AddButton( "Cancel" );

    // Prod Mode では tma プロセスがダミーなので nn::htc::Initialize() を呼ぶとブロックされてしまう
    if( devmenu::IsDebugModeEnabled() )
    {
        pMessageView->AddButton("To Host PC", [&](void*, nn::TimeSpan) {
            // %USERPROFILE% の取得
            nn::htc::Initialize();
            NN_UTIL_SCOPE_EXIT{ nn::htc::Finalize(); };
            size_t userProfileSize;
            char userProfile[128];
            auto getUserProfileResult = nn::htc::GetEnvironmentVariable(&userProfileSize, userProfile, sizeof(userProfile), "USERPROFILE");
            if( getUserProfileResult.IsFailure() )
            {
                char resultMessage[128];
                nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "Failed to get USERPROFILE environment variable.\nResult = 0x%08x", getUserProfileResult.GetInnerValueForDebug());
                ShowExportResultMessage(s_pRootSurfaceContext, resultMessage);
                return;
            }
            if( userProfileSize > sizeof(userProfile) )
            {
                char resultMessage[128];
                nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "USERPROFILE environment variable is too long.\nIt must be shorter than %u.", sizeof(userProfile));
                ShowExportResultMessage(s_pRootSurfaceContext, resultMessage);
                return;
            }
            // 書込み禁止箇所にアクセスした際のアボートを抑制し、失敗メッセージを表示させる。
            nn::fs::ScopedAutoAbortDisabler fsAutoAbortDisabler;
            // マウント
            const char* HostMountName = "host";
            auto mountHostResult = nn::fs::MountHost(HostMountName, userProfile);
            if( mountHostResult.IsFailure() )
            {
                char resultMessage[256];
                nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "Failed to mount %%USERPROFILE%% (%s).\nResult = 0x%08x", userProfile, mountHostResult.GetInnerValueForDebug());
                ShowExportResultMessage(s_pRootSurfaceContext, resultMessage);
                return;
            }
            NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(HostMountName); };
            // 出力先ディレクトリ作成
            const char* ExportDirectory = "Documents/Nintendo/NxErrorReports";
            char exportPath[256];
            int l = nn::util::SNPrintf(exportPath, sizeof(exportPath), "%s:/%s/", HostMountName, ExportDirectory);
            auto createExportDirectoryResult = devmenuUtil::CreateDirectoryRecursively(exportPath, false);
            if( createExportDirectoryResult.IsFailure() )
            {
                ShowCreateExportDirectoryFailureMessage(s_pRootSurfaceContext, exportPath, createExportDirectoryResult);
                return;
            }
            // エクスポート
#if defined( NN_DEVMENUSYSTEM )
            if( !s_Encrypt )
            {
                MakeExportReportName(exportPath + l, sizeof(exportPath) - l, "txt");
                char displayPath[256];
                nn::util::SNPrintf(displayPath, sizeof(displayPath), "%s\nin %s/%s.", exportPath + l, userProfile, ExportDirectory);
                DumpReportAndShowResultMessage(s_pRootSurfaceContext, exportPath, displayPath);
            }
            else
#endif
            {
                MakeExportReportName(exportPath + l, sizeof(exportPath) - l, "bin");
                char displayPath[256];
                nn::util::SNPrintf(displayPath, sizeof(displayPath), "%s\nin %s/%s.", exportPath + l, userProfile, ExportDirectory);
                ExportReportAndShowResultMessage(s_pRootSurfaceContext, exportPath, displayPath);
            }
        });
    }
    pMessageView->AddButton("To SD Card", [&](void*, nn::TimeSpan) {
        // 書込み禁止箇所にアクセスした際のアボートを抑制し、失敗メッセージを表示させる。
        nn::fs::ScopedAutoAbortDisabler fsAutoAbortDisabler;
        // マウント
        const char* SdMountName = "sd";
        auto mountSdResult = nn::fs::MountSdCardForDebug(SdMountName);
        if( mountSdResult.IsFailure() )
        {
            char resultMessage[128];
            nn::util::SNPrintf(resultMessage, sizeof(resultMessage), "Failed to mount the SD Card.\nResult = 0x%08x", mountSdResult.GetInnerValueForDebug());
            ShowExportResultMessage(s_pRootSurfaceContext, resultMessage);
            return;
        }
        NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(SdMountName); };
        // 出力先ディレクトリ作成
        const char* ExportDirectory = "NintendoSDK/NxErrorReports";
        char exportPath[128];
        int l = nn::util::SNPrintf(exportPath, sizeof(exportPath), "%s:/%s/", SdMountName, ExportDirectory);
        auto createExportDirectoryResult = devmenuUtil::CreateDirectoryRecursively(exportPath, false);
        if( createExportDirectoryResult.IsFailure() )
        {
            ShowCreateExportDirectoryFailureMessage(s_pRootSurfaceContext, exportPath, createExportDirectoryResult);
            return;
        }
        // エクスポート
#if defined( NN_DEVMENUSYSTEM )
        if( !s_Encrypt )
        {
            MakeExportReportName(exportPath + l, sizeof(exportPath) - l, "txt");
            char displayPath[128];
            nn::util::SNPrintf(displayPath, sizeof(displayPath), "%s\nin %s in the SD Card.", exportPath + l, ExportDirectory);
            DumpReportAndShowResultMessage(s_pRootSurfaceContext, exportPath, displayPath);
        }
        else
#endif
        {
            MakeExportReportName(exportPath + l, sizeof(exportPath) - l, "bin");
            char displayPath[128];
            nn::util::SNPrintf(displayPath, sizeof(displayPath), "%s\nin %s in the SD Card.", exportPath + l, ExportDirectory);
            ExportReportAndShowResultMessage(s_pRootSurfaceContext, exportPath, displayPath);
        }
    });

    s_pRootSurfaceContext->StartModal(pMessageView, true, false, true);
} // NOLINT(impl/function_size)

}} // ~namespace devmenu::error, ~namespace devmenu
