﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "DevMenu_DownloadProgressViewUnit.h"

namespace devmenu { namespace system { namespace download {

    ProgressViewUnit::ProgressViewUnit
(
    nn::ns::ApplicationView view,
    glv::space_t yPos,
    glv::space_t width,
    glv::space_t height,
    glv::space_t idPos,
    glv::space_t statePos,
    glv::space_t downloadedPos,
    glv::space_t sliderPos,
    glv::space_t sliderLength,
    glv::space_t buttonPos,
    glv::space_t buttonLength,
    std::function< void( const nn::ncm::ApplicationId& id ) > callback
) NN_NOEXCEPT
    : m_View(view)
    , m_IdLabel(nullptr)
    , m_ProgressSlider(nullptr)
    , m_Button(nullptr)
    , m_ButtonLabel(nullptr)
    , m_StateLabel(nullptr)
    , m_ProgressLabel(nullptr)
    , m_IsDownloadRequired(false)
    , m_DownloadButtonCallback( callback )
{
    m_Position.y = yPos;
    m_Position.id = idPos;
    m_Position.state = statePos;
    m_Position.progressLabel = downloadedPos;
    m_Position.progressSlider = sliderPos;
    m_Position.button = buttonPos;

    const int labelStringSize = 30;
    char labelString[labelStringSize];

    nn::util::SNPrintf(labelString, sizeof(labelString), "0x%016llx", view.id);
    m_IdLabel = new glv::Label(labelString, glv::Label::Spec(glv::Place::TL, m_Position.id, m_Position.y, CommonValue::InitialFontSize));

    nn::util::SNPrintf(labelString, sizeof(labelString), "%s", DecisionDownloadState(view));
    std::unique_ptr<glv::Label> pState(new glv::Label(labelString, glv::Label::Spec(glv::Place::TL, m_Position.state, m_Position.y, CommonValue::InitialFontSize)));
    m_StateLabel = std::move(pState);

    nn::util::SNPrintf(labelString, sizeof(labelString), "%lluKB / %lluKB", view.progress.downloaded / 1024, view.progress.total / 1024);
    std::unique_ptr<glv::Label> pProgress(new glv::Label(labelString, glv::Label::Spec(glv::Place::TL, m_Position.progressLabel, m_Position.y, CommonValue::InitialFontSize)));
    m_ProgressLabel = std::move(pProgress);
    m_ProgressSlider = new glv::Slider(glv::Rect(m_Position.progressSlider, m_Position.y, sliderLength, CommonValue::InitialFontSize));
    m_ProgressSlider->disable( glv::Property::Controllable | glv::Property::Focused | glv::Property::FocusHighlight );

    m_Button = new glv::Button(glv::Rect(m_Position.button, m_Position.y, buttonLength, CommonValue::InitialFontSize ), true);
    m_Button->attach([](const glv::Notification& n)->void { n.receiver<ProgressViewUnit>()->SetDownloadRequired(); }, glv::Update::Clicked, this);
    m_ButtonLabel = new glv::Label(GLV_TEXT_API_WIDE_STRING("Download"), glv::Label::Spec(glv::Place::CC, 0, 0, CommonValue::InitialFontSize));
    *m_Button << m_ButtonLabel;

    std::unique_ptr<glv::Table> table( new glv::Table( "<", 0.0f, 0.0f, glv::Rect( width, height ) ) );
    *table
        << GetIdLabel()
        << GetProgressSlider()
        << GetButton();
    m_ApplicationTable = std::move( table );
}

ProgressViewUnit::~ProgressViewUnit()
{
    m_IsDownloadRequired = false;

    //!< 更新しない描画オブジェクトの破棄
    DeleteViewObject(&m_IdLabel);
    DeleteViewObject(&m_ProgressSlider);
    DeleteViewObject(&m_ButtonLabel);
    DeleteViewObject(&m_Button);
}

template<class T> void ProgressViewUnit::DeleteViewObject(T** pViewObject) NN_NOEXCEPT
{
    if (*pViewObject != nullptr)
    {
        (*pViewObject)->remove();
        delete *pViewObject;
        *pViewObject = nullptr;
    }
}

void ProgressViewUnit::UpdateProgress(const nn::ns::ApplicationView& view) NN_NOEXCEPT
{
    if (view.HasMainEntity())   //!< アプリの実態が存在するかどうかの判定
    {
        m_ProgressSlider->setValue(1);
    }
    else
    {
        if (view.progress.total != 0)
        {
            m_ProgressSlider->setValue(static_cast<double>(view.progress.downloaded) / static_cast<double>(view.progress.total));
        }
        else
        {
            m_ProgressSlider->setValue(0);
        }
    }

    const int labelStringSize = 30;
    char labelString[labelStringSize];

    if (m_StateLabel != nullptr)
    {
        m_StateLabel->remove();
        nn::util::SNPrintf(labelString, sizeof(labelString), "%s", DecisionDownloadState(view));
        std::unique_ptr<glv::Label> pState(new glv::Label(labelString, glv::Label::Spec(glv::Place::TL, m_Position.state, m_Position.y, CommonValue::InitialFontSize)));
        m_StateLabel = std::move(pState);
    }

    if (m_ProgressLabel != nullptr)
    {
        m_ProgressLabel->remove();
        if( view.IsDownloading() )
        {
            // ダウンロード中の場合、総サイズとダウンロード済みサイズを表示する
            nn::util::SNPrintf( labelString, sizeof( labelString ), "%lluKB / %lluKB", view.progress.downloaded / 1024, view.progress.total / 1024 );
        }
        else
        {
            nn::util::SNPrintf( labelString, sizeof( labelString ), "               ", view.progress.downloaded / 1024, view.progress.total / 1024 );
        }
        std::unique_ptr<glv::Label> pProgress( new glv::Label( labelString, glv::Label::Spec( glv::Place::TL, m_Position.progressLabel, m_Position.y, CommonValue::InitialFontSize ) ) );
        m_ProgressLabel = std::move( pProgress );
    }
}

const char* ProgressViewUnit::DecisionDownloadState(nn::ns::ApplicationView view)
{
    switch (view.progress.state)
    {
    case nn::ns::ApplicationDownloadState::Runnable:
        {
            if ( view.HasMainEntity() )
            {
                return "Existed";    //!< すでに実態が存在している状態
            }
            else if ( view.IsDownloading() )
            {
                return "Downloading";   //!< ダウンロード中の状態
            }
            else
            {
                return "None";
            }
        }
    case nn::ns::ApplicationDownloadState::Fatal:
        {
            return "Fatal";   //!< ダウンロードが始まらない致命的なエラーが発生
        }
    case nn::ns::ApplicationDownloadState::Suspended:
        {
            return "Suspended";        //!< ダウンロード待ち状態
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}

}}} // ~namespace devmenu::system::download, ~namespace devmenu::system, ~namespace devmenu
