﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/nn_Abort.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_StandardLocalSystemClockPrivilegeApi.h>

#include "DevMenu_DeviceSettingsSceneType.h"
#include "DevMenu_DeviceSettingsSystem.h"
#include "../Common/DevMenu_CommonCheckBox.h"
#include "../Common/DevMenu_CommonDropDown.h"
#include "../Common/DevMenu_CommonIconButton.h"
#include "../Common/DevMenu_CommonIconLabel.h"
#include "../DevMenu_RootSurface.h"
#include "DevMenuCommand_TimeZone.h"

namespace devmenu { namespace devicesettings {

namespace {
    void SetLanguage( const std::string& language ) NN_NOEXCEPT
    {
        nn::settings::LanguageCode code;
        nn::util::Strlcpy( code.string, language.c_str(), sizeof( code.string ) );
        nn::settings::system::SetLanguageCode( code );
    }

    const std::string GetCurrentLanguage() NN_NOEXCEPT
    {
        nn::settings::LanguageCode currentCode;
        nn::settings::GetLanguageCode( &currentCode );
        return std::string( currentCode.string );
    }

    const std::vector< RebootRequiredSetting::valuePair >& GetDropDownItemsForLanguageSetting() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC( std::vector< RebootRequiredSetting::valuePair >, s_Items, =
            {
                RebootRequiredSetting::valuePair( std::string( "en-US" )  , std::string( "American English" ) ),
                RebootRequiredSetting::valuePair( std::string( "en-GB" )  , std::string( "British English" ) ),
                RebootRequiredSetting::valuePair( std::string( "ja" )     , std::string( "Japanese" ) ),
                RebootRequiredSetting::valuePair( std::string( "fr" )     , std::string( "French" ) ),
                RebootRequiredSetting::valuePair( std::string( "de" )     , std::string( "German" ) ),
                RebootRequiredSetting::valuePair( std::string( "es-419" ) , std::string( "Latin American Spanish" ) ),
                RebootRequiredSetting::valuePair( std::string( "es" )     , std::string( "Spanish" ) ),
                RebootRequiredSetting::valuePair( std::string( "it" )     , std::string( "Italian" ) ),
                RebootRequiredSetting::valuePair( std::string( "nl" )     , std::string( "Dutch" ) ),
                RebootRequiredSetting::valuePair( std::string( "fr-CA" )  , std::string( "Canadian French" ) ),
                RebootRequiredSetting::valuePair( std::string( "pt" )     , std::string( "Portuguese" ) ),
                RebootRequiredSetting::valuePair( std::string( "ru" )     , std::string( "Russian" ) ),
                RebootRequiredSetting::valuePair( std::string( "zh-Hans" ), std::string( "Simplified Chinese" ) ),
                RebootRequiredSetting::valuePair( std::string( "zh-Hant" ), std::string( "Traditional Chinese" ) ),
                RebootRequiredSetting::valuePair( std::string( "ko" )     , std::string( "Korean" ) ),
            }
        );

        return s_Items;
    }

    void SetRegion( const std::string& region ) NN_NOEXCEPT
    {
        nn::settings::system::RegionCode regionCode;

        if ( "Japan" == region )
        {
            regionCode = nn::settings::system::RegionCode_Japan;
        }
        else if ( "Usa" == region )
        {
            regionCode = nn::settings::system::RegionCode_Usa;
        }
        else if ( "Europe" == region )
        {
            regionCode = nn::settings::system::RegionCode_Europe;
        }
        else if ( "Australia" == region )
        {
            regionCode = nn::settings::system::RegionCode_Australia;
        }
        else
        {
            return;
        }

        nn::settings::system::SetRegionCode( regionCode );
    }

    const std::string GetCurrentRegion() NN_NOEXCEPT
    {
        nn::settings::system::RegionCode currentCode;
        nn::settings::system::GetRegionCode( &currentCode );

        switch ( currentCode )
        {
        case nn::settings::system::RegionCode_Japan:
            return std::string( "Japan" );
        case nn::settings::system::RegionCode_Usa:
            return std::string( "Usa" );
        case nn::settings::system::RegionCode_Europe:
            return std::string( "Europe" );
        case nn::settings::system::RegionCode_Australia:
            return std::string( "Australia" );
        case nn::settings::system::RegionCode_China:
            return std::string( "China" );
        case nn::settings::system::RegionCode_Korea:
            return std::string( "Korea" );
        case nn::settings::system::RegionCode_Taiwan:
            return std::string( "Taiwan" );
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const std::vector< RebootRequiredSetting::valuePair >& GetDropDownItemsForRegionSetting() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC( std::vector< RebootRequiredSetting::valuePair >, s_Items, =
            {
                RebootRequiredSetting::valuePair( std::string( "Japan" )    , std::string( "Japan" ) ),
                RebootRequiredSetting::valuePair( std::string( "Usa" )      , std::string( "Usa" ) ),
                RebootRequiredSetting::valuePair( std::string( "Europe" )   , std::string( "Europe" ) ),
                RebootRequiredSetting::valuePair( std::string( "Australia" ), std::string( "Australia" ) ),
                // 中韓台は設定不可の為リストに追加しない
            }
        );

        auto currentRegion = GetCurrentRegion();
        if ( "China" == currentRegion || "Korea" == currentRegion || "Taiwan" == currentRegion )
        {
            // 関数が 2 回実行された時に重複する要素が追加されるのを防ぐために、リストに存在するかをチェックする
            bool exists = false;
            for ( auto& item : s_Items )
            {
                if ( item.first == currentRegion )
                {
                    exists = true;
                    break;
                }
            }

            if ( !exists )
            {
                // リスト作成時の Region が中韓台の場合は DropDownList に表示する為に項目を追加する
                s_Items.push_back( RebootRequiredSetting::valuePair( currentRegion, currentRegion ) );
            }
        }

        return s_Items;
    }
} // end of anonymous namespace

/**************************************
 * class RebootRequiredSetting
 **************************************/
RebootRequiredSetting::RebootRequiredDropDown::RebootRequiredDropDown( const glv::Rect& rect, float textSize,
    const std::function< void( const std::string&, const std::string& ) >& callback,
    const std::function< void() >& adhocScrollCallback,
    const std::function< const std::vector< RebootRequiredSetting::valuePair >&() >& getDropDownItemsCallback,
    const std::function< const std::string() >& getCurrentSettingValueCallback ) NN_NOEXCEPT
    : DropDownStringKeyValueBase( rect, textSize, callback )
    , m_AdhocScrollCallback( adhocScrollCallback )
    , m_GetDropDownItemsCallback( getDropDownItemsCallback )
    , m_GetCurrentSettingValueCallback( getCurrentSettingValueCallback )
{
    NN_ASSERT_NOT_NULL( m_GetDropDownItemsCallback );
    NN_ASSERT_NOT_NULL( m_GetCurrentSettingValueCallback );

    mItemList.font().size( textSize * 0.8f );
    this->RegisterDropDownItems();
    this->Refresh();
    attach( []( const glv::Notification& notification )->void { notification.receiver< RebootRequiredDropDown >()->UpdateSettingsValue(); }, glv::Update::Action, this );
}

std::string RebootRequiredSetting::RebootRequiredDropDown::GetSettingsValue() NN_NOEXCEPT
{
    return m_GetCurrentSettingValueCallback();
}

void RebootRequiredSetting::RebootRequiredDropDown::RegisterDropDownItems() NN_NOEXCEPT
{
    for ( const auto& iter : m_GetDropDownItemsCallback() )
    {
        m_Values.push_back( DropDownStringKeyValueBase::valuePair( iter.first, iter.second ) );
        addItem( iter.second );
    }
}

void RebootRequiredSetting::RebootRequiredDropDown::UpdateSettingsValue() NN_NOEXCEPT
{
    DropDownStringKeyValueBase::UpdateSettingsValue();
}

bool RebootRequiredSetting::RebootRequiredDropDown::onEvent( glv::Event::t event, glv::GLV& glvRoot ) NN_NOEXCEPT
{
    if ( glv::Event::FocusGained == event && nullptr != m_AdhocScrollCallback )
    {
        m_AdhocScrollCallback();
    }

    return DropDownBase::onEvent( event, glvRoot );
}

RebootRequiredSetting::RebootRequiredSetting( const char* labelString, glv::space_t itemWidth, glv::space_t dropDownWidth, Page* pParentPage,
    const std::function< void() >& adhocScrollCallback,
    const std::function< const std::vector< RebootRequiredSetting::valuePair >&() >& getDropDownItemsCallback,
    const std::function< const std::string() >& getSettingsValueCallback,
    const std::function< void( const std::string& selected ) >& setSettingValueCallback ) NN_NOEXCEPT
    : m_LabelString( std::string( labelString ) )
    , m_pDropDown( nullptr )
    , m_pParentPage( pParentPage )
    , m_pRebootDialog( nullptr )
    , m_IsRebootDialogUnvisible( false )
    , m_SetSettingValueCallback ( setSettingValueCallback )
{
    NN_ASSERT_NOT_NULL( m_SetSettingValueCallback );

    auto pLabel = new glv::Label( labelString, DefaultLabelSpec );
    auto pDropDown = new RebootRequiredSetting::RebootRequiredDropDown( glv::Rect( dropDownWidth, 35.0f ), CommonValue::InitialFontSize,
        [&]( const std::string& selected, std::string current ){ this->DisplayRebootDialog( selected, current ); },
        adhocScrollCallback,
        getDropDownItemsCallback,
        getSettingsValueCallback );
    pDropDown->anchor( glv::Place::TR ).pos( -pDropDown->width(), 0.0f );
    pDropDown->enable( glv::Property::KeepWithinParent );

    auto pGroup = new glv::Group( glv::Rect( itemWidth, pDropDown->h ) );
    *pGroup << pLabel << pDropDown;

    *this << pGroup;
    arrange().fit( false );

    m_pDropDown = pDropDown;
}

void RebootRequiredSetting::DisplayRebootDialog( const std::string& selected, const std::string& current ) NN_NOEXCEPT
{
    auto pView = new MessageView( true );

    m_pDropDown->setValue( m_pDropDown->GetDropDownItemString( current ) );

    std::string lowerString;
    lowerString.resize( m_LabelString.size() );
    std::transform( m_LabelString.cbegin(), m_LabelString.cend(), lowerString.begin(), tolower );

#if !defined( APPLICATION_BUILD )
    std::string selectedMessage = std::string( "Need to reboot device to change " ) + lowerString + std::string( ".\n" )
        + std::string( "You selected \"" ) + std::string( m_pDropDown->GetDropDownItemString( selected ) ) + std::string( "\"." );
    pView->AddMessage( selectedMessage );
    pView->AddButton( "Cancel" );
    pView->AddButton(
        "Reboot",
        [ this, selected ]( void* pParam, nn::TimeSpan& timespan )
        {
            m_SetSettingValueCallback( selected );
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
            devmenu::RequestRebootDevice();
#endif
        },
        nullptr,
        MessageView::ButtonTextColor::Green
        ),

#else
    std::string selectedMessage = std::string( "Can't change " ) + lowerString + std::string( " in DevMenu Application." );
    pView->AddMessage( selectedMessage );
    pView->AddButton( "Close" );
#endif // !defined( APPLICATION_BUILD )

    m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true );
    m_IsRebootDialogUnvisible = true; // TORIAEZU: DropDown の処理の兼ね合いでボタン追加時は前面に表示されない
    m_pRebootDialog = pView;
}

void RebootRequiredSetting::ForceMoveFocusToRebootDialog() NN_NOEXCEPT
{
    if ( true == m_IsRebootDialogUnvisible )
    {
        NN_ABORT_UNLESS( nullptr != m_pRebootDialog );
        m_pParentPage->GetRootSurfaceContext()->DisplayModal( m_pRebootDialog );
        m_IsRebootDialogUnvisible = false;
    }
}

void RebootRequiredSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pDropDown, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pDropDown, pNextView );
}

glv::View* RebootRequiredSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

glv::View* RebootRequiredSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

/**************************************
 * class TimeZoneSetting
 **************************************/
TimeZoneSetting::TimeZoneSetting( glv::space_t width, Page* pParentPage ) NN_NOEXCEPT
    : m_pParentPage( pParentPage )
    , m_pItemGroup( nullptr )
    , m_pEditButton( nullptr )
    , m_pTimeZoneLabel( nullptr )
{
    auto pEditButton = new Button(
        "Edit",
        [&] { m_pParentPage->SwitchScene( SceneType_TimeZoneSelection, false ); },
        glv::Rect( 150.0f, 35.0f ),
        glv::Place::TR );
    pEditButton->anchor( glv::Place::TR ).pos( -( pEditButton->w ), 0.0f );

    m_pItemGroup = new glv::Group( glv::Rect( width, pEditButton->h ) );
    *m_pItemGroup << pEditButton;
    *this << m_pItemGroup;

    this->RefreshCurrentTimeZoneLabel();

    arrange().fit( false );

    m_pEditButton = pEditButton;
}

void TimeZoneSetting::RefreshCurrentTimeZoneLabel() NN_NOEXCEPT
{
    if ( nullptr != m_pTimeZoneLabel )
    {
        m_pTimeZoneLabel->remove();
        delete m_pTimeZoneLabel;
    }

    nn::time::LocationName locationName;
    nn::time::GetDeviceLocationName( &locationName );

    std::string title( "Time Zone ( " );
    title += devmenuUtil::GetTimeZoneDisplayString( locationName );
    title += " )";
    m_pTimeZoneLabel = new glv::Label( title, DefaultLabelSpec );

    *m_pItemGroup << m_pTimeZoneLabel;
}

void TimeZoneSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pEditButton, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pEditButton, pNextView );
}

glv::View* TimeZoneSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pEditButton;
}

glv::View* TimeZoneSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pEditButton;
}

/**************************************
 * class DateTimeSetting
 **************************************/
DateTimeSetting::DateTimeSetting( glv::space_t width, Page* pParentPage ) NN_NOEXCEPT
    : m_pParentPage( pParentPage )
    , m_pYearLabel( nullptr )
    , m_pMonthLabel( nullptr )
    , m_pDayLabel( nullptr )
    , m_pHourLabel( nullptr )
    , m_pMinuteLabel( nullptr )
    , m_pYearUp( nullptr )
    , m_pMonthUp( nullptr )
    , m_pDayUp( nullptr )
    , m_pHourUp( nullptr )
    , m_pMinuteUp( nullptr )
    , m_pYearDown( nullptr )
    , m_pMonthDown( nullptr )
    , m_pDayDown( nullptr )
    , m_pHourDown( nullptr )
    , m_pMinuteDown( nullptr )
    , m_pAutomaticCorrectionCheckBox( nullptr )
    , m_DisplayedDateClock( 0LL )
{
    auto pSetterTable = new glv::Table(
        "x . x . x . x . x,"
        "x x x x x x x x x,"
        "x . x . x . x . x" );
    {
        const glv::Rect defaultRect( 55.0f );
        const glv::Label::Spec iconLabelSpec( glv::Place::CC, 0.0f, 0.0f, 40.0f );

        auto pYearUp   = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );
        auto pMonthUp  = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );
        auto pDayUp    = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );
        auto pHourUp   = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );
        auto pMinuteUp = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );

        auto pYearLabel   = new glv::Label( "2017", CommonValue::DefaultLabelSpec );
        auto pYearSpacer  = new glv::Label( "/",    CommonValue::DefaultLabelSpec );
        auto pMonthLabel  = new glv::Label( "01",   CommonValue::DefaultLabelSpec );
        auto pMonthSpacer = new glv::Label( "/",    CommonValue::DefaultLabelSpec );
        auto pDayLabel    = new glv::Label( "01",   CommonValue::DefaultLabelSpec );
        auto pDaySpacer   = new glv::Label( " ",    CommonValue::DefaultLabelSpec );
        auto pHourLabel   = new glv::Label( "00",   CommonValue::DefaultLabelSpec );
        auto pHourSpacer  = new glv::Label( ":",    CommonValue::DefaultLabelSpec );
        auto pMinuteLabel = new glv::Label( "00",   CommonValue::DefaultLabelSpec );

        auto pYearDown   = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );
        auto pMonthDown  = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );
        auto pDayDown    = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );
        auto pHourDown   = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );
        auto pMinuteDown = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );

        *pSetterTable
            << pYearUp << pMonthUp << pDayUp << pHourUp << pMinuteUp
            << pYearLabel << pYearSpacer << pMonthLabel << pMonthSpacer << pDayLabel << pDaySpacer << pHourLabel << pHourSpacer << pMinuteLabel
            << pYearDown << pMonthDown << pDayDown << pHourDown << pMinuteDown;
        pSetterTable->arrange().fit( false );

        m_pYearLabel = pYearLabel;
        m_pMonthLabel = pMonthLabel;
        m_pDayLabel = pDayLabel;
        m_pHourLabel = pHourLabel;
        m_pMinuteLabel = pMinuteLabel;

        m_pYearUp = pYearUp;
        m_pMonthUp = pMonthUp;
        m_pDayUp = pDayUp;
        m_pHourUp = pHourUp;
        m_pMinuteUp = pMinuteUp;

        m_pYearDown = pYearDown;
        m_pMonthDown = pMonthDown;
        m_pDayDown = pDayDown;
        m_pHourDown = pHourDown;
        m_pMinuteDown = pMinuteDown;

        this->RegisterUpdateTimeFunction();
        this->UpdateTime();
    }

    // TORIAEZU: KeepWithinParent は初回フレームは効かないので height を調整して親の表示範囲内に収める
    pSetterTable->anchor( glv::Place::CR ).pos( -pSetterTable->width(), -74.0f );
    auto pLabel = new glv::Label( "Date and Time", DefaultLabelSpec );
    auto pSpacer = new Spacer( 65.0f, 0.0f );
    pSetterTable->enable( glv::Property::KeepWithinParent );

    auto pAutomaticCorrectionCheckboxTable = new glv::Table( "xx" );
    {
        auto pCheckbox = new CheckBoxButton( "Automatic Correction" );
        pCheckbox->SetValue( nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled() );
        *pAutomaticCorrectionCheckboxTable << pSpacer << pCheckbox;
        pAutomaticCorrectionCheckboxTable->arrange();
        pAutomaticCorrectionCheckboxTable->anchor( glv::Place::TL ).pos( 0.0f, pLabel->bottom() );
        pCheckbox->SetCallback( []( const glv::Notification& notification )->void { notification.receiver< DateTimeSetting >()->OnCheckBoxPressed( notification ); }, this );

        m_pAutomaticCorrectionCheckBox = pCheckbox;

        this->UpdateDateUpDownButtonsDisplay( nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled() );
    }

    auto pGroup = new glv::Group( glv::Rect( width, pSetterTable->h ) );
    *pGroup << pLabel << pSetterTable << pAutomaticCorrectionCheckboxTable;

    *this << pGroup;
    arrange().fit( false );
}

void DateTimeSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pYearUp    , pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pMonthUp   , pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pDayUp     , pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pHourUp    , pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pMinuteUp  , pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pYearUp    , m_pMonthUp );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pMonthUp   , m_pDayUp );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pDayUp     , m_pHourUp);
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pHourUp    , m_pMinuteUp );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pMinuteUp  , m_pHourUp );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pHourUp    , m_pDayUp );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pDayUp     , m_pMonthUp );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pMonthUp   , m_pYearUp );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pYearDown  , m_pMonthDown );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pMonthDown , m_pDayDown );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pDayDown   , m_pHourDown );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pHourDown  , m_pMinuteDown );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pMinuteDown, m_pHourDown );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pHourDown  , m_pDayDown );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pDayDown   , m_pMonthDown );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pMonthDown , m_pYearDown );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pYearUp    , m_pYearDown );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pMonthUp   , m_pMonthDown );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pDayUp     , m_pDayDown );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pHourUp    , m_pHourDown );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pMinuteUp  , m_pMinuteDown );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pYearDown  , m_pYearUp );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pMonthDown , m_pMonthUp );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pDayDown   , m_pDayUp );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pHourDown  , m_pHourUp );
    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pMinuteDown, m_pMinuteUp );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pYearDown  , pNextView );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pMonthDown , pNextView );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pDayDown   , pNextView );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pHourDown  , pNextView );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pMinuteDown, pNextView );

    pFocusManager->AddFocusSwitch< FocusButtonUp >   ( m_pAutomaticCorrectionCheckBox->GetButtonFocus(), pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown > ( m_pAutomaticCorrectionCheckBox->GetButtonFocus(), pNextView );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pAutomaticCorrectionCheckBox->GetButtonFocus(), m_pYearUp );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pYearUp    , m_pAutomaticCorrectionCheckBox->GetButtonFocus() );
    pFocusManager->AddFocusSwitch< FocusButtonLeft > ( m_pYearDown  , m_pAutomaticCorrectionCheckBox->GetButtonFocus() );
}

glv::View* DateTimeSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pYearUp;
}

glv::View* DateTimeSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pYearDown;
}

void DateTimeSetting::RegisterUpdateTimeFunction() NN_NOEXCEPT
{
    static auto IsUserClockOperationAcceptable = []() -> bool
    {
        // ユーザー時計の自動補正ONだったら、ユーザー操作で時計をいじれなくする
        return !nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled();
    };

    m_pYearUp->attach    ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 1 ); } }, glv::Update::Clicked, this );
    m_pMonthUp->attach   ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 0, 1 ); } }, glv::Update::Clicked, this );
    m_pDayUp->attach     ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 0, 0, 1 ); } }, glv::Update::Clicked, this );
    m_pHourUp->attach    ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 0, 0, 0, 1 ); } }, glv::Update::Clicked, this );
    m_pMinuteUp->attach  ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 0, 0, 0, 0, 1 ); } }, glv::Update::Clicked, this );

    m_pYearDown->attach  ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( -1 ); } }, glv::Update::Clicked, this );
    m_pMonthDown->attach ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 0, -1 ); } }, glv::Update::Clicked, this );
    m_pDayDown->attach   ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 0, 0, -1 ); } }, glv::Update::Clicked, this );
    m_pHourDown->attach  ( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 0, 0, 0, -1 ); } }, glv::Update::Clicked, this );
    m_pMinuteDown->attach( []( const glv::Notification& notification )->void { if ( IsUserClockOperationAcceptable() ) { notification.receiver< DateTimeSetting >()->UpdateTime( 0, 0, 0, 0, -1); } }, glv::Update::Clicked, this );
}

void DateTimeSetting::UpdateTime( int yearDelta, int monthDelta, int dayDelta, int hourDelta, int minuteDelta ) NN_NOEXCEPT
{
    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime( &posixTime ) );

    nn::time::CalendarTime carendarTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime( &carendarTime, nullptr, posixTime ) );

    carendarTime.year += yearDelta;
    carendarTime.month += monthDelta;
    carendarTime.day += dayDelta;
    carendarTime.hour += hourDelta;
    carendarTime.minute += minuteDelta;

    // 年補正
    if ( carendarTime.year <= 0 )
    {
        carendarTime.year = 1;
    }

    // 月補正
    if ( carendarTime.month <= 0 )
    {
        carendarTime.month = 12;
    }
    else if ( carendarTime.month > 12)
    {
        carendarTime.month = 1;
    }

    // 日補正
    const int maxDays = nn::time::GetDaysInMonth( carendarTime.year, carendarTime.month );
    if ( carendarTime.day <= 0 )
    {
        carendarTime.day = maxDays;
    }
    else if ( carendarTime.day > maxDays )
    {
        if ( dayDelta == 0 )
        {
            // day をいじってなければ、year や month をずらしたことによって、月の日数が減った場合
            carendarTime.day = maxDays;
        }
        else
        {
                // day をいじってここに来た場合は、月の最終日から + をしたので、初日に戻す
            carendarTime.day = 1;
        }
    }

    // 時刻補正
    if ( carendarTime.hour < 0 )
    {
        carendarTime.hour = 23;
    }
    carendarTime.hour %= 24;

    if ( carendarTime.minute < 0 )
    {
        carendarTime.minute = 59;
    }
    carendarTime.minute %= 60;

    if ( carendarTime.second < 0 )
    {
        carendarTime.second = 59;
    }
    carendarTime.second %= 60;

    // ユーザー操作で時計をいじったときにのみデバイスに時刻保存を行う
    const bool isSetToDeviceRequired = yearDelta != 0 || monthDelta != 0 || dayDelta != 0 || hourDelta != 0 || minuteDelta != 0;
    this->SetTime( carendarTime, isSetToDeviceRequired );
}

void DateTimeSetting::UpdateTimeOnLoop() NN_NOEXCEPT
{
    // CHECK: フォーカス中(編集中)のヘッダ更新停止の場合は、ここで停止/解除の切り替え。
    // 同フレーム中のヘッダの時計更新は、このメソッドの後で呼び出される。

    // ページ内、日時編集要素の時刻自動更新
    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime( &posixTime ) );

    if ( ( posixTime.value / 60 ) > m_DisplayedDateClock )
    {
        this->UpdateTime();    // execute at 1 minute after.
    }
}

void DateTimeSetting::SetTime( const nn::time::CalendarTime& carendarTime, bool isSetToDeviceRequired ) NN_NOEXCEPT
{
    int count;
    nn::time::PosixTime posixToSet;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToPosixTime( &count, &posixToSet, 1, carendarTime ) );

    // デバイスのタイムゾーンによっては count == 0 があり得る(ex.夏時間解除によるスキップする時間帯).
    // 本来は設定できない時刻であることを利用者に伝えたい.
    if ( count > 0 )
    {
        if ( isSetToDeviceRequired )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::SetStandardLocalSystemClockCurrentTime( posixToSet ) );
        }
        this->ScheduleRefreshDateTime( posixToSet.value );
    }

    m_pYearLabel->setValue( ToString( carendarTime.year, 4 ) );
    m_pMonthLabel->setValue( ToString( carendarTime.month, 2 ) );
    m_pDayLabel->setValue( ToString( carendarTime.day, 2 ) );
    m_pHourLabel->setValue( ToString( carendarTime.hour, 2 ) );
    m_pMinuteLabel->setValue( ToString( carendarTime.minute, 2 ) );
}

void DateTimeSetting::ScheduleRefreshDateTime( int64_t nowSecond ) NN_NOEXCEPT
{
    m_DisplayedDateClock = nowSecond / 60;   // convert to minutes.

    if ( nullptr != m_pParentPage )
    {
        auto pRootSurfaceContext = m_pParentPage->GetRootSurfaceContext();
        if ( nullptr != pRootSurfaceContext )
        {
            pRootSurfaceContext->RefreshDateTime();
        }
    }
}

void DateTimeSetting::UpdateDateUpDownButtonsDisplay( bool isEnabled ) NN_NOEXCEPT
{
    static_cast< IconLabel* >( m_pYearUp->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pMonthUp->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pDayUp->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pHourUp->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pMinuteUp->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pYearDown->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pMonthDown->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pDayDown->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pHourDown->child )->UpdateStyle( isEnabled );
    static_cast< IconLabel* >( m_pMinuteDown->child )->UpdateStyle( isEnabled );
}

void DateTimeSetting::OnCheckBoxPressed( const glv::Notification& notificaiton ) NN_NOEXCEPT
{
    NN_UNUSED( notificaiton );
    this->SetAutomaticCorrectionEnabledAndUpdateCheckBoxValue( m_pAutomaticCorrectionCheckBox->GetValue() );
}

void DateTimeSetting::SetAutomaticCorrectionEnabledAndUpdateCheckBoxValue( bool isEnabled ) NN_NOEXCEPT
{
    nn::time::SetStandardUserSystemClockAutomaticCorrectionEnabled( isEnabled );
    this->UpdateTime(); // ユーザー時計の表示が変わるので更新
    this->UpdateDateUpDownButtonsDisplay( !isEnabled );
}

void DateTimeSetting::Refresh() NN_NOEXCEPT
{
    m_DisplayedDateClock = 0LL;
    this->SetAutomaticCorrectionEnabledAndUpdateCheckBoxValue( nn::time::IsStandardUserSystemClockAutomaticCorrectionEnabled() );
}

/**************************************
 * class SystemSettings
 **************************************/
SystemSettings::SystemSettings( Page* pPage, glv::space_t width, ScrollableBoxView* pContainer ) NN_NOEXCEPT
    : SubsectionWithFocusUtility( pPage, "System Settings", width )
    , m_pLanguage( nullptr )
    , m_pRegion( nullptr )
    , m_pTimeZone( nullptr )
    , m_pDateTime( nullptr )
{
    m_pLanguage = new RebootRequiredSetting( "Language", this->GetItemWidth(), 320.0f, pPage,
        [ pContainer ] { pContainer->ScrollTo( 0.0f ); },
        GetDropDownItemsForLanguageSetting,
        GetCurrentLanguage,
        SetLanguage);
    m_pRegion = new RebootRequiredSetting( "Region", this->GetItemWidth(), 160.0f, pPage,
        nullptr,
        GetDropDownItemsForRegionSetting,
        GetCurrentRegion,
        SetRegion);
    m_pTimeZone = new TimeZoneSetting( this->GetItemWidth(), pPage );
    m_pDateTime = new DateTimeSetting( this->GetItemWidth(), pPage );

    *this << m_pLanguage << m_pRegion << m_pTimeZone << m_pDateTime;
    arrange().fit( false );
}

void SystemSettings::ForceMoveFocusToRebootDialog() NN_NOEXCEPT
{
    m_pLanguage->ForceMoveFocusToRebootDialog();
    m_pRegion->ForceMoveFocusToRebootDialog();
}

void SystemSettings::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousFocusItem, glv::View* pNextFocusItem ) const NN_NOEXCEPT
{
    m_pLanguage->SetFocusTransitionPath( pFocusManager, pPreviousFocusItem, m_pRegion->GetFirstFocusTargetView() );
    m_pRegion->SetFocusTransitionPath( pFocusManager, m_pLanguage->GetLastFocusTargetView(), m_pTimeZone->GetFirstFocusTargetView() );
    m_pTimeZone->SetFocusTransitionPath( pFocusManager, m_pRegion->GetLastFocusTargetView(), m_pDateTime->GetFirstFocusTargetView() );
    m_pDateTime->SetFocusTransitionPath( pFocusManager, m_pTimeZone->GetLastFocusTargetView(), pNextFocusItem );
}

glv::View* SystemSettings::GetFirstFocusTargetView() const NN_NOEXCEPT
{
    return m_pLanguage->GetFirstFocusTargetView();
}

glv::View* SystemSettings::GetLastFocusTargetView() const NN_NOEXCEPT
{
    return m_pTimeZone->GetLastFocusTargetView();
}

void SystemSettings::OnLoopBeforeSceneRenderer() NN_NOEXCEPT
{
    if ( nullptr != m_pDateTime )
    {
        m_pDateTime->UpdateTimeOnLoop();
    }
    this->ForceMoveFocusToRebootDialog();
}

void SystemSettings::OnLoopAfterSceneRenderer() NN_NOEXCEPT
{
    // Do Nothing
}

void SystemSettings::Refresh() NN_NOEXCEPT
{
    m_pDateTime->Refresh();
    m_pTimeZone->RefreshCurrentTimeZoneLabel();
}

}} // ~namespace devmenu::devicesettings, ~namespace devmenu
