﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <vector>
#include <nn/idle/idle_SystemApi.h>

#include "../Common/DevMenu_CommonDropDown.h"
#include "../Common/DevMenu_CommonSettingsApi.h"
#include "DevMenu_DeviceSettingsSleep.h"

namespace devmenu { namespace devicesettings {

/**************************************
 * class SleepModeSetting
 **************************************/

SleepModeSetting::SleepModeSetting( glv::space_t width ) NN_NOEXCEPT
{
    auto pLabel = new glv::Label( "Sleep Mode", DefaultLabelSpec );
    auto pDropDown = new DropDownComparableKeyValue< bool >(
        DefaultDropDownRect, CommonValue::InitialFontSize,
        [&]( void* pOutValues, DropDownBase* pDropDown ) { this->RegisterDropDownItems( pOutValues, pDropDown ); },
        [&]( void* pOutValue ) { this->GetSettingsValue( pOutValue ); },
        [&]( const bool& selected, bool current ) { this->SetSettingsValue( selected, current ); }
    );
    pDropDown->anchor( glv::Place::TR ).pos( -pDropDown->width(), 0.0f );
    pDropDown->enable( glv::Property::KeepWithinParent );

    auto pGroup = new glv::Group( glv::Rect( width, pLabel->h > pDropDown->h ? pLabel->h : pDropDown->h ) );
    *pGroup << pLabel << pDropDown;
    *this << pGroup;
    arrange().fit( false );

    m_pDropDown = pDropDown;
}

void SleepModeSetting::Refresh() NN_NOEXCEPT
{
    m_pDropDown->Refresh();
}

void SleepModeSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >  ( m_pDropDown, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pDropDown, pNextView );
}

glv::View* SleepModeSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

glv::View* SleepModeSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

void SleepModeSetting::GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
{
    auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
    *pIsEnabled = true;
    GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, SleepModeSetting::SettingName, SleepModeSetting::KeyName );
}

void SleepModeSetting::RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
{
    const valuePair valueList[] = {
        valuePair( true , "Default" ),
        valuePair( false ,  "Pseudo" ),
    };

    auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

    for ( const auto& iter: valueList )
    {
        pVectorValues->push_back( iter );
        pDropDown->addItem( iter.second );
     }
}

    void SleepModeSetting::SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( SleepModeSetting::SettingName, SleepModeSetting::KeyName, selected );
    }

const char* SleepModeSetting::SettingName = "systemsleep";
const char* SleepModeSetting::KeyName = "enter_sleep";

/**************************************
 * class IdleTimeDetectionDropDown
 **************************************/

IdleTimeDetectionSetting::IdleTimeDetectionSetting( glv::space_t width ) NN_NOEXCEPT
{
    // 一定時間無操作で AutoSleep する設定(無操作状態検出モード設定)
    auto pTitleLabel = new glv::Label( "Idle Time Detection", DefaultLabelSpec );
    auto pDropDown = new DropDownComparableKeyValue< bool >(
        DefaultDropDownRect, CommonValue::InitialFontSize,
        [&]( void* pOutValues, DropDownBase* pDropDown ) { this->RegisterDropDownItems( pOutValues, pDropDown ); },
        [&]( void* pOutValue ) { this->GetSettingsValue( pOutValue ); },
        [&]( const bool& selected, bool current ) { this->SetSettingsValue( selected, current ); }
    );
    pDropDown->anchor( glv::Place::TR ).pos( -pDropDown->width(), 0.0f );
    pDropDown->enable( glv::Property::KeepWithinParent );

    m_pDropDown = pDropDown;

    auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
    *pGroup << pTitleLabel << pDropDown;
    *this << pGroup;
    arrange().fit( false );
}

void IdleTimeDetectionSetting::Refresh() NN_NOEXCEPT
{
    m_pDropDown->Refresh();
}

void IdleTimeDetectionSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >  ( m_pDropDown, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pDropDown, pNextView );
}

glv::View* IdleTimeDetectionSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

glv::View* IdleTimeDetectionSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

void IdleTimeDetectionSetting::GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
{
    auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
    *pIsEnabled = true;
    GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, IdleTimeDetectionSetting::SettingName, IdleTimeDetectionSetting::KeyName );
}

void IdleTimeDetectionSetting::RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
{
    const valuePair valueList[] = {
        valuePair( false , "On    " ),
        valuePair( true ,  "Off    " ),
    };

    auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

    for ( const auto& iter: valueList )
    {
        pVectorValues->push_back( iter );
        pDropDown->addItem( iter.second );
    }
}

void IdleTimeDetectionSetting::SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
{
    SetFixedSizeFirmwareDebugSettingsItemValue( IdleTimeDetectionSetting::SettingName, IdleTimeDetectionSetting::KeyName, selected );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // 即時反映
    nn::idle::InitializeForSystem();
    nn::idle::LoadAndApplySettings();
    nn::idle::FinalizeForSystem();
#endif
}

const char* IdleTimeDetectionSetting::SettingName = "systemsleep";
const char* IdleTimeDetectionSetting::KeyName = "disable_auto_sleep";


/**************************************
 * class SleepSettings
 **************************************/

SleepSettings::SleepSettings( Page* pPage, glv::space_t width ) NN_NOEXCEPT
    : SubsectionWithFocusUtility( pPage, "Sleep Settings", width )
{
    m_pSleepMode = new SleepModeSetting( this->GetItemWidth() );
    m_pIdleTimeDetection = new IdleTimeDetectionSetting( this->GetItemWidth() );

    *this << m_pSleepMode << m_pIdleTimeDetection;
    arrange().fit( false );
}

void SleepSettings::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousFocusItem, glv::View* pNextFocusItem ) const NN_NOEXCEPT
{
    m_pSleepMode->SetFocusTransitionPath( pFocusManager, pPreviousFocusItem, m_pIdleTimeDetection->GetFirstFocusTargetView() );
    m_pIdleTimeDetection->SetFocusTransitionPath( pFocusManager, m_pSleepMode->GetLastFocusTargetView(), nullptr );
}

glv::View* SleepSettings::GetFirstFocusTargetView() const NN_NOEXCEPT
{
    return m_pSleepMode->GetFirstFocusTargetView();
}

glv::View* SleepSettings::GetLastFocusTargetView() const NN_NOEXCEPT
{
    return m_pIdleTimeDetection->GetLastFocusTargetView();
}

void SleepSettings::OnLoopBeforeSceneRenderer() NN_NOEXCEPT
{
    // Do Nothing
}

void SleepSettings::OnLoopAfterSceneRenderer() NN_NOEXCEPT
{
    // Do Nothing
}

void SleepSettings::Refresh() NN_NOEXCEPT
{
    m_pSleepMode->Refresh();
    m_pIdleTimeDetection->Refresh();
}

}} // ~namespace devmenu::devicesettings, ~namespace devmenu
