﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <vector>
#include <nn/nn_Abort.h>
#include <nn/lbl/lbl.h>
#include <nn/mem.h>
#include <nn/ae/ae_SystemAppletApi.h>
#include <nn/ae/ae_DisplayResolutionApi.h>

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    #include <nn/vi.private.h>
#endif

#include "../Common/DevMenu_CommonDropDown.h"
#include "../Common/DevMenu_CommonIconButton.h"
#include "../Common/DevMenu_CommonIconLabel.h"
#include "DevMenu_DeviceSettingsDisplay.h"

namespace devmenu { namespace devicesettings {

/**************************************
 * class BrightnessSetting
 **************************************/
BrightnessSetting::BrightnessSetting( glv::space_t width ) NN_NOEXCEPT
{
    auto pSliderLabel = new glv::Label( "Screen Brightness", DefaultLabelSpec );
    auto pSlider = new glv::Slider( glv::Rect( 600.0f, CommonValue::InitialFontSize ) );
    pSlider->anchor( glv::Place::TR ).pos( -pSlider->width(), 0.0f );
    pSlider->enable( glv::Property::KeepWithinParent );
    auto pSliderGroup = new glv::Group( glv::Rect( width, pSliderLabel->h ) );
    *pSliderGroup << pSliderLabel << pSlider;
    auto pSpacer = new Spacer( 65.0f, 0.0f );

    auto pCheckboxTable = new glv::Table( "xx" );
    m_pCheckbox = new CheckBoxButton( "Auto Brightness Control" );
    *pCheckboxTable << pSpacer << m_pCheckbox;
    pCheckboxTable->arrange();

    pSlider->attach( []( const glv::Notification& notification )->void { notification.receiver< BrightnessSetting >()->Update(); }, glv::Update::Value, this );
    m_pCheckbox->SetCallback( []( const glv::Notification& notification )->void { notification.receiver< BrightnessSetting >()->UpdateAutoBrightness( notification ); }, this );

    *this << pSliderGroup << pCheckboxTable;
    arrange().fit( false );

    m_pSlider = pSlider;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::lbl::LoadCurrentSetting();
    m_pSlider->setValue( nn::lbl::GetCurrentBrightnessSetting() );
    m_pCheckbox->SetValue( nn::lbl::IsAutoBrightnessControlEnabled() );
#endif
}

void BrightnessSetting::Update() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    // lbl::DisableAutoBrightness時に SetCurrentBrightnessSettingで設定した値に復帰してくれる模様なので、
    // Auto状態に依存せず常にスライダ値を設定。
    nn::lbl::SetCurrentBrightnessSetting( m_pSlider->getValue() );
    nn::lbl::SaveCurrentSetting();
#endif
}

void BrightnessSetting::UpdateAutoBrightness( bool isChecked ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    if ( isChecked )
    {
        nn::lbl::EnableAutoBrightnessControl();
    }
    else
    {
        nn::lbl::DisableAutoBrightnessControl();
    }
    // lbl::DisableAutoBrightness時に SetCurrentBrightnessSettingで設定した値に復帰してくれる模様。
    nn::lbl::SaveCurrentSetting();
#else
    NN_UNUSED( isChecked );
#endif
}

void BrightnessSetting::UpdateAutoBrightness( const glv::Notification& notification ) NN_NOEXCEPT
{
    NN_UNUSED( notification );
    UpdateAutoBrightness( m_pCheckbox->GetValue() );
}

void BrightnessSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >  ( m_pSlider, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pSlider, m_pCheckbox->GetButtonFocus() );
    pFocusManager->AddFocusSwitch< FocusButtonUp >  ( m_pCheckbox->GetButtonFocus(), m_pSlider );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pCheckbox->GetButtonFocus(), pNextView );
}

glv::View* BrightnessSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pSlider;
}

glv::View* BrightnessSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pCheckbox->GetButtonFocus();
}

/**************************************
 * class DisplayResolutionDropDown
 **************************************/
DisplayResolutionDropDown::DisplayResolutionDropDown( const glv::Rect& rect, float textSize ) NN_NOEXCEPT
    : DropDownBase( rect, textSize )
{
    attach( []( const glv::Notification& notification )->void { notification.receiver< DisplayResolutionDropDown >()->UpdateDisplayResolution(); }, glv::Update::Action, this );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::vi::OpenDisplay( &m_pExternalDisplay, "External" );

    if ( IsExternalDisplayConnected() )
    {
        InitializeExternalDropDownList();
    }
    else
#endif
    {
        InitializeDefaultDropDownList();
    }
}

void DisplayResolutionDropDown::UpdateDisplayResolution() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    if ( m_IsDisplayConnected )
    {
        nn::settings::system::TvSettings currentTvSettings;
        nn::settings::system::GetTvSettings( &currentTvSettings );

        currentTvSettings.tvResolution = m_ExternalModes[ mSelectedItem ].first;

        nn::settings::system::SetTvSettings( currentTvSettings );

        if ( nn::vi::SetDisplayMode( m_pExternalDisplay, &( m_ExternalModes[ mSelectedItem ].second ) ).IsFailure() )
        {
            // ToDo: ModalView で表示する
            DEVMENU_LOG( "[DevMenu] Failed to change display mode\n" );
        }
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
        nn::ae::UpdateDefaultDisplayResolution();
#endif
    }
#endif
}

// Checks for a change in the HDMI state and updates the drop down list
void DisplayResolutionDropDown::UpdateDisplayModes() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )

    if ( IsExternalDisplayConnected() )
    {
        if ( !m_IsDisplayConnected )
        {
            InitializeExternalDropDownList();
        }
    }
    else if ( m_IsDisplayConnected )
    {
        InitializeDefaultDropDownList();
    }
#endif
}

// Sets up the drop down list for the external display
void DisplayResolutionDropDown::InitializeExternalDropDownList() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    mItems.clear();
    m_ExternalModes.clear();

    nn::vi::DisplayModeInfo modes[ nn::vi::DisplayModeCountMax ];
    const int modeCount = nn::vi::ListDisplayModes( modes, sizeof( modes ) / sizeof( modes[ 0 ] ), m_pExternalDisplay );
    int maxResolutionIndex = 0;

    if ( modeCount == 0 )
    {
        // No modes were found, so assume display is disconnected.
        return;
    }
    else
    {
        m_IsDisplayConnected = true;

        int maxWidth = 0;

        for ( int i = 0; i < modeCount; ++i )
        {
            m_ExternalModes.push_back( std::pair< nn::settings::system::TvResolution, nn::vi::DisplayModeInfo >( GetTvResolutionFromDisplayMode( modes[ i ] ), modes[ i ] ) );

            if ( modes[ i ].width > maxWidth )
            {
                maxWidth = modes[ i ].width;
                maxResolutionIndex = i;
            }
        }
    }

    m_ExternalModes.push_back( std::pair< nn::settings::system::TvResolution, nn::vi::DisplayModeInfo >( nn::settings::system::TvResolution_Auto, modes[ maxResolutionIndex ] ) );

    for ( unsigned i = 0; i < m_ExternalModes.size(); ++i )
    {
        addItem( GetResolutionLabelString( m_ExternalModes[ i ].first ) );
    }

    nn::settings::system::TvSettings currentTvSettings;
    nn::settings::system::GetTvSettings( &currentTvSettings );
    setValue( GetResolutionLabelString( static_cast< nn::settings::system::TvResolution >( currentTvSettings.tvResolution ) ) );
#endif
}

// Sets up the drop down list for the native device display
void DisplayResolutionDropDown::InitializeDefaultDropDownList() NN_NOEXCEPT
{
    mItems.clear();
    m_IsDisplayConnected = false;
    addItem( "Auto" );
}

const char* DisplayResolutionDropDown::GetResolutionLabelString( nn::settings::system::TvResolution resolution ) NN_NOEXCEPT
{
    switch ( resolution )
    {
    case nn::settings::system::TvResolution::TvResolution_Auto:
        return "Auto";
    case nn::settings::system::TvResolution::TvResolution_1080p:
        return "1080p";
    case nn::settings::system::TvResolution::TvResolution_720p:
        return "720p";
    case nn::settings::system::TvResolution::TvResolution_480p:
        return "480p";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
nn::settings::system::TvResolution DisplayResolutionDropDown::GetTvResolutionFromDisplayMode( const nn::vi::DisplayModeInfo& mode ) NN_NOEXCEPT
{
     if ( mode.height == 1080 )
     {
        return nn::settings::system::TvResolution_1080p;
     }
     else if ( mode.height == 720 )
     {
        return nn::settings::system::TvResolution_720p;
     }
     else if ( mode.height == 480 )
     {
        return nn::settings::system::TvResolution_480p;
     }

     NN_ABORT( "Unknown Resolution Setting" );
     return nn::settings::system::TvResolution_Auto;
}
#endif


/**************************************
 * class DisplaySetting
 **************************************/
DisplaySetting::DisplaySetting( glv::space_t width ) NN_NOEXCEPT
{
    auto pLabel = new glv::Label( "Screen Resolution", DefaultLabelSpec );
    auto pDropDown = new DisplayResolutionDropDown( glv::Rect( 160.0f, 35.0f ), CommonValue::InitialFontSize );
    pDropDown->anchor( glv::Place::TR ).pos( -pDropDown->width(), 0.0f );
    pDropDown->enable( glv::Property::KeepWithinParent );

    auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
    *pGroup << pLabel << pDropDown;

    *this << pGroup;
    arrange().fit( false );

    m_pDropDown = pDropDown;
}

void DisplaySetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp > ( m_pDropDown, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pDropDown, pNextView );
}

glv::View* DisplaySetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

glv::View* DisplaySetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

void DisplaySetting::UpdateDropDownList() NN_NOEXCEPT
{
    m_pDropDown->UpdateDisplayModes();
}


/**************************************
 * class CecEnabledSetting
 **************************************/
CecEnabledSetting::CecEnabledDropDown::CecEnabledDropDown( const glv::Rect& rect, float textSize ) NN_NOEXCEPT
    : DropDownBase( rect, textSize )
{
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    addItem( "On    " );
    addItem( "Off    " );
    attach( []( const glv::Notification& notification )->void { notification.receiver< CecEnabledDropDown >()->UpdateSettingsValue(); }, glv::Update::Action, this );
#else
    // DeMenuApp では設定変更を行えないため、現在値を表示するのみとする
    setValue( GetSettingsValue() ? "On    " : "Off    " );
#endif
}

void CecEnabledSetting::CecEnabledDropDown::Refresh() NN_NOEXCEPT
{
    setValue( GetSettingsValue() ? "On    " : "Off    " );
}


bool CecEnabledSetting::CecEnabledDropDown::GetSettingsValue() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::settings::system::TvSettings settings;
    nn::settings::system::GetTvSettings( &settings );
    return settings.flags.Test< nn::settings::system::TvFlag::AllowsCec >();
#else
    return false;
#endif
}

void CecEnabledSetting::CecEnabledDropDown::SetSettingsValue( bool value ) NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    nn::settings::system::TvSettings currentTvSettings;
    nn::settings::system::GetTvSettings( &currentTvSettings );
    currentTvSettings.flags.Set< nn::settings::system::TvFlag::AllowsCec >( value );
    nn::settings::system::SetTvSettings( currentTvSettings );
    nn::ae::NotifyCecSettingsChanged();
#endif
}

void CecEnabledSetting::CecEnabledDropDown::UpdateSettingsValue() NN_NOEXCEPT
{
    const auto isEnabled = ( mSelectedItem == 0 );
    const auto current = GetSettingsValue();
    if ( current != isEnabled )
    {
        SetSettingsValue( isEnabled );
    }
}

CecEnabledSetting::CecEnabledSetting( glv::space_t width ) NN_NOEXCEPT
    : m_pDropDown( nullptr )
{
    auto pTitleLabel = new glv::Label( "Consumer Electronics Control (CEC)", DefaultLabelSpec );
    auto pDropDown = new CecEnabledDropDown( glv::Rect( 160.0f, 35.0f ), CommonValue::InitialFontSize );
    {
        pDropDown->anchor( glv::Place::TR ).pos( -pDropDown->width(), 0.0f );
        pDropDown->enable( glv::Property::KeepWithinParent );
    }

    m_pDropDown = pDropDown;

    auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
    *pGroup << pTitleLabel << pDropDown;
    *this << pGroup;
    arrange().fit( false );
}

void CecEnabledSetting::Refresh() NN_NOEXCEPT
{
    m_pDropDown->Refresh();
}

void CecEnabledSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >  ( m_pDropDown, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pDropDown, pNextView );
}

glv::View* CecEnabledSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

glv::View* CecEnabledSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pDropDown;
}

/**************************************
 * class UnderscanSetting
**************************************/
UnderscanSetting::UnderscanSetting( glv::space_t width ) NN_NOEXCEPT
    : m_pUnderscanLabel( nullptr )
    , m_pUnderscanUp( nullptr )
    , m_pUnderscanDown( nullptr )
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    , m_pExternalDisplay( nullptr )
#endif
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::vi::OpenDisplay(&m_pExternalDisplay, "External");
#endif

    auto pSetterTable = new glv::Table( "x . x . x" );
    {
        const glv::Rect defaultRect( 55.0f );
        const glv::Label::Spec iconLabelSpec( glv::Place::CC, 0.0f, 0.0f, 40.0f );

        nn::settings::system::TvSettings currentTvSettings;
        nn::settings::system::GetTvSettings( &currentTvSettings );

        *pSetterTable
            << ( m_pUnderscanLabel = new glv::Label( ToString( currentTvSettings.tvUnderscan, 2 ), CommonValue::DefaultLabelSpec ) )
            << ( m_pUnderscanUp = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) ) )
            << ( m_pUnderscanDown = new IconButton( defaultRect, false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) ) );

        pSetterTable->arrange().fit( false );
        pSetterTable->enable( glv::Property::KeepWithinParent );
        pSetterTable->anchor( glv::Place::CR ).pos( -pSetterTable->width(), -27.0f );

        this->RegisterUpdateUnderscanFunction();
    }

    auto pGroup = new glv::Group( glv::Rect( width, pSetterTable->h ) );
    *pGroup
        << new glv::Label( "TV Underscan Settings", glv::Label::Spec( glv::Place::TL, 5.0f, 15.0f, CommonValue::InitialFontSize ) )
        << pSetterTable;

    *this << pGroup;
    arrange().fit( false );
}

void UnderscanSetting::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousView, glv::View* pNextView ) NN_NOEXCEPT
{
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pUnderscanUp, pPreviousView );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pUnderscanUp, m_pUnderscanDown );
    pFocusManager->AddFocusSwitch< FocusButtonRight >( m_pUnderscanUp, m_pUnderscanDown );
    pFocusManager->AddFocusSwitch< FocusButtonUp >( m_pUnderscanDown, m_pUnderscanUp );
    pFocusManager->AddFocusSwitch< FocusButtonDown >( m_pUnderscanDown, pNextView );
    pFocusManager->AddFocusSwitch< FocusButtonLeft >( m_pUnderscanDown, m_pUnderscanUp );
}

glv::View* UnderscanSetting::GetFirstFocusTargetView() NN_NOEXCEPT
{
    return m_pUnderscanUp;
}

glv::View* UnderscanSetting::GetLastFocusTargetView() NN_NOEXCEPT
{
    return m_pUnderscanDown;
}

void UnderscanSetting::UpdateUnderscan( int underscanDelta ) NN_NOEXCEPT
{
    nn::settings::system::TvSettings currentTvSettings;
    nn::settings::system::GetTvSettings( &currentTvSettings );
    const auto currentUnderscan = static_cast< int >( currentTvSettings.tvUnderscan );
    auto nextUnderscan = currentUnderscan + underscanDelta;

    // 補正
    if ( nextUnderscan < UnderscanSetting::UnderscanValueMin )
    {
        nextUnderscan = UnderscanSetting::UnderscanValueMin;
    }
    else if ( nextUnderscan > UnderscanSetting::UnderscanValueMax )
    {
        nextUnderscan = UnderscanSetting::UnderscanValueMax;
    }

    if ( currentUnderscan == nextUnderscan )
    {
        // アンダースキャン値に変更がなければ終了
        return;
    }

    // アンダースキャン値の表示を更新
    m_pUnderscanLabel->setValue( ToString( nextUnderscan, 2 ) );

    // 実機内部の値を更新
    currentTvSettings.tvUnderscan = nextUnderscan;
    nn::settings::system::SetTvSettings( currentTvSettings );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    if ( nullptr != m_pExternalDisplay )
    {
        // 画面へ即時反映
        nn::vi::SetDisplayUnderscan( m_pExternalDisplay, nextUnderscan );
    }
#endif
}

void UnderscanSetting::RegisterUpdateUnderscanFunction() NN_NOEXCEPT
{
    m_pUnderscanUp->attach([](const glv::Notification& notification)->void { notification.receiver< UnderscanSetting >()->UpdateUnderscan(1); }, glv::Update::Clicked, this);
    m_pUnderscanDown->attach([](const glv::Notification& notification)->void { notification.receiver< UnderscanSetting >()->UpdateUnderscan(-1); }, glv::Update::Clicked, this);
}

/**************************************
 * class DisplaySettings
 **************************************/

DisplaySettings::DisplaySettings( Page* pPage, glv::space_t width ) NN_NOEXCEPT
    : SubsectionWithFocusUtility( pPage, "Display Settings", width )
{
    m_pBrightness = new BrightnessSetting( this->GetItemWidth() );
    m_pDisplay = new DisplaySetting( this->GetItemWidth() );
    m_pCec = new CecEnabledSetting( this->GetItemWidth() );
    m_pUnderscan = new UnderscanSetting( this->GetItemWidth() );

    *this << m_pBrightness << m_pDisplay << m_pCec << m_pUnderscan;
    arrange().fit( false );
}

void DisplaySettings::SetFocusTransitionPath( FocusManager* pFocusManager, glv::View* pPreviousFocusItem, glv::View* pNextFocusItem ) const NN_NOEXCEPT
{
    m_pBrightness->SetFocusTransitionPath( pFocusManager, pPreviousFocusItem, m_pDisplay->GetFirstFocusTargetView() );
    m_pDisplay->SetFocusTransitionPath( pFocusManager, m_pBrightness->GetLastFocusTargetView(), m_pCec->GetFirstFocusTargetView() );
    m_pCec->SetFocusTransitionPath( pFocusManager, m_pDisplay->GetLastFocusTargetView(), m_pUnderscan->GetFirstFocusTargetView() );
    m_pUnderscan->SetFocusTransitionPath( pFocusManager, m_pCec->GetLastFocusTargetView(), pNextFocusItem );
}

glv::View* DisplaySettings::GetFirstFocusTargetView() const NN_NOEXCEPT
{
    return m_pBrightness->GetFirstFocusTargetView();
}

glv::View* DisplaySettings::GetLastFocusTargetView() const NN_NOEXCEPT
{
    return m_pUnderscan->GetLastFocusTargetView();
}

void DisplaySettings::OnLoopBeforeSceneRenderer() NN_NOEXCEPT
{
    m_pDisplay->UpdateDropDownList();
}

void DisplaySettings::OnLoopAfterSceneRenderer() NN_NOEXCEPT
{
    // Do Nothing
}

void DisplaySettings::Refresh() NN_NOEXCEPT
{
    m_pCec->Refresh();
}

}} // ~namespace devmenu::devicesettings, ~namespace devmenu
