﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Optional.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/ntc/shim/ntc_Shim.h>
#include <nn/time.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_StandardNetworkSystemClockPrivilegeApi.h>
#include <nn/settings/system/settings_Clock.h>

#include <algorithm>
#include <memory>
#include <functional>
#include <sstream>
#include <iomanip>
#include <new>

#include "Common/DevMenu_CommonCheckBox.h"
#include "Common/DevMenu_CommonIconButton.h"
#include "Common/DevMenu_CommonIconLabel.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "DevMenu_Common.h"
#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"


//#define DEBUG_DATA

namespace devmenu { namespace timetest {

class Spacer
    : public glv::View
{
public:
    Spacer(glv::space_t width, glv::space_t height) NN_NOEXCEPT
        : glv::View(glv::Rect(width, height))
    {
        disable(glv::Property::Controllable | glv::Property::DrawBorder | glv::Property::DrawBack | glv::Property::HitTest);
        enable(glv::Property::AlwaysBubble);
    }
};

class Button
    : public glv::Button
{
private:
    glv::Label m_Label;
    std::function<void()> m_Callback;

public:
    Button(const glv::WideString& label, std::function<void()> callback, const glv::Rect& rect, glv::Place::t anchor = glv::Place::TL) NN_NOEXCEPT
        : glv::Button(rect, true)
        , m_Label(label, glv::Label::Spec(glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize))
        , m_Callback(callback)
    {
        this->anchor(anchor);
        *this << m_Label;
        changePadClickDetectableButtons(glv::BasicPadEventType::Button::Ok::Mask);
        changePadClickDetectableButtons(glv::DebugPadEventType::Button::Ok::Mask);
        attach([](const glv::Notification& n)->void { n.receiver< Button >()->m_Callback(); }, glv::Update::Clicked, this);
    }
};

class SteadyClockSetting : public glv::Table
{
public:
    SteadyClockSetting() NN_NOEXCEPT
        : m_Caption("Steady clock settings", glv::Label::Spec(glv::Place::TL, 0.0f, 8.0f, CommonValue::InitialFontSize))
        , m_IsSystemResetRequried(false)
        , m_IsClockResetRequested(false)
        , m_CurrentOffsetLabel("", glv::Label::Spec(glv::Place::TL, 0.0f, 8.0f, CommonValue::InitialFontSize))
        , m_ResetButton(GLV_TEXT_API_WIDE_STRING("Reset Steady Clock"), [&] {OnClickResetButton(); }, glv::Rect(400, ButtonHeight), glv::Place::TL)
    {
        auto table = new glv::Table("<", 100.0f);

        // Reset button
        {
            *table << m_ResetButton;
        }

        // Initialize current offset value
        {
            *table << m_CurrentOffsetLabel;

            uint32_t currentOffsetMinutes;
            GetFixedSizeFirmwareDebugSettingsItemValue(&currentOffsetMinutes, "time", "standard_steady_clock_test_offset_minutes");
            m_TotalOffsetTimeSpan = nn::TimeSpan::FromMinutes(static_cast<int64_t>(currentOffsetMinutes));
        }

        // Add offset
        {
            auto pAddOffsetTable = new glv::Table(
                "xxxx,"
                "xxxx",
                10.f, 10.f);
            {
                const glv::space_t ButtonWidth = 250;
                auto add1Minute     = new Button(GLV_TEXT_API_WIDE_STRING("Add   1 Minutes"), [&] {AddOffset(nn::TimeSpan::FromMinutes(1)); }, glv::Rect(ButtonWidth, ButtonHeight), glv::Place::TL);
                auto add30Minute    = new Button(GLV_TEXT_API_WIDE_STRING("Add  30 Minutes"), [&] {AddOffset(nn::TimeSpan::FromMinutes(30)); }, glv::Rect(ButtonWidth, ButtonHeight), glv::Place::TL);
                auto add12Hour      = new Button(GLV_TEXT_API_WIDE_STRING("Add  12   Hours"), [&] {AddOffset(nn::TimeSpan::FromHours(12)); }, glv::Rect(ButtonWidth, ButtonHeight), glv::Place::TL);
                auto add7Day        = new Button(GLV_TEXT_API_WIDE_STRING("Add   7    Days"), [&] {AddOffset(nn::TimeSpan::FromDays(7)); }, glv::Rect(ButtonWidth, ButtonHeight), glv::Place::TL);
                auto add30Day       = new Button(GLV_TEXT_API_WIDE_STRING("Add  30    Days"), [&] {AddOffset(nn::TimeSpan::FromDays(30)); }, glv::Rect(ButtonWidth, ButtonHeight), glv::Place::TL);
                auto add365Day      = new Button(GLV_TEXT_API_WIDE_STRING("Add 365    Days"), [&] {AddOffset(nn::TimeSpan::FromDays(365)); }, glv::Rect(ButtonWidth, ButtonHeight), glv::Place::TL);

                *pAddOffsetTable << new Spacer(30.f, 0.f);
                *pAddOffsetTable << add1Minute;
                *pAddOffsetTable << add30Minute;
                *pAddOffsetTable << add12Hour;

                *pAddOffsetTable << new Spacer(30.f, 0.f);
                *pAddOffsetTable << add7Day;
                *pAddOffsetTable << add30Day;
                *pAddOffsetTable << add365Day;

                pAddOffsetTable->arrange();
            }

            *table << pAddOffsetTable;
        }


        table->arrange();

        *this << m_Caption << table;
        this->arrange();
    }

    void RefreshUi() NN_NOEXCEPT
    {
        std::stringstream ss;

        nn::TimeSpan tempTimeSpan = m_TotalOffsetTimeSpan;
        ss << "Total offset : ";
        if(tempTimeSpan.GetDays() > 0)
        {
            ss << tempTimeSpan.GetDays() << " days, ";
            tempTimeSpan -= nn::TimeSpan::FromDays( tempTimeSpan.GetDays() );
        }
        if(tempTimeSpan.GetHours() > 0)
        {
            ss << tempTimeSpan.GetHours() << " hours, ";
            tempTimeSpan -= nn::TimeSpan::FromHours( tempTimeSpan.GetHours() );
        }
        ss << tempTimeSpan.GetMinutes() << " minutes";

        if(m_IsClockResetRequested)
        {
            NN_FUNCTION_LOCAL_STATIC(glv::Style, s_RedStyle);
            s_RedStyle.color.set(glv::StyleColor::BlackOnWhite);
            s_RedStyle.color.fore.set(0.8, 0.8, 0.8);
            s_RedStyle.color.back.set(0.6, 0.0, 0.0);
            m_CurrentOffsetLabel.enable(glv::Property::DrawBack);
            m_CurrentOffsetLabel.style(&s_RedStyle);

            ss << " *** Reset was requested. Cannot operate.";
        }

        m_CurrentOffsetLabel.setValue(ss.str().c_str());
    }

    bool IsSystemResetRequried() const NN_NOEXCEPT
    {
        return m_IsSystemResetRequried;
    }

    bool IsResetRequested() const NN_NOEXCEPT
    {
        return m_IsClockResetRequested;
    }

private:
    void OnClickResetButton() NN_NOEXCEPT
    {
        if(m_IsClockResetRequested)
        {
            return;
        }

        nn::settings::system::SetExternalSteadyClockSourceId(nn::util::InvalidUuid);

        // テストオフセットも初期化
        SetFixedSizeFirmwareDebugSettingsItemValue("time", "standard_steady_clock_test_offset_minutes", static_cast<uint32_t>(0UL));
        m_TotalOffsetTimeSpan = nn::TimeSpan(0);

        m_IsSystemResetRequried = true;
        m_IsClockResetRequested = true;
    }

    void AddOffset(nn::TimeSpan value) NN_NOEXCEPT
    {
        NN_ASSERT(value > nn::TimeSpan(0));
        const nn::TimeSpan MaxOffset = nn::TimeSpan::FromDays(365 * 30); // ひとまず30年制限

        if(m_IsClockResetRequested)
        {
            // リセットしたのでもう追加できない
            return;
        }

        if(m_TotalOffsetTimeSpan + value > MaxOffset)
        {
            return;
        }

        m_TotalOffsetTimeSpan += value;
        SetFixedSizeFirmwareDebugSettingsItemValue("time", "standard_steady_clock_test_offset_minutes", static_cast<uint32_t>(m_TotalOffsetTimeSpan.GetMinutes()));

        m_IsSystemResetRequried = true;
    }

private:
    static const int ButtonHeight = 40;

    glv::Label m_Caption;
    bool m_IsSystemResetRequried;
    bool m_IsClockResetRequested;

    glv::Label m_CurrentOffsetLabel;
    nn::TimeSpan m_TotalOffsetTimeSpan;
    Button m_ResetButton;
};

class NetworkClockSetting : public glv::Table
{
private:
    class AutoEnsureEnabledCheckbox : public glv::Table
    {
    public:
        AutoEnsureEnabledCheckbox() NN_NOEXCEPT
            : glv::Table( "<", 0.0f )
            , m_pCheckBox( new CheckBoxButton( "Auto Clock Ensure Enabled" ) )
            , m_IsSystemResetRequried(false)
            , m_CurrentValue(false)
        {
            bool isEnabled = true;
            GetFixedSizeFirmwareDebugSettingsItemValue( &isEnabled, "ntc", "is_autonomic_correction_enabled" );
            m_pCheckBox->SetValue( isEnabled );
            m_pCheckBox->SetCallback( [](const glv::Notification& notification)->void { notification.receiver<AutoEnsureEnabledCheckbox>()->UpdateAutoEnsureEnabled( notification ); }, this );

            m_CurrentValue = isEnabled;

            *this << m_pCheckBox;

            this->arrange();
        }

        bool IsSystemResetRequried() const NN_NOEXCEPT
        {
            return m_IsSystemResetRequried;
        }

        void RefreshValue() NN_NOEXCEPT
        {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
            bool currentValue = true;
            GetFixedSizeFirmwareDebugSettingsItemValue( &currentValue, "ntc", "is_autonomic_correction_enabled" );

            if ( m_CurrentValue != currentValue )
            {
                m_pCheckBox->SetValue( currentValue );
                m_IsSystemResetRequried = true;
                m_CurrentValue = currentValue;
            }
#endif
        }

    private:
        CheckBoxButton* m_pCheckBox;

        bool  m_IsSystemResetRequried;
        bool  m_CurrentValue;

        void UpdateAutoEnsureEnabled( const glv::Notification& notificaiton ) NN_NOEXCEPT
        {
            NN_UNUSED( notificaiton );
            SetFixedSizeFirmwareDebugSettingsItemValue("ntc", "is_autonomic_correction_enabled", m_pCheckBox->GetValue());
            m_IsSystemResetRequried = true;
        }
    };

    class DateTimeSettingUi : public glv::Table
    {
    private:
        nn::time::CalendarTime m_CurrentCalendar;
        glv::Label  m_CautionLabel;

        glv::Label* m_YearText;
        glv::Label* m_MonthText;
        glv::Label* m_DayText;
        glv::Label* m_HourText;
        glv::Label* m_MinuteText;

        glv::View* m_YearUp;
        glv::View* m_MonthUp;
        glv::View* m_DayUp;
        glv::View* m_HourUp;
        glv::View* m_MinuteUp;

        glv::View* m_YearDown;
        glv::View* m_MonthDown;
        glv::View* m_DayDown;
        glv::View* m_HourDown;
        glv::View* m_MinuteDown;
    public:
        DateTimeSettingUi() NN_NOEXCEPT
            : m_CurrentCalendar(NetworkClockSetting::InitialCalendar)
            , m_CautionLabel("", glv::Label::Spec(glv::Place::TL, 0.0f, 8.0f, CommonValue::InitialFontSize * 0.8f))
        {
            // カレンダー初期値を現在時刻にセット
            nn::time::PosixTime current;
            if(nn::time::StandardNetworkSystemClock::GetCurrentTime(&current).IsSuccess())
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&m_CurrentCalendar, nullptr, current) );
            }

            const glv::Label::Spec iconLabelSpec( glv::Place::CC, 0.0f, 0.0f, 40.0f );

            auto pSetterTable = new glv::Table(
                "x . x . x . x . x,"
                "x x x x x x x x x,"
                "x . x . x . x . x");
            auto yearUp = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );
            auto monthUp = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );
            auto dayUp = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );
            auto hourUp = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );
            auto minuteUp = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowUp, iconLabelSpec ) );

            auto yearText = new glv::Label("2017", glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
            auto yearSpacer = new glv::Label(("/"), glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
            auto monthText = new glv::Label("01", glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
            auto monthSpacer = new glv::Label(("/"), glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
            auto dayText = new glv::Label("01", glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
            auto daySpacer = new glv::Label((" "), glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
            auto hourText = new glv::Label("00", glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
            auto hourSpacer = new glv::Label((":"), glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));
            auto minuteText = new glv::Label("00", glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize));

            auto yearDown = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );
            auto monthDown = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );
            auto dayDown = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );
            auto hourDown = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );
            auto minuteDown = new IconButton( glv::Rect(55.0f), false, new IconLabel( IconCodePoint::ArrowDown, iconLabelSpec ) );

            *pSetterTable << yearUp << monthUp << dayUp << hourUp << minuteUp
                << yearText << yearSpacer << monthText << monthSpacer << dayText << daySpacer << hourText << hourSpacer << minuteText
                << yearDown << monthDown << dayDown << hourDown << minuteDown;
            pSetterTable->arrange();

            *this << m_CautionLabel << pSetterTable;
            this->arrange();

            m_YearText = yearText;
            m_MonthText = monthText;
            m_DayText = dayText;
            m_HourText = hourText;
            m_MinuteText = minuteText;

            m_YearUp = yearUp;
            m_MonthUp = monthUp;
            m_DayUp = dayUp;
            m_HourUp = hourUp;
            m_MinuteUp = minuteUp;

            m_YearDown = yearDown;
            m_MonthDown = monthDown;
            m_DayDown = dayDown;
            m_HourDown = hourDown;
            m_MinuteDown = minuteDown;

            m_YearUp->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(1); }, glv::Update::Clicked, this);
            m_MonthUp->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(0, 1); }, glv::Update::Clicked, this);
            m_DayUp->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(0, 0, 1); }, glv::Update::Clicked, this);
            m_HourUp->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(0, 0, 0, 1); }, glv::Update::Clicked, this);
            m_MinuteUp->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(0, 0, 0, 0, 1); }, glv::Update::Clicked, this);

            m_YearDown->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(-1); }, glv::Update::Clicked, this);
            m_MonthDown->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(0, -1); }, glv::Update::Clicked, this);
            m_DayDown->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(0, 0, -1); }, glv::Update::Clicked, this);
            m_HourDown->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(0, 0, 0, -1); }, glv::Update::Clicked, this);
            m_MinuteDown->attach([](const glv::Notification& notification)->void { notification.receiver<DateTimeSettingUi>()->UpdateTime(0, 0, 0, 0, -1); }, glv::Update::Clicked, this);
        }

        nn::time::CalendarTime GetCalendar() const NN_NOEXCEPT
        {
            return m_CurrentCalendar;
        }

        void RefreshUi() NN_NOEXCEPT
        {
            auto SetLabelText = [](glv::Label* pOut, int value, std::streamsize width) -> void
            {
                std::stringstream ss;
                ss << std::setw(width) << std::setfill('0') << static_cast<int>(value);
                pOut->setValue(ss.str().c_str());
            };

            NN_FUNCTION_LOCAL_STATIC(nn::time::CalendarTime, s_CalendarPrevious, = NetworkClockSetting::InitialCalendar);
            if(s_CalendarPrevious != m_CurrentCalendar)
            {
                SetLabelText(m_YearText,    m_CurrentCalendar.year, 4);
                SetLabelText(m_MonthText,   m_CurrentCalendar.month, 2);
                SetLabelText(m_DayText,     m_CurrentCalendar.day, 2);
                SetLabelText(m_HourText,    m_CurrentCalendar.hour, 2);
                SetLabelText(m_MinuteText,  m_CurrentCalendar.minute, 2);

                s_CalendarPrevious = m_CurrentCalendar;


                // セットできない日時だったら、注意を表示
                int outCount;
                nn::time::PosixTime posixToSet;
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToPosixTime(&outCount, &posixToSet, 1, m_CurrentCalendar) );
                if(outCount == 0)
                {
                    NN_FUNCTION_LOCAL_STATIC(glv::Style, s_RedStyle);
                    s_RedStyle.color.set(glv::StyleColor::BlackOnWhite);
                    s_RedStyle.color.fore.set(0.8, 0.8, 0.8);
                    s_RedStyle.color.back.set(0.6, 0.0, 0.0);
                    m_CautionLabel.enable(glv::Property::DrawBack);
                    m_CautionLabel.style(&s_RedStyle);

                    m_CautionLabel.setValue(" This date-time is not available and cannot set it. ");
                }
                else
                {
                    m_CautionLabel.setValue("");
                }
            }
        }

        void UpdateTime(int yearDelta = 0, int monthDelta = 0, int dayDelta = 0, int hourDelta = 0, int minuteDelta = 0) NN_NOEXCEPT
        {
            nn::time::CalendarTime carendarTime = m_CurrentCalendar;

            carendarTime.year += yearDelta;
            carendarTime.month += monthDelta;
            carendarTime.day += dayDelta;
            carendarTime.hour += hourDelta;
            carendarTime.minute += minuteDelta;

            // 年補正
            if (carendarTime.year <= 0)
            {
                carendarTime.year = 1;
            }

            // 月補正
            if(carendarTime.month <= 0)
            {
                carendarTime.month = 12;
            }
            else if (carendarTime.month > 12)
            {
                carendarTime.month = 1;
            }

            // 日補正
            const int maxDays = nn::time::GetDaysInMonth(carendarTime.year, carendarTime.month);
            if (carendarTime.day <= 0)
            {
                carendarTime.day = maxDays;
            }
            else if (carendarTime.day > maxDays)
            {
                if(dayDelta == 0)
                {
                    // day をいじってなければ、year や month をずらしたことによって、月の日数が減った場合
                    carendarTime.day = maxDays;
                }
                else
                {
                    // day をいじってここに来た場合は、月の最終日から + をしたので、初日に戻す
                    carendarTime.day = 1;
                }
            }

            // 時刻補正
            if (carendarTime.hour < 0)
            {
                carendarTime.hour = 23;
            }
            carendarTime.hour %= 24;

            if (carendarTime.minute < 0)
            {
                carendarTime.minute = 59;
            }
            carendarTime.minute %= 60;

            if (carendarTime.second < 0)
            {
                carendarTime.second = 59;
            }
            carendarTime.second %= 60;

            m_CurrentCalendar = carendarTime;
        }
    };

    class CorrectRightTimeButton : public glv::Table
    {
    public:
        CorrectRightTimeButton () NN_NOEXCEPT
            : glv::Table("x,<", 0.0f, 0.0f)
            , m_Button(GLV_TEXT_API_WIDE_STRING("Get Right Time From Server"), [&] {OnClickCorrectionButton(); }, glv::Rect(400.0f, ButtonHeight), glv::Place::TL)
            , m_ResultLabel("", glv::Label::Spec(glv::Place::TL, 0.0f, 8.0f, CommonValue::InitialFontSize * 0.8f))
            , m_IsThreadRunning(false)
            , m_IsThreadDestoryRequested(false)
        {
            m_ResultMsg[0] = '\n';

            auto pLabelTable = new glv::Table("xx");
            *pLabelTable << new Spacer(100.0f, 0.0f) << m_ResultLabel;
            pLabelTable->arrange();

            *this << m_Button << pLabelTable;
            this->arrange();
        }

        void RefreshUi() NN_NOEXCEPT
        {
            if (m_IsThreadDestoryRequested)
            {
                DestoryCorrectionThread();
                const char* p = &m_ResultMsg[0];
                m_ResultLabel.setValue(p);
            }
        }

    private:
        Button m_Button;
        glv::Label m_ResultLabel;
        char m_ResultMsg[128];

        static const size_t ThreadStackSize = 4 * 1024;
        void* m_pStack;
        nn::os::ThreadType m_Thread;
        bool m_IsThreadRunning;
        std::atomic<bool> m_IsThreadDestoryRequested;

        static void ThreadFunc(void* pArg) NN_NOEXCEPT
        {
            CorrectRightTimeButton* p = reinterpret_cast<CorrectRightTimeButton*>(pArg);
            p->ThreadFunctionImpl();
        }

        void ThreadFunctionImpl() NN_NOEXCEPT
        {
            NN_UTIL_SCOPE_EXIT
            {
                m_IsThreadDestoryRequested = true;
            };

            nn::nifm::NetworkConnection connection;
            connection.SubmitRequestAndWait();
            if (!connection.IsAvailable())
            {
                SetMessageResultLable("Network is not available.");
                return;
            }

            nn::ntc::shim::CorrectionNetworkClockAsyncTask
                task(nn::os::EventClearMode_AutoClear, nn::ntc::EnsureNetworkClockAvailabilityMode_ForcibleDownload);

            auto result = task.StartTask();
            if (result.IsFailure())
            {
                if (nn::time::ResultNetworkRequestNotAccepted::Includes(result))
                {
                    SetMessageResultLable("Network is not available.");
                }
                else
                {
                    SetErrorResultLabel(result);
                }
                return;
            }

            task.GetFinishNotificationEvent().Wait();
            if (task.GetResult().IsFailure())
            {
                SetErrorResultLabel(task.GetResult());
                return;
            }

            SetMessageResultLable("Success.");
        }

        void SetMessageResultLable(const char* msg) NN_NOEXCEPT
        {
            nn::util::SNPrintf(m_ResultMsg, sizeof(m_ResultMsg), msg);
        }
        void SetErrorResultLabel(const nn::Result& result) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(result.IsFailure());
            nn::util::SNPrintf(m_ResultMsg, sizeof(m_ResultMsg), "failed. Result(0x%08lx)", result.GetInnerValueForDebug());
        }

        void LaunchCorrectionThread() NN_NOEXCEPT
        {
            m_IsThreadRunning = true;
            m_ResultLabel.setValue("");

            m_pStack = MemoryAllocator::AllocAlignedMemory( nn::os::ThreadStackAlignment, ThreadStackSize );
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::os::CreateThread( &m_Thread, ThreadFunc, this, m_pStack, ThreadStackSize, nn::os::DefaultThreadPriority ) );
            nn::os::SetThreadNamePointer( &m_Thread, "DevMenuCorrectionRightTime" );
            nn::os::StartThread( &m_Thread );
        }

        void DestoryCorrectionThread() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_IsThreadDestoryRequested);
            nn::os::DestroyThread( &m_Thread );
            MemoryAllocator::FreeAlignedMemory(m_pStack);
            m_IsThreadDestoryRequested = false;
            m_IsThreadRunning = false;
        }

        void OnClickCorrectionButton() NN_NOEXCEPT
        {
            if (m_IsThreadRunning)
            {
                return;
            }

            LaunchCorrectionThread();
        }
    };

public:
    NetworkClockSetting() NN_NOEXCEPT
        : m_SettingCaption("Network clock settings", glv::Label::Spec(glv::Place::TL, 0.0f, 8.0f, CommonValue::InitialFontSize))
        , m_IsSystemResetRequried(false)
        , m_CurrentDateTime("", glv::Label::Spec(glv::Place::TL, 0.0f, 8.0f, CommonValue::InitialFontSize))
        , m_DateTimeCommitButton(GLV_TEXT_API_WIDE_STRING("Commit New Date-Time"), [&] { OnClickDateTimeCommitButton(); }, glv::Rect(400.0f, ButtonHeight), glv::Place::TL)
        , m_InvalidateButton(GLV_TEXT_API_WIDE_STRING("Invalidate Clock"), [&] { OnClickInvalidateButton(); }, glv::Rect(400.0f, ButtonHeight), glv::Place::TL)
    {
        auto table = new glv::Table("<", 100);

        m_pAutoEnsureEnabledCheckbox = new AutoEnsureEnabledCheckbox();
        m_pCorrectRightTimeButton = new CorrectRightTimeButton();
        m_pDateTimeSettingUi = new DateTimeSettingUi();

        auto pLeftTable = new glv::Table("<", 0.0f, 10.0f);
        *pLeftTable << m_pAutoEnsureEnabledCheckbox;
        *pLeftTable << m_InvalidateButton;
        *pLeftTable << m_DateTimeCommitButton;
        *pLeftTable << m_pCorrectRightTimeButton;
        pLeftTable->arrange();

        // Add date-time setting table
        {
            auto dateTimeSettings = new glv::Table("xx^");
            *dateTimeSettings
                << pLeftTable
                << new Spacer(50.0f, 0.0f)
                << m_pDateTimeSettingUi;
            dateTimeSettings->arrange();
            *table << dateTimeSettings ;
        }

        table->arrange();

        auto pCaptionTable = new glv::Table("xx");
        *pCaptionTable << m_SettingCaption << m_CurrentDateTime;
        pCaptionTable->arrange();

        *this << pCaptionTable << table;
        this->arrange();
    }

    void OnClickInvalidateButton() NN_NOEXCEPT
    {
        auto context = nn::settings::system::SystemClockContext();
        context.timeStamp.sourceId = nn::util::InvalidUuid;

        nn::settings::system::SetNetworkSystemClockContext(context);

        m_IsSystemResetRequried = true;
    }

    void OnClickDateTimeCommitButton() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::InitializeForSystem() );

        nn::time::PosixTime posixToSet;
        int outCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToPosixTime(&outCount, &posixToSet, 1, m_pDateTimeSettingUi->GetCalendar()) );

        if (outCount == 0)
        {
            // セットできないカレンダー
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::SetStandardNetworkSystemClockCurrentTime(posixToSet) );
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::InitializeForMenu() );
    }

    void RefreshUi() NN_NOEXCEPT
    {
        nn::time::PosixTime current;
        if(nn::time::StandardNetworkSystemClock::GetCurrentTime(&current).IsSuccess())
        {
            NN_FUNCTION_LOCAL_STATIC(nn::time::PosixTime, s_PosixPrevious, = { 0 });
            if(s_PosixPrevious != current)
            {
                s_PosixPrevious = current;

                std::stringstream ss;
                ss << " / Current Date-Time : ";

                nn::time::CalendarTime c;
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&c, nullptr, current) );

                ss << std::setfill('0')
                    << std::setw(4) << static_cast<int>(c.year) << "/"
                    << std::setw(2) << static_cast<int>(c.month) << "/"
                    << std::setw(2) << static_cast<int>(c.day) << " "
                    << std::setw(2) << static_cast<int>(c.hour) << ":"
                    << std::setw(2) << static_cast<int>(c.minute);
                m_CurrentDateTime.setValue(ss.str().c_str());
            }
        }
        else
        {
            m_CurrentDateTime.setValue(" / Clock is not available.");
        }

        m_pCorrectRightTimeButton->RefreshUi();
        m_pDateTimeSettingUi->RefreshUi();
    }

    // 毎フレーム呼ばない
    void RefreshValue() NN_NOEXCEPT
    {
        m_pAutoEnsureEnabledCheckbox->RefreshValue();
    }

    bool IsSystemResetRequried() const NN_NOEXCEPT
    {
        return m_IsSystemResetRequried || m_pAutoEnsureEnabledCheckbox->IsSystemResetRequried();
    }

public:
    static const nn::time::CalendarTime InitialCalendar;

private:
    static const int ButtonHeight = 40;

    glv::Label m_SettingCaption;

    bool m_IsSystemResetRequried;

    glv::Label m_CurrentDateTime;
    DateTimeSettingUi* m_pDateTimeSettingUi;
    Button m_DateTimeCommitButton;
    Button m_InvalidateButton;
    AutoEnsureEnabledCheckbox* m_pAutoEnsureEnabledCheckbox;
    CorrectRightTimeButton* m_pCorrectRightTimeButton;
};
const nn::time::CalendarTime NetworkClockSetting::InitialCalendar = {2017, 1, 1, 0, 0, 0};

/**
 * @brief 時計テスト機能のページです。
 */
class TimeTestPage : public Page
{
private:
    NetworkClockSetting* m_NetworkClockSetting;
    SteadyClockSetting* m_SteadyClockSetting;

public:
    /**
     * @brief コンストラクタです。
     */
    TimeTestPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        : Page(pageId, pageCaption, rect)
        , m_NetworkClockSetting(nullptr)
        , m_SteadyClockSetting(nullptr)
    {
    }

    virtual void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE
    {
        m_NetworkClockSetting->RefreshValue();
    }

    /**
     * @brief ページがコンテナに追加された後に呼び出されます。
     */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        auto table = new glv::Table("<", 30.0f, 5.0f);

        m_NetworkClockSetting = new NetworkClockSetting();
        m_SteadyClockSetting = new SteadyClockSetting();

        *table
            << m_NetworkClockSetting
            << m_SteadyClockSetting;
        table->arrange();

        // Set the Table to be able to detect clicks
        table->enable( glv::Property::HitTest | glv::Property::GestureDetectability );
        // Attach a Click handler to detect if the B button was pressed
        table->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

        *this << table;
    }

    /**
     * @brief ページがコンテナから削除された前に呼び出されます。
     */
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
    }

    /**
     * @brief ページがアクティブ( 選択により表示開始 )になる際に呼び出されます。
     */
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_NetworkClockSetting->RefreshValue();
    }

    /**
     * @brief ページがディアクティブ( 選択により非表示開始 )になる際に呼び出されます。
     */
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        NN_UNUSED( events );

        m_NetworkClockSetting->RefreshUi();
        m_SteadyClockSetting->RefreshUi();

        bool isSystemResetRequried =
            m_NetworkClockSetting->IsSystemResetRequried() ||
            m_SteadyClockSetting->IsSystemResetRequried();

        if(isSystemResetRequried)
        {
            // Display the header Reboot Warning message
            GetRootSurfaceContext()->DisplayRebootRequestText();
        }
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
     */
    virtual void OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        NN_UNUSED( events );
    }

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    virtual View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return this;
    }

private:
    /**
     * @brief 問い合わせ済プロパティデータの解放。
     */
    void FinalizeProperties() NN_NOEXCEPT
    {
    }
};

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class TimeTestPageCreator : PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit TimeTestPageCreator( const char* pPageName ) NN_NOEXCEPT
        : PageCreatorBase( ID, pPageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const glv::space_t width = static_cast< glv::space_t >( resolution[ 0 ] );
        const glv::space_t height = static_cast< glv::space_t >( resolution[ 1 ] );
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new TimeTestPage( ID, GLV_TEXT_API_WIDE_STRING( "Time Test" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) TimeTestPageCreator< _id > g_TimeTestPageCreator##_id( _name );
LOCAL_PAGE_CREATOR(DevMenuPageId_TimeTest, "TimeTestPage" );

}} // ~namespace devmenu::timetest, ~namespace devmenu
