﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_BitTypes.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Directory.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/ns/ns_Result.h>

#include "DevMenu_SystemProgramInstaller.h"
#include "DevMenu_SystemProgram.h"

//#define DEBUG_DATA

namespace devmenu { namespace system { namespace program {

const glv::space_t SdCardListView::HorizontalMargin_Name = 4.f;
const glv::space_t SdCardListView::HorizontalLength_Name = 768.f;

const size_t       InstallView::InitialActionCapacity = 4;
const glv::space_t InstallView::InstallViewWidth      = 800.f;
const glv::space_t InstallView::ProgressBarMargineX   = 8.f;
const glv::space_t InstallView::MessageButtonPaddingX = 32.f;
const glv::space_t InstallView::MessageButtonPaddingY = 16.f;

namespace {
    const size_t    InstallBufferSize = 1 * 1024 * 1024;
    const size_t    MaxComfirmNum = 5;

    // パッディング
    const glv::space_t MessageViewPaddingX = 32.f;
    const glv::space_t MessageViewPaddingY = 32.f;

    const size_t MaxModalMessageLength    = 55;
    const size_t MaxCatalogFileNameLength = 75;
}

/**
 * @brief       コンストラクタ
 */
InstallView::InstallView() NN_NOEXCEPT
    :   MessageView( false ),
    m_ProgressBarsTable( "", MessageViewPaddingX, MessageViewPaddingY, glv::Rect( 0 ) )
{
    Initialize();
}

InstallView::~InstallView() NN_NOEXCEPT
{
}

void InstallView::Initialize() NN_NOEXCEPT
{
    enable( glv::Property::DrawBorder | glv::Property::DrawBack );
    disable( glv::Property::FocusHighlight );
    m_Actions.reserve( InitialActionCapacity );
    m_ViewTable.arrangement( "<,x,x" );
    *this << ( m_ViewTable << m_MessagesTable << m_ProgressBarsTable << m_ButtonsTable );


    auto pProgressCaption = new glv::Label( "[Install Progress]", glv::Label::Spec( glv::Label::Spec( glv::Place::TL, 0.f, 0.f, CommonValue::InitialFontSize ) ) );
    auto pProgressFileName = new glv::Label( "Preparing...", glv::Label::Spec( glv::Label::Spec( glv::Place::TL, pProgressCaption->right() + 10.f, 0.f, CommonValue::InitialFontSize ) ) );
    auto pProgressLabel = new glv::Label( "Start installing...", glv::Label::Spec( glv::Place::TL, 25.f, pProgressCaption->bottom() + 20.f, CommonValue::InitialFontSize ) );
    AddProgressMessage( pProgressCaption, pProgressFileName, pProgressLabel );
    m_pProgressCaption = pProgressCaption;
    m_pProgressFileName = pProgressFileName;
    m_pProgressLabel = pProgressLabel;

    glv::Rect sliderRect( InstallViewWidth - ProgressBarMargineX - ProgressBarMargineX, CommonValue::InitialFontSize );
    auto pProgressBar = new glv::Slider( sliderRect, 0.f  );
    AddProgressBar( pProgressBar );
    m_pProgressBar = pProgressBar;

    auto pLabel = new glv::Label( "OK" );
    pLabel->size( CommonValue::InitialFontSize ).pos( glv::Place::CC ).anchor( glv::Place::CC );
    glv::Rect rect( pLabel->width() + MessageButtonPaddingX * 2, pLabel->height() + MessageButtonPaddingY * 2 );
    auto pButton = new glv::Button( rect, true );
    *pButton << pLabel;
    AddButton( pButton );
    m_pButton = pButton;
    m_pButton->disable(glv::Property::t::Visible);

    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    pProgressBar->disable( focusableProperty );

    m_InstallState = InstallState_NotStarted;
}

InstallView& InstallView::AddProgressMessage( glv::Label* pProgressCaption, glv::Label* pProgressFileName, glv::Label* pProgressLabel ) NN_NOEXCEPT
{
    m_pProgressCaption = pProgressCaption;
    m_pProgressFileName = pProgressFileName;
    m_pProgressLabel = pProgressLabel;

    // レイアウト調整
    m_MessagesTable << pProgressCaption;
    m_MessagesTable << pProgressFileName;
    m_MessagesTable << pProgressLabel;
    m_MessagesTable.arrange();
    m_ViewTable.arrange();

    return *this;
}

InstallView& InstallView::AddButton( glv::Button* pButton ) NN_NOEXCEPT
{
    return AddButton( pButton, nullptr, nullptr );
}
InstallView& InstallView::AddButton( glv::Button* pButton, ActionFunc pFunc, void* pParam ) NN_NOEXCEPT
{
    pButton->attach( ReceiveUpdateNotification, glv::Update::Clicked, this );

    // 配置方法指定
    std::string align = m_ButtonsTable.arrangement() + "x";
    m_ButtonsTable.arrangement( align.c_str() );

    // レイアウト調整
    m_ButtonsTable << pButton;
    m_ButtonsTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );

    // 実行情報を登録
    Action action;
    action.pButton = pButton;
    action.pFunc = pFunc;
    action.pParam = pParam;
    m_Actions.push_back( action );

    return *this;
}

InstallView& InstallView::AddProgressBar( glv::Slider* pProgressBar ) NN_NOEXCEPT
{
    m_pProgressBar = pProgressBar;

    // 配置方法指定
    std::string align = m_ProgressBarsTable.arrangement() + "x";
    m_ProgressBarsTable.arrangement( align.c_str() );

    // レイアウト調整
    m_ProgressBarsTable << pProgressBar;
    m_ProgressBarsTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );
    return *this;
}


// インストールスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_InstallStack[ 32 * 1024 ];

void ConvertWideStringToString( char* pOutStr, const glv::WideCharacterType* fileName ) NN_NOEXCEPT
{
    int fileListLength = 0;
    const uint16_t* pFileList = reinterpret_cast< const uint16_t* >( fileName );
    nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8( &fileListLength, pFileList, MaxDirectoryNameLength );
    nn::util::ConvertStringUtf16NativeToUtf8( pOutStr, fileListLength, pFileList );
}

/**
 * @brief コンストラクタです。
 */
InstallScene::InstallScene( SystemProgramPage* pParentPage, devmenu::Scene* pParentScene ) NN_NOEXCEPT
    : Scene( pParentPage->rect() ),
    m_pButtonAllCheckboxes( nullptr ),
    m_pNsps( nullptr ),
    m_pNspListView( nullptr ),
    m_pLabelNoNsp( nullptr ),
    m_pButtonToCatalog( nullptr ),
    m_pParentScene( pParentScene ),
    m_pParentPage( pParentPage ),
    m_pInstallTask( nullptr ),
    m_IsInstallAll( false ),
    m_pInstallView( nullptr )
{
    // シザーボックスの領域設定
    static const glv::space_t InstallSceneHeaderRegion = 48.f;
    static const glv::space_t InstallSceneFooterRegion = 48.f;

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    static const glv::space_t ListMarginL = 16.f;
    static const glv::space_t ListMarginR = 16.f;
    static const glv::space_t ListMarginY = 2.f;

    // 全 nsp の選択ボタン ( → 現状は選択できる実装になっていないので、Install All にしておく )
    m_pButtonAllCheckboxes = new Button( "Install All", [&]{ m_IsInstallAll = true; ConfirmInstall(); }, CommonValue::InitialFontSize, 16.f );
    m_pButtonAllCheckboxes->anchor( glv::Place::TL ).pos( glv::Place::TL, ListMarginL, 4.f );
    *this << m_pButtonAllCheckboxes;

    // ヘッダ
    auto pHeaderName = new glv::Label( "Name", glv::Label::Spec( glv::Place::TL, m_pButtonAllCheckboxes->right() + 16.f, 8.f, CommonValue::InitialFontSize ) );
    auto pHeaderSize = new glv::Label( "Size(Byte)", glv::Label::Spec( glv::Place::TR, -( ListMarginR + 16.f ), 8.f, CommonValue::InitialFontSize ) );
    {
        *this << pHeaderName;
        *this << pHeaderSize;
    }

    // リスト
    auto pListContainer = new glv::ScissorBoxView( 0, InstallSceneHeaderRegion, width(), height() - ( InstallSceneHeaderRegion + InstallSceneFooterRegion ) );
    {
        glv::Rect clipRegion( ListMarginL, ListMarginY, pListContainer->width() - ( ListMarginL + ListMarginR ), pListContainer->height() - ( ListMarginY * 2 ) );
        auto pNspListView = new SdCardListView( clipRegion );
        pNspListView->attach(
            []( const glv::Notification& notification )->void {
                auto pScene = notification.receiver< InstallScene >();
                const FilePropertyType* pProperty = notification.data< SdCardListView::ItemType >();
                if ( nullptr != pScene && nullptr != pProperty )
                {
                    pScene->m_IsInstallAll = false;
                    pScene->ConfirmInstall( pProperty );
                }
            },
            glv::Update::Action,
            this
        );

        m_pNspListView = pNspListView;
        *pListContainer << pNspListView;

        // NSP ファイルなしメッセージ
        auto pLabelNoNsp = new glv::Label( "No system programs are found.", glv::Label::Spec( glv::Place::CC, 0.f, 0.f, 32.f ) );
        {
            m_pLabelNoNsp = pLabelNoNsp;
            *pListContainer << pLabelNoNsp;
        }
        *this << pListContainer;
    }

    // フッタ
    auto pFooterExplaination = new glv::Label( "A: Install the selected file", glv::Label::Spec( glv::Place::BR, -( ListMarginL + 16.f ), -8.f, CommonValue::InitialFontSize ) );
    {
         *this << pFooterExplaination;
    }

    // Catalog に戻るボタン
    m_pButtonToCatalog = new Button( "Back", [&]{ this->OnBackButtonPressed(); }, CommonValue::InitialFontSize, 16.f );
    m_pButtonToCatalog->anchor( glv::Place::BL ).pos( glv::Place::TL, ListMarginL, -InstallSceneFooterRegion + 4.f );
    *this << m_pButtonToCatalog;

    this->SetFirstFocusTargetView( m_pNspListView );

    // インストールビューの初期化
    m_pInstallView = new InstallView();
}

// ToDo: どこで使うのかが決まっていない
void InstallScene::FinalizeProperties() NN_NOEXCEPT
{
    // プロパティデータの解放
    SdCardListView::CollectionType* pNsps;
    if ( nullptr != ( pNsps = m_pNsps ) )
    {
        m_pNsps = nullptr;
        delete pNsps;
    }
}

void InstallScene::Refresh() NN_NOEXCEPT
{

}

nn::Result InstallScene::EnumerateFiles( std::vector< FilePropertyType >& entries, const char* directoryPath ) NN_NOEXCEPT
{
    nn::fs::DirectoryHandle directoryHandle;
    NN_RESULT_DO( nn::fs::OpenDirectory( &directoryHandle, directoryPath, nn::fs::OpenDirectoryMode_All ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory( directoryHandle );
    };

    int64_t entryCount = 1;
    nn::fs::DirectoryEntry entry;

    while ( true )
    {
        NN_RESULT_DO( nn::fs::ReadDirectory( &entryCount, &entry, directoryHandle, 1 ) );
        if ( entryCount == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_File )
        {
            auto length = strlen(entry.name);
            if ( strcmp( entry.name + length - 4, ".nsp" ) == 0 && length > 4 )
            {
                FilePropertyType fileProp;
                fileProp.Prepare( entry, directoryPath );
                entries.push_back( fileProp );
            }
        }

        else if ( entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory )
        {
            char childDirectoryPath[ MaxDirectoryNameLength ];
            sprintf( childDirectoryPath, "%s%s/", directoryPath, entry.name );

            // 子ディレクトリに対して再帰的に関数を呼び出す
            NN_RESULT_DO( EnumerateFiles( entries, childDirectoryPath ) );
        }
    }
    NN_RESULT_SUCCESS;
}

/**
 * @brief SD カードをマウントしファイルリストを作成。
 */
nn::Result InstallScene::MakeNspList() NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    // SD カードをマウントしファイルリストを作成
    SdCardListView::CollectionType* pNsps;
    m_pNsps = pNsps = new SdCardListView::CollectionType();
    m_FileList.clear();

    auto result = nn::fs::MountSdCardForDebug( "sd" );
    if ( result.IsFailure() )
    {
        auto pView = new devmenu::MessageView( false );
        pView->AddMessage( "Failed to mount the SD card." );
        pView->AddButton( "Close" );
        m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true );

        DEVMENU_LOG( "Failed to mount the SD card.\n" );
        return result;
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };

    result = EnumerateFiles( m_FileList, "sd:/" );
    if ( result.IsFailure() )
    {
        DEVMENU_LOG( "Failed EnumerateFiles.\n" );
        return result;
    }
    // リストに反映
    pNsps->resize( m_FileList.size() );
    for ( size_t i = 0; i < m_FileList.size(); i++ )
    {
        pNsps->at( i ) = m_FileList[ i ];
        // wchar_t 型の標準関数( snwprintfなど )は、gccでは 4byteのためクロスプラットフォームでは使えないので
        // 一度 char* 型に変換して ClipMessage を行う。
        char fileName[ MaxDirectoryNameLength ];
        ConvertWideStringToString( fileName, pNsps->at( i ).name );

        BuildUtf16< MaxDirectoryNameLength >(  pNsps->at( i ).name, "%s", ClipMessage( fileName, MaxCatalogFileNameLength ) );
    }

    EntryProperties( pNsps );
#endif

    NN_RESULT_SUCCESS;
}

/**
 * @brief SD カード内のプロパティコレクションを登録します。
 * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
 */
void InstallScene::EntryProperties( const SdCardListView::CollectionType* pNsps ) NN_NOEXCEPT
{
    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    SdCardListView* pNspListView;
    auto* const pLabelNoNsp = m_pLabelNoNsp;
    if ( nullptr != ( pNspListView = m_pNspListView ) )
    {
        pNspListView->EntryCollection( *pNsps );
        if ( true == pNsps->empty() )
        {
            pNspListView->disable( focusableProperty );
            pLabelNoNsp->enable( glv::Property::t::Visible );
            pLabelNoNsp->bringToFront();
            this->SetFirstFocusTargetView( m_pButtonToCatalog );
        }
        else
        {
            pNspListView->enable( focusableProperty );
            pLabelNoNsp->disable( glv::Property::t::Visible );
            //this->SetFirstFocusTargetView( m_pButtonAllCheckboxes );
        }
    }
}

/**
 * @brief ダイアログを表示して確認を取った後、インストールを行います。
 */
void InstallScene::ConfirmInstall( const FilePropertyType* pProperty ) NN_NOEXCEPT
{
    auto pView = new devmenu::MessageView( false );
    auto fileListSize = m_FileList.size();
    auto overSize = 0;

    if ( fileListSize == 0 )
    {
        pView->AddMessage( "No file is selected. Select at least one nsp." );
        pView->AddButton( "Close" );
    }
    else
    {
        // リストから 1 つを選択したとき
        if( m_IsInstallAll == false )
        {
            fileListSize = 1;
        }
        // 確認表示時のファイル数が 1 つ
        if( fileListSize == 1 )
        {
            pView->AddMessage( "Install the selected file?" );
        }
        // 確認表示時のファイル数が複数
        else
        {
            pView->AddMessage( "Install these selected files?" );
            // 上限を MaxComfirmNum 個までとする。
            if ( fileListSize > MaxComfirmNum )
            {
                overSize = fileListSize - MaxComfirmNum;
                fileListSize = MaxComfirmNum;
            }
        }

        // ファイル名を表示
        for ( size_t i = 0; i < fileListSize; i++ )
        {
            char fileName[ MaxDirectoryNameLength ];
            // リストから 1 つを選択したとき
            if( m_IsInstallAll == false )
            {
                ConvertWideStringToString( fileName, pProperty->name );
            }
            else
            {
                ConvertWideStringToString( fileName, m_FileList[ i ].name );
            }
            // モーダルにインストールするファイル名を表示、文字数が多い場合は最初と最後から数文字だけを表示
            pView->AddMessage( ClipMessage( fileName, MaxModalMessageLength ) );
        }
        // 列挙するファイル数が MaxComfirmNum を超えた場合、残りの数を表示して省略
        if( overSize > 0 )
        {
            if ( overSize == 1 )
            {
                pView->AddMessage( "and another file." );
            }
            else
            {
                char message[ MaxModalMessageLength ];
                sprintf( message, "and the other %d files.", overSize );
                pView->AddMessage( message );
            }
        }

        pView->AddButton( "Cancel" );

        // 開始ボタンを追加
        pView->AddButton(
            "Start",
            []( void* pParam, nn::TimeSpan& timespan )
            {
                InstallScene *_this = static_cast< InstallScene * >( pParam );
                NN_ABORT_UNLESS_NOT_NULL( _this );

                if ( false == _this->MountSdCard() )
                {
                    return;
                }
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::Unmount("sd");
                };

                const FilePropertyType *currentProperty = _this->m_pNspListView->GetSelectedValue();
                if ( currentProperty == nullptr ) {
                    DEVMENU_LOG( "Could not install nsp - Selected Value was null.\n" );
                }
                else
                {
                    _this->InstallNspFile();
                }
            },
            this,
            MessageView::ButtonTextColor::Green
        );
    }
    m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true );
}

/**
 * @brief 選択された nsp ファイルのインストールを行います。
 */
void InstallScene::InstallNspFile() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread( &m_InstallTrigThread, ThreadFunc, this, g_InstallStack, sizeof( g_InstallStack ), nn::os::DefaultThreadPriority )
    );
    nn::os::StartThread( &m_InstallTrigThread );

    // インストール進捗ビューを登録
    auto pView = new InstallView();
    m_pInstallView = pView;
    m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true, true );
}

/**
 * @brief SD カードをマウントします。
 * @return      マウントに成功したら true
 */
bool InstallScene::MountSdCard() NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    auto pRootSurface = m_pParentPage->GetRootSurfaceContext();

    auto result = nn::fs::MountSdCardForDebug( "sd" );
    if ( result.IsFailure() )
    {
        auto pView = new devmenu::MessageView( false );
        pView->AddMessage( "Failed to mount the SD card." );
        pView->AddButton( "Close" );
        pRootSurface->StartModal( pView, true, true );

        DEVMENU_LOG( "Failed to mount the SD card.\n" );
        return false;
    }
#endif
    return true;
}

/**
  * @brief インストールする nsp ファイルをオープンします。
  */
bool InstallScene::OpenFile( nn::fs::FileHandle& handle, const char* fileName ) NN_NOEXCEPT
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
    auto result = nn::fs::OpenFile( &handle, fileName, nn::fs::OpenMode::OpenMode_Read );
    if ( !result.IsSuccess() )
    {
        auto pView = new devmenu::MessageView( false );
        pView->AddMessage( fileName );
        pView->AddMessage( "\n[Error] Failed to open the file." );
        pView->AddButton( "Close" );
        m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true, true );

        DEVMENU_LOG( "[Error] Failed to open the file. ( %s )\n", fileName );
        return false;
    }
#endif
    return true;
}

/**
 * @brief インストールスレッドのメイン関数です。
 */
void InstallScene::ThreadFunc( void* args ) NN_NOEXCEPT
{
    InstallScene* _this = reinterpret_cast< InstallScene* >( args );

    NN_ASSERT_NOT_NULL( _this );
    NN_ASSERT_EQUAL( _this->m_pInstallView->GetInstallState(), InstallState_NotStarted );

    if ( false == _this->MountSdCard() )
    {
        return;
    }

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( "sd" );
    };

    // 不要なアプリケーション実体を削除
    nn::ns::DeleteRedundantApplicationEntity();
    char fileName[ MaxDirectoryNameLength ];

    // Install All ボタンでの処理
    if( _this->m_IsInstallAll == true )
    {
        auto fileListSize = _this->m_FileList.size();
        for ( size_t i = 0; i < fileListSize; i++ )
        {
            ConvertWideStringToString( fileName, _this->m_FileList[ i ].name );
            while ( InstallState_NotStarted != _this->m_pInstallView->GetInstallState() )
            {
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
            }
            _this->ExecuteInstall( const_cast< const char* >( fileName ) );
        }
        _this->m_IsInstallAll = false;
    }

    // nsp リストから 1 つを選択した処理
    else
    {
        ConvertWideStringToString( fileName, _this->m_FileList[ _this->m_pNspListView->GetSelectedPosition() ].name);
        _this->ExecuteInstall( const_cast< const char* >( fileName ) );
    }

    _this->m_pInstallView->SetInstallState( InstallState_Completed );
}

/**
 * @brief nsp ファイルをインストールします。
 */
void InstallScene::ExecuteInstall( const char* fileName ) NN_NOEXCEPT
{
    if ( false == OpenFile( m_Handle, fileName ) )
    {
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile( m_Handle );
    };
    DEVMENU_LOG( "Preparing to install %s...\n", fileName );
    // モーダルにインストールするファイル名を表示
    m_pInstallView->SetProgressFileName( ClipMessage( fileName, MaxModalMessageLength ) );

    std::unique_ptr< char[] > buffer( new char[ InstallBufferSize ] );
    nn::ncm::SubmissionPackageInstallTask task;

    // TORIAEZU: Pad からの入力で Progress の ModalView が消えないようするための Work Aruond
    // ひとつ前の ModalView が EndModal してから呼ぶ必要があるが、タイミングは適当
    m_pParentPage->GetRootSurfaceContext()->setFocus( nullptr );

    // インストール先ストレージを内蔵 NAND に固定
    nn::util::optional< nn::ncm::StorageId > storage = nn::ncm::StorageId::BuildInSystem;
    task.Initialize( m_Handle, *storage, buffer.get(), InstallBufferSize );

    task.Prepare();

    // TORIAEZU: 念のため、もう一度呼んでおく
    m_pParentPage->GetRootSurfaceContext()->setFocus( nullptr );

    m_pInstallTask = &task;
    InstallTaskExecute( &task );
    m_pInstallView->SetLastNspTotalSize( task.GetProgress().totalSize );
    task.Commit();

    buffer.reset();
    m_pInstallView->SetInstallState( InstallState_Committed );
}

/**
 * @brief インストール処理を別スレッドで実行します。
 */
void InstallScene::InstallTaskExecute( nn::ncm::InstallTaskBase* task ) NN_NOEXCEPT
{
    static NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 s_Stack[ 16 * 1024 ];

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &m_InstallExecThread, []( void* p ) {
        reinterpret_cast< nn::ncm::InstallTaskBase* >( p )->Execute();
    }, task, s_Stack, sizeof( s_Stack ), nn::os::DefaultThreadPriority ) );
    nn::os::StartThread( &m_InstallExecThread );
    m_pInstallView->SetInstallState( InstallState_Running );

    while ( NN_STATIC_CONDITION( true ) )
    {
        auto isFinished = nn::os::TryWaitAny( &m_InstallExecThread ) >= 0;
        if ( isFinished )
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromSeconds( 1 ) );
    }
    nn::os::WaitThread( &m_InstallExecThread );
    nn::os::DestroyThread( &m_InstallExecThread );
}

/**
 * @brief インストール状況を更新します。
 */
void InstallScene::UpdateInstallProgress() NN_NOEXCEPT
{
    if ( InstallState_Running == m_pInstallView->GetInstallState() )
    {
        const int64_t pastTimeMills = nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() - m_StartTick ).GetMilliSeconds();
        if ( 1000 < pastTimeMills )
        {
            auto progress = m_pInstallTask->GetProgress();
            if( progress.installedSize == 0 || progress.totalSize == 0 ) return;
            char progressStr[ 48 ];
            auto installedSizeStr = devmenu::GetDelimitedNumberString( progress.installedSize );
            auto totalSizeStr = devmenu::GetDelimitedNumberString( progress.totalSize );

            float progressRate = static_cast< float >( progress.installedSize ) / static_cast< float >( progress.totalSize );
            float progressRatePercent = 100 * progressRate;
            nn::util::SNPrintf( progressStr, sizeof( progressStr ), "%s / %s (%.1f%)", installedSizeStr.c_str(), totalSizeStr.c_str(), progressRatePercent );
            m_pInstallView->SetProgressLabel( progressStr );
            m_pInstallView->SetProgressBarValue( progressRate );
            DEVMENU_LOG( "%s\n", progressStr );

            m_StartTick = nn::os::GetSystemTick();
        }
    }
    else if ( InstallState_Committed == m_pInstallView->GetInstallState() )
    {
        if ( true == m_IsInstallAll )
        {
            // ToDo: 進捗 100% を表示をするなど必要なら対応する
            m_pInstallView->SetInstallState( InstallState_NotStarted );
        }
    }
    else if ( InstallState_Completed == m_pInstallView->GetInstallState() )
    {
        m_pInstallView->GetButton()->enable( glv::Property::t::Visible );
        m_pParentPage->GetRootSurfaceContext()->setFocus( m_pInstallView->GetButton() );
        char progressStr[ 48 ];
        auto totalSizeStr = devmenu::GetDelimitedNumberString( m_pInstallView->GetLastNspTotalSize() );
        nn::util::SNPrintf( progressStr, sizeof( progressStr ), "%s / %s (100.0%)", totalSizeStr.c_str(), totalSizeStr.c_str() );
        m_pInstallView->SetProgressCaption( "[Install Complete]" );
        m_pInstallView->SetProgressLabel( progressStr );
        m_pInstallView->SetProgressBarValue( 1.f );
        DEVMENU_LOG( "%s\n", progressStr );

        nn::os::WaitThread( &m_InstallTrigThread );
        nn::os::DestroyThread( &m_InstallTrigThread );
        m_pInstallView->SetInstallState( InstallState_NotStarted );
    }
}

/**
 * @brief       表示する文字数に上限を設けて超えた場合は省略した形を生成します。
 *
 * @param[in]   message         文字列
 * @param[in]   maxLength       表示する文字の上限数( 9 以上 MaxDirectoryNameLength - 1 以内 )
 */
const char* InstallScene::ClipMessage( const char* message, const size_t maxLength ) NN_NOEXCEPT
{
    // 文字数が多い場合は、最初と最後から数文字だけを表示
    auto messageLength = std::string( message ).length();
    if( messageLength > maxLength )
    {
        static char omittedMessage[ MaxDirectoryNameLength ];
        size_t includeMessageLength = maxLength - 6;            // " ... " と終端文字の長さを引く
        // 表示する文字を前半:後半 = 1:2 とする
        size_t firstSize = includeMessageLength / 3.f;
        size_t endSize = std::ceilf( includeMessageLength * 2 / 3.f );
        char firstMessage[ MaxDirectoryNameLength ];
        nn::util::Strlcpy( firstMessage, message, firstSize );
        nn::util::SNPrintf( omittedMessage, MaxDirectoryNameLength, "%s ... %s", firstMessage, message + messageLength - endSize );
        return const_cast< const char* >( omittedMessage );
    }
    return message;
}

/**
 * @brief チェックボックスの全選択。
 */
void InstallScene::SelectAllCheckboxes() NN_NOEXCEPT
{

}

}}} // ~namespace devmenu::system::program, ~namespace devmenu::system, ~namespace devmenu

