﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <algorithm>
#include <memory>
#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <glv/utils/glv_MessageThread.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>

#include "Common/DevMenu_CommonScene.h"
#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"
#include "SystemProgram/DevMenu_ProgramIdNameMap.h"

using namespace nn;

namespace devmenu { namespace system { namespace program {

// Dev_SystemProgramPageInstaller.h との相互参照を防ぐためのクラス参照
class InstallScene;
class SystemProgramPage;

namespace {

    static const int MaxShowListCount = 1024;        //!< アプリケーションプロパティ最大表示上限。

    template< size_t TemporarySize, size_t OutBufferSize  >
    glv::WideCharacterType* BuildUtf16( glv::WideCharacterType ( &pOutBuffer )[ OutBufferSize ], const char* pFormat, ... ) NN_NOEXCEPT
    {
        va_list vaList;
        va_start( vaList, pFormat );

        char pTemp[ TemporarySize ];
        std::vsprintf( pTemp, pFormat, vaList );
        nn::util::ConvertStringUtf8ToUtf16Native( reinterpret_cast< uint16_t* >( pOutBuffer ), OutBufferSize, pTemp );

        va_end( vaList );
        return pOutBuffer;
    }
}

/**
 * @brief システムプログラムプロパティ型です。
 */
template< size_t N >
struct SystemProgramProperty
{
    glv::WideCharacterType   name[N];
    glv::WideCharacterType   id[N];
    glv::WideCharacterType   version[N];

    ncm::ContentMetaKey meta;

    bool                isQueried;



    static const size_t BufferSize = 128 * 1024;     //!< ApplicationControlProperty 受信用テンポラリバッファサイズ( バイト )。

    /**
     * @brief システムプログラムリスト用プロパティデータを構築します。
     */
    void Prepare( const ncm::ContentMetaKey& key, const char* programName ) NN_NOEXCEPT
    {
        BuildUtf16<N>( name, "%s", programName );
        BuildUtf16<N>( id, "0x%016llx", key.id );
        BuildUtf16<N>( version, "%4u", key.version );

        meta = key;
    }

    /**
     * @brief アプリケーションレコード総数を問い合わせます。
     * @details 問い合わせ時のテンポラリバッファにスタックを利用します。
     */

    /**
     * @brief アプリケーションレコードを問い合わせます。
     */
    static const int QueryApplicationRecordCount( std::vector< nn::ns::ApplicationRecord >& recordList ) NN_NOEXCEPT
    {
#if defined( DEBUG_DATA )

        return static_cast< int >( recordList.size() );

#else

        const int bufferCount = static_cast< int >( recordList.size() );
        auto count = nn::ns::ListApplicationRecord( recordList.data(), bufferCount, 0 );
        return std::min( bufferCount, count );

#endif // defined( DEBUG_DATA )
    }

};

/**
 * @brief アプリケーションプロパティ( 64文字上限 )型です。
 */
typedef SystemProgramProperty< 64 > SystemProgramPropertyType;

/**
 * @brief リストビュー
 */
class PropertyListView : public glv::CustomVerticalListView< SystemProgramPropertyType >
{
public:
    typedef glv::CustomVerticalListView< SystemProgramPropertyType > ParentType;

    static const glv::space_t HorizontalMargin_ProgramId;       //!< リスト要素中の[プログラムID]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_ProgramId;       //!< リスト要素中の[プログラムID]表示横幅( ピクセル単位 )
    static const glv::space_t HorizontalMargin_ProgramVersion;  //!< リスト要素中の[プログラムVersion]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_ProgramVersion;  //!< リスト要素中の[プログラムVersion]表示横幅( ピクセル単位 )

    /**
     * @brief コンストラクタです。
     */
    explicit PropertyListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT;

protected:
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE;
};

/**
 * @brief クエリ結果通知用メッセージ ( 192 byte )
 */
class QueryReply : public glv::utils::ObjectRecyclerBase< QueryReply >::BasedNode
{
public:
    static const unsigned MessageQueueCapacity = 8;
    typedef glv::utils::StaticObjectRecycler< QueryReply, MessageQueueCapacity > RecyclerType;
    typedef glv::utils::MessageReceiver< MessageQueueCapacity > ReplyReceiver;

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    explicit QueryReply( RecyclerType* pOwnerRecycler ) NN_NOEXCEPT
        : RecyclerType::BasedNode( pOwnerRecycler ) {}

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    void Initialize( const SystemProgramPropertyType& inPropoerty, const int index ) NN_NOEXCEPT
    {
        m_Index = index;
        m_Property = inPropoerty;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    const SystemProgramPropertyType& GetProperty() const NN_NOEXCEPT
    {
        return m_Property;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    const int GetIndex() const NN_NOEXCEPT
    {
        return m_Index;
    }

private:
    SystemProgramPropertyType m_Property;
    int                     m_Index;
};

/**
 * @brief クエリ要求用メッセージ ( 80 byte )
 */
class QueryRequest : public glv::utils::ObjectRecyclerBase< QueryRequest >::BasedNode, public glv::utils::ThreadMessage
{
public:
    static const unsigned MessageQueueCapacity = 1;
    static const size_t QueryThreadStackSize = nn::os::ThreadStackAlignment * 3;    // 12KB( 4096 * 3 )
    typedef glv::utils::StaticObjectRecycler< QueryRequest, MessageQueueCapacity > RecyclerType;
    typedef glv::utils::MessageThread< QueryThreadStackSize, MessageQueueCapacity > QueryThreadBaseType;

    //!--------------------------------------------------------------------------------------
    //! @brief
    //!--------------------------------------------------------------------------------------
    explicit QueryRequest( RecyclerType* pOwnerRecycler ) NN_NOEXCEPT
        : RecyclerType::BasedNode( pOwnerRecycler ) {}
};

/**
 * @brief クエリスレッド
 : @details 12KBスタック( from Heap ), 1キュー
 */
class QueryThread : public QueryRequest::QueryThreadBaseType
{
public:
    typedef QueryRequest::QueryThreadBaseType ParentType;

    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ
    //!--------------------------------------------------------------------------------------
    explicit QueryThread() NN_NOEXCEPT : m_pQueryRequest( nullptr )
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 初期化
    //!--------------------------------------------------------------------------------------
    void Initialize() NN_NOEXCEPT
    {
        m_QueryRecycler.Initialize();
        m_ReplyRecycler.Initialize();
        m_ReplyReceiver.Initialize();
        ParentType::InitializeWithRelativePriority( 0 );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 終了
    //!--------------------------------------------------------------------------------------
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        ClearReceivedReplyMessage();
        ParentType::Finalize();
        m_ReplyReceiver.Finalize();
        m_ReplyRecycler.Finalize();
        m_QueryRecycler.Finalize();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief クエリ開始
    //!--------------------------------------------------------------------------------------
    void Query() NN_NOEXCEPT
    {
        auto& query = m_QueryRecycler.Obtain();
        m_pQueryRequest = &query;
        query.Initialize( nullptr, &query );
        Send( query );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief クエリキャンセル
    //!--------------------------------------------------------------------------------------
    void CancelQuery() NN_NOEXCEPT
    {
        QueryRequest* pQuery;
        if ( nullptr != ( pQuery = m_pQueryRequest ) )
        {
            pQuery->Cancel();
        }
        ClearReceivedReplyMessage();        // Replyキューをクリーン
        m_QueryRecycler.WaitCanObtain();    // Queryコンテナ取得可能まで待機
        ClearReceivedReplyMessage();        // Query可能になったら、再度 Replyキューをクリーン
    }

    //!--------------------------------------------------------------------------------------
    //! @brief メッセージ受信通知
    //!--------------------------------------------------------------------------------------
    virtual void OnMessageReceived( glv::utils::ThreadMessage* pMessage ) NN_NOEXCEPT NN_OVERRIDE
    {
        if ( nullptr == pMessage )
        {
            return; // 終了要求
        }

        QueryRequest* pQuery;
        if ( nullptr != ( pQuery = static_cast< QueryRequest* >( pMessage ) ) )
        {
            DEVMENU_LOG_DEBUG( "Start of system program query.\n" );


            ProgramIdNameMap nameMap;
            NN_ABORT_UNLESS_RESULT_SUCCESS( nameMap.Initialize() );

            ncm::ContentMetaDatabase db;
            NN_ABORT_UNLESS_RESULT_SUCCESS(ncm::OpenContentMetaDatabase( &db, ncm::StorageId::BuildInSystem ) );

            std::vector< ncm::ContentMetaKey > list;
            list.resize( MaxShowListCount );
            auto listCount = db.ListContentMeta( list.data(), static_cast< int >( list.size() ), ncm::ContentMetaType::Unknown );
            list.resize( static_cast< size_t >( listCount.listed ) );

            std::vector< nn::ns::ApplicationRecord > recordList( MaxShowListCount );
            const int n = SystemProgramPropertyType::QueryApplicationRecordCount( recordList );
            for ( int i = 0; i < n; ++i )
            {
                if ( false == pQuery->IsCancelled() )
                {
                    SystemProgramPropertyType prop;
                    prop.Prepare( list[ i ], nameMap.GetName( list[ i ].id ) );

                    prop.isQueried = true;
                    QueryReply& reply = m_ReplyRecycler.Obtain();
                    reply.Initialize( prop, i );
                    m_ReplyReceiver.Send( reinterpret_cast< const uintptr_t >( &reply ) );
                }
                else
                {
                    // キャンセルにより中断。
                    DEVMENU_LOG_DEBUG( "Cancel of system program query.\n" );
                    break;
                }
            }
            m_ReplyReceiver.Send( 0U ); // 処理終端メッセージ送信
            pQuery->Recycle();          // クエリ要求ハンドルの返却

            DEVMENU_LOG_DEBUG( "End of system program query.\n" );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief 返答メッセージ受信
    //!--------------------------------------------------------------------------------------
    void ObserveReceiveProperty( PropertyListView::CollectionType* const pItems ) NN_NOEXCEPT
    {
        DoActionReceivedReplyMessage( [ pItems ]( QueryReply* pReply )
        {
            if ( nullptr != pItems )
            {
                pItems->at( pReply->GetIndex() ) = pReply->GetProperty();
            }
        } );
    }

private:
    //!--------------------------------------------------------------------------------------
    //! @brief リプライメッセージの共通処理
    //!--------------------------------------------------------------------------------------
    void DoActionReceivedReplyMessage( std::function< void( QueryReply* ) > doRunnable ) NN_NOEXCEPT
    {
        uintptr_t receive;
        while ( true == m_ReplyReceiver.TryReceive( receive ) )
        {
            if ( 0U == receive )
            {
                m_pQueryRequest = nullptr;
            }
            else
            {
                QueryReply* pReply = reinterpret_cast< QueryReply* >( receive );
                if ( nullptr != doRunnable )
                {
                    doRunnable( pReply );
                }
                pReply->Recycle();
            }
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief キュークリア
    //!--------------------------------------------------------------------------------------
    void ClearReceivedReplyMessage() NN_NOEXCEPT
    {
        DoActionReceivedReplyMessage( nullptr );
    }

    QueryRequest* volatile      m_pQueryRequest;    //!< アクティブなクエリ要求ハンドル
    QueryRequest::RecyclerType  m_QueryRecycler;    //!< クエリ要求ハンドルリサイクラ
    QueryReply::ReplyReceiver   m_ReplyReceiver;    //!< クエリ結果受信機
    QueryReply::RecyclerType    m_ReplyRecycler;    //!< クエリ結果受信ハンドルリサイクラ
};

/**
 * @brief カタログのシーン
 */
class CatalogScene : public Scene
{
public:
    /**
     * @brief コンストラクタ
     */
    CatalogScene( SystemProgramPage* pParentPage, const glv::Rect& rect, InstallScene* pChildScene = nullptr ) NN_NOEXCEPT;

    /**
     * @brief プロパティデータの問い合わせ。
     */
    void QueryProperties() NN_NOEXCEPT;

    /**
     * @brief アプリケーションプロパティコレクションを登録します。
     * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
     */
    void EntryProperties( const PropertyListView::CollectionType* pItems ) NN_NOEXCEPT;

    /**
     * @brief 問い合わせ済プロパティデータの解放。
     */
    void FinalizeProperties() NN_NOEXCEPT;

    /**
     * @brief TORIAEDZU: 返答メッセージ受信
     */
    void ObserveReceiveProperty() NN_NOEXCEPT
    {
        m_QueryThread.ObserveReceiveProperty( m_pItems );
    }

    /**
     * @brief TORIAEDZU: クエリスレッドの終了処理
     */
    void FinalizeQueryThread() NN_NOEXCEPT
    {
        m_QueryThread.Finalize();
    }

    /**
     * @brief 全て削除
     */
    void OnDeleteAllUnknown() NN_NOEXCEPT;

    /**
     * @brief 選択されたシステムプログラムを削除
     */
    void OnDelete() NN_NOEXCEPT;

    /**
     * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    View* GetFocusableChild() NN_NOEXCEPT;

    /**
     * @brief 子シーンを設定します。 TORIAEZU: 複数対応は要検討
     */
    void SetChildScene( InstallScene* pScene ) NN_NOEXCEPT
    {
        m_pChildScene = pScene;
    }

    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    virtual void Refresh() NN_NOEXCEPT final NN_OVERRIDE;

private:
    /**
     * @brief 指定した StorageId の空き容量・総容量の取得
     */
    void GetStorageSize( int64_t* pFreeSpaceSize, int64_t* pTotalSpaceSize, nn::ncm::StorageId id ) NN_NOEXCEPT;

    /**
     * @brief リスト選択結果の受信
     */
    static void OnPropertyListUpdateNotification( const glv::Notification& n ) NN_NOEXCEPT;

    PropertyListView::CollectionType*                  m_pItems;
    PropertyListView*                                  m_pItemListView;
    Button*                                            m_pSwitchButton;
    glv::Label*                                        m_pLabelNoItem;
    InstallScene*                                      m_pChildScene;
    SystemProgramPage*                                 m_pParentPage;
    QueryThread                                        m_QueryThread;
    std::function< void( const glv::Notification& ) >  m_ListUpdateCallback;
    ProgramIdNameMap                                   m_NameMap;
    nn::os::SystemEvent                                m_systemEvent;
};



class SystemProgramPage : public devmenu::Page
{
public:
    /**
     * @brief コンストラクタです。
     */
    SystemProgramPage( int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect ) NN_NOEXCEPT;

    /**
     * @brief ページがコンテナに追加された後に呼び出されます。
     */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページがコンテナから削除された前に呼び出されます。
     */
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページがアクティブ( 選択により表示開始 )になる際に呼び出されます。
     */
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページがディアクティブ( 選択により非表示開始 )になる際に呼び出されます。
     */
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief バックグラウンド遷移処理です。
     */
    void OnChangeIntoBackground() NN_NOEXCEPT NN_OVERRIDE;
    /**
     * @brief フォアグラウンド遷移処理です。
     */
    void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
     */
    virtual void OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    virtual View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief シーンを切替えます。
     */
    void SwitchScene( devmenu::Scene* current, devmenu::Scene* next, bool isBack ) NN_NOEXCEPT;

    /**
     * @brief プロパティデータの問い合わせ。
     */
    void QueryProperties() NN_NOEXCEPT;

private:
    void EntryProperties( const PropertyListView::CollectionType* pItems ) NN_NOEXCEPT;

    /**
     * @brief 問い合わせ済プロパティデータの解放。
     */
    void FinalizeProperties() NN_NOEXCEPT;

    CatalogScene*                                   m_pCatalogScene;
    InstallScene*                                   m_pInstallScene;
};

}}} // ~namespace devmenu::system::program, ~namespace devmenu::system, ~namespace devmenu
