﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>

#include "DevMenu_SoundUtil.h"

#define DEVMENU_SOUND_WAV_THROW_UNLESS(condition, result) \
    if (condition) {} \
    else \
    { \
        return (result); \
    }

namespace devmenu {

namespace {

    struct WavRiff
    {
        uint32_t tag;
        uint32_t size;
        uint32_t type;
    };

    struct WavChunk
    {
        uint32_t tag;
        uint32_t size;
    };

    struct WavFmt
    {
        uint16_t fmtId;
        uint16_t channel;
        uint32_t sampleRate;
        uint32_t transRate;
        uint16_t blockSize;
        uint16_t bitsPerSample;
    };

    inline uint16_t ReadLittleEndianUint16(const void* buffer)
    {
        auto p = static_cast<const uint8_t*>(buffer);
        return static_cast<uint16_t>(p[0] << 0)
            | static_cast<uint16_t>(p[1] << 8);
    }

    inline uint32_t ReadLittleEndianUint32(const void* buffer)
    {
        auto p = static_cast<const uint8_t*>(buffer);
        return static_cast<uint32_t>(p[0] << 0)
            | static_cast<uint32_t>(p[1] << 8)
            | static_cast<uint32_t>(p[2] << 16)
            | static_cast<uint32_t>(p[3] << 24);
    }

} // end of unnamed namespace

WavResult ParseWavFormat(WavFormat* pOutWavFormat, const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutWavFormat);
    NN_ASSERT_NOT_NULL(buffer);

    auto p = static_cast<const char*>(buffer);
    int64_t readOffset = 0;

    // riff header
    WavRiff riff;
    DEVMENU_SOUND_WAV_THROW_UNLESS(size - readOffset >= sizeof(riff), WavResult_InsufficientBuffer);
    riff.tag = ReadLittleEndianUint32(p + readOffset);
    readOffset += sizeof(riff.tag);
    riff.size = ReadLittleEndianUint32(p + readOffset);
    readOffset += sizeof(riff.size);
    riff.type = ReadLittleEndianUint32(p + readOffset);
    readOffset += sizeof(riff.type);
    DEVMENU_SOUND_WAV_THROW_UNLESS(riff.tag == ReadLittleEndianUint32("RIFF") && riff.type == ReadLittleEndianUint32("WAVE"), WavResult_InvalidDataFormat);

    int64_t dataOffset = 0;
    int64_t dataSize = 0;
    int32_t sampleRate = 0;
    int16_t channelCount = 0;
    int16_t bitsPerSample = 0;
    WavEndian endian = WavEndian_Little;
    bool isFmtChunkFound = false;
    bool isDataChunkFound = false;
    while (!isFmtChunkFound || !isDataChunkFound)
    {
        WavChunk chunk;
        DEVMENU_SOUND_WAV_THROW_UNLESS(size - readOffset >= sizeof(chunk), WavResult_InsufficientBuffer);
        chunk.tag = ReadLittleEndianUint32(p + readOffset);
        readOffset += sizeof(chunk.tag);
        chunk.size = ReadLittleEndianUint32(p + readOffset);
        readOffset += sizeof(chunk.size);

        // "fmt "
        if (chunk.tag == ReadLittleEndianUint32("fmt "))
        {
            WavFmt fmt;
            DEVMENU_SOUND_WAV_THROW_UNLESS(size - readOffset >= sizeof(fmt), WavResult_InsufficientBuffer);
            fmt.fmtId = ReadLittleEndianUint16(p + readOffset);
            readOffset += sizeof(fmt.fmtId);
            fmt.channel = ReadLittleEndianUint16(p + readOffset);
            readOffset += sizeof(fmt.channel);
            fmt.sampleRate = ReadLittleEndianUint32(p + readOffset);
            readOffset += sizeof(fmt.sampleRate);
            fmt.transRate = ReadLittleEndianUint32(p + readOffset);
            readOffset += sizeof(fmt.transRate);
            fmt.blockSize = ReadLittleEndianUint16(p + readOffset);
            readOffset += sizeof(fmt.blockSize);
            fmt.bitsPerSample = ReadLittleEndianUint16(p + readOffset);
            readOffset += sizeof(fmt.bitsPerSample);

            channelCount = fmt.channel;
            sampleRate = fmt.sampleRate;
            bitsPerSample = fmt.bitsPerSample;
            switch (fmt.fmtId)
            {
                case 0x1:
                {
                    endian = WavEndian_Little;
                    break;
                }
                case 0xfe02:
                {
                    endian = WavEndian_Big;
                    break;
                }
                default:
                {
                    return WavResult_InvalidDataFormat;
                }
            }

            // discard extention area
            if (chunk.size > sizeof(fmt))
            {
                uint16_t ext_size;
                DEVMENU_SOUND_WAV_THROW_UNLESS(size - readOffset >= sizeof(ext_size), WavResult_InsufficientBuffer);
                ext_size = ReadLittleEndianUint16(p + readOffset);
                DEVMENU_SOUND_WAV_THROW_UNLESS(size - readOffset >= sizeof(ext_size) + ext_size, WavResult_InsufficientBuffer);
                readOffset += sizeof(ext_size) + ext_size;
            }
            isFmtChunkFound = true;
        }
        // "data"
        else if (chunk.tag == ReadLittleEndianUint32("data"))
        {
            dataOffset = readOffset;
            dataSize = chunk.size;
            isDataChunkFound = true;
        }
        // skip others
        else
        {
            DEVMENU_SOUND_WAV_THROW_UNLESS(size - readOffset >= chunk.size, WavResult_InsufficientBuffer);
            readOffset += chunk.size;
        }
    }

    pOutWavFormat->dataOffset = dataOffset;
    pOutWavFormat->dataSize = dataSize;
    pOutWavFormat->sampleRate = sampleRate;
    pOutWavFormat->channelCount = channelCount;
    pOutWavFormat->bitsPerSample = bitsPerSample;
    pOutWavFormat->endian = endian;

    return WavResult_Success;
}

} // ~namespace devmenu
