﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>
#include <nn/am/service/am_SystemProgramIds.h>

#include "Accounts/DevMenu_AccountsSceneList.h"
#include "Common/DevMenu_CommonDropDown.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "DevMenu_RootSurface.h"
#include "DevMenu_ShopUserSelector.h"
#include "DevMenuCommand_DebugCommand.h"
#include "DevMenuCommand_ShopCommand.h"

#include "DevMenu_Common.h"

namespace devmenu { namespace shop {

namespace {
    // シザーボックスの領域設定
    const glv::space_t HeaderRegionLength = 80.0f;

    // リスト要素のパディング
    const glv::space_t ListMarginLeft = 4.0f;
    const glv::space_t ListMarginRight = 4.0f;
}

/*********************************
 * class SwitchShopDropDown
 *********************************/
class SwitchShopDropDown : public DropDownBase
{
public:
    SwitchShopDropDown( const glv::Rect& rect, float textSize ) NN_NOEXCEPT
        : DropDownBase( rect, textSize )
    {
        addItem( "Dev Shop" );
        addItem( "Shop" );

        char shopIdStr[32];
        char shopAppletIdStr[32];
        GetVariableSizeFirmwareDebugSettingsItemValue( shopIdStr, sizeof( shopIdStr ), "ns.applet", "shop_applet_id" );
        nn::util::SNPrintf( shopAppletIdStr, sizeof( shopAppletIdStr ), "0x%016llx", nn::am::service::ProgramId_LibraryAppletShop.value );
        if ( nn::util::Strncmp( shopIdStr, shopAppletIdStr, sizeof( shopIdStr ) ) == 0 )
        {
            setValue( "Shop" );
        }

        attach( []( const glv::Notification& notification )->void { notification.receiver< SwitchShopDropDown >()->UpdateEnabledShop(); }, glv::Update::Action, this );
    }

private:
    enum ShopType
    {
        ShopType_DevShop,
        ShopType_Shop
    };

private:
    nn::Result RequestSwitchShopCommand( bool* pOutValue, const char* shopName ) NN_NOEXCEPT
    {
        bool isSuccess = true;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // DevMenuCommand の "debug switch-shop コマンド" を実行する
        const char* argv[] = { "DevMenu", "debug", "switch-shop", shopName };
        Option option( NN_ARRAY_SIZE( argv ), const_cast<char**>(argv) );
        NN_RESULT_DO( DebugCommand( &isSuccess, option ) );
#endif
        *pOutValue = isSuccess;

        NN_RESULT_SUCCESS;
    }

    void UpdateEnabledShop() NN_NOEXCEPT
    {
        bool isSuccess = false;
        nn::Result result;
        if ( mSelectedItem == ShopType_DevShop )
        {
            // Dev Shop へ更新
            result = RequestSwitchShopCommand( &isSuccess, "devshop" );
        }
        else
        {
            // Shop へ更新
            result = RequestSwitchShopCommand( &isSuccess, "shop" );
        }

        if ( result.IsFailure() || isSuccess == false )
        {
            DEVMENU_LOG( "Switching shop failed\n" );
            if ( mSelectedItem == ShopType_DevShop )
            {
                // Shop へ戻す
                setValue( "Shop" );
            }
            else
            {
                // Dev Shop へ戻す
                setValue( "Dev Shop" );
            }
        }
    }
};

/*********************************
 * class ShopScene
 *********************************/
class ShopScene : public ProgressModalSceneBase
{
public:
    /**
     * @brief コンストラクタ
     */
    ShopScene( Page* pParentPage, const glv::Rect& rect )
        : ProgressModalSceneBase( pParentPage, rect, true )
    {
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        NN_UNUSED( events );

        this->UpdateExecutionProgress();
    }

    virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE
    {
    }
};

/*********************************
 * class ShopPage
 *********************************/
class ShopPage : public Page
{
public:
    /**
     * @brief コンストラクタ
     */
    ShopPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        : Page(pageId, pageCaption, rect ),
        m_ShopScene( this, rect ),
        m_SwitchShopDropDown( glv::Rect( 160.0f, 35.0f ), CommonValue::InitialFontSize ),
        m_ButtonShopStart( "Start", [&] { StartFirstAccessToShopAndRegisterDevice(); }, glv::Rect( 200.0f, 40.0f ), glv::Place::TR ),
        m_ButtonShopLinkDevice( "Start", [&] { LinkDevice(); }, glv::Rect( 200.0f, 40.0f ), glv::Place::TR ),
        m_ButtonShopUnlinkDevice( "Start", [&] { UnlinkDevice(); }, glv::Rect( 200.0f, 40.0f ), glv::Place::TR ),
        m_ButtonShopUnlinkDeviceAll( "Start", [&] { DisplayUnlinkDeviceAllDialog(); }, glv::Rect( 200.0f, 40.0f ), glv::Place::TR ),
        m_ButtonShopDeviceLinkStatus( "Start", [&] { RequestDeviceLinkStatus(); }, glv::Rect( 200.0f, 40.0f ), glv::Place::TR )
    {
        m_ShopScene.pos( m_ShopScene.left(), m_ShopScene.top() );
        *this << m_ShopScene;

        // Switching Shops ヘッダ
        auto pHeaderSwitchingShops = new Header( new glv::Label( "Switching Shops", glv::Label::Spec( glv::Place::TL, ListMarginLeft, 17.0f, CommonValue::InitialFontSize ) ),
            glv::Rect( width() - ( ListMarginLeft + ListMarginRight + 30.0f ), HeaderRegionLength ) );
        pHeaderSwitchingShops->anchor( glv::Place::TL ).pos( ListMarginLeft + 9.0f, 17.0f );
        m_ShopScene << pHeaderSwitchingShops;

        // ラベル
        auto pLabelSwitchShop = new glv::Label( "Valid Shop", glv::Label::Spec(glv::Place::TL, ListMarginLeft + 18.0f, pHeaderSwitchingShops->bottom(), CommonValue::InitialFontSize) );
        m_ShopScene << pLabelSwitchShop;

        // ドロップダウン
        m_SwitchShopDropDown.anchor( glv::Place::TR ).pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pLabelSwitchShop->top() - 4.0f );
        m_SwitchShopDropDown.enable( glv::Property::KeepWithinParent );
        m_ShopScene << m_SwitchShopDropDown;

        // Registering an Active Console ヘッダ
        auto pHeaderRegisteringAnShop = new Header( new glv::Label( "Registering an Active Console", glv::Label::Spec( glv::Place::TL, ListMarginLeft, pLabelSwitchShop->bottom() + 80.0f, CommonValue::InitialFontSize ) ),
            glv::Rect( width() - ( ListMarginLeft + ListMarginRight + 30.0f ), HeaderRegionLength ) );
        pHeaderRegisteringAnShop->anchor( glv::Place::TL ).pos( ListMarginLeft + 9.0f, pLabelSwitchShop->bottom() + 80.0f );
        *this << pHeaderRegisteringAnShop;

        // ラベル
        auto pLabelShopStart = new glv::Label( "Initial Settings and Registering an Active Console", glv::Label::Spec(glv::Place::TL, ListMarginLeft + 18.0f, pHeaderRegisteringAnShop->bottom(), CommonValue::InitialFontSize) );
        auto pLabelShopLinkDevice = new glv::Label( "Registering an Active Console", glv::Label::Spec(glv::Place::TL, ListMarginLeft + 18.0f, pLabelShopStart->bottom() + 20.0f, CommonValue::InitialFontSize) );
        auto pLabelShopUnlinkDevice = new glv::Label( "Unregistering an Active Console", glv::Label::Spec(glv::Place::TL, ListMarginLeft + 18.0f, pLabelShopLinkDevice->bottom() + 20.0f, CommonValue::InitialFontSize) );
        auto pLabelShopUnlinkDeviceAll = new glv::Label( "Unregistering All Active Consoles", glv::Label::Spec(glv::Place::TL, ListMarginLeft + 18.0f, pLabelShopUnlinkDevice->bottom() + 20.0f, CommonValue::InitialFontSize) );
        auto pLabelShopDeviceLinkStatus = new glv::Label( "Checking the Active Console Registration Status", glv::Label::Spec(glv::Place::TL, ListMarginLeft + 18.0f, pLabelShopUnlinkDeviceAll->bottom() + 20.0f, CommonValue::InitialFontSize) );
        m_ShopScene
            << pLabelShopStart
            << pLabelShopLinkDevice
            << pLabelShopUnlinkDevice
            << pLabelShopUnlinkDeviceAll
            << pLabelShopDeviceLinkStatus;

        // ボタン
        m_ButtonShopStart.pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pLabelShopStart->top() - 4.0f );
        m_ButtonShopLinkDevice.pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pLabelShopLinkDevice->top() - 4.0f );
        m_ButtonShopUnlinkDevice.pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pLabelShopUnlinkDevice->top() - 4.0f );
        m_ButtonShopUnlinkDeviceAll.pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pLabelShopUnlinkDeviceAll->top() - 4.0f );
        m_ButtonShopDeviceLinkStatus.pos( glv::Place::TR, -( ListMarginRight + 16.0f ), pLabelShopDeviceLinkStatus->top() - 4.0f );
#if defined( APPLICATION_BUILD )
        m_ButtonShopUnlinkDevice.UpdateFocusAndColor( false, false );
        m_ButtonShopUnlinkDeviceAll.UpdateFocusAndColor( false, false );
#endif
        m_ShopScene
         << m_ButtonShopStart
         << m_ButtonShopLinkDevice
         << m_ButtonShopUnlinkDevice
         << m_ButtonShopUnlinkDeviceAll
         << m_ButtonShopDeviceLinkStatus;
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        m_ShopScene.OnLoopBeforeSceneRenderer( context, events );
    }

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return static_cast< glv::View* >( &m_SwitchShopDropDown );
    }

private:
    /*********************************
     * class Header
     *********************************/
    class Header : public glv::Group
    {
    public:
        Header( glv::Label* pPageTitleLabel, const glv::Rect& rect ) NN_NOEXCEPT
            : glv::Group( rect )
        {
            // Update the header styles
            NN_FUNCTION_LOCAL_STATIC( glv::Style, s_HeaderStyle );
            s_HeaderStyle.color.text.set( 0.6f, 0.6f, 0.6f );
            auto pPageTitle = pPageTitleLabel;
            pPageTitle->style( &s_HeaderStyle );

            auto pTitleTable = new glv::Table( "<", 0.0f, 0.0f, glv::Rect( rect.width(), rect.height() ) );
            *pTitleTable << pPageTitle;

            // Creates a Divider
            auto pHeaderDivider = new glv::Divider( 10.0f, 2.0f );
            *pTitleTable << pHeaderDivider;
            pTitleTable->arrange();
            *this << pTitleTable;
        }
    };

private:
    void StartFirstAccessToShopAndRegisterDevice() NN_NOEXCEPT
    {
        auto pView = new UserAccountSelector(
            [&]( int userAccountIndex )
        {
            m_ShopScene.StartProgressModalThread(
                "First connection and registering",
                [ userAccountIndex ]( bool* pOutIsSuccess ) -> nn::Result
            {
                // DevMenuCommand の "shop start コマンド" を実行する
                std::string userAccountIndexString = std::to_string( userAccountIndex );
                const char* argv[] = { "DevMenu", "shop", "start", userAccountIndexString.data() };
                Option option( NN_ARRAY_SIZE( argv ), const_cast< char** >( argv ) );
                return ShopCommand( pOutIsSuccess, option );
            }, [&]( const nn::Result& result, bool isSuccess ) { OnCompleteShopStartProgress( result, isSuccess ); });
        }
        );
        GetRootSurfaceContext()->StartModal( pView, true );
    }

    void LinkDevice() NN_NOEXCEPT
    {
        auto pView = new UserAccountSelector( [&]( int userAccountIndex )
            {
                m_ShopScene.StartProgressModalThread(
                    "Registering",
                    [ userAccountIndex ]( bool* pOutIsSuccess ) -> nn::Result
                        {
                            // DevMenuCommand の "shop link-device コマンド" を実行する
                            std::string userAccountIndexString = std::to_string( userAccountIndex );
                            const char* argv[] = { "DevMenu", "shop", "link-device", userAccountIndexString.data() };
                            Option option( NN_ARRAY_SIZE( argv ), const_cast< char** >( argv ) );
                            return ShopCommand( pOutIsSuccess, option );
                        },
                    [&]( const nn::Result& result, bool isSuccess ){ OnCompleteShopLinkDeviceProgress( result, isSuccess ); }
                );
            }
        );
        GetRootSurfaceContext()->StartModal( pView, true );
    }

    void UnlinkDevice() NN_NOEXCEPT
    {
        auto pView = new UserAccountSelector([&]( int userAccountIndex )
            {
                m_ShopScene.StartProgressModalThread(
                    "Unregistering",
                    [ userAccountIndex ]( bool* pOutIsSuccess ) -> nn::Result
                        {
                            // DevMenuCommand の "shop unlink-device コマンド" を実行する
                            std::string userAccountIndexString = std::to_string( userAccountIndex );
                            const char* argv[] = { "DevMenu", "shop", "unlink-device", userAccountIndexString.data() };
                            Option option( NN_ARRAY_SIZE( argv ), const_cast< char** >( argv ) );
                            return ShopCommand( pOutIsSuccess, option );
                        },
                    [&]( const nn::Result& result, bool isSuccess ) { OnCompleteShopUnlinkDeviceProgress( result, isSuccess ); }
                );
            }
        );
        GetRootSurfaceContext()->StartModal( pView, true );
    }

    void RequestDeviceLinkStatus() NN_NOEXCEPT
    {
        auto pView = new UserAccountSelector([&]( int userAccountIndex )
            {
                m_ShopScene.StartProgressModalThread(
                    "Registration status request",
                    [ this, userAccountIndex ]( bool* pOutIsSuccess ) -> nn::Result
                        {
                            // DevMenuCommand の "shop device-link-status コマンド" を実行する
                            std::string userAccountIndexString = std::to_string( userAccountIndex );
                            const char* argv[] = { "DevMenu", "shop", "device-link-status", userAccountIndexString.data() };
                            Option option( NN_ARRAY_SIZE( argv ), const_cast< char** >( argv ) );
                            return DisplayLinkedDeviceStatus( pOutIsSuccess, m_RegistrationStatusStr, sizeof( m_RegistrationStatusStr ), option );
                        },
                    [&]( const nn::Result& result, bool isSuccess ) { OnCompleteShopDeviceLinkStatusProgress( result, isSuccess ); }
                );
            }
        );
        GetRootSurfaceContext()->StartModal( pView, true );
    }

    void DisplayFinishedMessage( const nn::Result& result, bool isSuccess, const std::string& completeMessage, const std::string& failureMessage ) NN_NOEXCEPT
    {
        if ( result.IsSuccess() && isSuccess == true )
        {
            // 成功メッセージを表示
            m_ShopScene.SetModalViewMessage( completeMessage );
        }
        else
        {
            // 失敗メッセージを表示
            if ( result.IsFailure() )
            {
                char errorMessage[64] = {};
                nn::util::SNPrintf( errorMessage, sizeof( errorMessage ), "%s (0x%08x)", failureMessage.c_str(), result.GetInnerValueForDebug() );
                m_ShopScene.SetModalViewMessage( std::string( errorMessage ) );
            }
            else
            {
                m_ShopScene.SetModalViewMessage( failureMessage );
            }
        }
    }

    void OnCompleteShopStartProgress( const nn::Result& result, bool isSuccess ) NN_NOEXCEPT
    {
        DisplayFinishedMessage( result, isSuccess, "First access to shop and registration of this device have completed.", "Failed to register this device to shop" );
    }

    void OnCompleteShopLinkDeviceProgress( const nn::Result& result, bool isSuccess ) NN_NOEXCEPT
    {
        DisplayFinishedMessage( result, isSuccess, "Registration has completed", "Failed to register" );
    }

    void OnCompleteShopUnlinkDeviceProgress( const nn::Result& result, bool isSuccess ) NN_NOEXCEPT
    {
        DisplayFinishedMessage( result, isSuccess, "Unregistration has completed", "Failed to unregister" );
    }

    void OnCompleteShopDeviceLinkStatusProgress( const nn::Result& result, bool isSuccess ) NN_NOEXCEPT
    {
        DisplayFinishedMessage( result, isSuccess, std::string( m_RegistrationStatusStr ), "Failed to get registration status" );
    }

    void DisplayUnlinkDeviceAllDialog() NN_NOEXCEPT
    {
        auto pView = new MessageView( true );
        pView->AddMessage( "Are you sure you want to unregister all active consoles registered on that system?" );

        pView->AddButton( "Cancel" );
        pView->AddButton(
            "OK",
            [&] ( void* pParam, nn::TimeSpan& timespan )
            {
                // DevMenuCommand の "shop unlink-device-all コマンド" を実行する
                bool isSuccess = false;
                const char* argv[] = { "DevMenu", "shop", "unlink-device-all" };
                Option option( NN_ARRAY_SIZE( argv ), const_cast< char** >( argv ) );
                auto result = ShopCommand( &isSuccess, option );

                auto pView = new MessageView( false );
                if ( result.IsSuccess() && isSuccess == true )
                {
                    pView->AddMessage( "Unlink Completed" );
                }
                else
                {
                    char resultValueStr[20] = { '\0' };
                    char errorMessage[64];
                    if ( result.IsFailure() )
                    {
                        nn::util::SNPrintf( resultValueStr, sizeof( resultValueStr ), " (0x%08x)", result.GetInnerValueForDebug() );
                    }
                    nn::util::SNPrintf( errorMessage, sizeof( errorMessage ), "Failed to unlink all devices%s", resultValueStr );
                    pView->AddMessage( std::string( errorMessage ) );
                }
                pView->AddButton( "Close" );
                GetRootSurfaceContext()->StartModal( pView, true, true );
            },
            nullptr,
            MessageView::ButtonTextColor::Green
        );

        GetRootSurfaceContext()->StartModal( pView, true );
    }

private:
    ShopScene                                       m_ShopScene;
    SwitchShopDropDown                              m_SwitchShopDropDown;
    devmenu::Button                                 m_ButtonShopStart;
    devmenu::Button                                 m_ButtonShopLinkDevice;
    devmenu::Button                                 m_ButtonShopUnlinkDevice;
    devmenu::Button                                 m_ButtonShopUnlinkDeviceAll;
    devmenu::Button                                 m_ButtonShopDeviceLinkStatus;
    char                                            m_RegistrationStatusStr[128];
};

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class ShopPageCreator : PageCreatorBase
{
public:
    explicit ShopPageCreator( const char* pageName ) NN_NOEXCEPT
        : PageCreatorBase( ID, pageName ) {}

protected:
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const auto& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const auto width = static_cast< glv::space_t >( resolution[ 0 ] );
        const auto height = static_cast< glv::space_t >( resolution[ 1 ] );
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new ShopPage( ID, GLV_TEXT_API_WIDE_STRING( "Shop" ), pageBounds );
    }
};

/**
 * @brief Declaration for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) ShopPageCreator< _id > g_ShopPageCreator##_id( _name );
LOCAL_PAGE_CREATOR( DevMenuPageId_Shop, "Shop" );

}} // ~namespace devmenu::shop, ~namespace devmenu
