﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>

#include "Common/DevMenu_CommonDropDown.h"
#include "SaveData/DevMenu_SaveData.h"
#include "SaveData/DevMenu_SaveDataProperty.h"
#include "SaveData/DevMenu_SaveDataListScene.h"
#include "SaveData/DevMenu_SaveDataDetailScene.h"
#include "SaveData/DevMenu_SaveDataImporter.h"

namespace devmenu { namespace savedata {

namespace {

    const char* SceneTitles[] = {
        "User",
#if defined ( NN_DEVMENUSYSTEM )
        "System Save Data",
#endif
        "Backup in SD card",
    };

}

/*********************************
 * clas SaveDataCatalogPage
 *********************************/

class SaveDataCatalogPage : public Page
{
public:
    /**
     * @brief コンストラクタです。
     */
    SaveDataCatalogPage( int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect ) NN_NOEXCEPT
        : Page( pageId, pageCaption, rect )
        , m_Op( *this )
        , m_DropDown(this, { SceneTitles, NN_ARRAY_SIZE(SceneTitles) })
        , m_ApplicationSaveListScene( m_Op, this, glv::Rect( w, h - SaveDataCatalogPage::VerticalLengthDropdown ) )
    #if defined ( NN_DEVMENUSYSTEM )
        , m_SystemSaveListScene( m_Op, this, glv::Rect( w, h - SaveDataCatalogPage::VerticalLengthDropdown ) )
    #endif
        , m_DetailScene( m_Op, this, rect, [&] { m_Op.CloseSaveDataDetails( m_DetailScene.GetParentScene() ); } )
        , m_TypeListScene( m_Op, this, glv::Rect( w, h - SaveDataCatalogPage::VerticalLengthDropdown + 20.0f )
            , [&] { m_Op.CloseSaveDataTypeListScene( SceneType_ApplicationSaveDataList ); } )
        , m_ImportScene( this, glv::Rect( w, h - SaveDataCatalogPage::VerticalLengthDropdown + 20.0f ) )
    {
    }

    /**
     * @brief ページがコンテナに追加された後に呼び出されます。
     */
    void OnAttachedPage() NN_NOEXCEPT
    {
        *this << m_DropDown;

        // Attach a Click handler to detect if the B button was pressed
        m_DropDown.attach( []( const glv::Notification& notification )->void {
                auto pPage = notification.receiver< Page >();

                auto& glvRoot = reinterpret_cast< glv::GLV& >( pPage->root() );
                if ( glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::B >() ||
                    glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::B >())
                {
                    // Call the given function when the B button is detected
                    pPage->GetRootSurfaceContext()->MoveFocusToMenuTabs();
                }
            }, glv::Update::Clicked, this );

        // Set the page to be able to detect clicks
        //this->enable( glv::Property::HitTest | glv::Property::GestureDetectability );

        this->SetActiveScene( SceneType_ApplicationSaveDataList );

        m_ApplicationSaveListScene.pos( m_ApplicationSaveListScene.left(), m_ApplicationSaveListScene.top() + SaveDataCatalogPage::VerticalLengthDropdown );
        m_ApplicationSaveListScene.enable( glv::Property::Visible );
        *this << m_ApplicationSaveListScene;

    #if defined ( NN_DEVMENUSYSTEM )
        m_SystemSaveListScene.pos( m_SystemSaveListScene.left(), m_SystemSaveListScene.top() + SaveDataCatalogPage::VerticalLengthDropdown );
        *this << m_SystemSaveListScene;
    #endif

        m_ImportScene.pos( m_ImportScene.left(), m_ImportScene.top() + SaveDataCatalogPage::VerticalLengthDropdown );
        *this << m_ImportScene;

        m_DetailScene.pos( m_DetailScene.left(), m_DetailScene.top() );
        *this << m_DetailScene;

        m_TypeListScene.pos( m_TypeListScene.left(), m_TypeListScene.top() );
        *this << m_TypeListScene;
    }

    /**
     * @brief ページがコンテナから削除された前に呼び出されます。
     */
    void OnDetachedPage() NN_NOEXCEPT
    {
        m_ApplicationSaveListScene.FinalizeProperties();
    }

    /**
     * @brief ページがアクティブ( 選択により表示開始 )になる際に呼び出されます。
     */
    void OnActivatePage() NN_NOEXCEPT
    {
        m_ApplicationSaveListScene.QueryProperties();
    }

    /**
     * @brief ページがディアクティブ( 選択により非表示開始 )になる際に呼び出されます。
     */
    void OnDeactivatePage() NN_NOEXCEPT
    {
        // TOEIAEZU: 一旦コメントアウト。見た目に影響しないなら不要。
        // m_pLabelNoItem->disable( glv::Property::t::Visible );
    }

    /**
     * @brief フォアグラウンド遷移処理です。
     */
    void OnChangeIntoForeground() NN_NOEXCEPT
    {
        m_ApplicationSaveListScene.QueryProperties();
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
    {
        m_ApplicationSaveListScene.OnLoopBeforeSceneRenderer( context, events );
    #if defined ( NN_DEVMENUSYSTEM )
        m_SystemSaveListScene.OnLoopBeforeSceneRenderer( context, events );
    #endif

        m_ImportScene.OnLoopBeforeSceneRenderer( context, events );
        m_TypeListScene.OnLoopBeforeSceneRenderer( context, events );
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
     */
    void OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
    {
        NN_UNUSED( context );
        NN_UNUSED( events );
    }

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    glv::View* GetFocusableChild() NN_NOEXCEPT
    {
        return &m_DropDown;
    }

    virtual Scene* GetScene( int sceneIndex ) NN_NOEXCEPT
    {
        switch ( sceneIndex )
        {
        case SceneType_ApplicationSaveDataList:
            return &m_ApplicationSaveListScene;
    #if defined ( NN_DEVMENUSYSTEM )
        case SceneType_SystemSaveDataList:
            return &m_SystemSaveListScene;
    #endif
        case SceneType_ExportedSaveDataList:
            return &m_ImportScene;
        case SceneType_SaveDataDetail:
            return &m_DetailScene;
        case SceneType_SaveDataTypeList:
            return &m_TypeListScene;
        default:
            NN_ABORT( "Invalid scene" );
        }
    }

private:
    /**
     * @brief These are the operations that the sub-scenes can perform, that are best handled at this level.
     *
     *        For example, when the UserSaveListScene wants to show the SaveDataDetailScene, it calls back through ShowSaveDataDetails(), which will do the appropriate action.
     */
    struct Operators : public AbstractOperators
    {
        SaveDataCatalogPage& parent;
        explicit Operators( SaveDataCatalogPage& parentPage ) NN_NOEXCEPT :
            parent( parentPage )
        {
        };

        virtual void UpdateSaveDataDetails( const SaveDataPropertyType* const pSaveDataProperty ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_DetailScene.UpdateDetails( pSaveDataProperty );
        }

        virtual void ShowSaveDataDetails( SceneType currentListScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_DropDown.disable( glv::Property::Visible );
            parent.m_DetailScene.SetParentScene( currentListScene );
            parent.SwitchScene( SceneType_SaveDataDetail, false );
        }

        virtual void CloseSaveDataDetails( SceneType previousScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            // Only enable the Drop Down if we are NOT going to the Save Data list page
            if ( previousScene != SceneType_SaveDataTypeList )
            {
                parent.m_DropDown.enable( glv::Property::Visible );
            }
            parent.SwitchScene( previousScene, true );
        }

        virtual void UpdateSaveDataTypeList( const nn::ncm::ApplicationId& applicationId ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_TypeListScene.UpdateSelectedApplicationInformation( applicationId );
        }

        virtual void ShowSaveDataTypeListScene( SceneType currentListScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            NN_UNUSED( currentListScene );
            parent.m_DropDown.disable( glv::Property::Visible );
            parent.SwitchScene( SceneType_SaveDataTypeList );
        }

        virtual void CloseSaveDataTypeListScene( SceneType previousScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            parent.m_DropDown.enable( glv::Property::Visible );
            parent.SwitchScene( previousScene );
        }

        virtual void ExportSaveData( SceneType currentListScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            switch ( currentListScene )
            {
            case SceneType_SaveDataTypeList:
                parent.m_TypeListScene.ExportSaveData();
                break;
#if defined ( NN_DEVMENUSYSTEM )
            case SceneType_SystemSaveDataList:
                parent.m_SystemSaveListScene.ExportSaveData();
                break;
#endif
            default:
                DEVMENU_LOG( "Warning:  Attempted to export save data from an invalid scene (%d)\n", currentListScene );
                break;
            }
        }

        virtual void VerifySaveData( SceneType currentListScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            switch ( currentListScene )
            {
            case SceneType_SaveDataTypeList:
                parent.m_TypeListScene.VerifySaveData();
                break;
#if defined ( NN_DEVMENUSYSTEM )
            case SceneType_SystemSaveDataList:
                parent.m_SystemSaveListScene.VerifySaveData();
                break;
#endif
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        virtual void DeleteSaveData( SceneType currentListScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            switch ( currentListScene )
            {
            case SceneType_SaveDataTypeList:
                parent.m_TypeListScene.DeleteSaveData();
                break;
#if defined ( NN_DEVMENUSYSTEM )
            case SceneType_SystemSaveDataList:
                parent.m_SystemSaveListScene.DeleteSaveData();
                break;
#endif
            default:
                DEVMENU_LOG( "Warning:  Attempted to delete save data from an invalid scene (%d)\n", currentListScene );
                break;
            }
        }

        virtual void CorruptSaveData( SceneType currentListScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
            // The functionality to actually corrupt save data should only be available in DevMenuSystem.
#if defined ( NN_DEVMENUSYSTEM )
            switch ( currentListScene )
            {
            case SceneType_SaveDataTypeList:
                parent.m_TypeListScene.CorruptSaveData();
                break;
            case SceneType_SystemSaveDataList:
                parent.m_SystemSaveListScene.CorruptSaveData();
                break;
            default:
                DEVMENU_LOG( "Warning:  Attempted to delete save data from an invalid scene (%d)\n", currentListScene );
                break;
            }
#endif
        }

        virtual void ExportRawSaveData( SceneType currentListScene ) const NN_NOEXCEPT final NN_OVERRIDE
        {
#if defined ( NN_DEVMENUSYSTEM )
            switch ( currentListScene )
            {
            case SceneType_SaveDataTypeList:
                parent.m_TypeListScene.ExportRawSaveData();
                break;
            case SceneType_SystemSaveDataList:
                parent.m_SystemSaveListScene.ExportRawSaveData();
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
#endif
        }

    } m_Op;


private:
    static const glv::space_t VerticalLengthDropdown;

private:
    SceneSwitchDropDown            m_DropDown;
    ApplicationSaveDataListScene   m_ApplicationSaveListScene;

#if defined ( NN_DEVMENUSYSTEM )
    SystemSaveDataListScene        m_SystemSaveListScene;
#endif

    DetailScene                    m_DetailScene;
    SaveDataTypeListScene          m_TypeListScene;
    ImportScene                    m_ImportScene;
};

const glv::space_t SaveDataCatalogPage::VerticalLengthDropdown = 48.0f;

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class SaveDataCatalogPageCreator : PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit SaveDataCatalogPageCreator( const char* pageName ) NN_NOEXCEPT
        : PageCreatorBase( ID, pageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const glv::space_t width = static_cast< glv::space_t >( resolution[ 0 ] );
        const glv::space_t height = static_cast< glv::space_t >( resolution[ 1 ] );
        //const glv::Rect pageBounds( width - ( ( 12.0f ) * 2.0f ), height - 118.0f );    // 横は 8 + 4 マージン
        const glv::Rect pageBounds( width - 218.0f, height - 118.0f );
        return new SaveDataCatalogPage( ID,  GLV_TEXT_API_WIDE_STRING( "Save Data" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) SaveDataCatalogPageCreator< _id > g_SaveDataCatalogPageCreator##_id( _name );
LOCAL_PAGE_CREATOR( DevMenuPageId_SaveData, "SaveDataCatalog" );

}} // ~namespace devmenu::savedata, ~namespace devmenu
