﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"

//#define DEVMENU_USE_SAMPLE_PAGE

#if defined( DEVMENU_USE_SAMPLE_PAGE )
//!--------------------------------------------------------------------------------------
//! GLVを使ったページサンプル
//!--------------------------------------------------------------------------------------
namespace {

//!--------------------------------------------------------------------------------------
//! @brief ラベルボタン定義
//!--------------------------------------------------------------------------------------
class LabelButton : public glv::Button
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //! @details 1バイト文字列( ASCII, Latin-1 )を対象にした初期化です.
    //!--------------------------------------------------------------------------------------
    explicit LabelButton( const std::string& text ) NN_NOEXCEPT : glv::Button( glv::Rect( 20 ), true )
    {
        this->add( FitButton( new glv::Label( text, false ), 25, 4 ) );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //! @details 2バイト文字列( UTF-16 )を対象にした初期化です.
    //!--------------------------------------------------------------------------------------
    explicit LabelButton( const glv::WideString& text ) NN_NOEXCEPT : glv::Button( glv::Rect( 20 ), true )
    {
        this->add( FitButton( new glv::Label( text, false ), 25, 4 ) );
    }

private:
    //!--------------------------------------------------------------------------------------
    //! @brief 自身( Button )の表示領域を指定のラベル情報にフィットさせます.
    //!--------------------------------------------------------------------------------------
    glv::Label* FitButton( glv::Label* pLabel, float fontSize, glv::space_t padding ) NN_NOEXCEPT
    {
        pLabel->size( fontSize );
        pLabel->paddingX( padding );
        pLabel->paddingY( padding );
        pLabel->pos( glv::Place::CC, 0, 0 ).anchor( glv::Place::CC );
        glv::space_t width = ( pLabel->width() + 1.f ) / 2.f * 2.f;    // ピクセルだから、センタリング時に2の倍数じゃないと...
        glv::space_t height = ( pLabel->height() + 1.f ) / 2.f * 2.f;  // ピクセルだから、センタリング時に2の倍数じゃないと...
        glv::Button::extent( width, height );
        return pLabel;
    }
};

//!--------------------------------------------------------------------------------------
//! @brief サンプルページ定義
//!--------------------------------------------------------------------------------------
class SamplePage : public devmenu::Page
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //!--------------------------------------------------------------------------------------
    SamplePage(int pageId, const glv::WideCharacterType* pageCaption, const glv::WideCharacterType* description, const glv::WideCharacterType* value = nullptr) NN_NOEXCEPT
        :   devmenu::Page(pageId, pageCaption, devmenu::DefaultPageRect),
            m_pDescription( nullptr ),
            m_pLabel( nullptr ),
            m_pLabelButton( nullptr )
    {
        InitializePage(description, value);
    }

    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //!--------------------------------------------------------------------------------------
    SamplePage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        :   devmenu::Page(pageId, pageCaption, rect),
            m_pDescription( nullptr ),
            m_pLabel( nullptr ),
            m_pLabelButton( nullptr )
    {
        InitializePage(pageCaption, nullptr);
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます.@n
    //! この時点ではまだ glvコンテキストのレンダリングは開始していません.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopAfterSceneRendererと同じです.@n
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        bool action = false;
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        if( true == dpad.HasAnyEvent() )
        {
            action = dpad.IsButtonDown<nn::hid::DebugPadButton::R>();
        }
        else if( events.GetAvailableBasicPadCount() > 0 )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad(0);
            action = bpad.IsButtonDown<glv::BasicPadEventType::Button::R>();
        }
        if( action != false )
        {
            DEVMENU_LOG( "OnLoopBeforeSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
        }
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラのレンダリングが終わった後に呼び出されます.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopBeforeSceneRendererと同じです.@n
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        bool action = false;
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        if( true == dpad.HasAnyEvent() )
        {
            action = dpad.IsButtonDown<nn::hid::DebugPadButton::R>();
        }
        else if( events.GetAvailableBasicPadCount() > 0 )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad(0);
            action = bpad.IsButtonDown<glv::BasicPadEventType::Button::R>();
        }
        if( action != false )
        {
            DEVMENU_LOG( "OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
        }
    }

    virtual View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pLabelButton;
    }

private:
    //!--------------------------------------------------------------------------------------
    //! @brief ページ初期化.
    //! @details キャプションと値のペアを表示するサンプル用のページを生成します.
    //!--------------------------------------------------------------------------------------
    void InitializePage(const glv::WideCharacterType* description, const glv::WideCharacterType* value)
    {
        if ( nullptr != value )
        {
            NN_ASSERT( description != nullptr );
            m_pDescription = description;

            glv::Label* vCaption;
            m_pLabel = vCaption = new glv::Label(description);
            *this << vCaption->size(25).pos(glv::Place::CC, 0, 0).anchor(glv::Place::CC);

            // 個別のスタイル
            glv::Style* pSytle = new glv::Style();
            pSytle->color.set(glv::StyleColor::BlackOnWhite);
            pSytle->color.fore.set(0.8, 0.8, 0.8);
            pSytle->color.back.set(0.5, 0.0, 0.0);

            LabelButton* vValue;
            m_pLabelButton = vValue = new LabelButton( value );
            vValue->attach(ReceiveUpdateNotification, glv::Update::Clicked, this);
            *this << vValue->pos(glv::Place::TC, 0, m_pLabel->bottom() + 10).anchor(glv::Place::TC).style(pSytle);
        }
        else
        {
            {   // 左上
                glv::Table* table = new glv::Table("p - -,< x >,b v d", 4, 4, glv::Rect(120, 120));
                *table << new glv::Button(glv::Rect(70, 20))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20));
                table->arrange();
                table->enable(glv::DrawBorder);
                *this << (table)->pos(glv::Place::TL, 300, 150).anchor(glv::Place::TL);
            }
            {   // 中上
                glv::Table* table = new glv::Table("p ^ q,< - -,b v d", 4, 4, glv::Rect(120, 120));
                *table << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                       << new glv::Button(glv::Rect(70, 20))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20));
                table->arrange();
                table->enable(glv::DrawBorder);
                *this << (table)->pos(glv::Place::TC, 0, 150).anchor(glv::Place::TC);
            }
            {   // 右上
                glv::Table* table = new glv::Table("p - -,< x >,b v |", 4, 4, glv::Rect(120, 120));
                *table << new glv::Button(glv::Rect(70, 20))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 45))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20));
                table->arrange();
                table->enable(glv::DrawBorder);
                *this << (table)->pos(glv::Place::TR, -300, 150).anchor(glv::Place::TR);
            }
            {   // 左下
                glv::Table* table = new glv::Table("p ^ q,| x >,| v -", 4, 4, glv::Rect(120, 120));
                *table << new glv::Button(glv::Rect(20, 70)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                                                             << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                                                             << new glv::Button(glv::Rect(45, 20));
                table->arrange();
                table->enable(glv::DrawBorder);
                *this << (table)->pos(glv::Place::BL, 300, -150).anchor(glv::Place::BL);
            }
            {   // 中下
                glv::Table* table = new glv::Table("p ^ q,< x >,b - -", 4, 4, glv::Rect(120, 120));
                *table << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                       << new glv::Button(glv::Rect(70, 20));
                table->arrange();
                table->enable(glv::DrawBorder);
                *this << (table)->pos(glv::Place::BC, 0, -150).anchor(glv::Place::BC);
            }
            {   // 右下
                glv::Table* table = new glv::Table("p ^ q,< x |,b - |", 4, 4, glv::Rect(120, 120));
                *table << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 70))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                       << new glv::Button(glv::Rect(45, 20));
                table->arrange();
                table->enable(glv::DrawBorder);
                *this << (table)->pos(glv::Place::BR, -300, -150).anchor(glv::Place::BR);
            }
            {   // 中央
                glv::Table* table = new glv::Table("p ^ q,< x >,b v d", 4, 4, glv::Rect(120, 120));
                *table << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20))
                       << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20)) << new glv::Button(glv::Rect(20, 20));
                table->arrange();
                table->enable(glv::DrawBorder);
                *this << (table)->pos(glv::Place::CC).anchor(glv::Place::CC);
            }
        }
    }

    void OnLabelButtonClicked()
    {
        auto pView = new devmenu::MessageView();

        // 表示メッセージ追加
        pView->AddMessage( "Message" );
        pView->AddMessage(m_pDescription);
        pView->AddMessage(GLV_TEXT_API_WIDE_STRING( "メッセージ" ));

        pView->AddButton( "Cancel" );

        // 実行ボタン追加
        pView->AddButton(
                GLV_TEXT_API_WIDE_STRING( "Create" ),
                []( void* pParam, nn::TimeSpan& timespan )
                {
                    DEVMENU_LOG( "Create button pushed.\n" );
                    timespan = nn::TimeSpan::FromMilliSeconds(600);
                }
            );
        pView->AddButton(
                GLV_TEXT_API_WIDE_STRING( "Delete" ),
                []( void* pParam, nn::TimeSpan& timespan )
                {
                    DEVMENU_LOG( "Delete button pushed.\n" );
                    timespan = nn::TimeSpan::FromMilliSeconds(400);
                }
            );
        pView->AddButton(
                "Check",
                []( void* pParam, nn::TimeSpan& timespan )
                {
                    DEVMENU_LOG( "Check button pushed.\n" );
                    timespan = nn::TimeSpan::FromMilliSeconds(200);
                }
            );

        GetRootSurfaceContext()->StartModal(pView, true);
    }

    static void ReceiveUpdateNotification(const glv::Notification& notification) NN_NOEXCEPT
    {
        auto pPage = notification.receiver<SamplePage>();
        const void* pSender = notification.sender();
        if( pSender == reinterpret_cast<void*>(pPage->m_pLabelButton) )
        {
            pPage->OnLabelButtonClicked();
        }
    }

    const glv::WideCharacterType* m_pDescription;
    glv::Label*                 m_pLabel;
    LabelButton*                m_pLabelButton;
};

//!--------------------------------------------------------------------------------------
//! @brief リソース定義
//!--------------------------------------------------------------------------------------
enum SamplePageId
{
    SamplePageId_Begin = DevMenuPageId_Sample,

    SamplePageId_Copyright = SamplePageId_Begin,
    SamplePageId_Configuration,
    SamplePageId_Application,
    SamplePageId_SaveData,
    SamplePageId_Private,

    SamplePageId_End,

    SamplePageId_TemplateCreator1,
    SamplePageId_TemplateCreator2,
    SamplePageId_TemplateCreator3,
    SamplePageId_TemplateCreator4,
};

struct SamplePageParam
{
    const glv::WideCharacterType* caption;
    const glv::WideCharacterType* description;
    const glv::WideCharacterType* value;
};
SamplePageParam g_SamplePageParam[] =
{
    {
        GLV_TEXT_API_WIDE_STRING( "コピーライト" ),
        GLV_TEXT_API_WIDE_STRING( "Copyright (C)Nintendo All rights reserved.\n" )
        GLV_TEXT_API_WIDE_STRING( "\n" )
        GLV_TEXT_API_WIDE_STRING( "These coded instructions, statements, and computer programs contain proprietary\n" )
        GLV_TEXT_API_WIDE_STRING( "information of Nintendo and/or its licensed developers and are protected by\n" )
        GLV_TEXT_API_WIDE_STRING( "national and international copyright laws. They may not be disclosed to third\n" )
        GLV_TEXT_API_WIDE_STRING( "parties or copied or duplicated in any form, in whole or in part, without the\n" )
        GLV_TEXT_API_WIDE_STRING( "prior written consent of Nintendo.\n" )
        GLV_TEXT_API_WIDE_STRING( "\n" )
        GLV_TEXT_API_WIDE_STRING( "The content herein is highly confidential and should be handled accordingly.\n" ),
        GLV_TEXT_API_WIDE_STRING( "OK" )
    },
    {
        GLV_TEXT_API_WIDE_STRING( "Configuration" ),
        GLV_TEXT_API_WIDE_STRING( "Firmware version" ),
        GLV_TEXT_API_WIDE_STRING( "?????????" )
    },
    {
        GLV_TEXT_API_WIDE_STRING( "アプリ一覧" ),
        GLV_TEXT_API_WIDE_STRING( "アプリ#0" ),
        GLV_TEXT_API_WIDE_STRING( "リマオ&イジール RPG" )
    },
    {
        GLV_TEXT_API_WIDE_STRING( "セーブデータ" ),
        GLV_TEXT_API_WIDE_STRING( "セーブデータ#0" ),
        GLV_TEXT_API_WIDE_STRING( "リマオ&イジール RPG セーブデータ-0" )
    },
    {
        GLV_TEXT_API_WIDE_STRING( "ひ・み・つ♪" ),
        GLV_TEXT_API_WIDE_STRING( "....." ),
        GLV_TEXT_API_WIDE_STRING( "いぇぇあぁぁぁおぇぁぁぁ" )
    },
};
const int SamplePageParamCountMax = sizeof( g_SamplePageParam ) / sizeof( g_SamplePageParam[0] );

//!--------------------------------------------------------------------------------------
//! @brief ページ生成
//!--------------------------------------------------------------------------------------
class SamplePageCreator : devmenu::PageCreatorBase
{
public:
    SamplePageCreator( int pageId, const char* pName ) NN_NOEXCEPT
        :   devmenu::PageCreatorBase( pageId, pName )
    {
    }

protected:
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        DEVMENU_LOG( "craete SamplePage(%d, %s)\n", id(), name() );

        const int pageId = id();
        const int index = pageId - SamplePageId_Begin;
        NN_ASSERT( index >= 0 && index < SamplePageParamCountMax );

        const SamplePageParam& param = g_SamplePageParam[ index ];
        return new SamplePage( pageId, param.caption, param.description, param.value );
    }

    virtual bool canCreate() NN_NOEXCEPT NN_OVERRIDE
    {
        // 環境条件にあわせて許可することが可能
        const int pageId = id();
        return ( pageId >= SamplePageId_Begin) && (pageId < SamplePageId_End );
    }
};
#define SAMPLE_PAGE_CREATOR( id, name ) SamplePageCreator g_SamplePageCreator##id( (id), (name) );

// 識別子順にページが表示される
devmenu::PageCreatorImpl< SamplePage > g_SamplePageCreator15( SamplePageId_TemplateCreator4, "Sample 4" );
devmenu::PageCreatorImpl< SamplePage > g_SamplePageCreator14( SamplePageId_TemplateCreator3, "Sample 3" );

SAMPLE_PAGE_CREATOR( SamplePageId_Copyright,     "Copyright" );
SAMPLE_PAGE_CREATOR( SamplePageId_Configuration, "Configuration" );
SAMPLE_PAGE_CREATOR( SamplePageId_Application,   "Application" );
SAMPLE_PAGE_CREATOR( SamplePageId_SaveData,      "SaveData" );
SAMPLE_PAGE_CREATOR( SamplePageId_Private,       "Private" );

DEVMENU_PAGE_CREATOR( SamplePage, SamplePageId_TemplateCreator2, "Sample 2" );
DEVMENU_PAGE_CREATOR( SamplePage, SamplePageId_TemplateCreator1, "Sample 1" );

} // end of unnamed namespace
#endif // defined( DEVMENU_USE_SAMPLE_PAGE )
