﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <vector>

#include <nn/nn_Abort.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/prepo/prepo_ApiConfig.h>
#include <nn/prepo/prepo_ApiDebug.h>
#include <nn/prepo/prepo_ResultPrivate.h>
#include <nn/dauth/dauth_Result.h>

#include "Common/DevMenu_CommonCheckBox.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "Common/DevMenu_RadioButtons.h"
#include "DevMenu_Common.h"
#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"

#include "DevMenu_PrepoSettingsPage.h"

namespace devmenu { namespace prepo {

class StatisticsView : public glv::Table
{
private:
    glv::Label  m_TotalUploadedSizeLabel;
    glv::Label  m_TotalUploadedCountLabel;
    glv::Label  m_TotalLostSizeLabel;
    glv::Label  m_TotalLostCountLabel;
    glv::Label  m_LossRateLabel;
public:
    StatisticsView() NN_NOEXCEPT
        : glv::Table("pqp,|qp,pqp,|qp,pqp,|qp,pqp", 10.0f, 4.0f)
        , m_TotalUploadedSizeLabel("", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
        , m_TotalUploadedCountLabel("", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
        , m_TotalLostSizeLabel("", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
        , m_TotalLostCountLabel("", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
        , m_LossRateLabel("", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
    {
        *this
            << new glv::Label("Uploaded:", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
            << m_TotalUploadedSizeLabel
            << new glv::Label("Bytes", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
            << m_TotalUploadedCountLabel
            << new glv::Label("Reports", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))

            << new glv::Label("Lost:", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
            << m_TotalLostSizeLabel
            << new glv::Label("Bytes", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
            << m_TotalLostCountLabel
            << new glv::Label("Reports", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))

            << new glv::Label("Loss Rate:", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
            << m_LossRateLabel
            << new glv::Label("%", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));

            disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);
    }
    void Update(const nn::prepo::Statistics& statistics) NN_NOEXCEPT
    {
        const auto category = nn::prepo::ReportCategory_Normal;

        m_TotalUploadedSizeLabel.setValue(std::to_string(statistics.groups[category].uploaded.size));
        m_TotalUploadedCountLabel.setValue(std::to_string(statistics.groups[category].uploaded.count));
        const auto totalLostSize = statistics.groups[category].lostByBufferShortage.size + statistics.groups[category].lostByStorageShortage.size + statistics.groups[category].lostByDisagree.size;
        const auto totalLostCount = statistics.groups[category].lostByBufferShortage.count + statistics.groups[category].lostByStorageShortage.count + statistics.groups[category].lostByDisagree.count;
        m_TotalLostSizeLabel.setValue(std::to_string(totalLostSize));
        m_TotalLostCountLabel.setValue(std::to_string(totalLostCount));
        const auto lossRate = totalLostSize > 0
            ? totalLostSize * 100.0 / (statistics.groups[category].uploaded.size + totalLostSize)
            : 0.0;
        m_LossRateLabel.setValue("              " + std::to_string(lossRate));
        arrange();
    }
};

class ThroughputView : public glv::Table
{
private:
    static const int ThroughputDataCount = 60;
    static const int XRageMax = ThroughputDataCount;
    static const int XRageMin = -2; // 0 が Y 軸と被るため -2 ずらす。
    static const int YRangeMax = 20;
    static const int YRangeMin = 0;
private:
    glv::Plot       m_Plot;
    nn::prepo::ThroughputRecord m_Data[ThroughputDataCount];
    int                         m_DataCount;
private:
    template<class T>
    T RoundUp(T value, T align) NN_NOEXCEPT
    {
        value += align - 1;
        value -= value % align;
        return value;
    }
    enum ViewIndex : int
    {
        ViewIndex_App,
        ViewIndex_Sys,
        ViewIndex_All,
        ViewIndex_Last,
    };
    std::atomic<int> m_ViewIndex;
public:
    ThroughputView() NN_NOEXCEPT
        : glv::Table("<>,<-", 8.0f)
        , m_Plot(glv::Rect(20, 20, 1000, 160), *new glv::PlotFunction1D(glv::Color(0.8), 4.0, glv::draw::Lines, glv::PlotFunction1D::ZIGZAG))
        , m_ViewIndex(0)
    {
        m_Plot.data().resize(glv::Data::INT, 1, ThroughputDataCount);
        m_Plot.range(XRageMin, XRageMax, 0);
        m_Plot.range(YRangeMin, YRangeMax, 1);
        m_Plot.showNumbering(true);
        m_Plot.major(5, 0);
        m_Plot.minor(5, 0);
        m_Plot.major(5, 1);
        m_Plot.minor(5, 1);

        const RadioButtons<int>::ButtonInfoList buttonInfoList = {
            { ViewIndex_App, ViewIndex_App, "App " },
            { ViewIndex_Sys, ViewIndex_Sys, "Sys" },
            { ViewIndex_All, ViewIndex_All, "All " }
        };

        *this
            << new glv::Label("Save Count / min:", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize))
            << new RadioButtons<int>( glv::Rect( 280, 20 ),
                buttonInfoList, ViewIndex_App, false, [this](int i) { m_ViewIndex = i; } )
            << m_Plot;

        disable(glv::Property::DrawBorder | glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);
        m_Plot.disable(glv::Property::Focused | glv::Property::HitTest | glv::Property::Controllable);
        arrange();

        Update();
    }
    void Update() NN_NOEXCEPT
    {
        const int viewIndex = m_ViewIndex;
        nn::prepo::GetThroughputHistory(&m_DataCount, m_Data, NN_ARRAY_SIZE(m_Data));

        int peek = 0;
        for (int i = 0; i < m_DataCount; i++)
        {
            int value = 0;
            if (viewIndex == ViewIndex_App || viewIndex == ViewIndex_All)
            {
                value += m_Data[i].app;
            }
            if (viewIndex == ViewIndex_Sys || viewIndex == ViewIndex_All)
            {
                value += m_Data[i].sys;
            }

            peek = std::max(peek, static_cast<int>(value));
            m_Plot.data().assign(static_cast<int>(value), i);
        }
        m_Plot.range(YRangeMin, std::max(YRangeMax, RoundUp(peek, 5)), 1);
    }
};
const int ThroughputView::ThroughputDataCount;
const int ThroughputView::XRageMax;
const int ThroughputView::XRageMin;
const int ThroughputView::YRangeMax;
const int ThroughputView::YRangeMin;

class CheckBox : public CheckBoxButton
{
public:
    CheckBox(const char* text, std::function<void(bool)> checkCanged, bool initialValue) NN_NOEXCEPT
        : CheckBoxButton(text, 20.0f, "< < <", false)
        , m_CheckChanged(checkCanged)
    {
        SetValue(initialValue);
        SetCallback(
            [](const glv::Notification& notification)
            {
                notification.receiver<CheckBox>()->m_CheckChanged(
                    notification.data<glv::ChangedValueOnGesture<bool>>()->newValue);
            }, this);
    }

public:
    glv::View* GetFirstFocusTargetView() NN_NOEXCEPT
    {
        return this;
    }

private:
    std::function<void(bool)>   m_CheckChanged;
};

class Button : public glv::Button
{
public:
    Button(const std::string& text, std::function<void()> onClick) NN_NOEXCEPT
        : glv::Button(glv::Rect(440, 50), true)
        , m_OnClick(onClick)
    {
        *this << new glv::Label(text, glv::Label::Spec(glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize));
        changePadClickDetectableButtons(glv::BasicPadEventType::Button::Ok::Mask);
        changePadClickDetectableButtons(glv::DebugPadEventType::Button::Ok::Mask);
        attach(
            [](const glv::Notification& notification)
            {
                notification.receiver<Button>()->m_OnClick();
            },
            glv::Update::Clicked, this);
    }

private:
    std::function<void()>   m_OnClick;
};

class PrepoSettingsPage : public Page
{
public:
    PrepoSettingsPage(int pageId, const glv::WideCharacterType* pageCaption, const glv::Rect& rect) NN_NOEXCEPT
        : Page(pageId, pageCaption, rect),
        m_pFirst( nullptr ),
        m_pMainTable( nullptr ),
        m_pTransmissionStatusLabel( nullptr ),
        m_pStorageUsageLabel( nullptr ),
        m_pStatisticsView( nullptr ),
        m_pThroughputView( nullptr ),
        m_PrevTick()
    {
    }

    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        auto transmissionStatus = new glv::Label("", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));
        auto storageUsage = new glv::Label("", glv::Label::Spec(glv::Place::TL, 0, 0, CommonValue::InitialFontSize));
        auto userAgreementCheckCheckBox = CreateUserAgreementCheckCheckBox();
        auto backgroundProcessingCheckBox = CreateBackgroundProcessingCheckBox();
        auto clearStorageButton = CreateClearStorageButton();
        auto requestImmediateTransmissionButton = RequestImmediateTransmissionButton();
        auto statistics = new StatisticsView();
        auto throughput = new ThroughputView();

        auto table = new glv::Table( "<-,<-,<p,<|,<|,<|,<-", 10, 20 );
        *table << transmissionStatus;
        *table << storageUsage;
        *table << userAgreementCheckCheckBox;
        *table << statistics;
        *table << backgroundProcessingCheckBox;
        *table << clearStorageButton;
        *table << requestImmediateTransmissionButton;
        *table << throughput;
        table->arrange();

        *this << table;

        this->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

        m_pFirst = userAgreementCheckCheckBox->GetFirstFocusTargetView();

        m_pMainTable = table;
        m_pTransmissionStatusLabel = transmissionStatus;
        m_pStorageUsageLabel = storageUsage;
        m_pStatisticsView = statistics;
        m_pThroughputView = throughput;

        UpdateTransmissionStatus();
        UpdateStorageUsage();
        UpdateStatistics();
        m_pThroughputView->Update();
        table->arrange();
    }

    virtual View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFirst;
    }

    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        NN_UNUSED( events );

        nn::os::Tick currentTick = nn::os::GetSystemTick();

        if ((currentTick - m_PrevTick).ToTimeSpan().GetSeconds() >= 1)
        {
            UpdateTransmissionStatus();
            UpdateStorageUsage();
            UpdateStatistics();
            m_pThroughputView->Update();
            m_PrevTick = currentTick;
        }
    }

private:
    glv::View* m_pFirst;

    glv::Table*     m_pMainTable;
    glv::Label*     m_pTransmissionStatusLabel;
    glv::Label*     m_pStorageUsageLabel;
    StatisticsView* m_pStatisticsView;
    ThroughputView* m_pThroughputView;

    nn::os::Tick m_PrevTick;

private:
    void UpdateTransmissionStatus() NN_NOEXCEPT
    {
        nn::prepo::TransmissionStatus status;
        nn::prepo::GetTransmissionStatus(&status);

        switch (status)
        {
        case nn::prepo::TransmissionStatus_Idle:
            {
                m_pTransmissionStatusLabel->setValue(" Transmission Status: Idle");
            }
            break;
        case nn::prepo::TransmissionStatus_Pending:
            {
                char text[128] = {};
                nn::util::SNPrintf(text, sizeof (text), " Transmission Status: Pending (%s)",
                    GetErrorStatusMessage(nn::prepo::GetLastUploadError()));

                m_pTransmissionStatusLabel->setValue(static_cast<const char*>(text));
            }
            break;
        case nn::prepo::TransmissionStatus_Processing:
            {
                m_pTransmissionStatusLabel->setValue(" Transmission Status: Processing");
            }
            break;
        default:
            break;
        }
    }

    void UpdateStorageUsage() NN_NOEXCEPT
    {
        int64_t used = 0;
        int64_t capacity = 0;

        if (nn::prepo::GetStorageUsage(&used, &capacity).IsSuccess())
        {
            // NN_LOG("used = %lld, capacity = %lld\n", used, capacity);
            const int barCountMax = 50;

            int barCount = static_cast<int>((used * barCountMax) / capacity) % (barCountMax + 1);

            char progress[barCountMax + 1] = {};

            for (int i = 0; i < barCount; i++)
            {
                progress[i] = '|';
            }

            int p = static_cast<int>((used * 1000) / capacity);

            // 少しでもデータがあれば 0.1% と表示させる。
            if (p == 0 && used > 0)
            {
                p = 1;
            }

            char text[128] = {};
            nn::util::SNPrintf(text, sizeof (text), " Storage Usage: %s%s%d.%d%%",
                progress, progress[0] ? " " : "", p / 10, p % 10);

            m_pStorageUsageLabel->setValue(static_cast<const char*>(text));
        }
    }

    void UpdateStatistics() NN_NOEXCEPT
    {
        nn::prepo::Statistics statistics;
        nn::prepo::GetStatistics(&statistics);
        m_pStatisticsView->Update(statistics);
    }

    CheckBox* CreateBackgroundProcessingCheckBox()
    {
        bool value = true;
        GetFixedSizeFirmwareDebugSettingsItemValue(&value, "prepo", "background_processing");

        return new CheckBox("Background Processing",
            [this](bool value)
            {
                SetFixedSizeFirmwareDebugSettingsItemValue("prepo", "background_processing", &value);
            #if defined(NN_BUILD_CONFIG_OS_HORIZON)
                GetRootSurfaceContext()->DisplayRebootRequestText();
            #endif
            },
            value);
    }

    CheckBox* CreateUserAgreementCheckCheckBox()
    {
        bool value = true;

    #if defined(NN_BUILD_CONFIG_OS_HORIZON)
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::IsUserAgreementCheckEnabled(&value));
    #endif

        return new CheckBox("User Agreement Check",
            [](bool value)
            {
            #if defined(NN_BUILD_CONFIG_OS_HORIZON)
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(value));
            #else
                NN_UNUSED(value);
            #endif
            },
            value);
    }

    glv::Button* CreateClearStorageButton()
    {
        return new Button(
            "Clear Storage",
            []
            {
                auto result = nn::prepo::ClearStorage();
                NN_ASSERT(result.IsSuccess());
                NN_UNUSED(result);
            });
    }

    glv::Button* RequestImmediateTransmissionButton()
    {
        return new Button(
            "Request Immediate Transmission",
            []
            {
                auto result = nn::prepo::RequestImmediateTransmission();
                NN_ASSERT(result.IsSuccess());
                NN_UNUSED(result);
            });
    }

    const char* GetErrorStatusMessage(nn::Result result)
    {
        if (nn::prepo::ResultTransmissionNotAgreed::Includes(result))
        {
            return "TransmissionNotAgreed";
        }
        if (nn::prepo::ResultNetworkUnavailable::Includes(result))
        {
            return "NetworkUnavailable";
        }
        if (nn::prepo::ResultHttpError::Includes(result))
        {
            if (nn::prepo::ResultHttpErrorCouldntResolveProxy::Includes(result))
            {
                return "HttpError:CouldntResolveProxy";
            }
            if (nn::prepo::ResultHttpErrorCouldntResolveHost::Includes(result))
            {
                return "HttpError:CouldntResolveHost";
            }
            if (nn::prepo::ResultHttpErrorCouldntConnect::Includes(result))
            {
                return "HttpError:CouldntConnect";
            }
            if (nn::prepo::ResultHttpErrorWriteError::Includes(result))
            {
                return "HttpError:WriteError";
            }
            if (nn::prepo::ResultHttpErrorReadError::Includes(result))
            {
                return "HttpError:ReadError";
            }
            if (nn::prepo::ResultHttpErrorOutOfMemory::Includes(result))
            {
                return "HttpError:OutOfMemory";
            }
            if (nn::prepo::ResultHttpErrorOperationTimedout::Includes(result))
            {
                return "HttpError:OperationTimedout";
            }
            if (nn::prepo::ResultHttpErrorSslConnectError::Includes(result))
            {
                return "HttpError:SslConnectError";
            }
            if (nn::prepo::ResultHttpErrorPeerFailedVerification::Includes(result))
            {
                return "HttpError:PeerFailedVerification";
            }
            if (nn::prepo::ResultHttpErrorGotNothing::Includes(result))
            {
                return "HttpError:GotNothing";
            }
            if (nn::prepo::ResultHttpErrorSendError::Includes(result))
            {
                return "HttpError:SendError";
            }
            if (nn::prepo::ResultHttpErrorRecvError::Includes(result))
            {
                return "HttpError:RecvError";
            }
            if (nn::prepo::ResultHttpErrorSslCertProblem::Includes(result))
            {
                return "HttpError:SslCertProblem";
            }
            if (nn::prepo::ResultHttpErrorSslCipher::Includes(result))
            {
                return "HttpError:SslCipher";
            }
            if (nn::prepo::ResultHttpErrorSslCaCert::Includes(result))
            {
                return "HttpError:SslCaCert";
            }
            return "HttpError";
        }
        if (nn::prepo::ResultServerError::Includes(result))
        {
            if (nn::prepo::ResultServerError400::Includes(result))
            {
                return "ServerError:Status400";
            }
            if (nn::prepo::ResultServerError401::Includes(result))
            {
                return "ServerError:Status401";
            }
            if (nn::prepo::ResultServerError403::Includes(result))
            {
                return "ServerError:Status403";
            }
            if (nn::prepo::ResultServerError500::Includes(result))
            {
                return "ServerError:Status500";
            }
            if (nn::prepo::ResultServerError503::Includes(result))
            {
                return "ServerError:Status503";
            }
            if (nn::prepo::ResultServerError504::Includes(result))
            {
                return "ServerError:Status504";
            }
            return "ServerError";
        }
        if (nn::dauth::ResultNdasStatus::Includes(result))
        {
            if (nn::dauth::ResultNdasStatusNo0007::Includes(result))
            {
                return "DeviceAuthenticationError:SystemUpdateRequired";
            }
            if (nn::dauth::ResultNdasStatusNo0008::Includes(result))
            {
                return "DeviceAuthenticationError:Banned";
            }
            if (nn::dauth::ResultNdasStatusNo0009::Includes(result))
            {
                return "DeviceAuthenticationError:InternalServerError";
            }
            if (nn::dauth::ResultNdasStatusNo0010::Includes(result))
            {
                return "DeviceAuthenticationError:UnderMaintenance";
            }
            return "DeviceAuthenticationError";
        }
        return "";
    } // NOLINT(impl/function_size)
};

// ページ作成するかどうかを動的に判断するため、グローバルなインスタンスはコメントアウトします。
//PageCreatorImpl< PrepoSettingsPage > g_PrepoSettingsPageCreator( DevMenuPageId_Prepo, "PlayReport" );

void CreatePrepoSettingsPage()
{
    static PageCreatorImpl< PrepoSettingsPage >* s_pPrepoSettingsPageCreator;
    s_pPrepoSettingsPageCreator = new PageCreatorImpl< PrepoSettingsPage >(DevMenuPageId_Prepo, "PlayReport");
}

}} // ~namespace devmenu::prepo, ~namespace devmenu

