﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <new>
#include <functional>   // std::hash
#include <string>       // std::hash< std::string >
#include <unordered_map>
#include <nn/gfx.h>
#include <nn/init.h>
#include <nn/mem.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/util/util_TypedStorage.h>

#include "DevMenu_PeakMemoryMeasurer.h"
#include "DevMenu_Config.h"

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
// For declearation of 'nv::InitializeGraphics' and 'nv::SetGraphicsAllocator'
#include <nv/nv_MemoryManagement.h>
#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
// For declearation of 'glslcSetAllocator'
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif
#endif

//! @brief  有効にした場合、動的アロケートのピーク値計測を行います。
//!         尚、本計測が有効になるのは NN_BUILD_CONFIG_OS_HORIZON が有効の時のみとなります。
//!         Windows環境では計測は行いません。
//#define ENABLE_PEAK_MEASURE

namespace devmenu {

namespace {

    //!--------------------------------------------------------------------------------------
    //! 定数定義
    //!--------------------------------------------------------------------------------------
    static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ

#if defined( NN_BUILD_CONFIG_OS_HORIZON )

#if defined( ENABLE_PEAK_MEASURE )

    class PeakRecordableAllocator
    {
        NN_DISALLOW_COPY( PeakRecordableAllocator );
        NN_DISALLOW_MOVE( PeakRecordableAllocator );

    public:
        PeakRecordableAllocator( void* addr, size_t size ) NN_NOEXCEPT
            : m_Capacity( size ), m_PeakSize( 0 )
        {
            m_Allocator.Initialize( addr, size );
        }

        inline void* Allocate( size_t size ) NN_NOEXCEPT
        {
            void* result = m_Allocator.Allocate( size );
            ComputePeak();
            return result;
        }

        inline void* Allocate( size_t size, size_t alignment ) NN_NOEXCEPT
        {
            void* result = m_Allocator.Allocate( size, alignment );
            ComputePeak();
            return result;
        }

        inline void* Reallocate( void* addr, size_t newSize ) NN_NOEXCEPT
        {
            void* result = m_Allocator.Reallocate( addr, newSize );
            ComputePeak();
            return result;
        }

        inline void Free( void* addr ) NN_NOEXCEPT
        {
            m_Allocator.Free( addr );
        }

        inline const size_t GetPeakSize() const NN_NOEXCEPT
        {
            return m_PeakSize;
        }

        inline nn::mem::StandardAllocator& GetAllocator() NN_NOEXCEPT
        {
            return m_Allocator;
        }

    private:
        void ComputePeak() NN_NOEXCEPT
        {
            const size_t remain = m_Allocator.GetTotalFreeSize();
            const size_t diff = m_Capacity - remain;
            m_PeakSize = ( diff > m_PeakSize ) ? diff : m_PeakSize;
        }

        nn::mem::StandardAllocator m_Allocator;
        const size_t m_Capacity;
        size_t m_PeakSize;
    };

    typedef PeakRecordableAllocator MeasureAllocatorType;

    nn::util::TypedStorage< MeasureAllocatorType, sizeof( MeasureAllocatorType ), NN_ALIGNOF( MeasureAllocatorType ) > g_MeasureAllocator;

    //! 管理ブロックは算出されないので概算用です。
    class PeakRecorder : public std::unordered_map< void*, size_t >
    {
    public:
        typedef void* KeyType;
        typedef size_t ValueType;
        typedef std::unordered_map< KeyType, ValueType > MapParentType;

        PeakRecorder() NN_NOEXCEPT : m_Required( 0 ), m_PeakSize( 0 )
        {
            MapParentType::reserve( 1024 );
        }

        inline void* Allocate( size_t size ) NN_NOEXCEPT
        {
            void* result = Get( devmenu::g_MeasureAllocator ).Allocate( size );
            AllocateRequired( size, result );
            return result;
        }

        inline void* Allocate( size_t size, size_t alignment ) NN_NOEXCEPT
        {
            void* result = Get( devmenu::g_MeasureAllocator ).Allocate( size, alignment );
            AllocateRequired( size, result );
            return result;
        }

        inline void* Reallocate( void* addr, size_t newSize ) NN_NOEXCEPT
        {
            FreeRequired( addr );
            void* result = Get( devmenu::g_MeasureAllocator ).Reallocate( addr, newSize );
            AllocateRequired( newSize, result );
            return result;
        }

        inline void Free( void* addr ) NN_NOEXCEPT
        {
            Get( devmenu::g_MeasureAllocator ).Free( addr );
            FreeRequired( addr );
        }

        inline const size_t GetPeakSize() const NN_NOEXCEPT
        {
            return m_PeakSize;
        }

    private:
        void AllocateRequired( const size_t size, void* pAssignedAddress ) NN_NOEXCEPT
        {
            const size_t newRequired = m_Required + size;
            m_Required = newRequired;
            m_PeakSize = ( newRequired > m_PeakSize ) ? newRequired : m_PeakSize;
            ( *this )[ pAssignedAddress ] = size;
        }

        void FreeRequired( void* addr ) NN_NOEXCEPT
        {
            auto it = MapParentType::find( addr );
            if ( it != MapParentType::end() )
            {
                m_Required -= it->second;
            }
        }

        size_t  m_Required;
        size_t  m_PeakSize;
    };

    PeakRecorder g_PeakRecorderNvn;

#endif // defined( ENABLE_PEAK_MEASURE )

#if defined( NN_BUILD_CONFIG_SPEC_NX ) || NN_GFX_IS_TARGET_NVN

    //!--------------------------------------------------------------------------------------
    //! @brief nvn アロケータ
    //!--------------------------------------------------------------------------------------
    static void* NvnAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED( userPtr );

#if defined( ENABLE_PEAK_MEASURE )
        return g_PeakRecorderNvn.Allocate( size, alignment );
#else
        return ::aligned_alloc(alignment, size);
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief nvn デアロケータ
    //!--------------------------------------------------------------------------------------
    static void NvnDeallocate(void* addr, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
#if defined( ENABLE_PEAK_MEASURE )
        g_PeakRecorderNvn.Free( addr );
#else
        ::free(addr);
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief nvn リアロケータ
    //!--------------------------------------------------------------------------------------
    static void* NvnReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
#if defined( ENABLE_PEAK_MEASURE )
        return g_PeakRecorderNvn.Reallocate( addr, newSize );
#else
        return ::realloc(addr, newSize);
#endif
    }

#endif // defined( NN_BUILD_CONFIG_SPEC_NX ) || NN_GFX_IS_TARGET_NVN

/* InitializeMeasurerOnStartup での
nn::os::SetMemoryAllocatorForThreadLocal( AllocatorForThreadLocal, DeallocatorForThreadLocal )のコメントアウトに伴い,以下もコメントアウトします*/
//#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    //!--------------------------------------------------------------------------------------
    //! @brief thread_local 用 デフォルトアロケータ
    //!--------------------------------------------------------------------------------------
    //void* AllocatorForThreadLocal( size_t size, size_t alignment ) NN_NOEXCEPT
    //{
    //    return ::aligned_alloc( size, alignment );
    //}

    //!--------------------------------------------------------------------------------------
    //! @brief thread_local 用 デフォルトデアロケータ
    //!--------------------------------------------------------------------------------------
    //void DeallocatorForThreadLocal( void* p, size_t size ) NN_NOEXCEPT
    //{
    //    NN_UNUSED( size );
    //    ::free( p );
    //}
//#endif // defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

#endif // defined( NN_BUILD_CONFIG_OS_HORIZON )

} // end of namespace anonymous

//!--------------------------------------------------------------------------------------
//! @brief メジャー初期化。( nninitStartup で呼び出してください )
//! @param[in] reservedHeapByteSize アプリケーションで予約するヒープメモリ上限サイズをバイト単位で指定します。
//! @details 内部でヒープメモリを準備し、アロケータを初期化します。@n
//! 通常モードでは、nn::init::InitializeAllocator を呼び出します。@n
//! 計測モードでは、計測用のアロケータを呼び出します。
//!--------------------------------------------------------------------------------------
void PeakMemoryMeasurer::InitializeMeasurerOnStartup( const size_t reservedHeapByteSize ) NN_NOEXCEPT
{
    DEVMENU_LOG( "InitializeMeasurerOnStartup reservedHeap[ %zd ] byte.\n", reservedHeapByteSize );

    NN_ABORT_UNLESS_RESULT_SUCCESS( ::nn::os::SetMemoryHeapSize( reservedHeapByteSize ) );
    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS( ::nn::os::AllocateMemoryBlock( &address, reservedHeapByteSize ) );

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( ENABLE_PEAK_MEASURE )

    new( &Get( devmenu::g_MeasureAllocator ) ) PeakRecordableAllocator( reinterpret_cast< void* >( address ), reservedHeapByteSize );

#else
    ::nn::init::InitializeAllocator( reinterpret_cast< void* >( address ), reservedHeapByteSize );

#endif

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

    // コンパイラのスレッドローカル実装用のメモリアロケータの登録
    // [TORIAEZU] DevMenuApp だと nn::os::SetMemoryAllocatorForThreadLocal 後に Assert に引っかかる
    // コメントアウトしてもいいが System Applet 時は呼ぶようにしておく
    //nn::os::SetMemoryAllocatorForThreadLocal( AllocatorForThreadLocal, DeallocatorForThreadLocal );// DevMenu 起動時にメモリアクセス違反になるためコメントアウト

#endif
}

//!--------------------------------------------------------------------------------------
//! @brief メジャー初期化( nnMain で呼び出してください )
//!--------------------------------------------------------------------------------------
void PeakMemoryMeasurer::InitializeMeasurerOnMain() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#if defined( NN_BUILD_CONFIG_SPEC_NX )
    // this memory allocation will be used from the nvn graphics systems at runtime.
    nv::SetGraphicsAllocator( NvnAllocate, NvnDeallocate, NvnReallocate, NULL );
    nv::InitializeGraphics( NvnAllocate( GraphicsSystemReservedMemorySize, sizeof( uintptr_t ), nullptr ), GraphicsSystemReservedMemorySize );
#endif // defined( NN_BUILD_CONFIG_SPEC_NX )

#if NN_GFX_IS_TARGET_NVN
    // this memory allocation interface will be used when compiling of shader code at runtime.
    glslcSetAllocator( NvnAllocate, NvnDeallocate, NvnReallocate, NULL );
#endif // defined( NN_GFX_IS_TARGET_NVN )
#endif // defined( NN_BUILD_CONFIG_OS_HORIZON )
}

//!--------------------------------------------------------------------------------------
//! @brief メジャー終了
//!--------------------------------------------------------------------------------------
void PeakMemoryMeasurer::FinalizeMeasurerOnMain() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( ENABLE_PEAK_MEASURE )

    DEVMENU_LOG( "Allocate.peak[ %zd ]\n", Get( devmenu::g_MeasureAllocator ).GetPeakSize() );
    DEVMENU_LOG( "NvnAllocate.peak[ %zd ]\n", g_PeakRecorderNvn.GetPeakSize() );

#endif
}

} // ~namespace devmenu


#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined( ENABLE_PEAK_MEASURE )

//!--------------------------------------------------------------------------------------
//! @brief malloc() オーバーライド関数の定義
//!--------------------------------------------------------------------------------------
extern "C" void* malloc( size_t size )
{
    return Get( devmenu::g_MeasureAllocator ).Allocate( size );
}

//!--------------------------------------------------------------------------------------
//! @brief free() オーバーライド関数の定義
//!--------------------------------------------------------------------------------------
extern "C" void free( void* p )
{
    if ( nullptr != p )
    {
        Get( devmenu::g_MeasureAllocator ).Free( p );
    }
}

//!--------------------------------------------------------------------------------------
//! @brief calloc() オーバーライド関数の定義
//!--------------------------------------------------------------------------------------
extern "C" void* calloc( size_t num, size_t size )
{
    size_t sum = num * size;
    void* p = std::malloc( sum );
    if ( nullptr != p )
    {
        std::memset( p, 0, sum );
    }
    return p;
}

//!--------------------------------------------------------------------------------------
//! @brief realloc() オーバーライド関数の定義
//!--------------------------------------------------------------------------------------
extern "C" void* realloc( void* p, size_t newSize )
{
    return Get( devmenu::g_MeasureAllocator ).Reallocate( p, newSize );
}

//!--------------------------------------------------------------------------------------
//! @brief aligned_alloc() オーバーライド関数の定義
//!--------------------------------------------------------------------------------------
extern "C" void* aligned_alloc( size_t alignment, size_t size )
{
    return Get( devmenu::g_MeasureAllocator ).Allocate( size, alignment );
}

//-----------------------------------------------------------------------------
#endif  // NN_BUILD_CONFIG_OS_HORIZON
