﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

#include <cstdlib>
#include <glv.h>
#include <mutex>

#include <nn/ae.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/applet/applet.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/vi/vi_DisplayEvents.h>

#include "DevMenu_Common.h"
#include "DevMenu_Config.h"
#include "DevMenu_Exhibition.h"
#include "DevMenu_Notification.h"
#include "DevMenu_NotificationMessageNotifier.h"
#include "DevMenu_RetailInteractiveDisplay.h"
#include "DevMenu_Sound.h"
#include "Launcher/DevMenu_Launcher.h"
#include "Launcher/DevMenu_LauncherLibraryAppletApis.h"

namespace devmenu {

namespace {

// SA モードでは、AE ライブラリを利用してメッセージ通知をコントロールします。

    // 現在のアプリケーションを再起動するメッセージ。エラービューアから発行される。
    const nn::Bit8 RelaunchCurrentApplicationMessage[] =
    {
        0x53, 0x41, 0x4d, 0x53, // Magic code = SAMS
        0x01, 0x00, 0x00, 0x00, // Base version
        0x12, 0x00, 0x00, 0x00, // Type = RelaunchCurrentApplication
        0x01, 0x00, 0x00, 0x00  // RelaunchCurrentApplication version
    };

    // 現在のアプリケーションを再起動するメッセージ。MyPage から発行される。
    const nn::Bit8 LaunchApplicationMessage[] =
    {
        0x53, 0x41, 0x4d, 0x53, // Magic code = SAMS
        0x01, 0x00, 0x00, 0x00, // Base version
        0x04, 0x00, 0x00, 0x00, // Type = LaunchCurrentApplication
        0x01, 0x00, 0x00, 0x00  // LaunchApplication version
    };

    // スレッド用スタック領域
    const int ThreadStackSize = 8 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];

    // メッセージ通知コントローラ
    class NotificationController
    {
        NN_DISALLOW_COPY( NotificationController );
        NN_DISALLOW_MOVE( NotificationController );

    public:
        // コンストラクタ
        NotificationController()  NN_NOEXCEPT
        :   m_IsInitialized( false ),
            m_IsForeground( false ),
            m_IsRequestedToExit( false ),
            m_IsRequestedToSleep( false ),
            m_IsDeviceRebootRequiredAfterApplicationExit( false ),
            m_IsRidExitMenuTriggered( false ),
            m_EventToGetForeground( nn::os::EventClearMode_ManualClear ),
            m_IsSdCardDetectionStateChanged( false ),
            m_IsHdcpAuthenticationFailed( false ),
            m_Mutex( true ),
            m_NotifierForSuspendAndResumeOnMainThread( nn::os::EventClearMode_AutoClear ),
            m_pExternalDisplay( nullptr ),
            m_IsExternalDisplayConnected( false )
        {
        }

        // デストラクタ
        ~NotificationController()  NN_NOEXCEPT
        {
            this->Finalize();
        }

        // メッセージコントローラを初期化します。
        void Initialize() NN_NOEXCEPT
        {
            NN_ASSERT( !m_IsInitialized );
            m_IsInitialized = true;

            // コントローラ終了イベントの初期化
            nn::os::InitializeEvent( &m_ExitEvent, false, nn::os::EventClearMode_AutoClear );

            // HDCP 認証失敗イベントの取得
            nn::ae::GetHdcpAuthenticationFailedEvent( &m_HdcpAuthenticationFailedEvent );

            if ( nn::vi::OpenDisplay(&m_pExternalDisplay, "External").IsSuccess() )
            {
                nn::vi::GetDisplayHotplugEvent( m_HotplugEvent.GetBase(), m_pExternalDisplay );
                this->UpdateExternalDisplayConnectedStatus();
            }

            // ae メッセージ通知イベントの初期化
            DEVMENU_LOG( "Invoke nn::ae::InitializeNotificationMessageEvent()\n" );
            nn::ae::InitializeNotificationMessageEvent( &m_NotifyEvent );
            m_NotifierForSuspendAndResumeOnMainThread.Clear();
            m_EventToGetForeground.Clear();

        #if defined( DEVMENU_ENABLE_SLEEP_HANDLING )
            // スリープハンドリングを開始
            DEVMENU_LOG( "Invoke nn::ae::BeginSleepHandling()\n" );
            nn::ae::BeginSleepHandling();
        #endif // defined( DEVMENU_ENABLE_SLEEP_HANDLING )

            // スレッド作成
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &m_Thread,
                    []( void* pArg ){ reinterpret_cast< NotificationController* >( pArg )->ThreadFunction(); },
                    this, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority ) );
            nn::os::SetThreadNamePointer( &m_Thread, "DevMenuNotificationControllerThread" );
            nn::os::StartThread( &m_Thread );

            // クレードル接続で起動しているならば、スリープする
            if ( nn::ae::ShouldSleepOnBoot() )
            {
                this->StartSleep();
            }
        }

        // メッセージコントローラを終了します。
        void Finalize() NN_NOEXCEPT
        {
            if ( m_IsInitialized )
            {
                m_IsInitialized = false;

            #if defined( DEVMENU_ENABLE_SLEEP_HANDLING )
                // スリープハンドリングを停止
                DEVMENU_LOG( "Invoke nn::ae::EndSleepHandling()\n" );
                nn::ae::EndSleepHandling();
            #endif // defined( DEVMENU_ENABLE_SLEEP_HANDLING )

                // スレッド破棄
                this->ReadyToExit();
                nn::os::WaitThread( &m_Thread );
                nn::os::DestroyThread( &m_Thread );

                // イベント破棄
                nn::os::FinalizeEvent( &m_ExitEvent );
                nn::os::DestroySystemEvent( &m_NotifyEvent );

                m_NotificationMessageNotifier.Finalize();

                if ( this->IsExternalDisplaySupported() )
                {
                    nn::vi::CloseDisplay(m_pExternalDisplay);
                }
            }
        }

        // 初回の FG 化まで待機する。
        void WaitForInitialize() NN_NOEXCEPT
        {
            m_EventToGetForeground.Wait();
        }

        // 終了要求されているか判定して返す。
        bool IsRequestedToExit() const NN_NOEXCEPT
        {
            return m_IsRequestedToExit;
        }

        // 終了準備完了を通知する。
        void ReadyToExit() NN_NOEXCEPT
        {
            nn::os::SignalEvent( &m_ExitEvent );
        }

        // スリープ要求されているか判定して返す。
        // ※スリープハンドリングする場合に使用します。
        bool IsRequestedToSleep() const NN_NOEXCEPT
        {
            return m_IsRequestedToSleep;
        }

        // スリープ準備完了を通知し、起床するまで待機する。
        // ※スリープハンドリングする場合に使用します。
        void ReadyToSleepAndWaitForAwake() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsRequestedToSleep );

            m_IsRequestedToSleep = false;
            DEVMENU_LOG( "Invoke nn::ae::AllowToEnterSleepAndWait()\n" );
            nn::ae::AllowToEnterSleepAndWait();
            m_LastAwakeTime = nn::os::GetSystemTick();
        }

        // サスペンド要求されているか判定して返す。
        bool IsRequestedToSuspend() NN_NOEXCEPT
        {
            return ( m_NotifierForSuspendAndResumeOnMainThread.TryWait() || !m_EventToGetForeground.TryWait() );
        }

        // サスペンド準備完了を通知し、レジュームするまで待機する。
        void ReadyToSuspendAndWaitForResume() NN_NOEXCEPT
        {
            m_EventToGetForeground.Wait();
        }

        // 本体再起動を要求する
        void RequestRebootDevice() NN_NOEXCEPT
        {
            if ( launcher::IsApplicationAlive() )
            {
                launcher::RequestApplicationExit();
                this->SetDeviceRebootRequiredAfterApplicationExit( true );
            }
            else
            {
                nn::ae::StartRebootSequence();
            }
        }

        // Rid デモアプリケーションの終了メニュー表示が要求されているかを判断して返す。
        bool IsRequestedToStartRidExitMenu() NN_NOEXCEPT
        {
            const auto isRidExitMenuTriggered = m_IsRidExitMenuTriggered;
            if ( isRidExitMenuTriggered )
            {
                std::lock_guard<nn::os::Mutex> lock( m_Mutex );
                m_IsRidExitMenuTriggered = false;
            }
            return isRidExitMenuTriggered;
        }

        // Rid デモアプリケーションの終了メニューのキャンセルボタン押下時の処理を行います。
        void CancelRidExitMenu() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsForeground );

            // BG に遷移する
            if ( !m_IsRidExitMenuTriggered )
            {
                this->RequestToSwitchForeground();
            }
        }

        // SD カードのマウント状態の変更有無を判定します。
        bool IsSdCardDetectionStateChanged() NN_NOEXCEPT
        {
            const auto isSdCardDetectionStateChanged = m_IsSdCardDetectionStateChanged;
            if ( isSdCardDetectionStateChanged )
            {
                std::lock_guard<nn::os::Mutex> lock( m_Mutex );
                m_IsSdCardDetectionStateChanged = false;
            }
            return isSdCardDetectionStateChanged;
        }

        // SD カードのマウント状態の変更有無を設定します。
        void SetSdCardDetectionStateChanged( bool isChanged ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );
            m_IsSdCardDetectionStateChanged = isChanged;
        }

        // HDCP 認証が失敗したかどうかを判定します。
        bool IsHdcpAuthenticationFailed() const NN_NOEXCEPT
        {
            return m_IsHdcpAuthenticationFailed;
        }

        bool IsExternalDisplayConnected() const NN_NOEXCEPT
        {
            return m_IsExternalDisplayConnected;
        }

        // HDCP 認証の失敗状態を設定します。
        void SetHdcpAuthenticationFailedStatus( bool isFailed ) NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );
            m_IsHdcpAuthenticationFailed = isFailed;
        }

        bool IsExternalDisplaySupported() const NN_NOEXCEPT
        {
            return m_pExternalDisplay != nullptr;
        }

        void UpdateExternalDisplayConnectedStatus() NN_NOEXCEPT
        {
            std::lock_guard<nn::os::Mutex> lock( m_Mutex );

            nn::vi::HotplugState state;
            m_IsExternalDisplayConnected = this->IsExternalDisplaySupported() &&
                                           nn::vi::GetDisplayHotplugState( &state, m_pExternalDisplay ).IsSuccess() &&
                                           state == nn::vi::HotplugState_Connected;
        }

        // レシーバー管理コンテキスト取得
        NN_FORCEINLINE NotificationMessageNotifier& GetNotificationMessageNotifier() NN_NOEXCEPT
        {
            return m_NotificationMessageNotifier;
        }

    private:
        // 終了を要求する。
        void RequestToExit() NN_NOEXCEPT
        {
            m_IsRequestedToExit = true;

            // レジューム待機中は再開させる
            m_NotifierForSuspendAndResumeOnMainThread.Clear();
            m_EventToGetForeground.Signal();
        }

        // シャットダウンを開始する
        static void StartShutdown() NN_NOEXCEPT
        {
            DEVMENU_LOG( "Invoke nn::ae::StartShutdownSequence()\n" );
            nn::ae::StartShutdownSequence();
            nn::os::SleepThread( nn::TimeSpan::FromDays( 1 ) );
        }

        // スリープ遷移を開始する。
        static void StartSleep() NN_NOEXCEPT
        {
        #if defined( DEVMENU_ENABLE_SLEEP_START )
            DEVMENU_LOG( "Invoke nn::ae::StartSleepSequence()\n" );
            nn::ae::StartSleepSequence();
        #endif // defined( DEVMENU_ENABLE_SLEEP_START )
        }

        // スリープ遷移を終了する。
        void FinishSleep() NN_NOEXCEPT
        {
            NN_ASSERT( IsForeground() );
            m_IsRequestedToSleep = false;

        #if defined( DEVMENU_ENABLE_SLEEP_HANDLING )

            // exhibition モード時は、スリープ復帰で実行中のアプリケーションを FG 遷移させる
            if ( exhibition::IsExhibitionModeEnabled() )
            {
                // 実行中アプリケーションが存在する
                if ( launcher::UpdateApplicationState() )
                {
                    launcher::RequestApplicationForeground();
                }
            }
           // Retail Interactive Display の場合はスリープ復帰で再起動する
            else if ( rid::IsRetailInteractiveDisplay() )
            {
                this->RequestRebootDevice();
            }
            else
            {
                // Retail Interactive Display メニューが BG で実行中の場合は FG 遷移させる（Rid メニューの開発効率を上げるため）
                if ( rid::IsRetailInteractiveDisplayMenuRunning( false ) )
                {
                    launcher::RequestApplicationForeground();
                }
            }
        #endif // defined( DEVMENU_ENABLE_SLEEP_HANDLING )
        }

        // サスペンドを要求し、準備完了するまで待機する。
        void RequestToSuspendAndWaitForReady() NN_NOEXCEPT
        {
            m_NotifierForSuspendAndResumeOnMainThread.Signal();
            m_EventToGetForeground.Clear();
        }

        // レジュームを要求する。
        void RequestToResume() NN_NOEXCEPT
        {
            m_EventToGetForeground.Signal();
        }

        // FG 遷移後の処理を実行する。
        void ChangeIntoForeground() NN_NOEXCEPT
        {
            NN_ASSERT( !m_IsForeground );
            m_IsForeground = true;

            launcher::UpdateApplicationLaunchProperty();
            this->GetNotificationMessageNotifier()
                .NotifyExecute( NotificationMessageReceiver::Message_ChangeIntoForeground );
            this->RequestToResume();

            // TORIAEZU : (SIGLO-68982)
            // DevMenu の状態遷移と am の通知はそれぞれ別スレッドで扱うのがよいが、ひとまずは見た目の問題に暫定対処を施す
            if ( this->IsRequestedToSleep() )
            {
                // 出画されるまで念のため数フレーム待つ
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 17 * 3 ) );
                this->ReadyToSleepAndWaitForAwake();
            }
        }

        // BG 遷移後の処理を実行する。
        void ChangeIntoBackground() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsForeground );
            m_IsForeground = false;

            this->GetNotificationMessageNotifier()
                .NotifyExecute( NotificationMessageReceiver::Message_ChangeIntoBackground );
            this->RequestToSuspendAndWaitForReady();
        }

        // 外部起動アプリを FG へ遷移するように要求する。
        void RequestToActivateFloatingApplication() const NN_NOEXCEPT
        {
            launcher::CheckFloatingApplication();
        }

        // FG / BG を切り替えるように要求する。
        void RequestToSwitchForeground() const NN_NOEXCEPT
        {
            if ( m_IsForeground )
            {
                launcher::RequestApplicationForeground();
            }
            else
            {
                DEVMENU_LOG( "Invoke nn::ae::RequestToGetForeground()\n" );
                nn::ae::RequestToGetForeground();
            }
        }

        // アプリケーション終了メッセージ受信時の処理を行う。
        void HandleApplicationExitedMessage() NN_NOEXCEPT
        {
            // アプリケーション終了イベントがシグナルされるのを待つが、通常は即座に終了する
            launcher::WaitForApplicationExited();

            // exhibition mode 時は必ず本体再起動してアプリケーションを自動起動する
            if ( exhibition::IsExhibitionModeEnabled() )
            {
                this->SetDeviceRebootRequiredAfterApplicationExit( true );
            }

            if ( this->IsDeviceRebootRequiredAfterApplicationExit() )
            {
                nn::ae::StartRebootSequence();
            }
            else
            {
                // Note:
                // exhibition mode でのアプリケーションジャンプをサポートするなら、適切な調整が必要になる
                // 愚直に対処するなら、アプリケーションジャンプ時は SetDeviceRebootRequiredAfterApplicationExit( true ) を呼ばなければよい
                launcher::UpdateApplicationStateAfterExit();
            }
        }

        // HOME ボタン短押し検出メッセージ受信時の処理を行う。
        void HandleShortPressingHomeButtonDetectionMessage() NN_NOEXCEPT
        {
            bool isIgnored = true;
            NN_UTIL_SCOPE_EXIT
            {
                if ( isIgnored )
                {
                    DEVMENU_LOG_AE( "(Message is ignored)\n" );
                }
            };

            // exhibition mode 時は HOME ボタン無効なので処理の対象外にする
            if ( exhibition::IsExhibitionModeEnabled() )
            {
                return;
            }

            // Rid モードで FG で Rid メニュー実行中の場合は、HOME ボタン無効なので処理の対象外にする
            if ( rid::IsRetailInteractiveDisplayMenuRunning() && !this->IsForeground() )
            {
                return;
            }

            // HDCP 認証失敗時は HOME ボタンで認証失敗状態をクリアする
            // ToDo: Home ボタンにアサインすべき処理ではないので別のボタンを割り当てる
            if ( this->IsHdcpAuthenticationFailed() )
            {
                this->SetHdcpAuthenticationFailedStatus( false );
                return;
            }

            // スリープ直前に投げたメッセージが復帰後に処理されてしまうケースをなるべく防ぐため、
            // 復帰直後に届いた重要でないイベントは無視する
            if ( this->IsRightAfterAwake() )
            {
                return;
            }

            // Rid モードで FG でデモアプリケーション実行中の場合は、終了メニュー表示のトリガをかける
            if ( rid::IsRetailInteractiveDisplay() && !this->IsForeground() )
            {
                this->StartRidExitMenu();
            }
            this->RequestToSwitchForeground();
            isIgnored = false;
        }

        void RequestToSleep() NN_NOEXCEPT
        {
            NN_ASSERT( !m_IsRequestedToSleep );

            m_IsRequestedToSleep = true;

        #if defined( DEVMENU_ENABLE_SLEEP_HANDLING )
            // LCD-OFF フェードアウト中に アプリ->SA 遷移が見えてしまうのを防ぐ
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 500 ) );

            if ( m_IsForeground )
            {
                // TORIAEZU : (SIGLO-68982)
                // DevMenu の状態遷移と am の通知はそれぞれ別スレッドで扱うのがよいが、ひとまずは見た目の問題に暫定対処を施す
                // この時点で Foreground 状態であった場合のみ、即座にスリープに遷移して待機する
                this->ReadyToSleepAndWaitForAwake();
            }
            else
            {
                // DevMenu が BG の場合は FG 遷移後にスリープする
                // ChangeIntoForeground() 内で ReadyToSleepAndWaitForAwake() が呼ばれる

                // DevMenu が起動した LA が FG で動作中
                if ( launcher::IsLibraryAppletLaunchedByDevMenu() )
                {
                    nn::ae::LockForeground();
                }
                else
                {
                    // 実行中アプリケーションが存在する
                    if ( launcher::IsApplicationAlive() )
                    {
                        // DevMenu が FG となるように要求する。
                        DEVMENU_LOG( "Invoke nn::ae::RequestToGetForeground()\n" );
                        nn::ae::RequestToGetForeground();
                    }
                }
            }
        #endif // defined( DEVMENU_ENABLE_SLEEP_HANDLING )
        }

        // 現在が時間的にスリープから復帰した直後の期間かどうかを返す
        bool IsRightAfterAwake() const NN_NOEXCEPT
        {
            return nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() - m_LastAwakeTime ) < HustTimeAfterAwake;
        }

        // 本体再起動が要求されている場合は本体を再起動する
        void RebootDeviceIfNecessary() NN_NOEXCEPT
        {
            if ( m_IsDeviceRebootRequiredAfterApplicationExit )
            {
                nn::ae::StartRebootSequence();
            }
        }

        // アプリケーション終了後の本体再起動を要求する
        void SetDeviceRebootRequiredAfterApplicationExit( bool isRequired ) NN_NOEXCEPT
        {
            m_IsDeviceRebootRequiredAfterApplicationExit = isRequired;
        }

        // アプリケーション終了後の本体再起動が要求されているかを確認する
        bool IsDeviceRebootRequiredAfterApplicationExit() const NN_NOEXCEPT
        {
            return m_IsDeviceRebootRequiredAfterApplicationExit;
        }

        // Rid デモアプリケーションの終了メニュー表示を開始する
        void StartRidExitMenu() NN_NOEXCEPT
        {
            m_IsRidExitMenuTriggered = true;
        }

        // Rid デモアプリケーションを終了して Rid メニューを起動する
        void ExitDemoApplicationAndLaunchRetailInteractiveDisplayMenu() NN_NOEXCEPT
        {
            m_IsRidExitMenuTriggered = true;
            rid::LaunchRetailInteractiveDisplayMenu( false );
        }

        // Foreground 状態であるかどうをか返す
        bool IsForeground() const NN_NOEXCEPT
        {
            return m_IsForeground;
        }

        // スレッド関数
        void ThreadFunction() NN_NOEXCEPT
        {
            DEVMENU_LOG( "NotificationController thread start.\n" );

            // 起動前に FloatingApplication があった場合には、メッセージが来ないので、一旦呼んでおく
            this->RequestToActivateFloatingApplication();

            nn::os::MultiWaitType multiWait;
            nn::os::InitializeMultiWait( &multiWait );

            // ae メッセージ向けのイベント
            nn::os::MultiWaitHolderType multiWaitHolderForNotification;
            nn::os::InitializeMultiWaitHolder( &multiWaitHolderForNotification, &m_NotifyEvent );
            nn::os::LinkMultiWaitHolder( &multiWait, &multiWaitHolderForNotification );

            // NotificationController の終了イベント
            nn::os::MultiWaitHolderType multiWaitHolderForExit;
            nn::os::InitializeMultiWaitHolder( &multiWaitHolderForExit, &m_ExitEvent );
            nn::os::LinkMultiWaitHolder( &multiWait, &multiWaitHolderForExit );

            // 本体機能から届くメッセージ向けのイベント
            auto pSystemGeneralChannelEvent = nn::ae::GetPopFromSystemGeneralChannelEvent();
            nn::os::MultiWaitHolderType multiWaitHolderForSystemGeneralChannelEvent;
            nn::os::InitializeMultiWaitHolder( &multiWaitHolderForSystemGeneralChannelEvent, pSystemGeneralChannelEvent );
            nn::os::LinkMultiWaitHolder( &multiWait, &multiWaitHolderForSystemGeneralChannelEvent );

            nn::applet::StorageHandle storageHandle;
            static nn::Bit8 s_StorageBuffer[ 1024 ]; // TORIAEZU : 本体機能から SA に Push される最大サイズ分のバッファ

            // SD カード挿抜イベント
            // 本当は nn::ns::GetSdCardMountStatusChangedEvent() で処理したいが、都合により fs から取得したイベントを直接使用する
            // ns の仕様により、SD カードが利用可能な状態から抜去を行うと、その後はカード挿抜でシグナルされなくなる
            std::unique_ptr<nn::fs::IEventNotifier> sdCardDetectionEventNotifier;
            nn::os::SystemEvent sdCardDetectionEvent;
            nn::fs::OpenSdCardDetectionEventNotifier( &sdCardDetectionEventNotifier );
            sdCardDetectionEventNotifier->BindEvent( sdCardDetectionEvent.GetBase(), nn::os::EventClearMode_ManualClear );
            nn::os::MultiWaitHolderType multiWaitHolderForSdCardDetectionEvent;
            nn::os::InitializeMultiWaitHolder( &multiWaitHolderForSdCardDetectionEvent, sdCardDetectionEvent.GetBase() );
            nn::os::LinkMultiWaitHolder( &multiWait, &multiWaitHolderForSdCardDetectionEvent );

            // HDCP 認証失敗イベント
            nn::os::MultiWaitHolderType multiWaitHolderForHdcpAuthenticationFailedEvent;
            nn::os::InitializeMultiWaitHolder( &multiWaitHolderForHdcpAuthenticationFailedEvent, m_HdcpAuthenticationFailedEvent.GetBase() );
            nn::os::LinkMultiWaitHolder( &multiWait, &multiWaitHolderForHdcpAuthenticationFailedEvent );

            // External Display Hotplug Event
            nn::os::MultiWaitHolderType multiWaitHolderForHotplugEvent;

            if ( this->IsExternalDisplaySupported() )
            {
                nn::os::InitializeMultiWaitHolder( &multiWaitHolderForHotplugEvent, m_HotplugEvent.GetBase() );
                nn::os::LinkMultiWaitHolder( &multiWait, &multiWaitHolderForHotplugEvent );
            }

            // メッセージ通知処理
            do
            {
                auto holder = nn::os::WaitAny( &multiWait );

                if ( holder == &multiWaitHolderForNotification &&
                     nn::os::TryWaitSystemEvent( &m_NotifyEvent ) )
                {
                    const auto message = nn::ae::GetNotificationMessage();
                    switch( message )
                    {
                    case nn::ae::Message_None:
                        break;

                    // FG ⇒ BG 遷移
                    case nn::ae::Message_ChangeIntoBackground:
                        {
                            DEVMENU_LOG( "Received Message_ChangeIntoBackground\n" );
                            this->ChangeIntoBackground();
                        }
                        break;

                    // BG ⇒ FG 遷移
                    case nn::ae::Message_ChangeIntoForeground:
                        {
                            DEVMENU_LOG( "Received Message_ChangeIntoForeground\n" );
                            this->ChangeIntoForeground();
                        }
                        break;

                    // アプリケーション起動リクエスト(アプリケーションジャンプ)
                    case nn::ae::Message_LaunchApplicationRequested:
                        {
                            DEVMENU_LOG( "Received LaunchApplicationRequested\n" );
                            this->RequestToSwitchForeground();
                            launcher::RequestApplicationJump();
                        }
                        break;

                    // アプリ外部起動
                    case nn::ae::Message_FloatingApplicationDetected:
                        {
                            DEVMENU_LOG( "Received FloatingApplicationDetected\n" );
                            this->RequestToActivateFloatingApplication();
                        }
                        break;

                    // Rid メニュー起動要求
                    case nn::ae::Message_RequestToGoBackQuestMenu:
                        {
                            NN_ASSERT( rid::IsRetailInteractiveDisplay() );
                            DEVMENU_LOG( "Received RequestToGoBackQuestMenu\n" );

                            // Rid メニューの起動
                            if ( !this->IsForeground() )
                            {
                                this->RequestToSwitchForeground();
                            }
                            this->ExitDemoApplicationAndLaunchRetailInteractiveDisplayMenu();
                        }
                        break;

                    // 権利喪失によるアプリケーションの強制中断
                    case nn::ae::Message_ApplicationSuspendedByRightsError:
                        {
                            DEVMENU_LOG( "Received Message_ApplicationSuspendedByRightsError\n" );
                            if ( !m_IsForeground )
                            {
                                nn::ae::RequestToGetForeground();
                            }
                            // ToDo: 権利復帰の処理を実施する（実装待ち）
                        }
                        break;

                    // アプリ終了(正常、異常の両方)
                    case nn::ae::Message_ApplicationExited:
                        {
                            DEVMENU_LOG( "Received Message_ApplicationExited\n" );
                            this->HandleApplicationExitedMessage();
                        }
                        break;

                    // HOME 短押下 ⇒ アプリと FG/BG 切替
                    case nn::ae::Message_DetectShortPressingHomeButton:
                        {
                            DEVMENU_LOG( "Received Message_DetectShortPressingHomeButton\n" );
                            this->HandleShortPressingHomeButtonDetectionMessage();
                        }
                        break;

                    // HOME 長押下
                    case nn::ae::Message_DetectLongPressingHomeButton:
                        {
                            DEVMENU_LOG( "Received Message_DetectLongPressingHomeButton\n" );
                        }
                        break;

                    // 電源短押下 ⇒ スリープ要求
                    case nn::ae::Message_DetectShortPressingPowerButton:
                        {
                            DEVMENU_LOG( "Received Message_DetectShortPressingPowerButton\n" );
                            // スリープ直前に投げたメッセージが復帰後に処理されてしまうケースをなるべく防ぐため、
                            // 復帰直後に届いた重要でないイベントは無視する
                            if ( !this->IsRightAfterAwake() )
                            {
                                this->StartSleep();
                            }
                            else
                            {
                                DEVMENU_LOG( "(Ignored)\n" );
                            }
                        }
                        break;

                    // 電源長(7sec)押下 ⇒ シャットダウン要求
                    case nn::ae::Message_DetectLongPressingPowerButton:
                        {
                            DEVMENU_LOG( "Received Message_DetectLongPressingPowerButton\n" );
                            this->StartShutdown();
                        }
                        break;

                    // CEC システムスタンバイ受信 ⇒ スリープ要求
                    case nn::ae::Message_DetectReceivingCecSystemStandby:
                        {
                            DEVMENU_LOG( "Received Message_DetectReceivingCecSystemStandby\n" );
                            // スリープ直前に投げたメッセージが復帰後に処理されてしまうケースをなるべく防ぐため、
                            // 復帰直後に届いた重要でないイベントは無視する
                            if ( !this->IsRightAfterAwake() )
                            {
                                this->StartSleep();
                            }
                            else
                            {
                                DEVMENU_LOG( "(Ignored)\n" );
                            }
                        }
                        break;

                    // SoC が高温状態 ⇒ スリープ要求
                    case nn::ae::Message_SleepRequiredByHighTemperature:
                        {
                            DEVMENU_LOG( "Received Message_SleepRequiredByHighTemperature\n" );
                            this->StartSleep();
                        }
                        break;

                    // バッテリー残量不足 ⇒ スリープ要求
                    case nn::ae::Message_SleepRequiredByLowBattery:
                        {
                            DEVMENU_LOG( "Received Message_SleepRequiredByLowBattery\n" );
                            this->StartSleep();
                        }
                        break;

                    // 一定時間無操作 ⇒ スリープ要求
                    case nn::ae::Message_AutoPowerDown:
                        {
                            DEVMENU_LOG( "Received Message_AutoPowerDown\n" );
                            // スリープ直前に投げたメッセージが復帰後に処理されてしまうケースをなるべく防ぐため、
                            // 復帰直後に届いた重要でないイベントは無視する
                            if ( !this->IsRightAfterAwake() )
                            {
                                this->StartSleep();
                            }
                            else
                            {
                                DEVMENU_LOG( "(Ignored)\n" );
                            }
                        }
                        break;

                    // スリープシーケンスの開始
                    case nn::ae::Message_RequestToPrepareSleep:
                        {
                            DEVMENU_LOG( "Received Message_RequestToPrepareSleep\n" );
                            this->RequestToSleep();
                        }
                        break;

                    // スリープシーケンスの終了
                    case nn::ae::Message_FinishedSleepSequence:
                        {
                            DEVMENU_LOG( "Received Message_FinishedSleepSequence\n" );
                            this->FinishSleep();

                            // DevMenu が起動した LA が FG で動作中
                            if ( launcher::IsLibraryAppletLaunchedByDevMenu() )
                            {
                                nn::ae::UnlockForeground();
                            }
                        }
                        break;

                    // 動作モード変化
                    case nn::ae::Message_OperationModeChanged:
                        {
                            DEVMENU_LOG( "Received Message_OperationModeChanged\n" );
                        }
                        break;

                    // 性能モード変化
                    case nn::ae::Message_PerformanceModeChanged:
                        {
                            DEVMENU_LOG( "Received Message_PerformanceModeChanged\n" );
                        }
                        break;

                    // VR モードの変更
                    case nn::ae::Message_VrModeChanged:
                        {
                            DEVMENU_LOG( "Received Message_VrModeChanged: VR-mode is %s\n", nn::ae::IsVrMode() ? "true" : "false" );
                        }
                        break;

                    // VR カーテンの要求
                    case nn::ae::Message_VrModeCurtainRequired:
                        {
                            DEVMENU_LOG( "Received Message_VrModeCurtainRequired\n" );
                            nn::applet::EndVrModeInternal();
                        }
                        break;

                    // シャットダウン要求
                    case nn::ae::Message_RequestToShutdown:
                        {
                            DEVMENU_LOG( "Received Message_RequestToShutdown\n" );
                            this->StartShutdown();
                        }
                        break;

                    // 再起動要求
                    case nn::ae::Message_RequestToReboot:
                        {
                            DEVMENU_LOG( "Received Message_RequestToReboot\n" );
                            this->RequestRebootDevice();
                        }
                        break;

                    // SD カードの抜去
                    case nn::ae::Message_SdCardRemoved:
                        {
                            DEVMENU_LOG( "Message_SdCardRemoved\n" );
                        }
                        break;

                    // 終了
                    case nn::ae::Message_Exit:
                        {
                            DEVMENU_LOG( "Received Message_Exit\n" );
                            this->RequestToExit();
                        }
                        break;

                    // 終了
                    case nn::ae::Message_RequestToLaunchApplicationForDebug:
                        {
                            DEVMENU_LOG( "Received Message_RequestToLaunchApplicationForDebug\n" );
                            nn::ncm::ApplicationId applicationId;
                            nn::account::Uid uid;
                            if (nn::ae::PopRequestLaunchApplicationForDebug(&applicationId, &uid))
                            {
                                DEVMENU_LOG("  RequestLaunchApplicationForDebug(0x%016llX, uid=%016llX,%016llX)\n", applicationId.value, static_cast<unsigned long long>(uid._data[0]), static_cast<unsigned long long>(uid._data[1]));
                                // TODO: applicationId のアプリケーションを uid のユーザーで起動する
                                NN_UNUSED(applicationId);
                                NN_UNUSED(uid);
                            }
                        }
                        break;

                    default:
                        {
                            DEVMENU_LOG( "Received unknown message= 0x%08x\n", message );
                        }
                        break;
                    }
                }
                else if ( holder == &multiWaitHolderForExit &&
                          nn::os::TryWaitEvent( &m_ExitEvent ) )
                {
                    // 終了準備完了
                    break;
                }
                else if ( holder == &multiWaitHolderForSystemGeneralChannelEvent &&
                         nn::ae::TryPopFromSystemGeneralChannel( &storageHandle ) )
                {
                    const auto size = nn::applet::GetStorageSize( storageHandle );
                    if ( size > sizeof( s_StorageBuffer ) )
                    {
                        DEVMENU_LOG(" Buffer size is insufficient to receive this message : Required = %llu, Prepared = %llu\n", size, sizeof( s_StorageBuffer ) );
                        continue;
                    }
                    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage( storageHandle, 0, s_StorageBuffer, size ) );

                    DEVMENU_LOG( "Received message from LA.\n" );

                    // ErrorViewer トリガの動作中のアプリケーションの再起動
                    if (  0 == std::memcmp( s_StorageBuffer, RelaunchCurrentApplicationMessage, sizeof( RelaunchCurrentApplicationMessage ) ) )
                    {
                        DEVMENU_LOG( "Received relaunch request message from ErrorViewer\n" );

                        const auto applicationId = launcher::GetActiveApplicationId();
                        DEVMENU_LOG( "RelaunchCurrentApplication requested (0x%016x).\n", applicationId.value );
                        NN_UNUSED( applicationId );
                        // TODO : 再起動する。
                        DEVMENU_LOG( "RelaunchCurrentApplication is not implemented yet. Use RequestApplicationExit instead for now.\n", applicationId.value );
                        launcher::RequestApplicationExit();
                    }
                    // MyPage トリガのアプリケーションの起動
                    else if (  0 == std::memcmp( s_StorageBuffer, LaunchApplicationMessage, sizeof( LaunchApplicationMessage ) ) )
                    {
                        DEVMENU_LOG( "Received launch request message from MyPage\n" );

                        nn::ncm::ApplicationId applicationId;
                        std::memcpy( &applicationId.value, s_StorageBuffer + sizeof( LaunchApplicationMessage ), sizeof( applicationId.value ) );
                        launcher::RequestLaunchTriggeredByLibraryApplet( applicationId );
                    }
                }
                else if ( holder == &multiWaitHolderForSdCardDetectionEvent )
                {
                    sdCardDetectionEvent.Clear();
                    this->SetSdCardDetectionStateChanged( true );
                }
                else if ( holder == &multiWaitHolderForHdcpAuthenticationFailedEvent )
                {
                    DEVMENU_LOG( "HDCP Authentication Failed.\n" );
                    this->m_HdcpAuthenticationFailedEvent.Clear();
                    this->SetHdcpAuthenticationFailedStatus( true );
                    if ( !this->IsForeground() )
                    {
                        this->RequestToSwitchForeground();
                    }
                }
                else if ( holder == &multiWaitHolderForHotplugEvent )
                {
                    DEVMENU_LOG( "Detected change in external display.\n" );
                    this->m_HotplugEvent.Clear();
                    this->UpdateExternalDisplayConnectedStatus();
                }
            } while( NN_STATIC_CONDITION( true ) );

            // 終了処理
            nn::os::UnlinkMultiWaitHolder( &multiWaitHolderForNotification );
            nn::os::FinalizeMultiWaitHolder( &multiWaitHolderForNotification );

            nn::os::UnlinkMultiWaitHolder( &multiWaitHolderForExit );
            nn::os::FinalizeMultiWaitHolder( &multiWaitHolderForExit );

            nn::os::UnlinkMultiWaitHolder( &multiWaitHolderForSystemGeneralChannelEvent );
            nn::os::FinalizeMultiWaitHolder( &multiWaitHolderForSystemGeneralChannelEvent );

            nn::os::UnlinkMultiWaitHolder( &multiWaitHolderForSdCardDetectionEvent );
            nn::os::FinalizeMultiWaitHolder( &multiWaitHolderForSdCardDetectionEvent );

            nn::os::UnlinkMultiWaitHolder( &multiWaitHolderForHdcpAuthenticationFailedEvent );
            nn::os::FinalizeMultiWaitHolder( &multiWaitHolderForHdcpAuthenticationFailedEvent );

            nn::os::UnlinkMultiWaitHolder( &multiWaitHolderForHotplugEvent );
            nn::os::FinalizeMultiWaitHolder( &multiWaitHolderForHotplugEvent );

            nn::os::FinalizeMultiWait( &multiWait );

            DEVMENU_LOG( "NotificationController thread finished.\n" );
        } // NOLINT(impl/function_size)

    private:
        const nn::TimeSpan  HustTimeAfterAwake = nn::TimeSpan::FromMilliSeconds( 500 );

    private:
        bool    m_IsInitialized;
        bool    m_IsForeground;
        bool    m_IsRequestedToExit;
        bool    m_IsRequestedToSleep;
        bool    m_IsDeviceRebootRequiredAfterApplicationExit;

        bool    m_IsRidExitMenuTriggered;
        RidDialogType m_RidDialogType;

        nn::os::Tick            m_LastAwakeTime;    // 最後に起床した時刻

        mutable nn::os::Event   m_EventToGetForeground; // FG 遷移時（と DevMenu 終了要求時）にシグナルされるイベント

        nn::os::EventType       m_ExitEvent;        // 終了待機イベント
        nn::os::SystemEventType m_NotifyEvent;      // メッセージ通知待機イベント
        nn::os::ThreadType      m_Thread;           // メッセージ通知処理スレッド

        bool m_IsSdCardDetectionStateChanged;       // SD カード状態の変化したか否か

        nn::os::SystemEvent m_HdcpAuthenticationFailedEvent; // HDCP 認証失敗イベント
        bool m_IsHdcpAuthenticationFailed;

        nn::os::Mutex           m_Mutex;            // 排他ロック

        mutable nn::os::LightEvent  m_NotifierForSuspendAndResumeOnMainThread;
        NotificationMessageNotifier m_NotificationMessageNotifier;

        nn::vi::Display* m_pExternalDisplay;
        nn::os::SystemEvent m_HotplugEvent;
        bool m_IsExternalDisplayConnected;
    };

    // 変数宣言
    NotificationController g_NotificationController;

} // end of unnamed namespace

void InitializeNotification() NN_NOEXCEPT
{
    g_NotificationController.Initialize();
}

void FinalizeNotification() NN_NOEXCEPT
{
    g_NotificationController.Finalize();
}

void WaitForNotificationInitialized() NN_NOEXCEPT
{
    g_NotificationController.WaitForInitialize();
}

bool IsRequestedToExit() NN_NOEXCEPT
{
    return g_NotificationController.IsRequestedToExit();
}

void ReadyToExit() NN_NOEXCEPT
{
    g_NotificationController.ReadyToExit();
}

bool IsRequestedToSleep() NN_NOEXCEPT
{
    return g_NotificationController.IsRequestedToSleep();
}

void ReadyToSleepAndWaitForAwake() NN_NOEXCEPT
{
    g_NotificationController.ReadyToSleepAndWaitForAwake();
}

bool IsRequestedToSuspend() NN_NOEXCEPT
{
    return g_NotificationController.IsRequestedToSuspend();
}

void ReadyToSuspendAndWaitForResume() NN_NOEXCEPT
{
    g_NotificationController.ReadyToSuspendAndWaitForResume();
}

void RequestRebootDevice() NN_NOEXCEPT
{
    g_NotificationController.RequestRebootDevice();
}

bool IsRequestedToStartRidExitMenu() NN_NOEXCEPT
{
    return g_NotificationController.IsRequestedToStartRidExitMenu();
}

void RequestToCancelRidExitMenu() NN_NOEXCEPT
{
    g_NotificationController.CancelRidExitMenu();
}

bool IsSdCardDetectionStateChanged() NN_NOEXCEPT
{
    return g_NotificationController.IsSdCardDetectionStateChanged();
}

bool IsHdcpAuthenticationFailed() NN_NOEXCEPT
{
    return g_NotificationController.IsHdcpAuthenticationFailed();
}

bool IsExternalDisplayConnected() NN_NOEXCEPT
{
    return g_NotificationController.IsExternalDisplayConnected();
}

bool NotificationMessageReceiver::EnableNotificationMessageReceiving( bool permission ) NN_NOEXCEPT
{
    const auto previousCondition = IsLinked();
    auto& notifier = g_NotificationController.GetNotificationMessageNotifier();
    if ( permission )
    {
        notifier.Register( this );
    }
    else
    {
        notifier.Unregister( this );
    }
    return previousCondition;
}

} // ~namespace devmenu

#endif
