﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/util/util_IntrusiveList.h>

namespace devmenu {

/**
 * @brief       メッセージ通知処理を初期化します。
 *
 * @details     メインスレッドから呼びます。メッセージ通知の受信を開始します。
 */
void InitializeNotification() NN_NOEXCEPT;

/**
 * @brief       メッセージ通知処理を終了します。
 *
 * @details     メインスレッドから呼びます。メッセージ通知の受信を終了します。
 */
void FinalizeNotification() NN_NOEXCEPT;

/**
 * @brief       メッセージ通知処理の初期化が完了するまで待機します。
 *
 * @details     メインスレッドから呼びます。
 *              メッセージ通知処理の初期化後、描画可能な状態になる（FG 遷移）までブロックされます。
 */
void WaitForNotificationInitialized() NN_NOEXCEPT;

/**
 * @brief       終了が要求されているか判定します。
 *
 * @return      終了が要求されているなら真を返します。
 *
 * @details     メインスレッドから呼びます。
 *              IsRequestedToExit() が真ならば、終了準備を行って ReadyToExit() を呼びます。
 */
bool IsRequestedToExit() NN_NOEXCEPT;

/**
 * @brief       終了の準備が完了したら呼び出します。
 *
 * @details     メインスレッドから呼びます。
 *              IsRequestedToExit() が真ならば、終了準備を行って ReadyToExit() を呼びます。
 *              ReadyToExit() を呼ぶと、メッセージ通知の受信を終了します。
 */
void ReadyToExit() NN_NOEXCEPT;

/**
 * @brief       スリープが要求されているか判定します。
 *
 * @return      スリープが要求されているなら真を返します。
 *
 * @details     スリープハンドルを行う場合（DEVMENU_ENABLE_SLEEP_HANDLING が有効）のみ、BG 遷移後も動作しているスレッドから呼び出します。
 *              スリープ遷移時は、スリープ要求より先に BG 遷移のサスペンド要求が発生しますので、メインスレッドは待機状態にあります。
 *              メインスレッド以外で BG 遷移後に動作しているスレッドがなければ、スリープハンドルを行う必要はありません。
 *              IsRequestedToSleep() が真ならば、スリープ準備を行って ReadyToSleepAndWaitForAwake() を呼びます。
 */
bool IsRequestedToSleep() NN_NOEXCEPT;

/**
 * @brief       スリープの準備が完了したら呼び出します。
 *              呼び出すとスリープ状態になり、起床するまで待機します。
 *
 * @details     スリープハンドルを行う場合（DEVMENU_ENABLE_SLEEP_HANDLING が有効）のみ、BG 遷移後も動作しているスレッドから呼び出します。
 *              スリープ遷移時は、スリープ要求より先に BG 遷移のサスペンド要求が発生しますので、メインスレッドは待機状態にあります。
 *              メインスレッド以外で BG 遷移後に動作しているスレッドがなければ、スリープハンドルを行う必要はありません。
 *              IsRequestedToSleep() が真ならば、スリープ準備を行って ReadyToSleepAndWaitForAwake() を呼びます。
 *              ReadyToSleepAndWaitForAwake() を呼ぶと、起床するまでブロックされます。
 */
void ReadyToSleepAndWaitForAwake() NN_NOEXCEPT;

/**
 * @brief       BG に遷移するためにサスペンドが要求されているか判定します。
 *
 * @return      サスペンドが要求されているなら真を返します。
 *
 * @details     メインスレッドから呼びます。
 *              IsRequestedToSuspend() が真ならば、サスペンド準備を行って ReadyToSuspendAndWaitForResume() を呼びます。
 */
bool IsRequestedToSuspend() NN_NOEXCEPT;

/**
 * @brief       サスペンドの準備が完了したら呼び出します。
 *              呼び出すとサスペンド状態になり、レジュームするまで待機します。
 *
 * @details     メインスレッドから呼びます。
 *              IsRequestedToSuspend() が真ならば、サスペンド準備を行って ReadyToSuspendAndWaitForResume() を呼びます。
 *              ReadyToSuspendAndWaitForResume を呼ぶと、再び FG 遷移または終了要求でレジュームするまでブロックされます。
 */
void ReadyToSuspendAndWaitForResume() NN_NOEXCEPT;

/**
 * @brief       本体の再起動を要求します。
 *
 * @details     メインスレッドから呼びます。
 *              活動中のアプリケーションが存在しない場合は即座に本体を再起動します。
 *              活動中のアプリケーションが存在する場合は、アプリケーション終了を行い、アプリケーション終了の通知が届いたら再起動します。
 *
 */
void RequestRebootDevice() NN_NOEXCEPT;

/**
 * @brief       RID デモアプリケーション向けのダイアログ種別です。
 */
enum RidDialogType
{
    RidDialogType_ExitMenu,
    RidDialogType_ThanksForPlaying,
};

/**
 * @brief       RID デモアプリケ－ショーンの終了メニュー表示要求があるか判定します。
 *
 * @details     メインスレッドから呼びます。
 *
 */
bool IsRequestedToStartRidExitMenu() NN_NOEXCEPT;

/**
 * @brief       RID デモアプリケ－ショーンの終了メニューの終了を要求します。
 *
 * @details     メインスレッドから呼びます。
 *
 */
void RequestToCancelRidExitMenu() NN_NOEXCEPT;

/**
 * @brief       SD カードの挿抜が発生したかを判定します。
 *
 * @details     メインスレッドから呼びます。
 *
 */
bool IsSdCardDetectionStateChanged() NN_NOEXCEPT;

/**
 * @brief       HDCP 認証が失敗したかどうかを判定します。
 *
 * @details     メインスレッドから呼びます。
 *
 */
bool IsHdcpAuthenticationFailed() NN_NOEXCEPT;

/**
 * @brief       外部ディスプレイが接続されたかどうかを判定します。
 *
 * @details     メインスレッドから呼びます。
 *
 */
bool IsExternalDisplayConnected() NN_NOEXCEPT;

/**
 * @brief       通知メッセージレシーバー。
 *
 * @details     ユーザレベルでの通知メッセージ受信機構です。
 *              本クラスインスタンスはユーザで管理されるものとします。
 *              システムフレームワーク側で自動的に破棄される事はありません。
 */
class NotificationMessageReceiver : public nn::util::IntrusiveListBaseNode< NotificationMessageReceiver >
{
public:
    /**
     * @brief   通知メッセージ種別列挙型。
     *
     * @details 基本的に nn::ae のメッセージを対象としています。
     *          oe のメッセージハンドリングも行う場合には拡張してください。
     *          FocusState_OutOfFocus ⇔ FocusState_Background 違いなど。
     */
    enum Message : uint8_t
    {
        Message_None                    = 0,
        Message_InFocus                 = 1,
        Message_OutOfFocus              = 2,

        Message_ChangeIntoForeground    = Message_InFocus,
        Message_ChangeIntoBackground    = Message_OutOfFocus,
    };

    /**
     * @brief       デストラクタです。
     */
    virtual ~NotificationMessageReceiver() NN_NOEXCEPT
    {
        EnableNotificationMessageReceiving( false );
    }

    /**
     * @brief       本クラスに対する通知メッセージ受信の許可状態を制御します。
     *
     * @param[in]   permission  true の場合、本クラスに対するメッセージ通知を許可します。
     *                          false の場合、本クラスに対するメッセージ通知を拒否します。
     *
     * @return      前回の通知許可状態を返します。
     *
     * @details     メッセージ通知の停止は、対象処理がメッセージ通知を不要とした時点で明示的に停止する事を推奨します。
     *              デストラクタで通知拒否処理を呼んでいますが、インスタンス寿命での保証ではなく処理スコープでの保証で制御してください。
     *              通知メッセージは受信許可された状態の時のみ通知されます。
     */
    bool EnableNotificationMessageReceiving( bool permission = true ) NN_NOEXCEPT;

    /**
     * @brief       通知メッセージ受信コールバック。
     *
     * @param[in]   message 受信した通知メッセージ。
     *
     * @details     本コールバックは通知メッセージ受信スレッド上で呼び出されます。
     *              通知メッセージ受信スレッド上での処理は以下の制約があります。
     *              - 受信スレッドを長時間停止させる処理を行わない。
     *                  停止保証値はありません、遅延が発生した場合システム状態との不和などの影響が発生します。
     *              - 高負荷な処理を行わない。( 他スレッドとの同期、演算処理など。)
     *                  他スレッドへの通知用フラグ( イベントオブジェクト )などの制御に留めてください。
     *              - スタックを大量に消費する処理を行わない。 ( 受信スレッドのスタックは 8 KiB です。 )
     */
    virtual void OnNotificationMessageReceived( Message message ) NN_NOEXCEPT = 0;

protected:
    /**
     * @brief       コンストラクタです。
     */
    NN_IMPLICIT NotificationMessageReceiver() NN_NOEXCEPT
    {
    }
};

} // end of namespace devmenu

