﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>

#include <nn/ae.h>
#include <nn/es.h>
#include <nn/fs.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_AccessLogPrivate.h>
#include <nn/init.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/os.h>
#include <nn/time/time_Api.h>

#if defined( NN_DEVMENUSYSTEM )
    #include <nn/nim/nim_NetworkInstallManagerApi.h>
#endif

#if !defined( NN_CUSTOMERSUPPORTTOOL )
    #include <nn/ncm/ncm_Service.h>
#endif

#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL )
    #include <curl/curl.h>
#endif

#include <glv_resources.h> // Need to include before socket.h and after curl.h to avoid build error.

#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL )
    #include <nn/account/account_ApiForAdministrators.h>
    #include <nn/lbl/lbl.h>
    #include <nn/ns/ns_DevelopApi.h>
    #include <nn/prepo.h>
    #include <nn/psm/psm_Api.h>
    #include <nn/psm/psm_System.h>
    #include <nn/socket.h>
    #include <nn/time/time_ApiForMenu.h>
#else
    #include <nn/account/account_ApiForSystemServices.h>
    #include <nn/settings/system/settings_SerialNumber.h>
    #include <nn/util/util_StringUtil.h>
#endif

#include "Common/DevMenu_CommonSettingsApi.h"
#include "DevMenu_Config.h"
#include "DevMenu_Eula.h"
#include "DevMenu_PeakMemoryMeasurer.h"
#include "DevMenu_PrepoSettingsPage.h"
#include "DevMenu_RootSurface.h"
#include "DevMenu_Sound.h"

//!--------------------------------------------------------------------------------------
//! プロセス確保総ヒープメモリ量
//!--------------------------------------------------------------------------------------
namespace {
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    static const size_t AppletHeapSize = 26ull * 1024 * 1024; //!< アプリケーション予約ヒープメモリサイズ
#elif defined( NN_BUILD_CONFIG_ADDRESS_64 )
    static const size_t AppletHeapSize = 3ull * 1024 * 1024 * 1024; //!< アプリケーション予約ヒープメモリサイズ
#else
    static const size_t AppletHeapSize = 512ull * 1024 * 1024; //!< アプリケーション予約ヒープメモリサイズ
#endif

#if defined(NN_DEVMENUSYSTEM)
    nn::socket::ConfigWithMemory<
        8,                                  // tcpSocketCountMax
        1,                                  // udpSocketCountMax
        32 * 1024,                          // tcpInitialSendBufferSize
        32 * 1024,                          // tcpInitialReceiveBufferSize
        64 * 1024,                          // tcpAutoSendBufferSizeMax
        64 * 1024,                          // tcpAutoReceiveBufferSizeMax
        9 * 1024,                           // udpSendBufferSize
        42240,                              // udpReceiveBufferSize
        1,                                  // socketBufferEfficiency,
        nn::socket::MinSocketAllocatorSize  // allocatorPoolSize
    > g_SocketConfig(nn::socket::DefaultConcurrencyLimit);
#elif !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL )
    // socket 初期化(tcpSocketCountMax:1, udpSocketCountMax:1, socketBufferEfficiency:1)
    // このメモリ設定ではパフォーマンスは出ません.
    // パフォーマンスが必要な場合は DevMenu としてどれだけメモリを使ってよいかなど要調整.
    nn::socket::ConfigDefaultWithConstrainedMemory< 1, 1, 1 > g_SocketConfig;
#endif

    //!--------------------------------------------------------------------------------------
    //! @brief ペリフェラルセットアップ
    //!--------------------------------------------------------------------------------------
    static void InitializePeripherals() NN_NOEXCEPT
    {
        devmenu::PeakMemoryMeasurer::InitializeMeasurerOnMain();

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    #if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        nn::psm::Initialize();  // 本体機能参照( 電池残量 etc )
        nn::lbl::Initialize();  // 輝度調整機能 etc
        nn::prepo::Initialize();
        nn::es::Initialize();
    #endif
    #if !defined( NN_CUSTOMERSUPPORTTOOL )
        #if !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
            nn::ncm::Initialize();
        #endif
        #if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::nifm::InitializeSystem() );
        #else
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        #endif
    #endif

#elif defined( NN_BUILD_CONFIG_OS_WIN )
        nn::ns::InitializeDependenciesForDfc();
#endif

#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER ) && !defined( NN_CUSTOMERSUPPORTTOOL )
        NN_ABORT_UNLESS_RESULT_SUCCESS( ::nn::ns::InitializeForDevelop() );
#endif
        ::nn::ns::Initialize();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ペリフェラル終了
    //!--------------------------------------------------------------------------------------
    static void FinalizePeripherals() NN_NOEXCEPT
    {
        ::nn::ns::Finalize();
#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER ) && !defined( NN_CUSTOMERSUPPORTTOOL )
        NN_ABORT_UNLESS_RESULT_SUCCESS( ::nn::ns::FinalizeForDevelop() );
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN )
        nn::ns::FinalizeDependenciesForDfc();

#elif defined( NN_BUILD_CONFIG_OS_HORIZON )
    #if !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        nn::ncm::Finalize();
    #endif
    #if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        nn::es::Finalize();
        nn::lbl::Finalize();
        nn::psm::Finalize();
    #endif

#endif

        devmenu::PeakMemoryMeasurer::FinalizeMeasurerOnMain();
    }

    //!--------------------------------------------------------------------------------------
    //! @brief GFX, HID 設定初期値
    //!--------------------------------------------------------------------------------------
    static const glv::InitialConfiguration LocalGfxConfiguration = glv::InitialConfiguration( 512 * 1024, 1 * 1024 * 1024, 8 );
    static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration(
        glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary,
#if !defined ( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        glv::HidInitialConfiguration::PadSamplingVariation_System
#else
        glv::HidInitialConfiguration::PadSamplingVariation_Normal
#endif
    );

    //!--------------------------------------------------------------------------------------
    //! 状況を見て PrepoSettingsPage を作成する
    //!--------------------------------------------------------------------------------------
    void CreatePrepoSettingsPage() NN_NOEXCEPT
    {
#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
#if defined( NN_DEVMENUSYSTEM )
        // DevMenuSystem の場合は常にページを表示
        bool isPrepoSettingsPageEnabled = true;
#else
        // DevMenu の場合は、ページの表示/非表示はツールでのフラグ値で判断
        bool isPrepoSettingsPageEnabled = false;
        devmenu::GetFixedSizeFirmwareDebugSettingsItemValue( &isPrepoSettingsPageEnabled, "prepo", "devmenu_prepo_page_view" );
#endif // defined( NN_DEVMENUSYSTEM )
        if ( isPrepoSettingsPageEnabled )
        {
            devmenu::prepo::CreatePrepoSettingsPage();
        }
#endif // !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    }
}

//!--------------------------------------------------------------------------------------
//! nninitStartup() is invoked before calling nnMain().
//! 256MB確保
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nninitStartup()
{
    devmenu::PeakMemoryMeasurer::InitializeMeasurerOnStartup( AppletHeapSize );
}

//!--------------------------------------------------------------------------------------
//! @brief システムアプレットエントリポイント
//!--------------------------------------------------------------------------------------
void SystemAppletMenuMain( nn::ae::SystemAppletParameters* pParam ) NN_NOEXCEPT
{
    NN_UNUSED( pParam );
    DEVMENU_LOG_AE( "Launched SystemAppletMenuMain\n" );

    // ペリフェラルを初期化
    InitializePeripherals();

#if !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    // ROM マウント
    size_t mountRomCacheBufferSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::QueryMountRomCacheSize( &mountRomCacheBufferSize ) );
    std::unique_ptr<char[]> mountRomCacheBuffer( new char[ mountRomCacheBufferSize ] );
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom( "Contents", mountRomCacheBuffer.get(), mountRomCacheBufferSize )
    );
#endif

#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    // 時刻ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::InitializeForMenu() );

    // Socket ライブラリ初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::socket::Initialize( g_SocketConfig ) );

    // curl 初期化
    curl_global_init( CURL_GLOBAL_DEFAULT );

    // サウンドシステム初期化
    devmenu::InitializeSound();

    // アカウントライブラリの初期化
    nn::account::InitializeForAdministrator();

    // 自動 EULA 同意
    devmenu::AcceptEulaInDevVersion();

#elif !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    // 時刻ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );

    // アカウントライブラリの初期化
    nn::account::InitializeForSystemService();
#endif

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    nn::ns::EnableAutoCommit();
#endif

#if !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    // リソースプロバイダ及びリソースの初期化
    glv::Resources::GetSharedResourceContext().Initialize( "Contents" );
#endif

    // GLVランタイムエンジンの初期化及びアプリケーションフレームワークの初期化
    glv::ApplicationFrameworkInitialize( LocalHidConfiguration, LocalGfxConfiguration );

    // PrepoSettings ページの作成
    CreatePrepoSettingsPage();

#if defined(NN_DEVMENUSYSTEM)
    // nim ネットワークマネージャの初期化
    nn::nim::InitializeForNetworkInstallManager();
    NN_UTIL_SCOPE_EXIT {nn::nim::FinalizeForNetworkInstallManager(); };
#endif

    // DevMenuエントリポイント呼び出し
    devmenu::DevMenuMain();

    // 時刻ライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );

#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    // サウンドシステム終了
    devmenu::FinalizeSound();

    // curl 終了
    curl_global_cleanup();

    // socket 終了
    nn::socket::Finalize();
#endif

#if !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    // ROMマウント解除
    nn::fs::Unmount( "Contents" );
    mountRomCacheBuffer.reset();
#endif

    // ペリフェラルを終了
    FinalizePeripherals();

    DEVMENU_LOG_AE( "Exit SystemAppletMenuMain\n" );
#if defined( DEVMENU_ENABLE_NOTIFICATION ) && !defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    // アプリとして動作している場合は、自動的に終了するまで待機
    nn::os::SleepThread( nn::TimeSpan::FromDays( 1 ) );
#endif // defined( DEVMENU_ENABLE_NOTIFICATION ) && !defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
}

extern "C" void nnMain() NN_NOEXCEPT
{
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    nn::fs::InitializeWithMultiSessionForTargetTool();

    nn::ae::InvokeSystemAppletMain( nn::ae::AppletId_SystemAppletMenu, SystemAppletMenuMain );
#else
    // DevMenuApp, DevMenuSytemApp で FS アクセスログが出力されないよう対処
    nn::fs::SetLocalAccessLog( false );

    SystemAppletMenuMain( nullptr );
#endif // defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
}
