﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/image/image_JpegDecoder.h>
#include "DevMenu_Config.h"
#include "DevMenu_Image.h"

namespace devmenu {

/**
 * @brief       画像データのユーティリティの定義です。
 *
 * @details     下記サンプルを参考に実装しています。
 *              ${SIGLO_SDK_ROOT}/Samples/Sources/Applications/GfxJpegViewer/GfxJpegViewer.cpp
 */
void LoadJpegImage(ImageDataInfo& image, const char* path) NN_NOEXCEPT
{
    NN_ASSERT( image.alignment > 0 );

    // ファイルから読み込み
    //DEVMENU_LOG( "Load JpegImage\n" );
    void* pCompressed = NULL;
    size_t compressedSize = 0;
    {
        nn::Result result;
        nn::fs::FileHandle file = {};
        result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());
        int64_t fileSize = 0;
        result = nn::fs::GetFileSize(&fileSize, file);
        NN_ASSERT(result.IsSuccess());
        compressedSize = static_cast<size_t>(fileSize);
        pCompressed = malloc(compressedSize);
        NN_ASSERT_NOT_NULL(pCompressed);
        result = nn::fs::ReadFile(file, 0, pCompressed, compressedSize);
        NN_ASSERT(result.IsSuccess());
        nn::fs::CloseFile(file);
    }
    // 画像を展開
    {
        nn::image::JpegStatus result;
        nn::image::JpegDecoder decoder;
        decoder.SetImageData(pCompressed, compressedSize);
        result = decoder.Analyze();
        NN_ASSERT(result == nn::image::JpegStatus_Ok);
        size_t workSize = decoder.GetAnalyzedWorkBufferSize();
        void* workBuffer = malloc(workSize);
        NN_ASSERT_NOT_NULL(workBuffer);
        image.format = GL_RGBA;
        image.type = GL_UNSIGNED_BYTE;
        image.width = decoder.GetAnalyzedDimension().width;
        image.height = decoder.GetAnalyzedDimension().height;
        image.stride = 4 * image.alignment * ((image.width + image.alignment - 1) / image.alignment);
        image.size = image.stride * image.height;
        image.data = reinterpret_cast<uint8_t*>(malloc(image.size));
        memset(image.data, 0, image.size);
        result = decoder.Decode(image.data, image.size, image.alignment, workBuffer, workSize);
        NN_ASSERT(result == nn::image::JpegStatus_Ok);
        free(workBuffer);
        workBuffer = NULL;
        workSize = 0;
    }
    free(pCompressed);
    //DEVMENU_LOG( "Image  W%d x H%d\n", image.width, image.height );
}

void ReleaseImage(ImageDataInfo& image) NN_NOEXCEPT
{
    if( image.data != nullptr )
    {
        free(image.data);
        image.data = nullptr;
    }
    image.format = 0;
    image.size = 0;
    image.width = 0;
    image.height = 0;
    image.stride = 0;
}


/**
 * @brief       テクスチャービューの定義です。
 *
 * @details     下記サンプルを参考に実装しています。
 *              ${SIGLO_SDK_ROOT}/Externals/glv/examples/textureView.cpp
 */

TextureView::TextureView() NN_NOEXCEPT
    :   View(glv::Rect(0)), m_pTexture(nullptr)
{
    disable(glv::DrawBorder);
}

TextureView::TextureView(const glv::Rect& rect) NN_NOEXCEPT
    :   View(rect), m_pTexture(nullptr)
{
    disable(glv::DrawBorder);
}

TextureView::~TextureView() NN_NOEXCEPT
{
    DestoryTexture();
}

TextureView& TextureView::CreateTexture(const ImageDataInfo& image, bool fitting) NN_NOEXCEPT
{
    NN_ASSERT( m_pTexture == nullptr );

    // Generate a pattern for the texture...
    m_pTexture = new glv::Texture2(image.width, image.height, image.format, image.type);
    unsigned char * pixs = m_pTexture->buffer< unsigned char >();
    for(int y = 0; y < image.height; y++)
    {
        int revY = image.height - 1 - y;
        memcpy(pixs + y * image.stride, image.data + revY * image.stride, image.stride);
    }
    m_pTexture->magFilter(GL_NEAREST);
    m_pTexture->recreate();             // Create texture on GPU
    m_pTexture->send();                 // Send over texel data

    if( fitting != false )
    {
        extent(image.width, image.height);
    }
    return *this;
}

TextureView& TextureView::DestoryTexture() NN_NOEXCEPT
{
    if( m_pTexture != nullptr )
    {
        m_pTexture->destroy();          // Destroy texture on GPU
        m_pTexture = nullptr;
    }
    return *this;
}

void TextureView::onDraw(glv::GLV& g) NN_NOEXCEPT
{
    NN_UNUSED( g );

    if( m_pTexture != nullptr )
    {
        glv::draw::enable(glv::draw::Texture2D);    // Enable texture mapping
        m_pTexture->begin();                        // Bind texture
        glv::draw::color(1, 1, 1, 1);               // Set current color
        m_pTexture->draw(0, 0, width(), height());  // Draw a textured quad filling the view's area
        m_pTexture->end();                          // Unbind texture
        glv::draw::disable(glv::draw::Texture2D);   // Disable texture mapping
    }
}

} // ~namespace devmenu
