﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <cstring>

#include <nn/arp/arp_Types.h>
#include <nn/nn_Assert.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include "DevMenu_Config.h"
#include "DevMenu_Footer.h"
#include "DevMenu_RootSurfaceEventHandler.h"
#include "Applications/DevMenu_ApplicationsCommon.h"
#include "Common/DevMenu_CommonIconButton.h"
#include "Launcher/DevMenu_Launcher.h"

namespace devmenu {

namespace {
    const glv::space_t FooterFontSize = 20.0f;
    const glv::Label::Spec FooterLabelSpec( glv::Place::CC, 0.0f, 0.0f, FooterFontSize );
}

FooterButton::FooterButton( const char* text, const std::function< void() >& callback, const glv::WideCharacterType* iconCodePoint, const glv::Rect& rect ) NN_NOEXCEPT
    : Button( text, callback, rect, FooterLabelSpec ),
    m_Table( "x x", 8.0f ),
    m_Icon( iconCodePoint, glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, FooterFontSize ) )
{
    auto& label = this->GetLabel();
    label.remove();
    m_Table << m_Icon << label;
    m_Table.arrange();
    *this << m_Table.pos( glv::Place::CC ).anchor( glv::Place::CC );
    this->enable( glv::Property::HitTest );
    this->disable( glv::Property::FocusHighlight | glv::Property::DrawBack | glv::Property::DrawBorder );
    this->extent( m_Table.width(), this->height() );
}

FooterControls::RunningApplicationController::InfoView::InfoView( const std::function< void() >& callback, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::Button( rect, true ),
    m_Table( "x x x", 3.0f, 0.0f ),
    m_Caption( "Running:" ),
    m_ApplicationIcon( 32, 32, 8.0f, 8.0f ),
    m_Storage( "" ),
    m_Callback( callback )
{
    this->disable( glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::FocusHighlight );
    this->attach( []( const glv::Notification& notification )->void{
        notification.receiver< FooterControls::RunningApplicationController::InfoView >()->m_Callback(); }, glv::Update::Clicked, this );
    m_Caption.size( FooterFontSize );
    m_Storage.size( FooterFontSize );
    m_Table << m_Caption << m_ApplicationIcon << m_Storage;
    *this << m_Table;
}

void FooterControls::RunningApplicationController::InfoView::Update( bool isApplicationLaunchedJustBefore, const std::string& str ) NN_NOEXCEPT
{
    if ( isApplicationLaunchedJustBefore )
    {
        this->SetApplicationIcon();
        m_Storage.setValue( str );
        m_Table.arrange();
        this->extent( m_Table.width(), m_Table.height() );
    }
    else
    {
        m_Storage.setValue( str );
    }
}

void FooterControls::RunningApplicationController::InfoView::SetApplicationIcon() NN_NOEXCEPT
{
    // Get Application Icon
    std::unique_ptr< char[] > buffer;
    size_t dataSize;
    const auto result = application::GetApplicationControlData(
        &dataSize, &buffer, nn::ns::ApplicationControlSource::Storage, launcher::GetActiveApplicationId() );

    if ( result.IsSuccess() )
    {
        nn::ns::ApplicationControlDataAccessor accessor( buffer.get(), dataSize );
        m_ApplicationIcon.SetTexture( accessor.GetIconData(), accessor.GetIconSize() );
        return;
    }

    // Clear the icon texture if valid data cannot be got.
    m_ApplicationIcon.ClearTexture();
}

FooterControls::RunningApplicationController::RunningApplicationController( const AbstructEventHandlers& handlers, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( rect )
    , m_Handlers( handlers )
    , m_Placer( *this, glv::Direction::E, glv::Place::TL, 0.0f, 0.0f, 0.0f )
    , m_InfoView( [&]{ this->OnAppliactionInfoClicked(); }, glv::Rect( rect.w * 0.6f, rect.h ) )
    , m_ResumeButton( "Resume", [&]{ this->OnResumeButtonClicked(); }, IconCodePoint::ControllerButtonFilledHome, glv::Rect( 0.0f, rect.h ) )
    , m_CloseButton( "Close", [&]{ this->OnCloseButtonClicked(); }, IconCodePoint::ControllerButtonFilledX, glv::Rect( 0.0f, rect.h ) )
    , m_Spacer( SpacerWidth, 0.0f )
    , m_IsShowing( false )
{
    this->disable( glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::Visible | glv::Property::FocusHighlight );
    m_Placer << m_InfoView << m_Spacer << m_ResumeButton << m_CloseButton;

    this->HideApplicationInfo();
}

void FooterControls::RunningApplicationController::ShowApplicationInfo() NN_NOEXCEPT
{
    m_IsShowing = true;
    this->enable( glv::Property::Visible );
}

void FooterControls::RunningApplicationController::HideApplicationInfo() NN_NOEXCEPT
{
    m_IsShowing = false;
    this->disable( glv::Property::Visible );
}

void FooterControls::RunningApplicationController::UpdateApplicationInfo() NN_NOEXCEPT
{
    // アプリケーションが非実行中の場合は非表示にする
    if ( !launcher::IsApplicationAlive() )
    {
        if ( m_IsShowing )
        {
            // 起動プロパティをクリアして、情報を非表示にする
            m_ApplicationLaunchProperty = launcher::GetApplicationLaunchProperty();
            this->HideApplicationInfo();
        }
        return;
    }

    auto& launchProperty = launcher::GetApplicationLaunchProperty();

    const std::string manualString( "R+Y: Show More Info" );
    NN_FUNCTION_LOCAL_STATIC( std::string, s_StorageInfoString, ( "(App: Unknown, Patch: Unknown)" ) );
    NN_FUNCTION_LOCAL_STATIC( nn::os::Tick, s_StartTick );

    // 現在表示中のアプリケーション情報から更新があるかをチェックする
    if ( 0 != memcmp( &launchProperty, &m_ApplicationLaunchProperty, sizeof( launchProperty ) ) )
    {
        // 念のため、アプリケーションのストレージが有効であるかを確認する
        if ( nn::ncm::StorageId::None == launchProperty.storageId )
        {
            return;
        }

        char buf[ 64 ];
        nn::util::SNPrintf( buf, sizeof ( buf ), "(App: %s, Patch: %s)",
            launcher::GetStorageName( launchProperty.storageId ), launcher::GetStorageName( launchProperty.patchStorageId ) );
        s_StorageInfoString = buf;

        m_InfoView.Update( true, s_StorageInfoString );
        m_ApplicationLaunchProperty = launchProperty;

        this->ShowApplicationInfo();

        s_StartTick = nn::os::GetSystemTick();
    }
    else
    {
        NN_FUNCTION_LOCAL_STATIC( bool, s_IsStorageInfoDisplayed, = true );
        const auto currentTick = nn::os::GetSystemTick();
        const auto intervalMiliSeconds = 4000LL;

        if ( intervalMiliSeconds < nn::os::ConvertToTimeSpan( currentTick - s_StartTick ).GetMilliSeconds() )
        {
            const std::string& nextStr = s_IsStorageInfoDisplayed ? manualString : s_StorageInfoString;
            m_InfoView.Update( false, nextStr );
            s_IsStorageInfoDisplayed = !s_IsStorageInfoDisplayed;
            s_StartTick = currentTick;
        }
    }
}

void FooterControls::RunningApplicationController::OnAppliactionInfoClicked() NN_NOEXCEPT
{
    if ( !launcher::IsApplicationAlive() )
    {
        return;
    }

    std::string applicationName( "Acquisition failure" );
    std::string applicationIdStr( "Acquisition failure" );
    std::string displayVersionStr( "Acquisition failure" );

    auto applicationControlProperty = std::make_shared< nn::ns::ApplicationControlProperty >();

    if ( launcher::GetApplicationControlProperty( applicationControlProperty.get() ) )
    {
        applicationName = std::string( applicationControlProperty->GetDefaultTitle().name );
        displayVersionStr = std::string( applicationControlProperty->displayVersion );
    }

    nn::ns::ApplicationView applicationView;
    const auto applicationId = launcher::GetActiveApplicationId();

    if ( nn::ns::GetApplicationView( &applicationView, &applicationId, 1 ).IsSuccess() )
    {
        char buf[ 32 ];
        nn::util::SNPrintf( buf, sizeof( buf ), "0x%016llx", applicationView.id.value );
        applicationIdStr = buf;
    }

    auto pMessageView = new MessageView( false );
    pMessageView->AddMessage( std::string( "Name: " ) + applicationName );
    pMessageView->AddMessage( std::string( "ID: " ) + applicationIdStr );
    pMessageView->AddMessage( std::string( "Version: " ) + displayVersionStr );

    pMessageView->AddButton( "Close" );

    m_Handlers.StartModal( pMessageView, true );
}

void FooterControls::RunningApplicationController::OnCloseButtonClicked() NN_NOEXCEPT
{
    launcher::RequestApplicationExit();
}

void FooterControls::RunningApplicationController::OnResumeButtonClicked() NN_NOEXCEPT
{
    launcher::RequestApplicationForeground();
}

FooterControls::FooterControls( const AbstructEventHandlers& handlers, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( rect )
    , m_ApplicationInfo( handlers, glv::Rect( rect.width(), rect.height() ) )
{
    this->disable(
        glv::Property::Controllable | glv::Property::HitTest | glv::Property::FocusHighlight |
        glv::Property::DrawBack | glv::Property::DrawBorder );
    this->pos( 0.0f, 0.0f );
    this->anchor( glv::Place::TL );
    *this << m_ApplicationInfo;
}

void FooterControls::UpdateApplicationInfo() NN_NOEXCEPT
{
    m_ApplicationInfo.UpdateApplicationInfo();
}

void FooterControls::InvokeCloseApplication() NN_NOEXCEPT
{
    m_ApplicationInfo.OnCloseButtonClicked();
}

void FooterControls::InvokeShowAppliactionDetailInfo() NN_NOEXCEPT
{
    m_ApplicationInfo.OnAppliactionInfoClicked();
}

} // ~namespace devmenu
