﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <string>

#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/ns/ns_ApplicationViewApi.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "DevMenu_Common.h"
#include "DevMenu_Config.h"
#include "DevMenu_Exhibition.h"
#include "DevMenu_Result.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "Launcher/DevMenu_Launcher.h"

#if defined (NN_DEVMENU_ENABLE_SYSTEM_APPLET)

namespace devmenu { namespace exhibition {

namespace {
    const std::string SaveDataFileName = "exhibition.dat";
    const std::string SaveDataPath = CommonValue::OwnSaveDataMeta.mountName + std::string( ":/" ) + SaveDataFileName;
    const int64_t SaveDataFileSize = 128;

    bool GetExhibitionModeSetting() NN_NOEXCEPT
    {
        static nn::os::Mutex s_Mutex( false );
        std::lock_guard< nn::os::Mutex > lock( s_Mutex );

        static nn::util::optional< bool > s_IsExhibitionModeEnabled( nn::util::nullopt );
        if ( !s_IsExhibitionModeEnabled )
        {
            bool isEnabled;
            GetFixedSizeFirmwareDebugSettingsItemValue( &isEnabled, "devmenu", "enable_exhibition_mode" );
            s_IsExhibitionModeEnabled = isEnabled;
        }
        return *s_IsExhibitionModeEnabled;
    }

    nn::Result WriteSaveData( const ExhibitionSaveData& data ) NN_NOEXCEPT
    {
        NN_RESULT_DO( nn::fs::MountSystemSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str(), CommonValue::OwnSaveDataMeta.id ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount( CommonValue::OwnSaveDataMeta.mountName.c_str() );
        };

        nn::fs::FileHandle handle;

        // Open -> Close
        {
            NN_RESULT_TRY( nn::fs::OpenFile( &handle, SaveDataPath.c_str(), nn::fs::OpenMode_Write ) );
                NN_RESULT_CATCH( nn::fs::ResultPathNotFound )
                {
                    NN_RESULT_DO( nn::fs::CreateFile( SaveDataPath.c_str(), exhibition::SaveDataFileSize ) );
                    DEVMENU_LOG( "Created %s\n", SaveDataFileName.c_str() );
                    NN_RESULT_DO( nn::fs::OpenFile( &handle, SaveDataPath.c_str(), nn::fs::OpenMode_Write ) );
                }
            NN_RESULT_END_TRY;
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile( handle );
            };

            NN_RESULT_DO( nn::fs::WriteFile( handle, 0, &data, exhibition::SaveDataFileSize, nn::fs::WriteOption() ) );
            NN_RESULT_DO( nn::fs::FlushFile( handle ) );
        }
        NN_RESULT_DO( nn::fs::CommitSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str() ) );

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadSaveData( ExhibitionSaveData* pOutData ) NN_NOEXCEPT
    {
        NN_RESULT_DO( nn::fs::MountSystemSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str(), CommonValue::OwnSaveDataMeta.id ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount( CommonValue::OwnSaveDataMeta.mountName.c_str() );
        };

        nn::fs::FileHandle handle;

        NN_RESULT_TRY( nn::fs::OpenFile( &handle, SaveDataPath.c_str(), nn::fs::OpenMode_Read ) )
            NN_RESULT_CATCH( nn::fs::ResultPathNotFound )
            {
                // セーブデータファイルが存在しない場合は初期値を設定する
                ExhibitionSaveData data = { { 0 } };
                nn::util::SNPrintf( data.applicationIdString, sizeof( data.applicationIdString ), "0x%016llx", nn::ncm::ApplicationId::GetInvalidId() );
                data.commandType = CancelCommandType_Default;
                std::memcpy( pOutData, &data, sizeof( data ) );
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( handle );
        };

        NN_RESULT_DO( nn::fs::ReadFile( handle, 0, pOutData, exhibition::SaveDataFileSize ) );
        NN_RESULT_SUCCESS;
    }
} // end of namespace anonymous

bool IsExhibitionModeEnabled() NN_NOEXCEPT
{
    return GetExhibitionModeSetting();
}

nn::Result DeleteSaveData() NN_NOEXCEPT
{
    NN_RESULT_DO( nn::fs::MountSystemSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str(), CommonValue::OwnSaveDataMeta.id ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( CommonValue::OwnSaveDataMeta.mountName.c_str() );
    };

    NN_RESULT_TRY( nn::fs::DeleteFile( SaveDataPath.c_str() ) );
        NN_RESULT_CATCH( nn::fs::ResultPathNotFound )
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO( nn::fs::CommitSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str() ) );

    NN_RESULT_SUCCESS;
}

nn::Result SetAutoBootApplicationId( const char* applicationIdStr ) NN_NOEXCEPT
{
    ExhibitionSaveData data;

    NN_RESULT_DO( ReadSaveData( &data ) );
    NN_ASSERT( sizeof( applicationIdStr ) < sizeof( data.applicationIdString ) );
    nn::util::Strlcpy( data.applicationIdString, applicationIdStr, sizeof( data.applicationIdString ) );
    NN_RESULT_DO( WriteSaveData( data ) );

    NN_RESULT_SUCCESS;
}

const std::string GetLaunchErrorMessage( nn::Result result, int64_t requiredSize ) NN_NOEXCEPT
{
    if ( nn::fs::ResultPathNotFound::Includes( result ) )
    {
        return "Auto boot application is not set. Press Y button and set an application.";
    }
    else if ( ResultAccountNotFound::Includes( result ) )
    {
        return "Account select is required before launching application.\nCreate at least one account to be selected.";
    }
    else if ( ResultApplicationNotLaunchable::Includes( result ) )
    {
        return "Application is not launchable. Set launchable application to be auto booted.";
    }
    else // 通常起動時と共通のエラーメッセージ
    {
        return launcher::GetLaunchRelatedErrorMessage( result, requiredSize );
    }
}

AutoBootExecutor::AutoBootExecutor() NN_NOEXCEPT
    : m_IsAutoBootDisabled( false ),
    m_CanReceiveCancelCommand( true )
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( ReadSaveData( &m_SaveData ) );

    if ( IsExhibitionModeEnabled() )
    {
        DEVMENU_LOG( "Exhibiton mode cancel command type is %d\n", m_SaveData.commandType );
    }
}

CancelCommandType AutoBootExecutor::GetCancelCommandType() NN_NOEXCEPT
{
    return m_SaveData.commandType < CancelCommandType_Unknown ? m_SaveData.commandType : CancelCommandType_Default;
}

void AutoBootExecutor::UpdateCancelCommand( CancelCommandType type ) NN_NOEXCEPT
{
    m_SaveData.commandType = type;

    NN_ABORT_UNLESS_RESULT_SUCCESS( WriteSaveData( m_SaveData ) );
    DEVMENU_LOG( "Exhibition mode cancel command has been changed.\n" );

    // 変更したキャンセルコマンドは再起動後に有効になるようにする
    m_CanReceiveCancelCommand = false;
    m_IsAutoBootDisabled = true;
}

bool AutoBootExecutor::IsAutoBootExecutable() NN_NOEXCEPT
{
    return ( IsExhibitionModeEnabled() && !CanReceiveCancelCommand() && !m_IsAutoBootDisabled );
}

bool AutoBootExecutor::CancelAutoBoot( bool isForced ) NN_NOEXCEPT
{
    if ( true == isForced || true == CanReceiveCancelCommand() )
    {
        DEVMENU_LOG( "Auto boot has been canceled.\n" );
        m_IsAutoBootDisabled = true;
        m_CanReceiveCancelCommand = false;
        return true;
    }
    return false;
}

bool AutoBootExecutor::CanReceiveCancelCommand() NN_NOEXCEPT
{
    if ( true == m_CanReceiveCancelCommand )
    {
        static const nn::os::Tick s_InitialTick( nn::os::GetSystemTick() );
        nn::os::Tick currentTick = nn::os::GetSystemTick();
        const int64_t pastTimeMilliSeconds = nn::os::ConvertToTimeSpan( currentTick - s_InitialTick ).GetMilliSeconds();
        if ( AutoBootExecutor::CancelCommandInputTime < pastTimeMilliSeconds )
        {
            m_CanReceiveCancelCommand = false;
        }
    }
    return m_CanReceiveCancelCommand;
}

nn::Result AutoBootExecutor::Launch() NN_NOEXCEPT
{
    NN_ASSERT( IsExhibitionModeEnabled() );

    m_IsAutoBootDisabled = true;
    const nn::ncm::ApplicationId autoBootApplicationId = { strtoull( m_SaveData.applicationIdString, 0, 16 ) };

    // 無効な Application ID が設定されている場合は、セーブデータが存在しないと見なしてエラーメッセージを調整する
    NN_RESULT_THROW_UNLESS( autoBootApplicationId != nn::ncm::ApplicationId::GetInvalidId(), nn::fs::ResultPathNotFound() );

    // 起動不可の場合はエラーメッセージを表示する
    nn::ns::ApplicationView applicationView;
    NN_RESULT_TRY( nn::ns::GetApplicationView( &applicationView, &autoBootApplicationId, 1 ) );
    NN_RESULT_CATCH_ALL
    {
        NN_RESULT_THROW( ResultApplicationNotLaunchable() );
    }
    NN_RESULT_END_TRY;

    NN_RESULT_THROW_UNLESS( true == applicationView.IsLaunchable(), ResultApplicationNotLaunchable() );

    DEVMENU_LOG( "Try to launch the auto boot application (0x%016llx)\n", autoBootApplicationId.value );
    launcher::RequestApplicationLaunchOrResume( autoBootApplicationId, true );
    NN_RESULT_SUCCESS;
}

}} // ~namespace devmenu::exhibition, ~namespace devmenu

#endif // defined (NN_DEVMENU_ENABLE_SYSTEM_APPLET)

