﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>
#include <nn/fs.h>
#include <nn/nifm.h>
#include <nn/ec/ec_Result.h>
#include <nn/ec/system/ec_DeviceAccountApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_DownloadTaskSystemApi.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_RetailInteractiveDisplayApi.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "DevMenu_RootSurface.h"
#include "DevMenu_Config.h"
#include "DevMenuCommand_ShopCommand.h"
#include "DevMenuCommand_ApplicationCommand.h"

#include "Common/DevMenu_CommonScrollBox.h"
#include "Download/DevMenu_DownloadProgressViewUnit.h"

using namespace nn;

namespace devmenu { namespace system { namespace download
{
namespace
{
    // ボタンのサイズ
    const glv::space_t ButtonOffset = 10.f;
    const glv::space_t ButtonHeight = 48.f;

    // リスト要素のパディング ( ボーダーを表示する場合は 2以上必要 )
    const glv::space_t ListMarginL = 16;
    const glv::space_t ListMarginR = 16;

    // VewUnit の表示高さ
    const glv::space_t UnitHeight = 30.f;
    const glv::space_t UnitMargin = 5.f;

    // Label の表示位置
    const glv::space_t LabelOffset = ButtonHeight + 20.f;
    const glv::space_t IdMargin = 10.f;
    const glv::space_t IdLength = 300.f;
    const glv::space_t StateMargin = 10.f;
    const glv::space_t StateLength = 160.f;
    const glv::space_t ProgressMargin = 10.f;
    const glv::space_t ProgressLength = 280.f;
    const glv::space_t SliderMargin = 10.f;
    const glv::space_t SliderLength = 140.f;
    const glv::space_t DownloadButtonMargin = 10.f;
    const glv::space_t DownloadButtonLength = 130.f;

    const glv::space_t IdPos = IdMargin;
    const glv::space_t StatePos = IdPos + IdLength + StateMargin;
    const glv::space_t ProgressPos = StatePos + StateLength + ProgressMargin;
    const glv::space_t SliderPos = ProgressPos + ProgressLength + SliderMargin;
    const glv::space_t DownloadButtonPos = SliderPos + SliderLength + DownloadButtonMargin;

    // Scrollの表示位置
    const glv::space_t ScrollPosAdjustment = -IdMargin;
    const glv::space_t ScrollOffset = LabelOffset + 30.f;

    // ダウンロード進捗の確認間隔 (ms)
    const uint64_t CheckInterval = 1000;

#if defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    // アプリケーションリスト
    const nn::ncm::ApplicationId ApplicationIdList[] =
    {
        // 各Nup用のDevmenuLotCheckのIDリスト
        { 0x01000000000020F0 },
        { 0x01000000000020F1 },
        { 0x01000000000020F2 },
        { 0x01000000000020F3 },
        { 0x01000000000020F4 },
        { 0x01000000000020F5 },
        { 0x01000000000020F6 },
        { 0x01000000000020F7 },
        { 0x01000000000020F8 },
        { 0x01000000000020F9 },
        { 0x01000000000020FA },
        { 0x01000000000020FB },
        { 0x01000000000020FC },
        { 0x01000000000020FD },
        { 0x01000000000020FE },
        { 0x01000000000020FF },
    };
#endif
}

/**
* @brief ダウンロード状況の一覧を表示するページです。
*/
class DownloadPage
    : public devmenu::Page
{
public:
    /**
    * @brief コンストラクタです。
    */
    DownloadPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
        : devmenu::Page(pageId, pageCaption, rect)
        , m_ListProgressView()
        , m_ProgressCheckEvent(nn::os::EventClearMode::EventClearMode_ManualClear)
        , m_ScrollContainer()
    {
    }

    /**
    * @brief ページがコンテナに追加された後に呼び出されます。
    */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        // ヘッダ
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("Application ID"), glv::Label::Spec(glv::Place::TL, IdPos, LabelOffset, CommonValue::InitialFontSize));
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("State"), glv::Label::Spec(glv::Place::TL, StatePos, LabelOffset, CommonValue::InitialFontSize));
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("Progress"), glv::Label::Spec(glv::Place::TL, SliderPos, LabelOffset, CommonValue::InitialFontSize));
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("Downloaded / Total"), glv::Label::Spec(glv::Place::TL, ProgressPos, LabelOffset, CommonValue::InitialFontSize));
        *this << new glv::Label(GLV_TEXT_API_WIDE_STRING("Button"), glv::Label::Spec(glv::Place::TL, DownloadButtonPos, LabelOffset, CommonValue::InitialFontSize));

        // アプリケーションリスト
        std::unique_ptr<ScrollableBoxView> pScrollContainer( new ScrollableBoxView( "<", glv::Rect( 0.0f, 0.0f, width(), height() - ScrollOffset ), 0.f, ScrollOffset ) );
        m_ScrollContainer = std::move( pScrollContainer );
        *this << *m_ScrollContainer;

        CreateProgressViewList();

        const auto ButtonWidth = (width() - ListMarginL - ListMarginR) / 2;
        auto buttonRegister = new glv::Button( glv::Rect( ListMarginL, ButtonOffset, ButtonWidth, ButtonHeight ), true );
        buttonRegister->attach( []( const glv::Notification& n )->void { n.receiver<DownloadPage>()->OnRegisterDevice(); }, glv::Update::Clicked, this );
        auto addLabelRegister = new glv::Label( GLV_TEXT_API_WIDE_STRING( "Register Device" ), glv::Label::Spec( glv::Place::CC, 0, 0, CommonValue::InitialFontSize ) );
        *buttonRegister << addLabelRegister;
        *this << buttonRegister;

        auto buttonSync = new glv::Button( glv::Rect( ListMarginL + ButtonWidth, ButtonOffset, ButtonWidth, ButtonHeight ), true );
        buttonSync->attach( []( const glv::Notification& n )->void { n.receiver<DownloadPage>()->OnSyncTicket(); }, glv::Update::Clicked, this );
        auto addLabelSync = new glv::Label( GLV_TEXT_API_WIDE_STRING( "Sync Ticket" ), glv::Label::Spec( glv::Place::CC, 0, 0, CommonValue::InitialFontSize ) );
        *buttonSync << addLabelSync;
        *this << buttonSync;

        this->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

        nn::TimeSpan span(nn::TimeSpanType::FromMilliSeconds(CheckInterval));
        m_ProgressCheckEvent.StartPeriodic(span, span);
        m_ProgressCheckEvent.Signal();
    }

    /**
    * @brief アプリケーションメインループからのコールバックです。
    *
    * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
    * この時点ではまだ glvコンテキストのレンダリングは開始していません。
    */
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(context);
        NN_UNUSED(events);

        //!< 1 秒間に 1 回ダウンロード情報を更新する
        if (!m_ProgressCheckEvent.TryWait())
        {
            return;
        }
        m_ProgressCheckEvent.Clear();

        static const size_t ApplicationRecordListBufferSize = 2014;
        std::unique_ptr<nn::ns::ApplicationRecord[]> recordBuffer(new nn::ns::ApplicationRecord[ApplicationRecordListBufferSize]);
        auto recordList = recordBuffer.get();
        auto numApp = nn::ns::ListApplicationRecord(recordList, ApplicationRecordListBufferSize, 0);

        // ダウンロード中のアプリケーションが存在する場合
        if (0 < numApp)
        {
            for (int i = 0; i < numApp; i++)
            {
                auto& record = recordList[i];
                nn::ns::ApplicationView view;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::GetApplicationView(&view, &record.id, 1));

                for( int j = 0; j < static_cast<int>(m_ListProgressView.size()) ; j++ )
                {
                    // 毎秒更新するラベル
                    if( m_ListProgressView[j] != nullptr && m_ListProgressView[j]->GetView().id == view.id )
                    {
                        m_ListProgressView[j]->UpdateProgress( view );
                        auto table = m_ListProgressView[j]->GetApplicationTable();
                        *table
                            << m_ListProgressView[j]->GetStateLabel()
                            << m_ListProgressView[j]->GetProgressLabel();
                    }
                }
            }
        }
    }

private:
    /**
    * @brief ネットワーク接続をシステムに要求する
    */
    nn::Result SubmitNetworkRequestAndWait() NN_NOEXCEPT
    {
        nn::nifm::SubmitNetworkRequestAndWait();
        if( !nn::nifm::IsNetworkAvailable() )
        {
            DEVMENU_LOG( "Network not available\n" );
            auto pView = new MessageView( false );
            CreateErrorMessageView( pView, "Network not available.");
            GetRootSurfaceContext()->StartModal( pView, true );
            return nn::ns::ResultInternetRequestNotAccepted();
        }

        NN_RESULT_SUCCESS;
    }

    /**
    * @brief アプリケーションをダウンロードする
    */
    void OnDownloadApplication( const nn::ncm::ApplicationId& id ) NN_NOEXCEPT
    {
#if defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        if( nn::ns::IsAnyApplicationEntityInstalled( id ) )
        {
            // アプリケーション記録が存在する場合はなにもしない
            return;
        }

        // ネットワーク接続をシステムに要求する
        if( SubmitNetworkRequestAndWait().IsFailure() )
        {
            return;
        }

        nn::ns::AsyncResult asyncResult;
        auto result = nn::ns::RequestDownloadApplication( &asyncResult, id, nn::ncm::StorageId::BuildInUser );
        if( nn::ns::ResultAlreadyDownloading::Includes( result ) )
        {
            if( nn::nifm::IsNetworkAvailable() )
            {
                // バックグラウンドダウンロードをする為にネットワーク利用要求を取り下げる
                nifm::CancelNetworkRequest();
            }
            return;
        }

        if( result.IsFailure() || asyncResult.Get().IsFailure())
        {
            auto pView = new MessageView( false );
            CreateErrorMessageView( pView, "Failed to start install.", result );
            GetRootSurfaceContext()->StartModal( pView, true );
        }
        else
        {
            if( nn::nifm::IsNetworkAvailable() )
            {
                // バックグラウンドダウンロードをする為にネットワーク利用要求を取り下げる
                nifm::CancelNetworkRequest();
            }
            DEVMENU_LOG( "Succeeded to start install.\n" );
        }
#else
        NN_UNUSED( id );

        DEVMENU_LOG( "Unsupported.\n" );
        auto pView = new MessageView( false );
        CreateErrorMessageView( pView, "Unsupported." );
        GetRootSurfaceContext()->StartModal( pView, true );
#endif
    }

#if defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    nn::Result RequestShopCommand( bool& outValue, const char* command, const char* successMessage, const char* failedMessage ) NN_NOEXCEPT
    {
        // DevMenuSystemの "shop コマンド" 機能を実行する
        outValue = false;
        const char* argv[] = { "DevMenuLotcheckDownloader", "shop", command };
        Option option( NN_ARRAY_SIZE( argv ), const_cast<char**>(argv) );
        auto result = ShopCommand( &outValue, option );
        if( result.IsFailure() || outValue == false )
        {
            if( successMessage != nullptr )
            {
                auto pView = new MessageView( false );
                CreateErrorMessageView( pView, failedMessage, result );
                GetRootSurfaceContext()->StartModal( pView, true );
            }
        }
        else
        {
            if( failedMessage != nullptr )
            {
                auto pView = new MessageView( false );
                pView->AddMessage( successMessage );
                pView->AddButton( "Close" );
                GetRootSurfaceContext()->StartModal( pView, true );
            }
        }

        return result;
    }

    nn::Result UnregisterDeviceAccountIfNotRegistered()
    {
        nn::Result result;

        // デバイスアカウント情報を取得
        {
            nn::ec::system::AsyncDeviceRegistrationInfo async;
            result = nn::ec::system::RequestDeviceRegistrationInfo( &async );
            if( result.IsSuccess() )
            {
                nn::ec::system::DeviceRegistrationInfo regInfo;
                result = async.Get( &regInfo );
            }
            if( nn::ec::ResultDeviceNotRegistered::Includes( result ) )
            {
                // デバイスアカウントが未登録の場合はUnregisterを実行しない
                DEVMENU_LOG( "Device not registered\n" );
                NN_RESULT_SUCCESS;
            }
        }

        // DevMenuSystemの "shop unregister-device-account" 機能を実行する
        bool outValue = false;
        result = RequestShopCommand( outValue, "unregister-device-account", nullptr, nullptr );
        if ( result.IsSuccess() && outValue == false )
        {
            // shopコマンドが実行されない場合なのでエラーをキャンセルとする
            DEVMENU_LOG( "Device not registered\n" );
            result = nn::ec::ResultCanceled();
        }

        return result;
    }
#endif

    /**
    * @brief デバイス登録をECIサーバに対して行う
    */
    void OnRegisterDevice() NN_NOEXCEPT
    {
#if defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        nn::Result result;
        nn::ec::system::DeviceAccountInfo deviceAccountInfo;
        result = nn::ec::system::GetDeviceAccountInfo( &deviceAccountInfo );
        if( result.IsSuccess() )
        {
            // 既にデバイスアカウントがある場合はメッセージを表示して終了
            auto pView = new MessageView( false );
            pView->AddMessage( "Device is already registered." );
            pView->AddButton( "Close" );
            GetRootSurfaceContext()->StartModal( pView, true );
            return;
        }

        // ネットワーク接続をシステムに要求する
        if( SubmitNetworkRequestAndWait().IsFailure() )
        {
            return;
        }

        // サーバーにデバイスアカウントが登録されている場合は削除する
        result = UnregisterDeviceAccountIfNotRegistered();
        if( result.IsFailure() )
        {
            DEVMENU_LOG( "Failed to register device account.(0x%08x)\n", result.GetInnerValueForDebug() );
            // 失敗した場合はエラーメッセージを表示して終了
            // Unregisterの失敗だが RegisterDevice機能の実行失敗なのでメッセージはregisterの失敗として表示する
            auto pView = new MessageView( false );
            CreateErrorMessageView( pView, "Failed to register device account.", result );
            GetRootSurfaceContext()->StartModal( pView, true );
            return;
        }

        // DevMenuSystemの "shop register-device-account" 機能を実行する
        bool outValue;
        RequestShopCommand( outValue, "register-device-account", "Succeeded to register device account.", "Failed to register device account." );
#else
        DEVMENU_LOG( "Unsupported.\n" );
        auto pView = new MessageView( false );
        CreateErrorMessageView( pView, "Unsupported.");
        GetRootSurfaceContext()->StartModal( pView, true );
#endif
    }

    /**
    * @brief ダウンロード状況を参照して m_ProgressViewList を構築する
    */
    void CreateProgressViewList() NN_NOEXCEPT
    {
        //!< ViewList のリセット
        m_ListProgressView.clear();

#if defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        static const size_t ApplicationRecodeListBufferSize = 2014;

        std::unique_ptr<nn::ns::ApplicationRecord[]> recordBuffer(new nn::ns::ApplicationRecord[ApplicationRecodeListBufferSize]);
        auto recordList = recordBuffer.get();
        auto numApp = nn::ns::ListApplicationRecord(recordList, ApplicationRecodeListBufferSize, 0);

        for ( const auto& applicationId : ApplicationIdList )
        {
            nn::ns::ApplicationView view = { };
            bool isExistAppView = false;
            for( int id = 0; id < numApp; id++ )
            {
                if( recordList[id].id == applicationId )
                {
                    // タスクが存在する場合はタスクの情報を取得する
                    auto& record = recordList[id];
                    if( nn::ns::GetApplicationView( &view, &record.id, 1 ).IsSuccess() )
                    {
                        isExistAppView = true;
                    }
                    break;
                }
            }

            if( !isExistAppView )
            {
                // アプリケーションビューが存在しない場合は画面表示用に情報を設定する
                view.id = applicationId;
            }

            const auto innerRect = m_ScrollContainer->GetInnerRect();

            std::unique_ptr<ProgressViewUnit> viewUnit( new ProgressViewUnit(
                view
                , 0.0f
                , innerRect.width()
                , UnitHeight + UnitMargin
                , IdPos + ScrollPosAdjustment
                , StatePos + ScrollPosAdjustment
                , ProgressPos + ScrollPosAdjustment
                , SliderPos + ScrollPosAdjustment
                , SliderLength
                , DownloadButtonPos + ScrollPosAdjustment
                , DownloadButtonLength
                , [&]( const nn::ncm::ApplicationId& id ) { OnDownloadApplication( id ); }
            ));

            m_ListProgressView.push_back( std::move( viewUnit ) );
            *m_ScrollContainer << m_ListProgressView.back()->GetApplicationTable();
        }
#else
        NN_UNUSED( UnitHeight );
        NN_UNUSED( UnitMargin );
        NN_UNUSED( DownloadButtonLength );
        NN_UNUSED( ScrollPosAdjustment );
#endif
    }

    /**
    * @brief ECIサーバとチケットの同期を行う
    */
    void OnSyncTicket() NN_NOEXCEPT
    {
#if defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        // ネットワーク接続をシステムに要求する
        if( SubmitNetworkRequestAndWait().IsFailure() )
        {
            return;
        }

        // DevMenuSystemの "application sync-ticket" 機能を実行する
        bool outValue;
        RequestShopCommand( outValue, "sync-ticket", "Succeeded to sync ticket.", "Failed to sync ticket." );
#else
        DEVMENU_LOG( "Unsupported.\n" );
        auto pView = new MessageView( false );
        CreateErrorMessageView( pView, "Unsupported." );
        GetRootSurfaceContext()->StartModal( pView, true );
#endif
    }

    std::vector< std::unique_ptr<ProgressViewUnit> > m_ListProgressView;    //!< ダウンロード状況を描画するコンテナ
    nn::os::TimerEvent m_ProgressCheckEvent;                                //!< ダウンロード状況を確認する間隔
    std::unique_ptr<ScrollableBoxView> m_ScrollContainer;                   //!< アプリケーションリスト表示用のスクロールコンテナ
};

/**
* @brief ページ生成 ( 専用クリエイター )
*/
template< size_t ID >
class DownloadPageCreator : devmenu::PageCreatorBase
{
public:
    explicit DownloadPageCreator(const char* pPageName) NN_NOEXCEPT
        : devmenu::PageCreatorBase(ID, pPageName) {}

protected:
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[2];
        const glv::DisplayMetrics& d = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        d.GetResolution(resolution[0], resolution[1]);
        const glv::space_t width = static_cast< glv::space_t >(resolution[0]);
        const glv::space_t height = static_cast< glv::space_t >(resolution[1]);
        const glv::Rect pageBounds(width - 218.f, height - 118.0f);
        return new DownloadPage(ID, GLV_TEXT_API_WIDE_STRING("Download"), pageBounds);
    }
};

/**
* @brief Declearation for the statical instance of page creator.
*/
#define LOCAL_PAGE_CREATOR( _id, _name ) DownloadPageCreator< _id > g_DownloadPageCreator##_id( _name );
LOCAL_PAGE_CREATOR(DevMenuPageId_Download, "Download");

}}} // ~namespace devmenu::system::download, ~namespace devmenu::system, ~namespace devmenu
