﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include "DeviceSettings/DevMenu_DeviceSettingsItemListScene.h"
#include "DeviceSettings/DevMenu_DeviceSettingsSceneType.h"
#include "DeviceSettings/DevMenu_DeviceSettingsTimeZoneSelectorScene.h"

namespace devmenu { namespace devicesettings {

/**************************************
 * class DeviceSettingsPage
 **************************************/

class DeviceSettingsPage : public Page
{
public:
    DeviceSettingsPage( int pageId, const glv::WideCharacterType* pageCaption, const glv::Rect& rect ) NN_NOEXCEPT
        : Page( pageId, pageCaption, rect )
        , m_ItemListScene( this, rect )
        , m_TimeZoneSelectorScene( this, rect, [&]{ this->SwitchScene( SceneType_ItemList, true ); } )
    {
    }

    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        *this << m_ItemListScene << m_TimeZoneSelectorScene;
        m_ItemListScene.PrepareSubsections();
        this->SetActiveScene( SceneType_ItemList );
    }

    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ASSERT_NOT_NULL( this->GetActiveScene() );
        return this->GetActiveScene()->GetFirstFocusTargetView();
    }

    virtual void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE
    {
        m_ItemListScene.OnChangeIntoForeground();
    }

    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_ItemListScene.OnActivatePage();
    }

    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        m_ItemListScene.OnLoopBeforeSceneRenderer( context, events );
    }

    virtual Scene* GetScene( int sceneIndex ) NN_NOEXCEPT NN_OVERRIDE
    {
        switch ( sceneIndex )
        {
        case SceneType_ItemList:
            return &m_ItemListScene;
        case SceneType_TimeZoneSelection:
            return &m_TimeZoneSelectorScene;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

private:
    ItemListScene                    m_ItemListScene;
    tzselect::TimeZoneSelectorScene  m_TimeZoneSelectorScene;
};

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class DeviceSettingsPageCreator : PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit DeviceSettingsPageCreator( const char* pPageName ) NN_NOEXCEPT
        : PageCreatorBase( ID, pPageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const glv::space_t width = static_cast< glv::space_t >( resolution[ 0 ] );
        const glv::space_t height = static_cast< glv::space_t >( resolution[ 1 ] );
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new DeviceSettingsPage( ID, GLV_TEXT_API_WIDE_STRING( "Device" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) DeviceSettingsPageCreator< _id > g_DeviceSettingsPageCreator##_id( _name );
LOCAL_PAGE_CREATOR ( DevMenuPageId_Device, "Device" );

}} // ~namespace devmenu::devicesettings, ~namespace devmenu
