﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <functional>
#include <iomanip>
#include <sstream>
#include <string>
#include <map>

#include <glv_core.h>
#include <glv_textview.h>

#include <nn/util/util_FormatString.h>

#include "Common/DevMenu_CommonIconLabel.h"
#include "DevMenu_Config.h"
#include "DevMenu_ModalView.h"

namespace devmenu {

// 相互参照を防ぐためのクラス参照
class Page;

/**
 * @brief UTF-8 文字列から UTF-16 文字列を作成します。
 */
template< size_t TemporarySize, size_t OutBufferSize >
glv::WideCharacterType* BuildUtf16( glv::WideCharacterType( &pOutBuffer )[ OutBufferSize ], const char* pFormat, ... ) NN_NOEXCEPT
{
    va_list vaList;
    va_start( vaList, pFormat );

    char pTemp[ TemporarySize ];
    std::vsprintf( pTemp, pFormat, vaList );
    nn::util::ConvertStringUtf8ToUtf16Native( reinterpret_cast< uint16_t* >( pOutBuffer ), OutBufferSize, pTemp );

    va_end( vaList );
    return pOutBuffer;
}

/**
 * @brief バイト数を整数で表現可能な最大の単位(KB, MB etc.)の文字列に変換します。
 */
void ConvertBytesToLargestType( char* outConvertedString, size_t stringSize, double byteSize, bool isSuffixAdded = true ) NN_NOEXCEPT;

/**
 * @brief 再起動を要求するダイアログを作成します。
 */
MessageView* CreateRebootRequestDialog() NN_NOEXCEPT;

/**
 * @brief エラーメッセージと result 値を含むメッセージを作成します。
 */
void CreateErrorMessage( std::string* pOutStr, const std::string& errorStr, const nn::Result& result, bool addsNewLine = true ) NN_NOEXCEPT;

/**
 * @brief エラーメッセージを作成します。
 */
void CreateErrorMessage( std::string* pOutStr, const std::string& errorStr, bool addsNewLine = true ) NN_NOEXCEPT;

/**
 * @brief エラーメッセージと result 値を表示する MessageView を作成します。
 */
void CreateErrorMessageView( MessageView* pOutView, const std::string& errorStr, const nn::Result& result, const std::string& buttonStr = "Close" ) NN_NOEXCEPT;

/**
 * @brief エラーメッセージを表示する MessageView を作成します。
 */
void CreateErrorMessageView( devmenu::MessageView* pOutView, const std::string& errorStr, const std::string& buttonStr = "Close" ) NN_NOEXCEPT;

/**
 * @brief 数字を 3 桁区切りの文字列に変換します。
 */
std::string GetDelimitedNumberString( int64_t number ) NN_NOEXCEPT;

/*********************************
 * class MemoryAllocator
 *********************************/

class MemoryAllocator
{
public:
    static void* AllocAlignedMemory( size_t alignment, size_t size ) NN_NOEXCEPT;
    static void FreeAlignedMemory( void* address ) NN_NOEXCEPT;
};

/*********************************
 * class RebootAttentionLabel
 *********************************/

class RebootAttentionLabel : public glv::Table
{
public:
    explicit RebootAttentionLabel( const std::string& text = "Reboot device to reflect changes",
                                   const glv::Label::Spec& spec = glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize ) ) NN_NOEXCEPT
    : glv::Table( "xx", 10.0f )
    {
        m_pRebootIcon = new IconLabel( IconCodePoint::Warning, glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, 20.0f ) );
        m_pRebootText = new glv::Label( text, spec );
        *this << m_pRebootIcon << m_pRebootText;
        this->arrange();

        ApplyInitialStyle();
    }

    void SetStyle( const glv::Style& style ) NN_NOEXCEPT
    {
        m_Style = style;
        this->style( &m_Style );
        m_pRebootIcon->style( &m_Style );
        m_pRebootText->style( &m_Style );
    }

    void UpdateSpec( const glv::Label::Spec& spec ) NN_NOEXCEPT
    {
        m_pRebootIcon->stroke( spec.stroke * 0.75f );
        m_pRebootIcon->vertical( spec.vert );
        m_pRebootIcon->size( spec.size * 0.75f );
        m_pRebootText->stroke( spec.stroke );
        m_pRebootText->vertical( spec.vert );
        m_pRebootText->size( spec.size );
        this->arrange();
        pos( spec.posAnch, spec.dx, spec.dy ).anchor( spec.posAnch );
    }

private:
    void ApplyInitialStyle() NN_NOEXCEPT;
    IconLabel* m_pRebootIcon;
    glv::Label* m_pRebootText;
    glv::Style m_Style;
};

/*********************************
 * class Button
 *********************************/

typedef std::map< glv::Event::t, glv::EventHandler* > EventHandlerMap;

struct EventHandler : glv::EventHandler
{
    using EventCallback = std::function< bool( glv::View&, glv::GLV& ) >;

    explicit EventHandler( const EventCallback& callback ) NN_NOEXCEPT : eventCallback( callback ) {};

    virtual bool onEvent(glv::View& view, glv::GLV& glvRoot) NN_NOEXCEPT
    {
        return eventCallback( view, glvRoot );
    }

    EventCallback eventCallback;
};

class Button : public glv::Button
{
public:

    /**
     * @brief コンストラクタ.
     * @details 1バイト文字列( ASCII, Latin-1 )を対象にした初期化です.
     */
    explicit Button( const std::string& text, std::function< void() > callback = nullptr, float fontSize = CommonValue::InitialFontSize, glv::space_t paddingX = 8.0f, glv::space_t paddingY = 4.0f ) NN_NOEXCEPT;

    Button( const char* caption, std::function< void() > callback, const glv::Rect& rect, glv::Place::t anchor = glv::Place::TL ) NN_NOEXCEPT;

    Button( const char* caption, std::function< void() > callback, const glv::Rect& rect, const glv::Label::Spec& spec ) NN_NOEXCEPT;

    /**
     * @brief コンストラクタ.
     * @details 2バイト文字列( UTF-16 )を対象にした初期化です.
     */
    explicit Button( const glv::WideString& text, std::function< void() > callback = nullptr, float fontSize = CommonValue::InitialFontSize, glv::space_t paddingX = 8.0f, glv::space_t paddingY = 4.0f ) NN_NOEXCEPT;

    Button( const glv::WideString& caption, std::function< void() > callback, const glv::Rect& rect, glv::Place::t anchor = glv::Place::TL ) NN_NOEXCEPT;

    Button( const glv::WideString& caption, std::function< void() > callback, const glv::Rect& rect, const glv::Label::Spec& spec ) NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT;

    virtual void OnClicked() NN_NOEXCEPT;

    void UpdateFocusAndColor( bool isValid, bool isFocusable = false ) NN_NOEXCEPT;

    void UpdateLabelText( const char* text ) NN_NOEXCEPT;

    void UpdateLabelText( const glv::WideString& text ) NN_NOEXCEPT;

    void SetCallback( std::function< void() > callback ) NN_NOEXCEPT;

    void EnableLabelView() NN_NOEXCEPT { m_Label.enable( glv::Property::Visible ); };

    void DisableLabelView() NN_NOEXCEPT { m_Label.disable( glv::Property::Visible ); };

    void IncreaseLabelSize( glv::space_t increasedX, glv::space_t increasedY ) NN_NOEXCEPT;

    void ChangeLabelSize( glv::space_t width, glv::space_t height ) NN_NOEXCEPT;

    Button& AddEventHandler( glv::Event::t event, EventHandlerMap* pHandlerMap ) NN_NOEXCEPT;

protected:
    glv::Label& GetLabel() NN_NOEXCEPT;

private:

    enum ButtonStyle : nn::Bit8
    {
        ButtonStyle_Focusable,
        ButtonStyle_NotFocusable,
        ButtonStyle_FocusableButInvalid,
    };

private:
    /**
     *@brief 自身( Button )の表示領域を指定のラベル情報にフィットさせます.
     */
    void FitButton( float fontSize, glv::space_t paddingX, glv::space_t paddingY ) NN_NOEXCEPT;

private:
    glv::Label              m_Label;
    std::function< void() > m_Callback;
    glv::Style              m_StyleLabel[ 3 ];
    glv::Style              m_StyleButton[ 3 ];
};

/*********************************
 * class Table
 *********************************/

class Table : public glv::Table
{
public:
    Table( const char* arrangement = "<", glv::space_t paddingX = 3.0f, glv::space_t paddingY = 3.0f, const glv::Rect& rect = glv::Rect( 0.0f ) ) NN_NOEXCEPT
        : glv::Table( arrangement, paddingX, paddingY, rect ) {}

    void RemoveAllChildren() NN_NOEXCEPT;

    void RemoveAllChildren( std::vector< glv::View* >* pOutChildren ) NN_NOEXCEPT;

    void ReAddOwnselfToParent( glv::View* pParentView,  bool isChildrenMoved = false ) NN_NOEXCEPT;
};

/*********************************
 * class Spacer
 *********************************/

class Spacer : public glv::View
{
public:
    Spacer( glv::space_t width, glv::space_t height ) NN_NOEXCEPT
        : glv::View( glv::Rect( width, height ) )
    {
        disable( glv::Property::Controllable | glv::Property::DrawBorder | glv::Property::DrawBack | glv::Property::HitTest );
        enable( glv::Property::AlwaysBubble );
    }
};

} // ~namespace devmenu
